#ifndef SNAZZYLIST_H
#define SNAZZYLIST_H

/* Instructions:
        This is a container module for implementing sets & sequences.
1. This module is extremely efficient: there's only a 4 byte overhead
   (plus the malloc() overhead) for each array.
2. NULL means 'empty set' and NULL is also how you initialise sets - there's
   no distinction between undefined and empty set.
3. If you create an empty set by deleting elements after inserting them, then
   you end up back with a NULL pointer (this is guaranteed).
4. Some of the functions give added value to sets & sequences of _pointers_:
   ListInsP(), ListDelP() and ListFindP() all work on sets of pointers.
*/

#ifndef HEADER_H
#include "header.h"
#endif

void msort(void *base, size_t nmemb, size_t size, int (*compar) (const void *, const void *));

#define ListSize(A)				((A) ? (((int*)A)[-1]) : 0)
#define ListNext(A)				aListNextClear((void**)&(A), sizeof(*A))
#define ListAdd(A,x)            (aListNext((void**)&(A), sizeof(*A)), (A)[ListSize(A)-1]=(x))
#define ListIdx(A,i)            (aListIdx((void**)&(A), (i), sizeof(*A)), (A)[i])
#define ListCopy(A)				aListCopy((void*)(A), sizeof(*A))
#define ListFree(A)				(aListFree(A), A=NULL)
#define ListClear(A)            (aListFree(A), A=NULL)
#define ListInsP(A,p)           aListInsP((void**)&A, p, sizeof(*A))
#define ListInsN(A,i,v)			(aListInsN((void**)&A, i, sizeof(*A)), A[i] = v)
#define ListDelN(A,i)           aListDelN(&A, i, sizeof(*A))
#define ListDelP(A,p)           aListDelP(&A, p, sizeof(*A))
#define ListConcat(A,B)			aListConcat((void*)(&A), (void*)B, sizeof(*B))
#define ListMerge(A,B)			aListMerge((void**&)A, (void**)B)
#define ListHasP(A,p)           aListHasP(A,p)
#define ListHasN(A,n)           aListHasN(A,n)
#define ListFindP(A,p)			aListFindP(A,p)
#define ListFindN(A,n)			aListFindN(A,n)
#define ListSort(A,cmp)			msort(A,ListSize(A),sizeof(A[0]),(cmp_fn)(cmp))
#define ListSetSize(A,s)		aListSetSize((void**)&A, s, sizeof(*A))
#define ListFreeFree(A,are_dynarrays)   (aListFreeFree((void**)A,are_dynarrays), A=NULL)

#define each_aeli(x,A)          i=0; ((A) and i < ((int*)(A))[-1]) ? (x=(A)[i],yes) : no; i++
#define each_ael(x,A,i)         i=0; ((A) and i < ((int*)(A))[-1]) ? (x=(A)[i],yes) : no; i++
#define each_oeli(x,A)			i=0; ((A) and i < ((int*)(A))[-1]) ? (x=&(A)[i],yes) : no; i++
#define each_oel(x,A,i)         i=0; ((A) and i < ((int*)(A))[-1]) ? (x=&(A)[i],yes) : no; i++
#define each_aeli_backwards(x,A)            \
                            i=(A) ? ((int*)(A))[-1] - 1: -1; i >= 0 ? (x=(A)[i],yes) : no; i--
#define each_ael_backwards(x,A,i)            \
                            i=(A) ? ((int*)(A))[-1] - 1: -1; i >= 0 ? (x=(A)[i],yes) : no; i--
#define each_oeli_backwards(x,A)    \
                            i=(A) ? ((int*)(A))[-1] - 1: -1; i >= 0 ? (x=&(A)[i],yes) : no; i--


void* aListNext(void** A, int elsize);
void* aListNextClear(void** A, int elsize);
void* aListIdx(void** A, int idx, int elsize);
void* aListCopy(void* A, int elsize);
void  aListInsP(void** A, const void* p, int elsize);
void* aListInsN(void** A, int idx, int elsize);
void  aListDelN(void* A, int idx, int elsize);
void  aListDelP(void* A, const void *p, int elsize);
void  aListFree(void* A);
void  aListFreeFree(void** A, bool are_arrays);
void  aListConcat(void* Ap, void *B, int elsize);
void aListMerge(void **&A, void **B);
bool aListHasP(void* A, const void* p);
bool aListHasN(void* A, int n);
int aListFindP(void* A, void* p);
int aListFindN(void* A, int n);
void  aListSetSize(void** Ap, int size, int elsize);


void MallocCheck();
	/* Checks the heap for consistency.  If there's an error, it calls */
	/* 'assert_failed()'. */

void MallocLeakReport(char *filename);
	/* Print a report on memory usage.  filename==NULL means stdout. */

int MallocMemUsed();
	/* How much memory is used? */

int MallocMemTotal();
	/* How much memory is allocated by the operating system? */
	/* = MallocMemUsed() + free tiles */

void MallocSetHeap(int n);
        /* Instead of using the main heap, use an alternative one for all */
        /* subsequent allocs/deallocs. Later you can return to the main   */
        /* heap using MallocSetHeap(0).  Important: don't try to free()   */
        /* something allocated in another heap! */

void MallocOptimise();
		/* Optimise the heap. Call this after freeing a bunch of stuff. */

void MallocFreeHeap();
        /* Free everything in the current heap. */

void snazzyFree(void *ptr);
char* snazzyStrdup(const char* s);
void* snazzyMalloc(size_t n);
		/* Faster versions of them. */


namespace nstd {
	template<typename _Ty> class vector
	{
		_Ty *A;

	public:
		vector() { A = NULL; }
		void push_back(_Ty object) {	aListNext((void**)&A, sizeof(*A));
										A[ListSize(A)-1] = object; }
		void resize(int n) { aListSetSize((void**)&A, n, sizeof(*A)); }
		_Ty& operator[](int idx) { return A[idx]; }
		int size() { return ListSize(A); }
		_Ty& begin() { return A[0]; }
		_Ty& back() { return A[ListSize(A)-1]; }
		_Ty& end() { return A[ListSize(A)]; }
		void clear() { ListFree(A); }
		bool empty() { return A == NULL; }
		void pop_back() { ListSetSize(A, ListSize(A)-1); }
	};

};

#endif