#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <direct.h>
#include "tfc.h"
#include "rowcolumn.h"
#include "list.h"
#include "print.h"
#include "editor.h"


/* Explanation:

Deep-copy of ScrollWin's:
-------------------------
  With great reluctance, I've set up a system whereby whenever you
want to print a ScrollWin, you have to do a deep-copy of your derived
object, i.e. use a copy-constructor, a C(C &orig) constructor which calls
ScrollWin(ScrollWin &orig).   Users like to be able to see the window
they're printing as a normal on-screen window, as well as on the print
preview screen.  This means there can be 2 copies of the same ScrollWin
object visible at the same time, with different attributes in the size,
scrollX/Y, font etc. fields.  So there's no alternative really (except
to remove the object from its on-screen window, resize it to the huge
printer resolution, display it in print preview, and then resize it
back after it has been both viewed and printed).  So every application-
writer has to write these copy constructors.


Mapping modes:
--------------
        We want the print preview to provide a simulation of what will come
out on the printer as accurately as possible.  Therefore, we resize the
object to the huge resolutions we see on printers (e.g. 4000 pixels by
6000 pixels) and use the Win32 API feature of 'mapping modes' to bring
this back to something we can reasonably view in Print Preview.  Mapping
modes allow us to scale all the coordinates by an arithmetic transformation
which the Win32 API calls do internally.
  So what we do is implement the PrintPreviewCell::Paint() function
to (a) paint the page borders and margins and grey background using the
plain mapping (1 pixel=1 pixel), and then (b) go into the scaled mapping
mode to paint the application ScrollWin and then (c) return to the plain
mapping mode.


Large ScrollWins:
-----------------
  When one ScrollWin spans more than one A4 sheet, (or whatever size
you're using), we actually produce multiple PrintPreviewCell's - one for
each sheet that will be needed, but they will each be pointing to the same
'child', (child=the deep copied application object).  That child's scrollY
(and scrollX if needed) fields are simply updated each time we paint a
sheet.



Fonts in the scaled mapping mode:
---------------------------------
  Kieran noticed a problem with fonts in the PrintPreview screen.
The TrueType (TT) fonts, which are initially represented using shape
coordinates, come out right but the raster fonts come out wrong.
  The EditorWin::Paragrapher will calculate word widths correctly
using printer coordinates, but when the words are painted, they are
painted using the closest matching font which might cause the words to
be clipped.

*/




class PrintPreviewGrid : public GridWin {
public:
	TfcPrintJob *printJob;
    float zoomX, zoomY;                 // The width and height in screen pixels of the print preview
                                        // cells, including everything:  desk, natural margins, user
                                        // margins and data.        
    int exitStatus;                     // -1=cancel, 1=go ahead, 2=redo pages.
    short dmDuplex;                     // Is it double-sided printing?  If so, flip on which side?

    class PrintPreviewCell **OrigPPList, **PPList; // The list of pages we want to print.

    PrintPreviewGrid(TfcPrintJob *job, HANDLE DevMode, class PrintPreviewCell **PPList);
    ~PrintPreviewGrid();
    void Arrange();                                     // Redisplay all the printpreview cells.
    virtual void Paint(int x1, int y1, int x2, int y2); // ...because the rebar gets overwritten

    // User operations
    bool Keystroke(int key);
    void FromHere();
    void UptoHere();
    void NotThisOne();
    void OnlyThisOne();
    void Reset();
    void Zoom(int m, int d);
    void ZoomIn();
    void ZoomOut();
    void HeadersFooters();
    virtual void Resized();
    void Print();
    void Cancel();
};


class PrintPreviewCell : public GridCell {
	// One PrintPreviewCell represents one A4 piece of paper.

	void ExpandHeaderFooter(str header, char dest[]);    

public:
    ScrollWin *child;
    int page_num;

    int InPrinterSpace;							// is the DC in screen coords or printer coords? 
												// 0=screen, 1=header/footer, 2=scrollwin content.

	// Fields in screen coords (i.e. pixel offset from PrintPreviewCell):
    TfcRect sheet_s;							// The sheet in screen (PrintPreviewCell) coords
	TfcRect printable_s;						// The printable area from the PrintPreviewCell
	int naturalMarginX_s, naturalMarginY_s;		// The margin between the edge of the sheet and the printable area

	// Fields in printer coords:
	TfcRect viewport_p;
	TfcRect content_p;							// The region used by the ScrollWin as opposed to ???
	TfcRect previewFrame_p;						// Stops printer-space commands painting the toolbar.

	enum al_enum { al_centre, al_top, al_bottom, al_left, al_right } alignment; // Alignment has to do with duplex printing

    PrintPreviewCell(ScrollWin* sw, int left, int top, int right, int bottom, int n, int zoomX, int zoomY); // Constructor
    PrintPreviewCell(PrintPreviewCell &orig); // Deep-copy constructor
    void SetAlignment(short dmDuplex, int page_num); // Set the alignment according to duplex printing
    void SetDC(void* DC) { child->hDC = DC;}
    void Paint(int x1, int y1, int x2, int y2);
    void CursorPaint();
    void Measure(int *widthp, int *heightp);
    bool Keystroke(int key);
    bool Mousestroke(int op, int x, int y) {
        ((GridWin*)parent)->MoveTo(this);
        return no;
    }
    Litewin* DuplicateMe();
	void GetPaintDetails(TfcPaintDetails *paintDetails, int x, int y);
    void Print(HDC DC);		// <-- This is all done in printer coordinates.
};


PrintPreviewCell::PrintPreviewCell(ScrollWin* sw, int left, int top, int right, int bottom, 
										int page_num, int zoomX, int zoomY)
{   
    child = sw;
    child->ShowState = tfc_shown;
    child->parent = this;
	this->viewport_p.left = left;
	this->viewport_p.top = top;
	this->viewport_p.right = right;
	this->viewport_p.bottom = bottom;
    this->page_num = page_num;
    req_width = zoomX;
    req_height = zoomY;
    InPrinterSpace = 0;
}


PrintPreviewCell::PrintPreviewCell(PrintPreviewCell &orig)
{
    child = (ScrollWin*)child->DuplicateMe();
    child->ShowState = tfc_shown;
    child->parent = this;
    viewport_p = orig.viewport_p;
    page_num = orig.page_num;
    req_width = orig.req_width;
    req_height = orig.req_height;
    InPrinterSpace = orig.InPrinterSpace;        
}


void PrintPreviewCell::GetPaintDetails(TfcPaintDetails *paintDetails, int x, int y)
{
	PrintPreviewGrid *PPgrid = (PrintPreviewGrid*)parent;
	TfcPrintJob *job = PPgrid->printJob;
    if (InPrinterSpace) {
		// Page contents are printed in the printer coordinate system:
		if (InPrinterSpace == 2) {
			// Doing the scrollwin contents:
			paintDetails->winX = content_p.left;
			if (x >= child->dontScrollX)
				paintDetails->winX -= viewport_p.left;
			paintDetails->winY = content_p.top;
			if (y >= child->dontScrollY)
				paintDetails->winY -= viewport_p.top;
			paintDetails->clipRegion.left = x < child->dontScrollX ? 
						content_p.left : content_p.left + child->dontScrollX;
			paintDetails->clipRegion.top = y < child->dontScrollY ? 
						content_p.top : content_p.top + child->dontScrollY;
			paintDetails->clipRegion.right = content_p.left + viewport_p.right - viewport_p.left;
			paintDetails->clipRegion.bottom = content_p.top + viewport_p.bottom - viewport_p.top;
		}
		else {
			// Doing the header/footer:
			paintDetails->winX = 0;
			paintDetails->winY = 0;
			paintDetails->clipRegion.left = 0;
			paintDetails->clipRegion.top = 0;
			paintDetails->clipRegion.right = job->printerWidth_p;
			paintDetails->clipRegion.bottom = job->printerHeight_p;
		}
		paintDetails->clipRegion.intersect(previewFrame_p);
		paintDetails->hDC = child->hDC;
		paintDetails->colourMode = job->colourmode;
		paintDetails->lineWidth = child->clientWidth / 1000;
		if (paintDetails->lineWidth < 1)
			paintDetails->lineWidth = 1;
	}
	else {
		// The page border, white margins and desk background are printed in screen coordinates.
		GridCell::GetPaintDetails(paintDetails, x, y);
	}
}


Litewin* PrintPreviewCell::DuplicateMe()
{
    return new PrintPreviewCell(*this);
}


void PrintPreviewCell::CursorPaint()
{
    int w = Width(), h = Height();
    DrawRectangle(0, 0, w, h, NOCOLOUR, RED);
    DrawRectangle(1, 1, w - 1, h - 1, NOCOLOUR, RED);
}


void TfcPrintJob::ExpandHeaderFooter(kstr s, PrintPreviewCell *PPcell, char dest[])
{
    str b;

    b = dest;
	*b = '\0';
    while(*s) {
        if (*s == '%') {
			if (strbegins(s, "%pagetotal")) { // Note: need to check this before we check for 'page'
                b += sprintf(b, "%d", ListSize(PPgrid->PPList));
                s += 10;
            }
            else if (strbegins(s, "%page")) {
                b += sprintf(b, "%d", PPcell->page_num + 1);
                s += 5;
            }
            else if (strbegins(s, "%title")) {
				ScrollWin *win = PPcell->child;
				if (win->pageTitle or win->windowTitle) {
					b += sprintf(b, "%s", win->pageTitle and *win->pageTitle ? win->pageTitle : win->windowTitle);
                    s += 6;
                }
                else
                    s += 6;
            }
			else if (*s == '%' and ExpandHeaderFooterToken(s, PPcell->child, b)) {
				b += strlen(b);
			}
			else 
                *b++ = *s++;
        }
        else
            *b++ = *s++;
    }
    *b = '\0';
}


void PrintPreviewCell::Print(HDC DC)
{   PrintPreviewGrid *grid = (PrintPreviewGrid*)parent;
	TfcPrintJob *job = grid->printJob;
    char buf2[2048];
	int w,h;

    /*** What area will we use for the ScrollWin content? ***/
	content_p.Init(job->pixMargins.left, 
					job->pixMargins.top + job->headerHeight,
					job->printerWidth_p - job->pixMargins.right, 
					job->printerHeight_p - job->footerHeight - job->pixMargins.bottom);

    // Paint Header
	child->hDC = DC;		// This is needed for the drawing commands below.
    if (*job->header) {
        job->ExpandHeaderFooter(job->header, this, buf2);
		TextDimensions(buf2,-1,job->headerFont, &w,&h, TFC_WORDBREAK);
		DrawString(buf2, -1, job->headerFont, 
							0,(content_p.top-h)/2, job->printerWidth_p, content_p.top, 
							BLACK, NOCOLOUR, TFC_WORDBREAK | TFC_CENTRE);
    }

    // Paint footer
    if (*job->footer and not job->continuousRollPaper) {
        job->ExpandHeaderFooter(job->footer, this, buf2);
		TextDimensions(buf2,-1,job->footerFont, &w,&h, TFC_WORDBREAK);
		DrawString(buf2, -1, job->footerFont, 
							0,(content_p.bottom+job->printerHeight_p-h)/2, 
								job->printerWidth_p, job->printerHeight_p, 
							BLACK, NOCOLOUR, TFC_WORDBREAK | TFC_CENTRE);
    }

    /* Paint the child: */
	child->hDC = DC;
    child->ShowState = tfc_shown;
    child->parent = this;
	child->scrollX = viewport_p.left;
	child->scrollY = viewport_p.top;
    int sx = viewport_p.left > 0 ? viewport_p.left : 0;
    int sy = viewport_p.top > 0 ? viewport_p.top : 0;
    child->inPrint = yes;
	InPrinterSpace = 2;
	child->PaintWithUnscrollables(sx, sy, sx+child->clientWidth, sy+child->clientHeight);
	InPrinterSpace = 1;
    child->inPrint = no;
}


void PrintPreviewCell::Paint(int x1, int y1, int x2, int y2)
{
    PrintPreviewGrid *grid = (PrintPreviewGrid*)parent;
	TfcPrintJob *job = grid->printJob;
    int W=Width(), H=Height();

    grid->parent = grid; /* Set parent of PrintPreviewGrid to NULL to avoid affecting 
                        of parent's GetWinXY and GetWinRect to present drawing*/

    /*** Where shall we display the sheet?: ***/
    // backgroundMarginX/Y_s is the 'background margin': the width of the grey
    // area around the page.  
    int backgroundMarginX_s = W / 20;
    int backgroundMarginY_s = H / 20;
    switch (alignment) {
        case al_top:        sheet_s.Init(backgroundMarginX_s, 0,
										W - backgroundMarginX_s, H - 2*backgroundMarginY_s);
                            break;
        case al_bottom:     sheet_s.Init(backgroundMarginX_s, 2*backgroundMarginY_s,
										W - backgroundMarginX_s, H);
                            break;
        case al_left:       sheet_s.Init(0, backgroundMarginY_s,
										W - 2*backgroundMarginX_s, H - backgroundMarginY_s);
                            break;
        case al_right:      sheet_s.Init(2*backgroundMarginX_s, backgroundMarginY_s,
										W, H - backgroundMarginY_s);
                            break;
		default:
		case al_centre:     sheet_s.Init(backgroundMarginX_s, backgroundMarginY_s,
										W - backgroundMarginX_s, H - backgroundMarginY_s);
                            break;
    }


	/*** What is the printable area? ***/
    // The 'natural margin' is the width from the printable region of the A4 sheet
    // to the edge of the paper.  This is purely a PrintPreview concept, hence they're
	// in screen coords, not printer coords.        
    naturalMarginX_s = W/40;
    naturalMarginY_s = H/40;
	printable_s.Init(sheet_s.left + naturalMarginX_s, sheet_s.top + naturalMarginY_s,
					sheet_s.right - naturalMarginX_s, sheet_s.bottom - naturalMarginY_s);

    /*** Draw the desk: ***/
	DrawRectangle(0,0,sheet_s.left,H, DARK_GREY,NOCOLOUR);
	DrawRectangle(sheet_s.right,0,W,H, DARK_GREY,NOCOLOUR);
	DrawRectangle(sheet_s.left,0,sheet_s.right,sheet_s.top, DARK_GREY,NOCOLOUR);
	DrawRectangle(sheet_s.left,sheet_s.bottom,sheet_s.right,H, DARK_GREY,NOCOLOUR);

    /*** Draw the sheet: ***/
    DrawRectangle(sheet_s, WHITE,BLACK);


	/*** Draw the header,footer and contents: ***/
	// Enter the printer's coordinate system:
	TfcPaintDetails paintDetails;
    POINT ViewO, WindO;
    SIZE ViewE, WindE;

	GridCell::GetPaintDetails(&paintDetails, 0,0);
	child->hDC = paintDetails.hDC;
	HDC DC = (HDC)child->hDC;
	int prevMode = SetMapMode(DC, MM_ANISOTROPIC);
	SetWindowExtEx(DC, job->printerWidth_p, job->printerHeight_p, &WindE);
    SetWindowOrgEx(DC, 0, 0, &WindO);
    SetViewportExtEx(DC, printable_s.right - printable_s.left, printable_s.bottom - printable_s.top, &ViewE);
	SetViewportOrgEx(DC, printable_s.left + paintDetails.winX, printable_s.top + paintDetails.winY, &ViewO);
    InPrinterSpace = 1;

	// Work out the clipping rectangle for the PrintPreviewGrid in printer coordinates:
	POINT Pts[2] = { 
						{ paintDetails.clipRegion.left, paintDetails.clipRegion.top },
						{ paintDetails.clipRegion.right, paintDetails.clipRegion.bottom }
				};
	DPtoLP(DC, Pts, 2);
	previewFrame_p.Init(Pts[0].x, Pts[0].y, Pts[1].x, Pts[1].y);

	// Print in printer coords:
    Print(DC);

	// Restore the mapping mode:
    SetViewportExtEx(DC, ViewE.cx, ViewE.cy, NULL);
    SetWindowExtEx(DC, WindE.cx, WindE.cy, NULL);
    SetViewportOrgEx(DC, ViewO.x, ViewO.y, NULL);
    SetWindowOrgEx(DC, WindO.x, WindO.y, NULL);
    SetMapMode(DC, prevMode);
    InPrinterSpace = 0;
}


bool PrintPreviewCell::Keystroke(int key)
{
    if (key == DEL) {
        PrintPreviewGrid *Grid = (PrintPreviewGrid*)parent;
        Grid->NotThisOne();
        return yes;
    }
    else
        return no;
}


void PrintPreviewCell::Measure(int *widthp, int *heightp)
{
    PrintPreviewGrid *grid = (PrintPreviewGrid*)parent;
    *widthp = grid->zoomX;
    *heightp = grid->zoomY;
}


PrintPreviewGrid::PrintPreviewGrid(TfcPrintJob *_job, HANDLE hDevMode, PrintPreviewCell **_PPList)
    : GridWin("Print Preview", 800, 600, DARK_GREY, tfc_colshugcells, _job->owner)
{
    DEVMODE *DevMode=(DEVMODE*)GlobalLock(hDevMode);
	printJob = _job;
	printJob->PPgrid = this;
    exitStatus = 0;
    dmDuplex = DevMode->dmDuplex;
    if (DevMode->dmOrientation == 2) {
        /* If it's landscape view, swap over the duplex mode: */
        if (dmDuplex == 2)
            dmDuplex = 3;
        else if (dmDuplex == 3)
            dmDuplex = 2;
    }
    PPList = _PPList;
    OrigPPList = (PrintPreviewCell**)ListCopy(PPList);
    GlobalUnlock(hDevMode);
}


PrintPreviewGrid::~PrintPreviewGrid()
{
    ListFree(PPList);
    ListFree(OrigPPList);
}


void PrintPreviewGrid::Paint(int x1, int y1, int x2, int y2)
{    
    GridWin::Paint(x1, y1, x2, y2);
}


void PrintPreviewGrid::FromHere()
{
    PrintPreviewCell *GridCell = (PrintPreviewCell*)GetCurrentCell();
    if (GridCell) {
        int n = GridCell->page_num - 1;
        while(n-- >= 0)
            ListDelN(PPList, 0);
        Arrange();
        MoveTo(GridCell);
    }
}


void PrintPreviewGrid::UptoHere()
{
    PrintPreviewCell *GridCell = (PrintPreviewCell*)GetCurrentCell();
    if (GridCell) {
        int n = GridCell->page_num+1;
        while(ListSize(PPList) > n)
            ListDelN(PPList, n);
        Arrange();
    }
}


void PrintPreviewGrid::NotThisOne()
{
    PrintPreviewCell *GridCell = (PrintPreviewCell*)GetCurrentCell();
    if (GridCell) {
        ListDelP(PPList, GridCell);
        Arrange();
    }
}


void PrintPreviewGrid::OnlyThisOne()
{
    PrintPreviewCell *GridCell = (PrintPreviewCell*)GetCurrentCell();
    if (GridCell) {
        ListFree(PPList);
        ListAdd(PPList, GridCell);
        Arrange();
    }
}


void PrintPreviewGrid::Reset()
{
    ListFree(PPList);
    PPList = (PrintPreviewCell**)ListCopy(OrigPPList);
    Arrange();
}


void PrintPreviewGrid::Zoom(int m, int d)
{
    float x,y;

    SuppressPaints();        
    x = zoomX;
    y = zoomY;
    zoomX *= m;
    zoomX /= d;
    zoomY *= m;
    zoomY /= d;
    // (ksh) don't dissappear completely!
    if (zoomX < 1 or zoomY < 1) 
        zoomX = x, zoomY = y;
    if ((m > d) && (x == zoomX || y == zoomY))
    {
        zoomX++;
        zoomY++;
    }
    for (x = 0; x < numcols; x++)
    {
        for (y = 0; y < numrows; y++)
        {
            GridCell* cell=Get(x, y);
            if (cell)
                cell->Resize(zoomX, zoomY);
        }
    }
    MakeMeasurements();
}


void PrintPreviewGrid::ZoomIn()
{
    float CurrentX, CurrentY;

    try {
        CurrentX = (float)scrollX / realWidth;
        CurrentY = (float)scrollY / realHeight;
    }
    catch(...) {
        CurrentX = CurrentY = 0;
    }
    Zoom(3,2);
    MoveTo(GetCurrentCell());
    UpdateScrollbars(CurrentX * realWidth, CurrentY * realHeight);
}


void PrintPreviewGrid::ZoomOut()
{
    float CurrentX, CurrentY;

    // Preserve the focus position
    try {
        CurrentX = (float)scrollX / realWidth;
        CurrentY = (float)scrollY / realHeight;
    }
    catch(...)
    {
        CurrentX = CurrentY = 0;
    }
    Zoom(2, 3);
    MoveTo(GetCurrentCell());
    scrollX = CurrentX * realWidth;
    scrollY = CurrentY * realHeight;
}


void TfcPrintJob::HelpOnHeadersAndFooters()
{
	char buf[65536], *s=buf;

	s += sprintf(s, "\1Header + Footer text variables\1\n\n"
                "Use tokens to show 'variable' information about the current print job. "
                "Tokens may be placed anywhere you choose in the header and footer strings. "
				"You can list multiple tokens in a line, or mix with normal text. E.g 'Term 1', "
				"followed by Student name and year.\n\n"
                "\1Available tokens are:\1\n");
	s += sprintf(s, "\3%%page\3 : Current page number.\n");
	s += sprintf(s, "\3%%pagetotal\3 : Total numbers of pages to be printed.\n");
	s += sprintf(s, "\3%%title\3 : Page title\n");
	AddHelpOnHeaderAndFooterTokens(s);
	s += strlen(s);
	s += sprintf(s, "\n\n");
    ViewHelp(buf);
}


void PrintPreviewGrid::HeadersFooters()
{
	TfcPrintJob::Margin_node margins = printJob->mmMargins;
	char header[512], footer[512];
	strcpy(header, printJob->header);
	strcpy(footer, printJob->footer);

	int result = DoChildDialog("Print parameters", 
				printJob->ParametersGui()
						-
				FillerControl(0,30)
						-
				OkButton() / CancelButton());
	if (result != 1) {
		printJob->mmMargins = margins;
		strcpy(printJob->header, header);
		strcpy(printJob->footer, footer);
	}
	else {
		printJob->SaveToRegistry();
		QuitEventLoop = yes;
		exitStatus = 2;
	}
}


void PrintPreviewGrid::Resized()
{
    float CurrentX, CurrentY;

    // Preserve the focus position
    CurrentX = realWidth ? (float)scrollX / realWidth : 0;
    CurrentY = realHeight ? (float)scrollY / realHeight : 0;
    GridWin::Resized();
    scrollX = CurrentX * realWidth;
    scrollY = CurrentY * realHeight;
}


void PrintPreviewGrid::Print()
{
    QuitEventLoop = yes;
    exitStatus = 1;
}


void PrintPreviewGrid::Cancel()
{
    QuitEventLoop = yes;
    exitStatus = -1;
}


bool PrintPreviewGrid::Keystroke(int key)
{
    if (key == ZOOM_IN)
        ZoomIn();
    else if (key == ZOOM_OUT)
        ZoomOut();
    else if (key == WINDOW_QUIT or key == ESC)
        Cancel();
    else if (key == ENTER)
        Print();
    else
        return GridWin::Keystroke(key);
    return yes;
}


void PrintPreviewCell::SetAlignment(short dmDuplex, int page_num)
/* Set the alignment according to duplex printing. */
{
    enum al_enum al;

    if (dmDuplex == 1)
        al = al_centre;
    else if (dmDuplex == 2)
        al = (page_num&1) ? al_left : al_right;
	else if (dmDuplex == 3) {
		/*if (LorP == 'P')
			al = (page_num & 1) ? al_top : al_bottom;
		else
			- in theory, we never get portrait with duplex=3.
		*/
			al = (page_num & 1) ? al_left : al_right;
	}
    else
        al = al_centre;
    if (alignment == al)
        return;
    alignment = al;
}

void PrintPreviewGrid::Arrange()
/* Redisplay all the cells after a page has been added or deleted. */
{
    PrintPreviewCell *PPCell = NULL, *Focus;
    int i,focus_x,focus_y;

    Focus = (PrintPreviewCell*)GetCurrentCell();
    if (Focus)
        focus_x = Focus->i, focus_y = Focus->j;
    else
        focus_x = focus_y = 0;
    SuppressPaints();
    TakeOwnershipAndClear();        // We don't want to call the destructor on any cell.
    if (ListSize(PPList) == 0)
    {
        Cancel();
        return;
    }
    int cells_across = (int)(sqrt((double)ListSize(PPList)) * 1.6);
    if (cells_across < 1)
        cells_across = 1;
    if (dmDuplex <= 1) // No duplex
    {
        for (each_aeli(PPCell, PPList))
        {
            PPCell->page_num = i;
            PPCell->SetAlignment(1,0);
            Set(i%cells_across, i/cells_across, PPCell);
        }
    }
    else if (dmDuplex == 2) // Flip on long side
    {
        if (cells_across & 1)
            cells_across++;
        for (each_aeli(PPCell, PPList))
        {
            PPCell->page_num = i;
            PPCell->SetAlignment(2,i);
            Set(i%cells_across, i/cells_across, PPCell);
        }
    }
    else if (dmDuplex == 3)
    {   // Flip on short side

		// Kennant> dmDuplex == 3 only when we're doing landscape mode, TFC
		// ensures that.

		if (cells_across & 1)
			cells_across++;		// If we're doing landscape and flipping
			// on the short side, then we must have an even number across,
			// because we are doing horizontally-joined pairs.

		// Flip on short side
		for ( each_aeli(PPCell, PPList) ) {
			int j = i & 1;
			int k = i >> 1;
			PPCell->page_num = i;
			PPCell->SetAlignment(3, i);
			int x = k % (cells_across/2) * 2 + j;
			int y = k / (cells_across/2);
			assert(Get(x,y) == NULL);
			Set(x, y, PPCell);
		}
    }
    if (Get(focus_x, focus_y) == NULL)
        MoveTo(PPList[ListSize(PPList)-1]);
    else
        MoveTo(focus_x, focus_y);
    UpdateScrollbars(scrollX, scrollY);
}




/*------------------------ Customised PrintDialogBoxes: ---------------------------*/

#define grp1        0x0430
#define grp2        0x0431
#define grp4        0x0433

#define rad1        0x0420
#define rad2        0x0421
#define rad3        0x0422

#define edt1        0x0480
#define edt2        0x0481
#define edt3        0x0482

#define stc2        0x0441
#define stc3        0x0442
#define stc5        0x0444
#define stc6        0x0445
#define stc7        0x0446
#define stc8        0x0447
#define stc9        0x0448
#define stc10       0x0449
#define stc11       0x044a
#define stc12       0x044b
#define stc13       0x044c
#define stc14       0x044d

#define cmb4        0x0473

#define psh2        0x0401

#define chx2        0x0411

/*
    "GROUPBOX        "Printer",grp4,8,4,272,84,WS_GROUP
    LTEXT           "&Name:",stc6,16,20,36,8
    COMBOBOX        cmb4,52,18,152,152,CBS_DROPDOWNLIST | CBS_SORT |
                    WS_VSCROLL | WS_GROUP | WS_TABSTOP
    PUSHBUTTON      "&Properties",psh2,212,17,60,14,WS_GROUP
    LTEXT           "Status:",stc8,16,36,36,10,SS_NOPREFIX
    LTEXT           "",stc12,52,36,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP
    LTEXT           "Type:",stc7,16,48,36,10,SS_NOPREFIX
    LTEXT           "",stc11,52,48,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP
    LTEXT           "Where:",stc10,16,60,36,10,SS_NOPREFIX
    LTEXT           "",stc14,52,60,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP
    LTEXT           "Comment:",stc9,16,72,36,10,SS_NOPREFIX
    LTEXT           "",stc13,52,72,152,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP
    GROUPBOX        "Paper",grp2,8,92,164,56,WS_GROUP
    LTEXT           "Size:",stc2,16,108,36,8
    COMBOBOX        cmb2,52,106,112,112,CBS_DROPDOWNLIST | CBS_SORT |
                    WS_VSCROLL | WS_GROUP | WS_TABSTOP
    LTEXT           "Source:",stc3,16,128,36,8
    COMBOBOX        cmb3,52,126,112,112,CBS_DROPDOWNLIST | CBS_SORT |
                    WS_VSCROLL | WS_GROUP | WS_TABSTOP
    GROUPBOX        "Orientation",grp1,180,92,100,56,WS_GROUP
    ICON            "",ico1,195,112,18,20,WS_GROUP
    CONTROL         "P&ortrait",rad1,"Button",BS_AUTORADIOBUTTON | WS_GROUP |
                    WS_TABSTOP,224,106,52,12
    CONTROL         "L&andscape",rad2,"Button",BS_AUTORADIOBUTTON,224,126,52,
                    12
*/

struct PRNITEMTEMPLATE
{
    DLGITEMTEMPLATE templ;
    WORD   cl[2];    
    str    name;
};


static PRNITEMTEMPLATE Item(str name, int id,int x,int y, int cx, int cy, DWORD style, WORD cl)
{
    PRNITEMTEMPLATE it;
        
    it.templ.style = WS_CHILD | WS_VISIBLE | style;
    it.templ.dwExtendedStyle = 0;
    it.templ.x = x;it.templ.y = y;
    it.templ.cx = cx;it.templ.cy = cy;
    it.templ.id = id;
    it.cl[0] = 0xffff;it.cl[1] = cl;
    it.name = strdup(name);

    return it;
}


static PRNITEMTEMPLATE* StandardPrintDlg()
{
    PRNITEMTEMPLATE *items = NULL;

    if (items)
        return items;

    ListAdd(items,Item("Printer",grp4,8,4,272,84,BS_GROUPBOX | WS_GROUP, 0x0080));
    ListAdd(items,Item("&Name:",stc6,16,20,36,8 ,SS_LEFT , 0x0082));
    ListAdd(items,Item("",cmb1,52,18,152,152,CBS_DROPDOWNLIST | CBS_SORT |WS_VSCROLL | WS_GROUP | WS_TABSTOP, 0x0085));    
    ListAdd(items,Item("&Properties",psh2,212,17,60,14,WS_GROUP,0x0080));    
    ListAdd(items,Item( "Status:",stc8,16,36,36,10,SS_NOPREFIX | SS_LEFT | WS_TABSTOP, 0x0082));    
    ListAdd(items,Item( "",stc12,52,36,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP | SS_LEFT | WS_TABSTOP, 0x0082));
    ListAdd(items,Item( "Type:",stc7,16,48,36,10,SS_NOPREFIX | SS_LEFT | WS_TABSTOP, 0x0082));    
    ListAdd(items,Item( "",stc11,52,48,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP | SS_LEFT | WS_TABSTOP, 0x0082));    
    ListAdd(items,Item( "Where:",stc10,16,60,36,10,SS_NOPREFIX | SS_LEFT , 0x0082));    
    ListAdd(items,Item( "",stc14,52,60,224,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP | SS_LEFT , 0x0082));     
    ListAdd(items,Item( "Comment:",stc9,16,72,36,10,SS_NOPREFIX | SS_LEFT , 0x0082));        
    ListAdd(items,Item( "",stc13,52,72,152,10,SS_NOPREFIX | SS_LEFTNOWORDWRAP | SS_LEFT , 0x0082));    
    ListAdd(items,Item( "Paper",grp2,8,92,164,56,BS_GROUPBOX | WS_GROUP, 0x0080));        
    ListAdd(items,Item( "Size:",stc2,16,108,36,8,SS_NOPREFIX | SS_LEFT , 0x0082));        
    ListAdd(items,Item( "",cmb2,52,106,112,112,CBS_DROPDOWNLIST | CBS_SORT |WS_VSCROLL | WS_GROUP | WS_TABSTOP, 0x0085));        
    ListAdd(items,Item( "Source:",stc3,16,128,36,8,SS_NOPREFIX | SS_LEFT , 0x0082));        
    ListAdd(items,Item( "",cmb3,52,126,112,112,CBS_DROPDOWNLIST | CBS_SORT |WS_VSCROLL | WS_GROUP | WS_TABSTOP, 0x0085));        
    ListAdd(items,Item( "Orientation",grp1,180,92,100,56,BS_GROUPBOX | WS_GROUP, 0x0080));            
    ListAdd(items,Item( "",ico1,195,112,18,20, SS_ICON | WS_GROUP | SS_CENTERIMAGE, 0x0082));                
    ListAdd(items,Item( "P&ortrait",rad1,224,106,52,12,BS_AUTORADIOBUTTON , 0x0080));    
    ListAdd(items,Item( "L&andscape",rad2,224,126,52,12,BS_AUTORADIOBUTTON , 0x0080));    
    ListAdd(items,Item( "OK",IDOK,180,156,48,14,WS_GROUP, 0x0080));
    ListAdd(items,Item( "Cancel",IDCANCEL,232,156,48,14,BS_PUSHBUTTON | WS_TABSTOP, 0x0080));

    return items;
}



UINT CALLBACK PrintSetupHookProc(  HWND hdlg,  UINT uiMsg,  WPARAM wParam, LPARAM lParam );
HGLOBAL TfcPrintTemplate(control rootcontrol, bool RightAlign,int standardSize,void** ctrls);

static PRINTDLG PrtDlg;

static bool PrintDialogBox(TfcPrintJob *job, HWND Owner, char LorP/*='\0'*/, control c /*= nullcontrol*/, bool RightAlign /*= 0*/)
/* Display the standard Windows dialog box. */
/* 'hDC' and other important values are     */
/* returned via 'PrtDlg'. */
{	HGLOBAL defaultDevNames;
    extern void* TfcGetInstance();

    /*** Initialise PrtDlg: ***/
    clearS(PrtDlg);
    PrtDlg.lStructSize = sizeof(PRINTDLG);
    PrtDlg.hwndOwner = Owner;
    PrtDlg.hDC = (HDC) NULL;
    PrtDlg.nFromPage = 1;
    PrtDlg.nToPage = 1;
    PrtDlg.nMinPage = 1;
    PrtDlg.nMaxPage = 1;
    PrtDlg.nCopies = 1;

	if (LorP == '\0' and job->landscape)
		LorP = 'L';

	/*** Initialise DevNames and DevMode to the system defaults: ***/
    PrtDlg.Flags = PD_RETURNDEFAULT;
    if (PrintDlg(&PrtDlg) == 0)
        TfcMessage("Printing", 'x', "Can't even get the printer default settings!");

    // Lock 'DevMode' and modify it acording to 'LorP':
    if (DEVMODE* pDevMode = (DEVMODE*)::GlobalLock(PrtDlg.hDevMode)) {
        // Change printer settings in here.
        pDevMode->dmFields |= DM_ORIENTATION;
        pDevMode->dmOrientation = (LorP == 'L') ?
                        DMORIENT_LANDSCAPE : DMORIENT_PORTRAIT;
		if (job->paperSize != 0)
			pDevMode->dmPaperSize = job->paperSize;
        ::GlobalUnlock(PrtDlg.hDevMode);
    }
	defaultDevNames = PrtDlg.hDevNames;
	if (*job->printerName and job->rememberPrinter) {
		PrtDlg.hDevNames = GlobalAlloc(2/*GMEM_MOVABLE*/, sizeof(DEVNAMES) + strlen(job->printerName) + 1);
		DEVNAMES* pDevNames = (DEVNAMES*)::GlobalLock(PrtDlg.hDevNames);
		str s = (char*)(pDevNames + 1);
		strcpy(s, job->printerName);
		pDevNames->wDefault = 1;
		pDevNames->wDriverOffset = s - (char*)pDevNames;
		while (*s and *s != '\1')
			s++;
		if (*s == '\1')
			*s++ = '\0';
		pDevNames->wDeviceOffset = s - (char*)pDevNames;
		while (*s and *s != '\1')
			s++;
		if (*s == '\1')
			*s++ = '\0';
		pDevNames->wOutputOffset = s - (char*)pDevNames;
	}


    /*** Call the dialog box: ***/
    PrtDlg.Flags = PD_RETURNDC | PD_NOPAGENUMS | PD_PRINTSETUP | PD_HIDEPRINTTOFILE ;
	PrtDlg.Flags |= PD_USEDEVMODECOPIES;	// TFC doesn't implement multiple copies, 
											// hopefully the printer/printer driver does.
    PrtDlg.hDC = (HDC) NULL;
    if (c != nullcontrol) {
        /* Template size for standard Print Dialog */
        PRNITEMTEMPLATE *items , *pit = NULL;
        int size = 0, i;
        wchar_t* Wp;
        str s;

        items = StandardPrintDlg();
        for (each_oeli(pit,items)) {
            size += sizeof(DLGITEMTEMPLATE) + 3*2;
            size += 2*strlen(pit->name);
        }

        HGLOBAL htempl = TfcPrintTemplate(c, RightAlign, size, (void**) &Wp);
        ((DLGTEMPLATE*)htempl)->cdit += ListSize(items);

        /* Add items of standard dialog */
        for (each_oeli(pit,items)) {
            memcpy(Wp, &(pit->templ), sizeof(DLGITEMTEMPLATE));
            Wp = (wchar_t*)((char*)Wp + sizeof(DLGITEMTEMPLATE));
            *Wp++ = pit->cl[0];
            *Wp++ = pit->cl[1];
            for (s=pit->name; *s; )
                *Wp++ = *s++;
            *Wp++ = 0;
            *Wp++ = 0;

            if ((int)Wp & 2)
                *Wp++ = 0;
        }

        PrtDlg.lpfnSetupHook = (LPSETUPHOOKPROC) PrintSetupHookProc;
        PrtDlg.hSetupTemplate = (HANDLE) htempl;
        PrtDlg.Flags |= PD_ENABLESETUPTEMPLATEHANDLE | PD_ENABLESETUPHOOK;
    }
    
    bool success = PrintDlg(&PrtDlg);
	
	// If the Print dialog box failed due to incorrect initialisation, then this might
	// be related to the 'rememberPrinter' feature.  So let's go back to the default printer.
	if (not success and CommDlgExtendedError() != 0 and PrtDlg.hDevNames != defaultDevNames) {
		PrtDlg.hDevNames = defaultDevNames;
		success = PrintDlg(&PrtDlg);
	}

	// Save the settings
	if (DEVNAMES* pDevNames = (DEVNAMES*)::GlobalLock(PrtDlg.hDevNames)) {
		kstr output = (char*)pDevNames + pDevNames->wOutputOffset;
		kstr driver = (char*)pDevNames + pDevNames->wDriverOffset;
		kstr device = (char*)pDevNames + pDevNames->wDeviceOffset;
		sprintf(job->printerName, "%s\1%s\1%s", driver, device, output);
        ::GlobalUnlock(PrtDlg.hDevNames);
    }
	if (DEVMODE* pDevMode = (DEVMODE*)::GlobalLock(PrtDlg.hDevMode)) {
		job->paperSize = pDevMode->dmPaperSize;
		GlobalUnlock(PrtDlg.hDevMode);
	}

    /*** Clean up: ***/
    return success;
}


bool TfcPrintJob::AsIfPrintDlg(void *OwnerHwnd)
// Initialise 'PrintDC' without using a dialog box. 
{
	LoadFromRegistry();
    clearS(PrtDlg);
    PrtDlg.lStructSize = sizeof(PRINTDLG);
    PrtDlg.hwndOwner = (HWND)OwnerHwnd;
    PrtDlg.hDC = (HDC) NULL;
    PrtDlg.nFromPage = 1;
    PrtDlg.nToPage = 1;
    PrtDlg.nMinPage = 1;
    PrtDlg.nMaxPage = 1;
    PrtDlg.nCopies = 1;
    PrtDlg.Flags = PD_RETURNDEFAULT | PD_RETURNDC;

	return PrintDlg(&PrtDlg);
#if 0
	// I can't get this to work.  The DevMode stuff might work but remembering the printer
	// doesn't work.  The PrintDlg() method just gives me an unhelpful error code.
    if (DEVMODE* pDevMode = (DEVMODE*)::GlobalLock(PrtDlg.hDevMode)) {
        // Change printer settings in here.
        pDevMode->dmFields |= DM_ORIENTATION;
		pDevMode->dmOrientation = landscape ? DMORIENT_LANDSCAPE : DMORIENT_PORTRAIT;
		if (paperSize != 0)
			pDevMode->dmPaperSize = paperSize;
        ::GlobalUnlock(PrtDlg.hDevMode);
    }
	if (*printerName and rememberPrinter) {
		PrtDlg.hDevNames = GlobalAlloc(2/*GMEM_MOVABLE*/, sizeof(DEVNAMES) + strlen(printerName) + 1);
		DEVNAMES* pDevNames = (DEVNAMES*)::GlobalLock(PrtDlg.hDevNames);
		str s = (char*)(pDevNames + 1);
		strcpy(s, printerName);
		pDevNames->wDefault = 1;
		pDevNames->wDriverOffset = s - (char*)pDevNames;
		while (*s and *s != '\1')
			s++;
		if (*s == '\1')
			*s++ = '\0';
		pDevNames->wDeviceOffset = s - (char*)pDevNames;
		while (*s and *s != '\1')
			s++;
		if (*s == '\1')
			*s++ = '\0';
		pDevNames->wOutputOffset = s - (char*)pDevNames;
		return PrintDlg(&PrtDlg);
	}
#endif
}



/*----------------------------------------------------------------*/

void TfcPrintJob::SetRegistryBranch(kstr appName)
{
	PrintKey = TfcRegistryKey(yes, TFC_CURRENT_USER, "Software", appName, NULL);
}


TfcRegistryKey& TfcPrintJob::GetRegistryBranch()
{
	if (PrintKey == NULL)
		PrintKey = TfcRegistryKey(no, TFC_CURRENT_USER, "Software", "TFC", NULL);
	return PrintKey; 
}


void TfcPrintJob::LoadFromRegistry()
{	char buf[512];

	GetRegistryBranch();
    mmMargins.left = 0;
    mmMargins.right = 0;
    mmMargins.top = 0;
    mmMargins.bottom = 0;
    if (str s = PrintKey.Subkey(no,"PrintMargins").Get(buf, sizeof(buf))) {
        sscanf(s, "%d,%d,%d,%d",
                &mmMargins.top,
                &mmMargins.left,
                &mmMargins.right,
                &mmMargins.bottom);
    }
    if (str s = PrintKey.Subkey(no,"PrintColours").Get(buf, sizeof(buf)))
        colourmode = (*s == 'C') ? tfc_colour : (*s == 'S') ? tfc_shadesofgrey : tfc_blackandwhite;
    else colourmode = tfc_unknownmode;
    if (PrintKey.Subkey(no,"PrintHeader").Get(header, sizeof(header)) == NULL)
		strcpy(header, "%title");
    if (PrintKey.Subkey(no,"PrintFooter").Get(footer, sizeof(footer)) == NULL)
		strcpy(footer, "Page %page/%pagetotal   -    printed %printdate");
	if (PrintKey.Subkey(no,"PrintLandscape").Get(buf, sizeof(buf)))
		landscape = (*buf == 'Y');
	else landscape = no;
    if (colourmode == tfc_unknownmode)
        colourmode = tfc_colour;
	PrintKey.Subkey(no,"Printer").Get(printerName, sizeof(printerName));
	PrintKey.Subkey(no,"PaperSize").Get(buf, sizeof(buf));
	paperSize = atoi(buf);
}


void TfcPrintJob::SaveToRegistry()
{	char buf[512];

	if (PrintKey == NULL)
		PrintKey = TfcRegistryKey(yes, TFC_LOCAL_MACHINE, "Software", "TFC", NULL);
    sprintf(buf, "%d,%d,%d,%d",
                mmMargins.top,
                mmMargins.left,
                mmMargins.right,
                mmMargins.bottom);
    PrintKey.Subkey(yes, "PrintMargins").Set(buf);
    PrintKey.Subkey(yes, "PrintColours").Set(
                (colourmode == tfc_colour) ? "C" :
                (colourmode == tfc_shadesofgrey) ? "S" : "W");
	PrintKey.Subkey(yes,"PrintLandscape").Set(landscape ? "Y" : "N");
	PrintKey.Subkey(yes,"PrintHeader").Set(header);
	PrintKey.Subkey(yes,"PrintFooter").Set(footer);
	PrintKey.Subkey(yes,"Printer").Set(printerName);
	sprintf(buf, "%d", paperSize);
	PrintKey.Subkey(yes,"PaperSize").Set(buf);
}


static void HelpOnHeadersAndFooters(TfcPrintJob *job)
{
	job->HelpOnHeadersAndFooters();
}


control TfcPrintJob::ParametersGui()
// This gives you controls for the long-term printing parameters.
{
	return GridLayout(
			StaticText("Margins") | Control(&mmMargins.top, "     &Top (mm)", 0, 100)
									-
			Control(&mmMargins.left, "&Left", 0, 100) / AlignXRight(StaticText("Right    ")) /
							Control(&mmMargins.right, NULL, 0, 100)
									-
			StaticText("") / Control(&mmMargins.bottom, "        &Bottom", 0, 100)
		)
				-
		FillerControl(0,30)
				-
		Control(header, sizeof(header), "Header text", 50, 0)
				-
		(continuousRollPaper ? nullcontrol : Control(footer, sizeof(footer), "Footer text", 50, 0))
				-
				Button("&Header + footer: Print variables", TfcCallback(::HelpOnHeadersAndFooters, this), 0);
		;
}


bool TfcPrintJob::DlgBox(control AppOptions, char LorP/*='\0' i.e. don't care*/)
/* Display the print dialog box(es). */
{
	LoadFromRegistry();
    control colours_c;
    colours_c =
        EnumControl(&colourmode, tfc_colour, "&Colour",0) |
        EnumControl(&colourmode, tfc_shadesofgrey, "Shades of &grey",0) |
        EnumControl(&colourmode, tfc_blackandwhite, "Black && &white",0);
	
    control c = colours_c - FillerControl(20,20);
    //c = c - (OkButton() | CancelButton());
    c = AppOptions - c;


    /*** The standard print dialog box combined with app-level controls: ***/
    if (not PrintDialogBox(this, (HWND)TfcTopLevelHwnd(yes), LorP, c, yes))
        return no;            // Failed because user cancelled.


    /* Save the current choices in the registry: */
	SaveToRegistry();
    return yes;
}

void TfcPrintJob::GetDeviceSpec(void* _DC)
{
	HDC DC = (HDC)_DC;
    MMwidth = GetDeviceCaps(DC, HORZSIZE);
    printerWidth_p = GetDeviceCaps(DC, HORZRES);
    printerHeight_p = GetDeviceCaps(DC, VERTRES);
	logPixelsY = GetDeviceCaps(DC,LOGPIXELSY);
}

bool TfcPrintJob::Prepare(void* _DC)
{
	// Query the device specs:
	
	GetDeviceSpec(_DC);

    if (printerHeight_p == 0 or printerWidth_p == 0)
        return no;

	// Calculate the white margins:
    double ppix_per_mm = (double)printerWidth_p / MMwidth;
    pixMargins.top = int(mmMargins.top * ppix_per_mm);
    pixMargins.left = int(mmMargins.left * ppix_per_mm);
    pixMargins.right = int(mmMargins.right * ppix_per_mm);
    pixMargins.bottom = int(mmMargins.bottom * ppix_per_mm);

	// Choose a header/footer font
	headerHeight = ppix_per_mm * 8;
	if (headerHeight*40 > printerHeight_p)
		headerHeight = printerHeight_p / 40;
    headerFont = TfcFindFont(headerHeight, 0, "MS San Serif");
	footerHeight = ppix_per_mm * 5;
	if (footerHeight*64 > printerHeight_p)
		footerHeight = printerHeight_p / 64;
    footerFont = TfcFindFont(footerHeight, 0, "MS San Serif");

	// How many pixels can we devote to the scrollwin content?
	innerWidth = printerWidth_p - pixMargins.left - pixMargins.right;
    innerHeight = printerHeight_p - pixMargins.top - pixMargins.bottom - (headerHeight+footerHeight);
	if (innerWidth < pixMargins.left or innerWidth < pixMargins.right) {
		pixMargins.left = pixMargins.right = 0;
		innerWidth = printerWidth_p;
	}
	if (innerHeight < pixMargins.top or innerHeight < pixMargins.bottom) {
		pixMargins.top = headerHeight;
		pixMargins.bottom = footerHeight;
		innerHeight = printerHeight_p - pixMargins.top - pixMargins.bottom - (headerHeight+footerHeight);
	}
	return yes;
}


void TfcPrintJob::Stats()
{
	TfcMessage("Printer statistics", 'i', "Horitontal size: %d\nHorizontal resolution: %d\nVertical resolution: %d\n",
				MMwidth, printerWidth_p, printerHeight_p);
}


static void Stats(TfcPrintJob *job)
{
	job->Stats();
}


// A structure to hold all the pre-printing state of the ScrollWins we are about to print
class SaveScrollWin2 {
public:
    SaveScrollWin2(ScrollWin &sw);
    void RestoreScrollWin(ScrollWin &sw);

    Litewin              *parent;
    int                   clientWidth, clientHeight;
    tfccolourmode_enum    colourmode;
    int                   Background;
    static SaveScrollWin2  **SaveList(ScrollWin **List);
    static void           RestoreList(ScrollWin **List, SaveScrollWin2 **SaveList);
};


SaveScrollWin2::SaveScrollWin2(ScrollWin &sw)
{
    parent = sw.parent;
    clientWidth = sw.clientWidth;
    clientHeight = sw.clientHeight;
    colourmode = sw.colourmode;
    Background = sw.Background;
}


void SaveScrollWin2::RestoreScrollWin(ScrollWin &sw)
{
    sw.parent = parent;
    sw.clientWidth = clientWidth;
    sw.clientHeight = clientHeight;
    sw.colourmode = colourmode;
    sw.Background = Background;
}


SaveScrollWin2 **SaveScrollWin2::SaveList(ScrollWin **List)
{
    SaveScrollWin2 **SaveList = NULL;
    ScrollWin     *sw = NULL;
    int           i;

    for (each_aeli(sw, List)) {
        ListAdd(SaveList, new SaveScrollWin2(*sw));
    }
    return SaveList;
}


void SaveScrollWin2::RestoreList(ScrollWin **List, SaveScrollWin2 **SaveList)
{   ScrollWin *sw = NULL;

    for (int each_aeli(sw, List)) {
        SaveList[i]->RestoreScrollWin(*sw);
        delete SaveList[i];
    }
    ListFree(SaveList);
}


bool TfcPrintJob::PrintWithOrWithoutPreview(bool withPreview)
{
    SaveScrollWin2 **SaveList = NULL;
    ScrollWin *sw = NULL;

    if (List == NULL)
        return no;            // There's nothing to print!
	if (PrtDlg.hDC == NULL) {
		AsIfPrintDlg(NULL);
	}

    SaveList = SaveScrollWin2::SaveList(List);
	tfccolourmode_enum saveMode = colourmode;
	if (List[0]->Background == BLACK)
		colourmode = tfc_blackandwhite;

	/*** Step 3: Calculate all the printer-specific fields: ***/
REDO_PAGES:
	if (not Prepare(PrtDlg.hDC)) {
        TfcMessage("Bad printer", '!', "Edval cannot print, due to a problem with printer setup.\n\n"
					"Check that the printer drivers are properly set up and try print again.\n");
		return no;
	}
    TfcBusyOn();


	/*** Step 4: Resize all the windows. ***/
    for (int each_aeli(sw, List)) {
        sw->clientWidth = innerWidth;
        sw->clientHeight = innerHeight;
        sw->colourmode = colourmode;
        sw->hDC = PrtDlg.hDC;
        sw->Background = WHITE;
        sw->inPrint = yes;
        assert(sw->hWnd == NULL);
        sw->Resized();
    }

    /*** Step 5: Break the ScrollWin's into pages: ***/
    PrintPreviewCell **PPList=NULL, *PPCell = NULL;
    double S = sqrt((double)ListSize(List));
    if (S > 2)
        S = 2;
    int zoomX = (int)(400 / S);
    int zoomY = zoomX * printerHeight_p / printerWidth_p;
    for (int each_aeli(sw, List)) {
        if (sw->realWidth <= 0 or sw->realWidth > 100000) {
			TfcMessage("Error", '!', "One of your pages is too large: > 100,000 pixels.\nAborting.");
            break;
        }

        /* Produce 1 or more pages for this one scrollwin: */
		int nextX,nextY,bottomY,rightX;
		for (int topY = 0; topY < sw->realHeight; topY = nextY) {
			if (topY + innerHeight >= sw->realHeight)
				bottomY = nextY = 999999999;
			else {
				sw->DecideOnPageBreakPosY(topY + innerHeight, &bottomY, &nextY);
				nextY -= sw->dontScrollY;
				if (nextY <= topY)
					nextY = topY + innerHeight;	// The application level is abusing the system.
			}
			for (int leftX = 0; leftX < sw->realWidth; leftX = nextX) {
				if (leftX + innerWidth >= sw->realWidth)
					rightX = nextX = 999999999;
				else {
					sw->DecideOnPageBreakPosX(leftX + innerWidth, &rightX, &nextX);
					nextX -= sw->dontScrollX;
					if (nextX <= leftX)
						nextX = leftX + innerWidth;	// The application level is abusing the system.
				}

				// Note: we won't print a page if the only thing on it is going to be
				// the column/row headings - hence "- sw->dontScroll*".
                if (ListSize(PPList) > 2000) {
                    TfcMessage("Error", '!', "You are not allowed to print > 2000 pages.");
					TfcBusyOff();
                    return no;
                }
                ListAdd(PPList, new PrintPreviewCell(sw,
                        leftX, topY, rightX, bottomY,
                        ListSize(PPList),
                        zoomX, zoomY));
                /* If we use a scrollwin more than once, because it doesn't fit onto*/
                /* a single page, then the sw->parent pointers will not really be        */
                /* valid - the sw will point to the last PrintPreviewCell to        */
                /* reference it.  But this doesn't matter in this case because we        */
                /* never need to go from sw to parent in PrintPreview, printing is        */
                /* always done top-down. */
            }
        }			
    }
    if (ListSize(PPList) == 0) {
        SaveScrollWin2::RestoreList(List, SaveList);
        TfcMessage("Print Preview", '!', "All the ScrollWin's have zero size!");
        return no;
    }
    PrintPreviewGrid *PPGrid = new PrintPreviewGrid(this, PrtDlg.hDevMode, PPList);
    PPGrid->FontHeight = 160;
    PPGrid->colourmode = tfc_colour;
    PPGrid->zoomX = zoomX;
    PPGrid->zoomY = zoomY;
    PPGrid->Arrange();
    PPGrid->MakeMeasurements();
    PPGrid->SetCursorBlinkRate(0, 0);


	/*** Step 6: The Preview screen ***/
	if (withPreview) {
		control bar, subset_c;
		if (ListSize(PPList) == 1)
			subset_c = nullcontrol;
		else
			subset_c =  FillerControl(20,0)
						| Button("&From here", (SwVoidVoid_fn)&PrintPreviewGrid::FromHere)
						| Button("&Upto here", (SwVoidVoid_fn)&PrintPreviewGrid::UptoHere)
						| Button("&Not this one", (SwVoidVoid_fn)&PrintPreviewGrid::NotThisOne)
						| Button("&Only this one", (SwVoidVoid_fn)&PrintPreviewGrid::OnlyThisOne)
						| Button("&Reset", (SwVoidVoid_fn)&PrintPreviewGrid::Reset);

		bar =   Button("&Print!", (SwVoidVoid_fn)&PrintPreviewGrid::Print)
				| Button("&Cancel", (SwVoidVoid_fn)&PrintPreviewGrid::Cancel)
				| FillerControl(20,0)
				| Button("Zoom &in (ctrl-PageUP)", (SwVoidVoid_fn)&PrintPreviewGrid::ZoomIn, TFC_SKINNYBUTTON)
				| Button("&Zoom out (ctrl-PageDown)", (SwVoidVoid_fn)&PrintPreviewGrid::ZoomOut, TFC_SKINNYBUTTON)
				| Button("&Stats", TfcCallback(::Stats, this))
				| FillerControl(20,0)
				| Button("&Header + Footer text", (SwVoidVoid_fn)&PrintPreviewGrid::HeadersFooters)
				| subset_c;

		PPGrid->SetRebar(bar, NULL);
		TfcBusyOff();

		PPGrid->EventLoop();
		if (PPGrid->exitStatus == -1) {
			SaveScrollWin2::RestoreList(List, SaveList);
			delete PPGrid;              // This will free the PrintPreviewCell's.
			return no;                  // The user cancelled.
		}
		else if (PPGrid->exitStatus == 2) {
			delete PPGrid;
			for (int each_aeli(sw, List))
				sw->parent = sw;
			goto REDO_PAGES;
		}
	}

	/*** Step 7: The actual printing. ***/
    DOCINFO di;
    int error;
    int job;

    di.cbSize = sizeof(DOCINFO);
    di.lpszDocName = jobName;
    di.lpszOutput = (LPTSTR) NULL;
    di.lpszDatatype = (LPTSTR) NULL;
    di.fwType = 0;
    job = StartDoc(PrtDlg.hDC, &di);
	bool success = no;
    if (job <= 0) {
        PRINT_ERROR:
        error = GetLastError();
		inActualPrinting = no;
        TfcMessage("Print job", '!',
                    "We had trouble starting your print job: error=%d",
                    error);
    }
	else {
		inActualPrinting = yes;
		PPgrid->hDC = PrtDlg.hDC;
		for (int each_aeli(PPCell, PPgrid->PPList)) {
			error = StartPage(PrtDlg.hDC);
			if (error <= 0)
				goto PRINT_ERROR;
			PPCell->child->parent = PPCell->child;
			PPCell->child->scrollX = PPCell->viewport_p.left;
			PPCell->child->scrollY = PPCell->viewport_p.top;
			PPCell->InPrinterSpace = 1;      
			PPCell->previewFrame_p.Init(-32000,-32000,32000,32000);
			PPCell->Print((HDC)PrtDlg.hDC);
			error = EndPage(PrtDlg.hDC);
			if (error <= 0)
				goto PRINT_ERROR;
		}
		error = EndDoc(PrtDlg.hDC);
		if (error <= 0)
			goto PRINT_ERROR;
		success = yes;
	}

    /*** Step 8: Cleaning up: ***/
	inActualPrinting = no;
    delete PPgrid;        // This will free the PrintPreviewCells.
	TfcBusyOff();
	SaveScrollWin2::RestoreList(List, SaveList);
	colourmode = saveMode;
	PrtDlg.hDC = NULL;
	return success;
}


/*--------------------------------------------------------------*/
TfcContinuousRollPrintJob::TfcContinuousRollPrintJob(kstr jobName):
					TfcPrintJob(jobName)
{

}

void TfcContinuousRollPrintJob::GetDeviceSpec(void* _DC)
{
	HDC DC = (HDC)_DC;
	
    printerHeight_p = GetDeviceCaps(DC, VERTRES);
	logPixelsY = GetDeviceCaps(DC,LOGPIXELSY);
	int logPixelsX = GetDeviceCaps(DC, LOGPIXELSY);
	MMwidth = 88;
	printerWidth_p = 3.5 * logPixelsX;

}
	
/*--------------------------------------------------------------*/

TfcPrintJob::TfcPrintJob(kstr _jobName)
{
	List = NULL;
	owner = NULL;
	jobName = _jobName;
	rememberPrinter = no;
	continuousRollPaper = no;
	inActualPrinting = no;
	paperSize = 0;
	printerName[0] = '\0';
}


TfcPrintJob::~TfcPrintJob()
{	ScrollWin *win;

	for (int each_aeli(win, List))
		delete win;
	ListFree(List);
}


void TfcPrintJob::Add(ScrollWin *win)
{
	win->printJob = this;
	ListAdd(List, win);
}


void TfcPrintJob::TakeBackOwnership(ScrollWin *win)
{
	win->printJob = NULL;
	ListDelP(List, win);
}


void EditorWin::Print(str jobname)
{   
	TfcPrintJob printJob(jobname ? jobname : pageTitle ? pageTitle : windowTitle);
	void *SaveHWnd;

    SaveHWnd = _hWnd();
    Hide();
    if (SaveHWnd)
        DetachFromWindow();
    printJob.Add(this);
    if (printJob.DlgBox())
		printJob.PreviewAndThenPrint();
    inPrint = no;
    if (SaveHWnd)
        AttachToWindow(SaveHWnd);
    Resized();
    Show();
}


