/* $Id: rowcolumn.cpp 623 2005-10-26 06:33:05Z jla $ */


#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include <math.h>
#include "tfc.h"
#include "list.h"
#include "RowColumn.h"
#include "print.h"



void TfcSplitCurOff();
void TfcSplitCurOn();


void RowColumnWin::SetColumnsToDefaults()
{   
	kstr help;
	char status;

	ListFree(X);
	ListFree(U);
	ListFree(ColumnIds);
	 
	ReadConfigDefCols();

	if (ColumnIds == NULL) {
		for (column_id cid=0; GetHeading(cid,&help,&status) != NULL; cid++) {
			if (status == 'D')
				ListAdd(ColumnIds, cid);
		}
	}
}


void RowColumnWin::SetColumnsToDefaultsAndRedraw()
/* a callback for the menu item to restore default cols */
{
	SetColumnsToDefaults();
	Calibrate(yes);
	Resized();
	PaintWhole();
}


void RowColumnWin::ShowAllColumns()
{
	char status;
	kstr help;

	ListFree(X);
	ListFree(U);
	ListFree(ColumnIds);
	    
	for (column_id cid=0; GetHeading(cid,&help,&status) != NULL; cid++) {
		if (status != '\0')
			ListAdd(ColumnIds, cid);
	}
	Calibrate(yes);
	Resized();
	PaintWhole();
}


str RowColumnWin::ColumnsToString(str buf)
/* return a string showing which columns are in use */
{
    int N;
    str s;
    N = ListSize(ColumnIds);
    s = buf;
    for (int i=0; i < N; i++) {
        char dummy2; 
		kstr dummy1;
        char *heading = strdup(GetHeading(ColumnIds[i],&dummy1,&dummy2));
        char *h = heading;
        while ((h = strchr(h, '\n')))
            *h = '\2';
        s += sprintf(s, "%s\1", heading);
        free(heading);
    }
    *s = 0;
    return buf;
}


void RowColumnWin::StringToColumns(str buf, bool call_default)
/* set the columns according to the string */
{
    char status; 
	kstr h,dummy1;
	str t, s2, s = strdup(buf);
    s2 = t = s;
    while ((s2 = strchr(s2, '\2')))
        *s2 = '\n';
    s2 = s;
    ListFree(X);
    ListFree(U);
    ListFree(ColumnIds);
    do {
		while (*s and *s != '\1' and *s != ';') 
			s++;
        if (*s == 0)
			break;
        *s = 0;
        for (column_id cid=0; (h = GetHeading(cid,&dummy1,&status)) != NULL; cid++) {
            if (h && streq(h, t)) {
                ListAdd(ColumnIds, cid);
                break;
            }
        }
        t = ++s;
    } while (true);
    if (ColumnIds == NULL && call_default) /* bad sign */
        SetColumnsToDefaults();
    free (s2);
}


str RowColumnWin::SortsToString(str buf)
{
    char *s = buf;
    for (int i = 2; i >= 0; --i) {
        if (sort[i] == 0)
            continue;
        char dummy2;
		kstr dummy1;
        s += sprintf(s, "%s|", GetHeading(sort[i] - 1, &dummy1, &dummy2));
    }
    if (*(s - 1) == '|')
        --s;
    *s = '\0';
    return buf;
}


void RowColumnWin::StringToSorts(str buf)
/* set the sort column according to the string */
{
    sort[0] = sort[1] = sort[2] = 0;
    char *s, *o, *currColumnName;
    currColumnName = s = o = strdup(buf);
    while (*s != '\0') {
        if (*s == '|' || *(s + 1) == '\0') {
            if (*s == '|')
                *s = '\0';
            bool isReverse = (*currColumnName == '-');
            if (isReverse)
                ++currColumnName;
            kstr heading, dummy1;
			char dummy2;
            for (column_id cid = 0;  (heading = GetHeading(cid, &dummy1, &dummy2)) != NULL; ++cid) {
                if (streq(currColumnName, heading)) {
                    ++cid;
                    sort[2] = sort[1];
                    sort[1] = sort[0];
                    sort[0] = isReverse ? -cid : cid;
                    break;
                }
            }
            currColumnName = s + 1;
        }
        ++s;
    }
    free(o);
    Sort();
}


str RowColumnWin::ColumnWidthsToString(str buf)
{
    int N = ListSize(ColumnIds);
    char *s = buf;
    for (int i = 0; i < N; ++i)
        s += sprintf(s, "%d\1", X[i]);
    *s = 0;
    return buf;
}


void RowColumnWin::StringToColumnWidths(str buf)
{
    int N = ListSize(ColumnIds);
    char *copy, *s, *width;
    s = width = copy = strdup(buf);
    int i = 0;
    while (s && *s && i < N) {
        if (*s == '\1') {
            *s = '\0';
            X[i] = atoi(width);
            ++i;
            width = ++s;
        } else {
            ++s;
        }
    }
    free(copy);
}


RowColumnWin::RowColumnWin(str caption, int width, int height, int flag, ScrollWin *owner)
            : ScrollWin(caption, width, height, flag, owner), mpExportDialog(NULL)
{
    font = NULL;
    sort[0] = sort[1] = sort[2] = 0;
    X = NULL;
    U = NULL;
    Rows = NULL;
    RowHeight = 16;
    SummaryHeight = 0;
    TopHeight = HeadingHeight = 20;
    ColumnGap = 10;
    RowJumpers = 2;
    HighlightedRows = NULL;
    CurrentRow = NULL;
    CurrentRowId = -1;
    BeingDragged = -1;
    HeadingClicked = false;
    HighlightedOnly = no;
    AllowNoSelection = no;
    AllowNoMultiSelection = no;
    mRepaintOnMousestroke = true;
    mCalibrateUsingFullWin = false;
    mAllowSorting = true;
    LeftmostAlwaysVisible = 0;
    summary = NULL;
    BeingResized = no;
    FirstShifted = -1;
    realHeight = TopHeight;
    SetCursorBlinkRate(0,0);
    ScrollbarMode = 'X';
    ColumnIds = NULL;
    one = 1;
    // NB: Can't call any pure virtual function yet.
    FromMousestroke = false;
    searchstring[0] = '\0';
    bHighlightOnMouse2 = false;
	SetUnscrollableAreas();
}


void RowColumnWin::SetHighlightOnMouse2(bool bHighlight)
{
    bHighlightOnMouse2 = bHighlight;
}


void RowColumnWin::Clear()
{
    SetRows(NULL);
}


void RowColumnWin::SetRows(void** _Rows)
/* Install a new or updated dynamic-array of rows.      */
/* The caller still owns this list (i.e. must eventually free it).*/
{   int lastrow;

    if (HighlightedRows) {
        lastrow = CurrentRowId;
        ListFree(HighlightedRows);
    } else {
        lastrow = -1;
    }

    ListFree(Rows); //RowColumn owns Rows
    Rows = _Rows;
    if (CurrentRowId >= 0 && CurrentRowId < ListSize(Rows))
        CurrentRow = Rows[CurrentRowId];
    else CurrentRowId = -1, CurrentRow = NULL;
    Calibrate(yes);
    if (ListSize(X) == 0)
        realWidth = 0;
    else realWidth = X[ListSize(X)-1];
    CalcHeight();

    UpdateScrollbars(0,0);

    sort[0] = sort[1] = sort[2] = 0;        // discard the sorting
    if (lastrow > -1 and not AllowNoSelection and Rows and ListSize(Rows) > lastrow and X)
        MoveToRow(lastrow);

    if (ShowState == tfc_shown)
        PaintWhole();
}


void RowColumnWin::AddRow(void* Row)
{
    ListAdd(Rows, Row);
    CalcHeight();
    SuppressPaints();
}


void RowColumnWin::DeleteRow(void* Row)
{   int moveto = -1;

    if (Row == NULL)
        return;

    if (Row == GetFocusRow()) {
        if (CurrentRowId == (NumRows()-1))
            moveto = CurrentRowId-1;
        else
            moveto = CurrentRowId;
    }

    ListDelP(HighlightedRows, Row);
    ListDelP(Rows, Row);
    CalcHeight();
    if (ShowState == tfc_shown)
        PaintWhole();
    if (moveto != -1)
		MoveToRow(moveto, yes);
}


void RowColumnWin::PaintHeadingRow(int x2)
{   int x,y,i,N, k, xStart = 0;

    if (inPrint)
        DrawRectangle(0, SummaryHeight, scrollX + clientWidth, HeadingHeight, WHITE, NOCOLOUR);
    y = SummaryHeight;
    N = ListSize(ColumnIds);
    if (LeftmostAlwaysVisible) {        
        for (k=0; k < LeftmostAlwaysVisible; k++) {        
            char dummy2;
            kstr dummy1;
            i = 0;
            DrawString(GetHeading(ColumnIds[k],&dummy1,&dummy2), -1, font,
                        X[k] + scrollX, y+1, X[k+1]+ scrollX-one, y+HeadingHeight,
                        BLACK, LIGHT_GREY, TFC_CENTRE);
            DrawLine(X[k]+ scrollX, y+HeadingHeight-2*one, X[k+1]+ scrollX-one, y+HeadingHeight-2*one, GREY);
            DrawLine(X[k+1]+ scrollX-one, y, X[k+1]+ scrollX-one, y+HeadingHeight, BLACK);
            DrawLine(X[k+1]+ scrollX-2*one, y, X[k+1]+ scrollX-2*one, y+HeadingHeight, GREY);
            DrawLine(X[k]+ scrollX, y, X[k+1]-one+ scrollX, y, WHITE);
            DrawLine(X[k]+ scrollX, y, X[k]+ scrollX, y+HeadingHeight, WHITE);
            if (ColumnIds[i] == sort[0]-1) {
                // We're sorting on this field: display the triangle.
                int h1=y+5*one, h2=y+12*one;
                x = X[k+1] + scrollX- 10*one;  // (x,h1) is the apex
                DrawLine(x,h1,x+4*one,h2,WHITE,one);
                DrawLine(x,h1,x-4*one,h2,GREY,one);
                DrawLine(x-4*one,h2,x+4*one,h2,WHITE,one);
            }
            else if (ColumnIds[i] == -1-sort[0]) {
                // We're reverse-sorting on this field: display the triangle.
                int h1=y+12*one, h2=y+5*one;
                x = X[k+1] + scrollX- 10*one;  // (x,h1) is the upside-down apex
                DrawLine(x,h1,x+4*one,h2,WHITE,one);
                DrawLine(x,h1,x-4*one,h2,GREY,one);
                DrawLine(x-4*one,h2,x+4*one,h2,GREY,one);

            }
        }
        xStart = X[k] + scrollX;
    }
    
    for (i=LeftmostAlwaysVisible; i < N; i++) {
        char dummy2;
        kstr dummy1;
        if (LeftmostAlwaysVisible and scrollX and i < LeftmostAlwaysVisible)
            continue;
        //SetClipper(X[i] < xStart ? xStart : X[i], y, inPrint ? 99999 : x2, y + HeadingHeight);
                // ^ When printing, we go out of our bounds by up to one column.
        DrawString(GetHeading(ColumnIds[i],&dummy1,&dummy2), -1, font,
                        X[i], y+1, X[i+1]-one, y+HeadingHeight,
                        BLACK, LIGHT_GREY,
                        TFC_CENTRE);
        DrawLine(X[i], y+HeadingHeight-2*one, X[i+1]-one, y+HeadingHeight-2*one, GREY);
        DrawLine(X[i+1]-one, y, X[i+1]-one, y+HeadingHeight, BLACK);
        DrawLine(X[i+1]-2*one, y, X[i+1]-2*one, y+HeadingHeight, GREY);
        DrawLine(X[i], y, X[i+1]-one, y, WHITE);
        DrawLine(X[i], y, X[i], y+HeadingHeight, WHITE);

        if (X[i+1] >= x2) {   
            i++;
            break;
        }

        if (ColumnIds[i] == sort[0]-1) {
            // We're sorting on this field: display the triangle.
            int h1=y+5*one, h2=y+12*one;
            x = X[i+1] - 10*one;  // (x,h1) is the apex
            DrawLine(x,h1,x+4*one,h2,WHITE,one);
            DrawLine(x,h1,x-4*one,h2,GREY,one);
            DrawLine(x-4*one,h2,x+4*one,h2,WHITE,one);
        }
        else if (ColumnIds[i] == -1-sort[0]) {
            int h1=y+12*one, h2=y+5*one;
            // We're reverse-sorting on this field: display the triangle.
            x = X[i+1] - 10*one;  // (x,h1) is the upside-down apex
            DrawLine(x,h1,x+4*one,h2,WHITE,one);
            DrawLine(x,h1,x-4*one,h2,GREY,one);
            DrawLine(x-4*one,h2,x+4*one,h2,GREY,one);
        }
    }
    if (X[i] < x2) {
        DrawRectangle(X[i], y, x2, y+HeadingHeight-2*one, LIGHT_GREY, NOCOLOUR);
        DrawLine(X[i], y+HeadingHeight-1-one, x2, y+HeadingHeight-1-one, GREY, one);
    }
    //SetClipper();
    if (not inPrint)
        DrawLine(0, y+HeadingHeight-1, x2, y+HeadingHeight-1, BLACK, one);
}


void RowColumnWin::Paint(int x0, int y0, int x1, int y1)
{   int j0, j1, y;

    if (X == NULL) {
        if (ColumnIds == NULL) {
            DrawString("You must call \"SetColumnsToDefaults()\" from your constructor\n"
                    "(and have some \"*statusp='D'\" before a \"return NULL\")", -1,
                    NULL, 0,0,clientWidth,clientHeight, BLACK, WHITE);
            return;
        }
        Calibrate(yes);     // This includes a call to PaintWhole().
        return;
    }

    ForExcel = no;

	// What rows will we paint?
    j0 = (y0-TopHeight) / RowHeight;
    j1 = (y1-TopHeight + RowHeight-1) / RowHeight;
    if (j1 >= ListSize(Rows))
        j1 = ListSize(Rows)-1;
    if (j0 < 0)
        j0 = 0;

    // Paint the summary line: */
    if (summary and y0 < SummaryHeight)
        DrawString(summary,-1,font, 0,0,clientWidth<x1?x1:clientWidth,
                    SummaryHeight, summaryFg,summaryBg);
    
    // Paint them one by one:
    y = j0 * RowHeight + TopHeight;
    for (paint_rownum=j0; paint_rownum <= j1; paint_rownum++) {
        if (inPrint and HighlightedOnly and not isHighlighted(Rows[paint_rownum]))
            continue;
        if (y >= scrollY)
            PaintRow(Rows[paint_rownum], y, x0, x1);
        y += RowHeight;
    }

    // paint header row
    PaintHeadingRow(x1);

    // Paint the bottom area and side area:        
    if (y < y1)
        DrawRectangle(x0,y,x1,y1,Background);
    if (inPrint and realWidth < x1)
        DrawRectangle(realWidth,0,x1,y1,Background);
}


static void getPrintColours(int* fgcolor, int* bgcolor, char status)
{
    if (status == 'F' or status == 'X') // the field is focused
        *fgcolor = BLACK, *bgcolor = GREY;
    else
    if (status == 'H') // the field is highlighted
        *fgcolor = WHITE, *bgcolor = LIGHT(GREY);
    else        
        *bgcolor = WHITE, *fgcolor = BLACK;
}


void RowColumnWin::PaintCell(void* row, column_id cid, int x1, int x2, int y, char status)
/* Paint this cell at this location. */
{   int fgcolor,bgcolor,rm;
    char alignment;
    char buf[8192];
    const char* s;

    paint_x = cid;
    if (x2 - x1 <= 4) {
        // This is a hidden column
        DrawRectangle(x1, y, x2, y + RowHeight, Background);
        return;
    }
    paint_x = cid;
    cellfont = font;
    s = GetField(row, ColumnIds[cid], buf, status,
                        &fgcolor, &bgcolor, &alignment);
    
    if (inPrint)        // ensure that while printing background is always white
        getPrintColours(&fgcolor, &bgcolor, status);

    DrawRectangle(x1, y, x2, y + RowHeight, bgcolor, NOCOLOUR);
    if (s) {
        if (alignment == 'L') {
            DrawRectangle(x1, y, x1+5*one, y+RowHeight, bgcolor);
            DrawStringU(s, strlen(s), cellfont,
                x1+5*one, y, x2, y + RowHeight,
                fgcolor, bgcolor, 0);
            rm = (clientWidth > 1500) ? 4 : 2;
            DrawRectangle(x2-rm/2, y, x2, y+RowHeight, bgcolor);
        }
        else if (alignment == 'C') {
            DrawStringU(s, strlen(s), cellfont,
                x1,
                y, x2, y + RowHeight,
                fgcolor, bgcolor,
                TFC_CENTRE);
            rm = (clientWidth > 1500) ? 4 : 2;
            DrawRectangle(x2-rm/2, y, x2, y+RowHeight, bgcolor);
        }
        else {
            rm = (clientWidth > 1500) ? 4 : 2;
            DrawStringU(s, strlen(s), cellfont,
                x1, y, x2-rm, y + RowHeight,
                fgcolor, bgcolor,
                TFC_RIGHTALIGN);
            DrawRectangle(x2-rm, y, x2-rm/2, y+RowHeight, bgcolor);
        }
    }
    else
        PaintField(row, ColumnIds[cid], rowStatus(row),
                x1, y, x2, y + RowHeight);

    if (not inPrint and cid == ListSize(ColumnIds) - 1 and x2 < clientWidth + scrollX)
        DrawRectangle(x2,y,clientWidth + scrollX, y + RowHeight,bgcolor);
}


void RowColumnWin::PaintRow(void *row, int y, int x0, int x1)
{   int cid, N, c0;
    char status;

    N = ListSize(ColumnIds);
    status = rowStatus(row);
    c0 = LeftmostAlwaysVisible and scrollX ? LeftmostAlwaysVisible : 0;
    for (cid=c0; cid < N-1 and X[cid+1] < x0; cid++)
        ;
    for (; cid < N and X[cid] < x1; cid++)
        PaintCell(row, cid, X[cid], X[cid+1], y, status);
    PerRowPaint(row, x0, y, x1, y+RowHeight);
    if (c0) {
        for (cid=0; cid< LeftmostAlwaysVisible; cid++)
            PaintCell(row, paint_x=cid, X[cid]+scrollX, X[cid+1]+scrollX, y, status);
        PerRowPaint(row, x0, y, x1, y+RowHeight);
    }
}


void RowColumnWin::PaintRow(int j)
{
    if (j < 0 or j >= ListSize(Rows))
        return;     // e.g. if a highlighted row is not found.
    paint_rownum = j;
    PaintRow(Rows[j], j*RowHeight + TopHeight, 0, realWidth);
}


void RowColumnWin::PaintRowViaVirtual(int j)
/* Use this if it's possible the user has overridden the Paint() function. */
{   int y=j*RowHeight + TopHeight;

    Paint(0, y, realWidth, y + RowHeight);
}


void RowColumnWin::Calibrate(bool AllowShrinkage)
// This function constructs the X[] array for field positions
// by examining the widest field in each column in the current
// viewport. Set CurViewptOnly to false to conside the whole window when calibrating
// We either allow the fields to suck back, or we
// only allow column widths to increase, depending on AllowShrinkage.
// Return 'yes' if we just did a PaintWhole.
{    
    int j0,j1,y0,y1,i,w,h,N;
    bool something_changed;
    int M[8192];
    kstr s;
    bool CurViewptOnly = !mCalibrateUsingFullWin;

    h = 0;

    // Shortcuts:
    N = ListSize(ColumnIds);
    if (N == 0)
        return;
    if (ListSize(X) <= N) {
        ListIdx(X,N); // One more entry than fields.
        ListIdx(U,N); // Initialises to 0.
    }

    // What's our viewport?
    y0 = scrollY;
    y1 = scrollY + clientHeight;

    if (CurViewptOnly) {
        // What rows are in the current viewport?
        j0 = (y0-TopHeight) / RowHeight;
        j1 = (y1-TopHeight + RowHeight-1) / RowHeight;
        if (j1 >= ListSize(Rows))
            j1 = ListSize(Rows)-1;
        if (j0 < 0)
            j0 = 0;
    }
    else {
        j0 = 0;
        j1 = ListSize(Rows)-1;
    }
    ForExcel = no;

    // Get the maximum width of each column:
    HeadingHeight = 10;
    for (i=0; i < N; i++) {
        char dummy2;
        kstr dummy1;
        s = GetHeading(ColumnIds[i], &dummy1,&dummy2);
        if (s == NULL)
            w = 10;
        else TextDimensions(s, -1, font, &w, &h);
        h += 7;
        if (h > HeadingHeight)
            HeadingHeight = h;
        M[i] = w + ColumnGap;
    }
    RowHeight = TfcFontHeight(font) + 1;
    if (inPrint)
        HeadingHeight = HeadingHeight * 10 / 8;
    if (summary) {
		int summarywidth;
        TextDimensions(summary,-1,font, &summarywidth, &SummaryHeight);
        SummaryHeight = SummaryHeight * 10 / 8;
    }
    else SummaryHeight = 0;
    TopHeight = SummaryHeight + HeadingHeight;
	SetUnscrollableAreas();
    CalcHeight();

    for (int j=j0; j <= j1; j++) {
        for (i=0; i < N; i++) {
            /* Warning: GetField() can (and does) modify paint_rownum;
            e.g. repaints sometimes happen inside GetField() even
            if it's a single-threaded program. */
            paint_x = i;
            w = GetFieldWidth(Rows[paint_rownum=j], ColumnIds[i]);
            w += ColumnGap;
            if (w > M[i])
                M[i] = w;
        }
    }
    something_changed = no;
    for (i=1; i <= N; i++) {
        // kch commenting this out, as I can't see why we shouldn't adjust column widths if users had changed it.
        // They are asking it to be re-calibrated anyway
        if (!mCalibrateAllCols and U[i-1] and BeingDragged==-1 and !inPrint)
            continue;
        int nxx = X[i-1] + M[i-1];
        if (AllowShrinkage ? X[i] != nxx : X[i] < nxx) {
            X[i] = nxx;
            something_changed = yes;
        }
    }
    if (realWidth != X[N]) {
        realWidth = X[N];
        UpdateScrollbars(scrollX, scrollY);
    }

    // If something has changed, then paint the whole window:
    if (something_changed and (ShowState == tfc_shown || ShowState ==tfc_dirtyshouldpaint))
        PaintWhole();
}


int RowColumnWin::GetFieldWidth(void* row, column_id cid)
{   int fgcolor, bgcolor;
    char buf[8192];
    char alignment;
    int w,h;
    const char* s;

	cellfont = font;
    s = GetField(row, cid, buf, ' ', &fgcolor, &bgcolor, &alignment);
    if (s == NULL)
        return FieldWidth(row,cid);
    TextDimensions(s,-1,cellfont,&w,&h);
    return w;
}


void RowColumnWin::DecideOnPageBreakPosX(int x, int *endX, int *startOfNextX)
{   int N = ListSize(ColumnIds);
	int Overlap;

    if (N == 0)
        Overlap = 400;
    else {
        if (LeftmostAlwaysVisible > 0)
            Overlap = X[LeftmostAlwaysVisible];
        else Overlap = X[N] / N;
    }
	*endX = x;
	*startOfNextX = x - Overlap;			// 5% overlap
}


void RowColumnWin::DecideOnPageBreakPosY(int y, int *endY, int *startOfNextY)
{
	*endY = y;
	*startOfNextY = y - HeadingHeight;			// 1% overlap
}


void RowColumnWin::ExpandRightmostColumn()
/* Rightmost column expands to clientWidth. */
{
    if (ColumnIds == NULL or X == NULL)
        return;
    int N = ListSize(ColumnIds);
    if (X[N] < clientWidth)
        X[N] = clientWidth;
}


void RowColumnWin::SetUnscrollableAreas()
{
	setDontScrollMargins(LeftmostAlwaysVisible < ListSize(X) ? X[LeftmostAlwaysVisible] : realWidth, 
					TopHeight);
}


void RowColumnWin::SetLeftmostColumnAlwaysVisible(int set)
/* If set, then it means we never scroll off the leftmost column. */
{
    LeftmostAlwaysVisible = set;
	SetUnscrollableAreas();
}


void RowColumnWin::SetRowJumpers(int set)
{
    RowJumpers = set;
}


void RowColumnWin::Measure(int *xp, int *yp)
{
    if (ListSize(X) == 0)
        *xp = 30;
    else *xp = X[ListSize(X)-1];
    *yp = ListSize(Rows) * RowHeight;
}


void RowColumnWin::CalcHeight()
{
    if (inPrint and HighlightedOnly)
        realHeight = ListSize(HighlightedRows) * RowHeight + TopHeight;
    else realHeight = ListSize(Rows)*RowHeight + TopHeight;
}


void RowColumnWin::CursorPaint()
{   void **row = NULL;
    int i;

    for (each_oeli(row, HighlightedRows)) {
        if (*row < Rows or *row >= Rows + ListSize(Rows))
            continue;
        PaintRowViaVirtual(ListFindP(Rows,row));
    }
    if (GetFocusRow())
        PaintRowViaVirtual(ListFindP(Rows, GetFocusRow()));
}


void RowColumnWin::FocusRowIntoView()
/* Adjust scrollX and scrollY until the focus row comes into view. */
{   int NewScrollY,top,bottom,marginY;

    if (CurrentRowId == -1)
        return;
    NewScrollY = scrollY;
    top = CurrentRowId*RowHeight + TopHeight;
    bottom = top + RowHeight*(RowJumpers+1);
    marginY = TopHeight;
    if (bottom - NewScrollY > clientHeight - marginY)
        NewScrollY = bottom - clientHeight + marginY;
    if (top - NewScrollY < marginY)
        NewScrollY = top - marginY;
    if (NewScrollY < 0)
        NewScrollY = 0;
    UpdateScrollbars(scrollX,NewScrollY);
}


void RowColumnWin::ClearSelection()
{   int tmpId;

    // remove focus
    tmpId = CurrentRowId;
    CurrentRowId = -1, CurrentRow = NULL;
    if (X != NULL)
        PaintRow(tmpId);
    ClearHighlight();
}


void RowColumnWin::ClearHighlight()
{   void** HighlightedCopy, *focusrow = NULL;
    int i;

    if (ListSize(HighlightedRows) <= 5) {
        HighlightedCopy = HighlightedRows;
        HighlightedRows = NULL;
        if (X) {
            for (each_aeli(focusrow,HighlightedCopy))
                PaintRow(ListFindP(Rows,focusrow));
        }
        ListFree(HighlightedCopy);
    }
    else {
        ListFree(HighlightedRows);
        SuppressPaints();
    }
}


bool RowColumnWin::HighlightRow(int i, bool clearprevioushighlight)
/* (aha) This function is primarily to move to AND highlight a row programmatically.
MoveToRow function is not usable for moving to rows programmatically.
It primarily acts like a CTRL+Click process */
{   bool repaintFocused = no;
    void* focusrow = NULL;
    int  j;

    if (ListSize(Rows) == 0)
        return no;

    if (i < 0) { // Wants to clear focus
        CurrentRow = NULL;
        CurrentRowId = -1;
        return yes;
    }
    else if (i >= ListSize(Rows)-1) // if more than what we have, then return no
        return no;

    if (clearprevioushighlight)
        ClearSelection();

    if (!AllowNoMultiSelection) {
        if (isHighlighted(i) and ListSize(HighlightedRows) > 1) {
            /* If the row is already in multiply focus - unfocus it */
            for (each_ael(focusrow,HighlightedRows,j))
                if (focusrow == Rows[i]) {
                    if (focusrow == CurrentRow)
                        // we've removed current row - repaint the new one
                        repaintFocused = yes;
                    break;
                }

            ListDelP(HighlightedRows,Rows[i]);

            CurrentRow = HighlightedRows ? HighlightedRows[ListSize(HighlightedRows)-1] : NULL;
            CurrentRowId = HighlightedRows ? ListFindP(Rows,HighlightedRows[ListSize(HighlightedRows)-1]) : -1;
            PaintRow(i);

            if (repaintFocused)
                PaintRow(CurrentRowId);

            return yes;
        }
    }

    /* Move Current row */
    CurrentRow = Rows[i];
    CurrentRowId = i;
    FocusRowIntoView();
    ListAdd(HighlightedRows, Rows[i]);
    PaintRowViaVirtual(i);
    return yes;
}


bool RowColumnWin::SetRowWithoutRedraw(int i)
{
    if (i < 0) {
        CurrentRow = NULL;
        CurrentRowId = -1;
        return yes;
    }
    else if (i >= ListSize(Rows)-1)
        i = ListSize(Rows)-1;
    if (ListSize(Rows) == 0)
        return no;
    CurrentRow = Rows[i];
    CurrentRowId = i;
    return yes;
}


void RowColumnWin::SetFocusRow(void *row)
{   int oldCurrent=CurrentRowId;

    CurrentRow = row;
    CurrentRowId = ListFindP(Rows, row);
	if (ListSize(HighlightedRows) == 1)
		ClearHighlight();
	if (X == NULL)
		return;
    if (CurrentRowId >= 0)
        PaintRow(CurrentRowId);
    if (oldCurrent)
        PaintRow(oldCurrent);
}


bool RowColumnWin::MoveToRow(int i, bool Highlight)
/* Move to a new row. */
{   bool repaintFocused = no;
    int oldCurrent = -1, j;
    void* focusrow = NULL;

    bool CtrlDown = TfcCtrlStatus(), ShiftDown = TfcShiftStatus();
        
    // Check first that this move is okay:
    if (i < 0) {
        CurrentRow = NULL;
        CurrentRowId = -1;
        return yes;
    }
    else if (i >= ListSize(Rows)-1)
        i = ListSize(Rows)-1;
    if (ListSize(Rows) == 0)
        return no;

    // if there is no multiselection and user clicks on the
    // already focused row - do nothing
    if (AllowNoMultiSelection and CurrentRowId == i and CurrentRow == Rows[i])
        return no;
    oldCurrent = CurrentRowId;
    if (!AllowNoMultiSelection) {
        if (ShiftDown and FirstShifted == -1)
            FirstShifted = CurrentRowId;
        if (not ShiftDown)
            FirstShifted = -1;

        if (not (CtrlDown and FromMousestroke)) {
            /* if Ctrl is not pressed - remove old focus and repaint rows */
            /* note: IFF mouse hasn't called MoveToRow, otherwise we will */
            /* encounter strange behaviour with the CTRL-KEY calls        */
            ClearHighlight();
        }
        else if ((Highlight and not ShiftDown and isHighlighted(i)) and
            (ListSize(HighlightedRows) > 1 or AllowNoSelection)) {
            /* If the row is already in multiply focus - unfocus it */
            for (each_ael(focusrow,HighlightedRows,j))
                if (focusrow == Rows[i]) {
                    if (focusrow == CurrentRow)
                        // we've removed current row - repaint the new one
                        repaintFocused = yes;
                    break;
                }

            ListDelP(HighlightedRows,Rows[i]);

            CurrentRow = HighlightedRows ? HighlightedRows[ListSize(HighlightedRows)-1] : NULL;
            CurrentRowId = HighlightedRows ? ListFindP(Rows,HighlightedRows[ListSize(HighlightedRows)-1]) : -1;
            PaintRow(i);

            if (repaintFocused)
                PaintRow(CurrentRowId);

            return yes;
        }
    }
    else if (CurrentRowId == i)
        return no;
    else
        ClearHighlight();

    /* Move Current row */
    CurrentRow = Rows[i];
    CurrentRowId = i;

    if (FromMousestroke && !mRepaintOnMousestroke)
        return yes;

    // If necessary, scroll until this row is properly in view:
    // Set RowJumpers to 0 to stop the scrollbar moving unless it
    // it is really needed
    FocusRowIntoView();

    /* If Shift was pressed - highlight all rows between first and last */
    if (!AllowNoSelection and ShiftDown) {
        if (FirstShifted == -1)
            FirstShifted = 0;   // This can happen e.g. if the app manually
            // highlights rows.
        for (j=FirstShifted; j!=i; FirstShifted < i ? j++ : j--) {
            if (not isHighlighted(j)) {
                ListAdd(HighlightedRows, Rows[j]);
                if (ShowState == tfc_shown)
                    PaintRow(j);
            }
        }
    }

    /* Highlight new row (if was requested) */
    if (Highlight)
        ListInsP(HighlightedRows, Rows[i]);

    /* Repaint old row */
    if (ShowState == tfc_shown) {
        if (oldCurrent >= 0)
            PaintRowViaVirtual(oldCurrent);
        PaintRowViaVirtual(i);
    }

    return yes;
}


void RowColumnWin::PaintObj(void *obj)
{	int rowIdx;

	rowIdx = ListFindP(Rows, obj);
	if (rowIdx >= 0)
		PaintRow(rowIdx);
}


bool RowColumnWin::Keystroke(int key)
{
    switch (key) {
        case UP:			if (CurrentRowId > 0)
                                return MoveToRow(CurrentRowId - 1,yes);
							else return yes;
        case SHIFT(UP):		
		case CTRL_(UP):     if (CurrentRowId > 0)
                                return MoveToRow(CurrentRowId - 1,no);
                            else return yes;
		
        case DOWN:          return MoveToRow(CurrentRowId + 1,yes);
        case SHIFT(DOWN):		
		case CTRL_(DOWN):   return MoveToRow(CurrentRowId + 1,no);
        case ' ':           return MoveToRow(CurrentRowId,yes);
        case PG_UP:         return MoveToRow(CurrentRowId > 16 ?
                                    CurrentRowId-16 : 0,yes);
        case PG_DOWN:       return MoveToRow(CurrentRowId+16,yes);

        case CTRL_(HOME):
        case HOME:
        case CTRL_(PG_UP):  return MoveToRow(0,yes);

        case END:
        case CTRL_(END):
        case CTRL_(PG_DOWN):return MoveToRow(ListSize(Rows)-1,yes);

        case WINDOW_QUIT:   QuitEventLoop = yes;
                            return yes;
        case 'c':
        case 'C':           Calibrate(yes);
                            return yes;

		case CTRL('A'):		HighlightAll();
							return true;

        default:            return no;
    }
}


int RowColumnWin::XtoCid(int x)
/* Which column are we in?  Returns -1 if 'none'. */
{   int i,N;

    N = ListSize(X)-1;
    for (i=N; i >= 0; i--) {
        if (X[i] < x)
          break;
    }
    if (i == N)
        return -1;
    else return ColumnIds[i];
}


str RowColumnWin::MakePopupName(int cid, char tmp[30], str prefix)
{   kstr name,help;
	str d;
    char status;

    name = GetHeading(cid, &help, &status);
    d = strcpy(tmp, prefix);
    d += strlen(tmp);
    *d++ = ' ';
    *d++ = '"';
    kstr s = name;
    while (*s) {
        *d++ = *s++;
        if (*s == ' ' or *s == '\n' or d > tmp + 25) {
            strcpy(d, "...\"");
            return tmp;
        }
    }
    strcpy(d, "\"");
    return tmp;
}


static int compar_menuitem(TfcGenericMenuItem *A, TfcGenericMenuItem *B)
{
    return stricmp(A->name, B->name);
}


TfcGenericMenuItem *RowColumnWin::ExtraColList(int i)
{   TfcGenericMenuItem *ExtraColList=NULL;
    char status, buf[512];
    kstr name, help;
	str s;

    for (int eid=0; (name=GetHeading(eid,&help,&status)) != NULL; eid++) {
        if (ListHasP(ColumnIds, (void*)eid) or not status)
            continue;
        strcpy(buf, name);
        for (s=buf; *s; s++)
            if (*s == '\n')
                *s++ = ' ';
        ListAdd(ExtraColList, TfcMenuItem(strdup(buf),
                    TfcCallback(&RowColumnWin::AddCol_1int,(eid<<8)|i)));
    }
    if (ListSize(ExtraColList) == 0)
        ListAdd(ExtraColList, TfcMenuItem("(all fields are "
            "already displayed)", 0, TFC_GREYED));
    else ListSort(ExtraColList, compar_menuitem);
    return ExtraColList;
}


TfcGenericMenuItem *RowColumnWin::RightButtonMenu(int x, int y)
{   static char tmp1[30], tmp2[30], tmp3[30];
    TfcGenericMenuItem *List=NULL;
    bool cidflags;
    int i,N,cid;

    // Which column are we in?
    N = ListSize(ColumnIds);
    for (i=N; i >= 0; i--) {
        if (X[i] < x)
          break;
    }
    if (i == N)
        cid = -1;
    else cid = ColumnIds[i];

    // Start building the menu:
    if (i < N and x > (X[i]+X[i+1]) / 2)
        i++;  // Depending on which side of the midpoint
        // of a field they click, the new field will be inserted
        // either before or after this field.
    cidflags = cid < 0 ? 1 : 0;
    MakePopupName(cid, tmp1, "Hide");
    MakePopupName(cid, tmp2, "Sort on");
    MakePopupName(cid, tmp3, "Help on");
    ListAdd(List, TfcMenuItem(tmp1, TfcCallback(&RowColumnWin::HideCol,cid), cidflags));
    ListAdd(List, TfcMenuItem(tmp2, TfcCallback(&RowColumnWin::SortOn,cid), cidflags));
    ListAdd(List, TfcMenuItem(tmp3, TfcCallback(&RowColumnWin::HelpOn,cid), cidflags));
    ListAdd(List, TfcMenuItem("Fit to width", TfcCallback(&RowColumnWin::Calibrate,yes)));
    ListAdd(List, TfcMenuItem("Copy", TfcCallback(&RowColumnWin::Export,(str)"__clipboard__")));
    if (realWidth > clientWidth)
        ListAdd(List, TfcMenuItem("FreezeColumn", TfcCallback(&RowColumnWin::LockCol,cid)));
    ListAdd(List, TfcMenuItem("Export", TfcCallback(&RowColumnWin::Export,(str)NULL)));
    ListAdd(List, TfcMenuItem("Save as *.csv/xml/etc", TfcCallback(&RowColumnWin::SaveAsCsv,(str)NULL)));
    ListAdd(List, TfcMenuItem("Show All Cols", &RowColumnWin::ShowAllColumns));
    ListAdd(List, TfcMenuItem("Restore Default Cols", &RowColumnWin::SetColumnsToDefaultsAndRedraw));
    ListAdd(List, TfcSubMenu("Add...", ExtraColList(i)));
    return List;
}


static void ShowTooltip(RowColumnWin* win)
{
    win->ShowTooltip();
}

void RowColumnWin::ShowTooltip()
{   kstr name,help;
    char status;
    int x,y,i;

    if (tooltip_tim > 0) {
        TfcKillTimer(tooltip_tim);
        tooltip_tim = -1;
        if (tooltip_cid >= 0) {
            y = 34+scrollY;
            i = ListFindP(ColumnIds,(void*)tooltip_cid);
            x = X[i>=0?i:0];
            name = GetHeading(tooltip_cid,&help,&status);
            if (help)
                tooltip_id = PopUpBox(x,y,no,(TfcCallback)0,name,"%s",help);
        }
    }
}


bool RowColumnWin::Mousemove(int x, int y)
{
#if 0
    /* (aha) This is taken out for now... */

    int N, cid, i;
    if (x == _x and y == _y) // mouse didn't move
       return no;
    //if (GetFocus() != _hWnd())
    //    return no;
    _x = x; _y = y;
    if (tooltip_tim > 0)
        TfcKillTimer(tooltip_tim);

    tooltip_tim = -1;

    PopDown(tooltip_id);
    // Which row are we in?
    if (y - scrollY < TopHeight) {
         N = ListSize(ColumnIds);
         for (i=N; i >= 0; i--) {
            if (X and X[i] < x)
               break;
         }
        if (i == N)
            cid = -1;
        else if (i == -1)
            return yes;
        else cid = ColumnIds[i];
        tooltip_cid = cid;
        tooltip_tim = TfcStartTimer(500,TfcCallback(::ShowTooltip, this));
        return yes;
    }

    if (tooltip_tim > 0)
        TfcKillTimer(tooltip_tim);
    tooltip_tim = -1;
    tooltip_id  = -1;

#endif
    return no;
}


#define IS_NEAR(x,A)    x<=A+NEARPIXEL and x>=A-NEARPIXEL

bool RowColumnWin::Mousestroke(int op, int x, int y)
/* Handles a MouseStroke message and processes the request. */
{   int j,i,N,cid;

    HeadingClicked = false;

    // Which column are we in?
    N = ListSize(ColumnIds);
    for (i=N; i >= 0; i--) {
        if (X and X[i] < x)
            break;
    }
    if (i == N)
        cid = -1;
    else if (i == -1)
        return yes;
    else cid = ColumnIds[i];

#if 0
    /* (aha) This is taken out for now... */
    if (tooltip_tim > 0)
        TfcKillTimer(tooltip_tim);
    if (tooltip_id > 0)
        PopDown(tooltip_id);
    tooltip_tim = -1;
    tooltip_id  = -1;
#endif

    // Which row are we in?
    if (y < SummaryHeight)
        return no;
    if (y - scrollY < HeadingHeight or y < TopHeight) {

        // The heading row:
        if (op == MOUSE2_PRESS) {
            int newFieldPos = i;
            if (i < N and x > (X[i]+X[i+1]) / 2)
                newFieldPos++;  // Depending on which side of the midpoint
                // of a field they click, the new field will be inserted
                // either before or after this field.
            bool cidflags = cid < 0 ? 1 : 0;
            PopupMenu(x,y,
                    TfcMenuItem("Hide", TfcCallback(&RowColumnWin::HideCol,cid), cidflags),
                    mAllowSorting? 
                        TfcMenuItem("Sort", TfcCallback(&RowColumnWin::SortOn,cid), cidflags) :
                        TfcMenuItem("", 0),
                    TfcMenuItem("Help", TfcCallback(&RowColumnWin::HelpOn,cid), cidflags),
                    TfcMenuItem("Fit to width", TfcCallback(&RowColumnWin::Calibrate,yes)),
                    realWidth <= clientWidth ? 
                        TfcMenuItem("", 0) :
                    scrollX == 0 && i < LeftmostAlwaysVisible ? 
                        TfcMenuItem("Unfreeze", TfcCallback(&RowColumnWin::LockCol,-1), cidflags) :
                        TfcMenuItem("Freeze", TfcCallback(&RowColumnWin::LockCol,cid), cidflags || scrollX > 0),
                    TfcMenuItem("Copy", TfcCallback(&RowColumnWin::Export,(str)"__clipboard__")),
                    TfcMenuItem("Export", TfcCallback(&RowColumnWin::Export,(str)NULL)),
                    TfcMenuItem("Save as *.csv/xml/etc", TfcCallback(&RowColumnWin::SaveAsCsv,(str)NULL)),
                    TfcMenuItem("Show All Cols", &RowColumnWin::ShowAllColumns),
                    TfcMenuItem("Restore Default Cols", &RowColumnWin::SetColumnsToDefaultsAndRedraw),
                    TfcSubMenu("Add...", ExtraColList(newFieldPos)),
                            NULL);
            HeadingClicked = true;
            return yes;
        }
        else if (op == MOUSE_PRESS) 
        {
            if (i == -1)
                return no;
            HeadingClicked = true;
            if (i != 0 and IS_NEAR(x,X[i])) {
                BeingDragged = ColumnIds[i-1];
                goto RESIZECOL;
            }
            else if (IS_NEAR(x,X[i+1])) {
                BeingDragged = cid;
                RESIZECOL:
                TfcSplitCurOn();
                BeingResized = yes;
                U[i] = yes; // Column being moved
            }
            else {
                BeingDragged = cid;
                BeingResized = no;
            }
        }
        else if (op == MOUSE_DOUBLECLICK) 
        {
            HeadingClicked = true;
            if (mAllowSorting)
                SortOn(cid);
            BeingResized = no;
            BeingDragged = -1;
        }
        else if (op == MOUSE_DRAG) 
        {
            if (i == -1 or BeingDragged == -1)
                return no;
            else if (BeingResized) {
                TfcSplitCurOn();
                ResizeColumn(BeingDragged,x);
            }
            else if (BeingDragged != cid and i < N) {
                MoveColumn(BeingDragged,i, true);
            }
        }
        else if (op == MOUSE_RELEASE) 
        {
            //U[BeingDragged] = no; // Column finished moved
            BeingDragged = -1;
        }
    }
    else {
        // The table body:
        j = (y-TopHeight)/RowHeight;
        if (j > ListSize(Rows)) {
            if (AllowNoSelection)
                ClearSelection();
            return no;
        }
        if ((op == MOUSE_PRESS and j != -1) or (bHighlightOnMouse2 and op == MOUSE2_PRESS and j != -1)){
            FromMousestroke = yes;
            MoveToRow(j, yes);
            FromMousestroke = no;
            return yes;
        }
    }

    return yes;
}


void* RowColumnWin::GetFocusRow()
{
    return (CurrentRowId != -1) ? CurrentRow : NULL;
}


char RowColumnWin::rowStatus(void * row)
{   bool focus, highlight;

    focus = (row and row == GetFocusRow());
    highlight = isHighlighted(row);
    if (focus and highlight)
        return 'X';
    if (focus)
        return 'F';
    if (highlight)
        return 'H';
    return ' ';
}


bool RowColumnWin::isHighlighted(int row)
{   void* jrow = NULL;
    int i;

    if (HighlightedRows == NULL)
        return no;
    if (ListSize(HighlightedRows) == ListSize(Rows))
        return yes;
    for (each_aeli(jrow,HighlightedRows))
        if (jrow == Rows[row])
            return yes;
    return no;
}


bool RowColumnWin::isHighlighted(void* row)
{
    return (ListFindP(HighlightedRows, row) != -1);
}


void** RowColumnWin::GetHighlightedData()
// return data that in focus
{
    return HighlightedRows;
}


void RowColumnWin::LockCol(column_id cid)
{   column_id id = 0;

    if (cid == -1) {
        SetLeftmostColumnAlwaysVisible(0);
        return;
    }

    for (int each_aeli(id, ColumnIds))
        if (id == cid) {
            SetLeftmostColumnAlwaysVisible(i+1);
            PaintWhole();
            break;
        }
}


void RowColumnWin::HideCol(column_id cid)
{   int i,k, delta, N;

    if (ListSize(ColumnIds) == 1)
        return;     // Can't hide the last column.
    k = ListFindP(ColumnIds, (void*)cid);
    if (k < 0)
        return;
    if (X == NULL)
        return;
    delta = X[k+1] - X[k];
    i = k;
    N = ListSize(ColumnIds);
    while (++i <= N)
        X[i] -= delta;
    ListDelN(ColumnIds, k);
    ListDelN(X, k);
    ListDelN(U, k);
    realWidth = X[N];
    UpdateScrollbars(scrollX, scrollY);
    PaintWhole();
}


void RowColumnWin::SortOn(column_id cid)
{
    cid++;
    if (cid == sort[0] or cid == -sort[0])
        sort[0] = -sort[0];
    else {
        sort[2] = sort[1];
        sort[1] = sort[0];
        sort[0] = (DescendingByDefault(cid-1) ? -cid : cid);
    }
    Sort(false);
}





/*--------------------------- Sorting: ---------------------------*/

void RowColumnWin::MakeSortKey(void* row, column_id cid, SortKey *sokey)
{   int fgcolour, bgcolour;
    char buf[1024];
    char alignment;

    number = NoNum;
    sokey->s = GetField(row, cid, buf, ' ', &fgcolour, &bgcolour, &alignment);
    sokey->f = number;
    if (number != NoNum) {
        sokey->s = NULL;
        sokey->strdup_s = no;
    }
    else {
        sokey->strdup_s = (sokey->s >= buf and sokey->s < buf+sizeof(buf));
        if (sokey->strdup_s)
            sokey->s = strdup(sokey->s);
    }
}


int RowColumnWin::CompareSortKey(column_id cid, SortKey *A, SortKey *B)
{   int n1, n2;

    if (A->s and B->s) {
        n1 = atoi(A->s);
        n2 = atoi(B->s);
        if (n1 != n2)
            return n1 - n2;
        else return stricmp(A->s, B->s);
    }
    else return A->f < B->f ? -1 : A->f > B->f ? +1 : 0;
}


void RowColumnWin::FreeSortKey(column_id cid, SortKey *sokey)
{
    if (sokey->strdup_s)
        free((void*)sokey->s);
}


static int compar_sortkeyptr(RowColumnWin::SortKey **Ap, RowColumnWin::SortKey **Bp)
{   RowColumnWin::SortKey *A=*Ap, *B=*Bp;

    return int(A) - int(B);
}


void RowColumnWin::InSituSort(SortKey **AA, int n, int sortcolm)
{   SortKey *tmp, *pivot;
    int a,b,c,cmp;
    column_id cid;
    bool reverse;

    if (n <= 1)
        return;
    pivot = AA[n/2];
    a = b = 0;
    c = n;
    if (sortcolm > 0)
        cid = sortcolm - 1, reverse = no;
    else cid = -1-sortcolm, reverse = yes;
    while (b < c) {
        cmp = CompareSortKey(cid, AA[b], pivot);
        if (cmp == 0)
            b++;
        else if ((cmp < 0) != reverse) {
            if (a < b)
                tmp=AA[a], AA[a]=AA[b], AA[b]=tmp;
            a++;
            b++;
        }
        else {
            c--;
            tmp=AA[c], AA[c]=AA[b], AA[b]=tmp;
        }
    }
    // AA[a..c-1]  equals the pivot.
    InSituSort(AA, a, sortcolm);
    InSituSort(AA+c, n-c, sortcolm);
    if (sortcolm == sort[0]) {
        DoSort(AA, a, c, 1);
        for (int i=a+1; i < c; i++)
            AA[i]->transition &= ~1;
    }
    else if (sortcolm == sort[1]) {
        DoSort(AA, a, c, 2);
        for (int i=a+1; i < c; i++)
            AA[i]->transition &= ~2;
    }
    else {
        /* This quaternary sort gives us stable sorts. */
        qsort(AA+a, c - a, sizeof(*AA), (cmp_fn)compar_sortkeyptr);
        for (int i=a+1; i < c; i++)
            AA[i]->transition &= ~4;
    }
}


void RowColumnWin::DoSort(SortKey **AA, int a, int b, int sortary)
/* Sort A[a..b-1] according to sort[sortcolm]. */
{   column_id cid;

    /* Terminating the recursion: */
    if (b - a <= 1)
        return;
    if (sortary >= 3)
        return;

    /* What column? */
    cid = sort[sortary];
    if (cid == 0)
        return;
    if (cid > 0)
        cid--;
    else cid = -1-cid;

    /* Make the sort-keys: */
    for (int i=a; i < b; i++)
        MakeSortKey(AA[i]->row, cid, AA[i]);

    /* Quicksort: */
    InSituSort(AA+a, b-a, sort[sortary]);
}


void RowColumnWin::Sort(bool bFocusRowIntoView/*=false */)
{   SortKey* A=NULL, **AA=NULL;
    int N;

    SuppressPaints();
    ForExcel = no;
    
    /* Don't allow repeated sort keys: */
    if (sort[1] == sort[0])
        sort[1] = 0;
    if (sort[2] == sort[0] or sort[2] == sort[1])
        sort[2] = 0;

    /* Doing the sort proper: */
    N = ListSize(Rows);
    ListSetSize(A, N);
    ListSetSize(AA, N);
    for (int i=N-1; i >= 0; i--) {
        A[i].row = Rows[i];
        A[i].transition = 15;
        AA[i] = &A[i];
    }
    DoSort(AA, 0, N, 0);
    for (int i=N-1; i >= 0; i--)
        Rows[i] = AA[i]->row;
    ListFree(A);
    ListFree(AA);
    
    CurrentRowId = ListFindP(Rows, CurrentRow);
    // (awa) This is really annoying to most people. It should
    // only sort not focus into view. This is a secondary function, so
    // it is now an option that defaults to false.
    if (bFocusRowIntoView)
        FocusRowIntoView();
}


char* RowColumnWin::SortAndReturnTransitions(bool bFocusRowIntoView/*=false */)
{   SortKey* A=NULL, **AA=NULL;
    str Transitions=NULL;
    int N;

    SuppressPaints();
    ForExcel = no;
    
    /* Don't allow repeated sort keys: */
    if (sort[1] == sort[0])
        sort[1] = 0;
    if (sort[2] == sort[0] or sort[2] == sort[1])
        sort[2] = 0;

    /* Doing the sort proper: */
    N = ListSize(Rows);
    ListSetSize(A, N);
    ListSetSize(AA, N);
    ListSetSize(Transitions, N);
    for (int i=N-1; i >= 0; i--) {
        A[i].row = Rows[i];
        A[i].transition = 15;
        AA[i] = &A[i];
    }
    DoSort(AA, 0, N, 0);
    for (int i=N-1; i >= 0; i--) {
        Rows[i] = AA[i]->row;
        Transitions[i] = AA[i]->transition;
    }
    ListFree(A);
    ListFree(AA);
    
    /* */
    CurrentRowId = ListFindP(Rows, CurrentRow);
    // (awa) This is really annoying to most people. It should
    // only sort not focus into view. This is a secondary function, so
    // it is now an option that defaults to false.
    if (bFocusRowIntoView)
        FocusRowIntoView();
    return Transitions;
}




/*------------------------------------------------------*/

void RowColumnWin::HelpOn(column_id cid)
{   kstr name,help;
    char status;
    int x,y,i;

    y = 34+scrollY;
    i = ListFindP(ColumnIds,(void*)cid);
    x = X[i>=0?i:0];
    name = GetHeading(cid,&help,&status);
	PopUpBox(x,y,no,(TfcCallback)0,name,"%s",help);
}


void RowColumnWin::AddCol_1int(int cid8i)
{
    AddCol(cid8i>>8, cid8i&255);
    Calibrate(yes);
    Resized();
    PaintWhole();
}


void RowColumnWin::AddCol(column_id cid, int i)
/* Add this column at this location. */
{   char dummy2;
    kstr dummy1;
    int w,h,N;

    assert(not ListHasP(ColumnIds, (void*)cid));
    ListInsN(ColumnIds,i,cid);
	if (X) {
		TextDimensions(GetHeading(cid,&dummy1,&dummy2),-1,font,&w,&h);
		w += ColumnGap;
		ListInsN(X,i+1,w);
		ListInsN(U,i+1,0);
		X[++i] += X[i-1];

		N = ListSize(ColumnIds);
		while (++i <= N)
			X[i] += w;

		realWidth = X[N];
		UpdateScrollbars(scrollX, scrollY);
		PaintWhole();
	}
}


void RowColumnWin::ResizeColumn(column_id cid, int x)
{   int curCol, colwidth, k, dx;

    curCol = ListFindP(ColumnIds, (void*)cid);
    assert(curCol != -1);
    dx = x - X[curCol+1];
    colwidth = X[curCol+1] - X[curCol];

    /*if column is to small = do nothing*/
    if (colwidth + dx <  2)
        return;

    /*move all other columns*/
    for (k=curCol+1;k <= ListSize(ColumnIds);k++)
        X[k] += dx;
    realWidth = X[ListSize(ColumnIds)];
    UpdateScrollbars(scrollX, scrollY);
    PaintWhole();
}


void RowColumnWin::MoveColumn(column_id cid, int newpos, bool moveMouse /*=false*/)
{   int colwidth,oldpos,i;

    oldpos = ListFindP(ColumnIds, (void*)cid);
    assert(oldpos != -1 and newpos != -1);
    colwidth = X[oldpos+1] - X[oldpos];
    if (newpos < oldpos) {
        memmove(&ColumnIds[newpos+1],&ColumnIds[newpos],
            (oldpos-newpos)*sizeof(column_id));
        ColumnIds[newpos] = cid;
        memmove(&X[newpos+1],&X[newpos],(oldpos-newpos)*sizeof(X[0]));
        for (i=newpos+1; i <= oldpos; i++)
            X[i] += colwidth;
        U[newpos] = U[oldpos];
        for (i=oldpos; i >=newpos+1; i--)
            U[i+1]=U[i];
    }
    else if (newpos > oldpos) {
        memmove(&ColumnIds[oldpos],&ColumnIds[oldpos+1],
                            (newpos-oldpos)*sizeof(column_id));
        ColumnIds[newpos] = cid;
        memmove(&X[oldpos+1],&X[oldpos+2],(newpos-oldpos)*sizeof(X[0]));
        for (i=oldpos+1; i <= newpos; i++)
            X[i] -= colwidth;
        U[newpos] = U[oldpos];
        for (i=oldpos+1; i <= newpos; i++)
            U[i]=U[i+1];
    }
    else assert(false);
    if (moveMouse) {
        if (scrollY < SummaryHeight)
            SetMousePos((X[newpos] + X[newpos+1])/2, SummaryHeight - scrollY + (HeadingHeight/2));
        else
            SetMousePos((X[newpos] + X[newpos+1])/2, (HeadingHeight/2));
    }
    PaintWhole();
}


static str stristr(str bigs, str smalls)
{
    while (*bigs) {
        if (strbegins(bigs,smalls))
            return bigs;
        bigs++;
    }
    return NULL;
}


void RowColumnWin::Search(str s)
/* Move the focus row to the next matching row. */
{
	char tmp[512], dummych;
	const char* text;
    int i,j,cnt,cid = 0;
    int dummy;
    void *row;

    i = CurrentRowId;
    cnt = ListSize(Rows);
    assert(cnt);
    do {
        if (++i == ListSize(Rows))
            i = 0;
        paint_rownum = i;
        row = Rows[i];

        /* Does this row match? */
        for (each_ael(cid,ColumnIds,j)) {
            text = GetField(row, cid, tmp, ' ', &dummy, &dummy, &dummych);
            if (stristr(const_cast<char*>(text), s))
                goto FOUND;
        }
    } until (--cnt == 0);
    TfcMessage("Search", '!', "No matching row found.");
    return; // Nothing found.

    FOUND:
    MoveToRow(i);
}


void RowColumnWin::Search(bool Continuation)
/* Ask the user for a search string, then call 'Search(s)'. */
/* 'Continuation' is like 'search next'. */
{   int result;

    if (Rows == NULL) {
        TfcMessage("Empty table", 'i', "There are no rows to search.");
        return;
    }
    if (Continuation and *searchstring)
        result = 1;
    else {
        result = DoDialog("Search for...",
                Control(searchstring, sizeof(searchstring), "Search for: ", 20)
                        -
                OkButton() / CancelButton());
    }
    if (result == 1)
        Search(searchstring);
}


void RowColumnWin::HighlightAll()
{
	ListFree(HighlightedRows);
	HighlightedRows = (void**)ListCopy(Rows);
	PaintWhole();
}


RowColumnWin::~RowColumnWin()
{
    ListFree(X);
    ListFree(U);
    ListFree(ColumnIds);
    ListFree(Rows);
}


void RowColumnWin::Resized()
{   int dummy,summarywidth;
    static bool Resizing;
    
    if (Resizing)
        return;
    Resizing = yes;
    if (inPrint) { //clientHeight > 1500) {
        int minheight,maxheight,goodheight;
        GetFontRange(&minheight, &maxheight);
        goodheight = (minheight + maxheight) / 2;
        do {
            font = TfcFindFont(goodheight);
            RowHeight = TfcFontHeight(font) + 1;
            HeadingHeight = RowHeight * 10 / 8;
            if (summary) {
                TextDimensions(summary,-1,font, &summarywidth, &SummaryHeight);
                SummaryHeight = SummaryHeight * 10 / 8;
            }
            else SummaryHeight = summarywidth = 0;
            TopHeight = HeadingHeight + SummaryHeight;
            ColumnGap = RowHeight / 2;
            one = clientHeight / 1000;

            /* Calibrate _all_ rows: */
            dummy = clientHeight;
            clientHeight = TopHeight + (ListSize(Rows)+5)*RowHeight;
            Calibrate(yes);
            clientHeight = dummy;

            /* Did we get it to fit? */
            realWidth = X[ListSize(X)-1];
            if (summarywidth > realWidth)
                realWidth = summarywidth;
            if (realWidth < clientWidth)
                break;
            goodheight -= 2;
        } until (goodheight <= minheight+1);
        CalcHeight();
    }
    else {
            //(jla) This just causes us to lose our font setting ???
            //font = NULL;
            ScrollWin::Resized();
    }
    one = clientHeight / 1000;
    if (one == 0)
        one = 1;
    Resizing=no;
}


RowColumnWin::RowColumnWin(RowColumnWin& orig)
					: ScrollWin(orig)
{
    ColumnIds = (int*)ListCopy(orig.ColumnIds);
    HighlightedRows = (void**)ListCopy(orig.HighlightedRows);
    Rows = (void**)ListCopy(orig.Rows);
    X = (int*)ListCopy(orig.X);
    U = (int*)ListCopy(orig.U);
    sort[0] = orig.sort[0];
    sort[1] = orig.sort[1];
    sort[2] = orig.sort[2];
    RowHeight = orig.RowHeight;
    SummaryHeight = orig.SummaryHeight;
    TopHeight = orig.TopHeight;
    AllowNoSelection = orig.AllowNoSelection;
    summary = orig.summary ? strdup(orig.summary) : NULL;
	summaryFg = BLACK;
    summaryBg = WHITE;
    AllowNoMultiSelection = orig.AllowNoMultiSelection;
    mRepaintOnMousestroke = orig.mRepaintOnMousestroke;
    mCalibrateUsingFullWin = orig.mCalibrateUsingFullWin;
    mAllowSorting = orig.mAllowSorting;
    ColumnGap = orig.ColumnGap;
    CurrentRow = orig.CurrentRow;
    CurrentRowId = orig.CurrentRowId;
    BeingDragged = orig.BeingDragged;
    HighlightedOnly = orig.HighlightedOnly;
    realHeight = orig.realHeight;
    FirstShifted = orig.FirstShifted;
    font = orig.font;
    LeftmostAlwaysVisible = orig.LeftmostAlwaysVisible;
	SetUnscrollableAreas();
}


void RowColumnWin::Print()
{   ScrollWin **List=NULL;

    RowColumnWin *Copy = Clone();
	if (Copy == NULL) {
		TfcMessage("Print", '!', "This type of table can't be printed, sorry.");
		return;
	}
	TfcPrintJob printJob(windowTitle);
	if (not printJob.DlgBox(nullcontrol))
		return;
	printJob.Add(Copy);
	printJob.PreviewAndThenPrint();
}





//----------------------------------------------------------------------------------

#undef interface
#include <ole2.h> // OLE2 Definitions
#include <stdio.h>

// AutoWrap() - Automation helper function...
HRESULT AutoWrap(int autoType, VARIANT *pvResult, IDispatch *pDisp, LPOLESTR ptName, int cArgs...)
{
    if (!pDisp) {
        MessageBox(NULL, "NULL IDispatch passed to AutoWrap()", "Error", 0x10010);
        exit(0);
    }

    // Variables used...
    DISPPARAMS dp = { NULL, NULL, 0, 0 };
    DISPID dispidNamed = DISPID_PROPERTYPUT;
    DISPID dispID;
    HRESULT hr;
    char buf[200];
    char szName[200];

    // Convert down to ANSI
    WideCharToMultiByte(CP_ACP, 0, ptName, -1, szName, 256, NULL, NULL);

    // Get DISPID for name passed...
    hr = pDisp->GetIDsOfNames(IID_NULL, &ptName, 1, LOCALE_USER_DEFAULT, &dispID);
    if (FAILED(hr)) {
        sprintf(buf, "IDispatch::GetIDsOfNames(\"%s\") failed w/err 0x%08lx", szName, hr);
        MessageBox(NULL, buf, "AutoWrap()", 0x10010);
        exit(0);
        return hr;
    }

    // Allocate memory for arguments...
    VARIANT *pArgs = new VARIANT[cArgs+1];
    // Extract arguments...
    va_list marker;
    va_start(marker, cArgs);
    for (int i=0; i<cArgs; i++) {
        pArgs[i] = va_arg(marker, VARIANT);
    }

    // Build DISPPARAMS
    dp.cArgs = cArgs;
    dp.rgvarg = pArgs;

    // Handle special-case for property-puts!
    if (autoType & DISPATCH_PROPERTYPUT) {
        dp.cNamedArgs = 1;
        dp.rgdispidNamedArgs = &dispidNamed;
    }

    // Make the call!
    if (pvResult)
        VariantInit(pvResult);
    hr = pDisp->Invoke(dispID, IID_NULL, LOCALE_SYSTEM_DEFAULT, autoType, &dp, pvResult, NULL, NULL);
    if (FAILED(hr)) {
        sprintf(buf, "IDispatch::Invoke(\"%s\"=%08lx) failed w/err 0x%08lx", szName, dispID, hr);
        MessageBox(NULL, buf, "AutoWrap()", 0x10010);
        exit(0);
        return 0;
    }
    va_end(marker);

    delete [] pArgs;

    return hr;
}


void RowColumnWin::SetSummary(str summary, int fgcolour, int bgcolour)
/* 'summary' is a multi-line string (contains \n's). This is output at the */
/* top of any Excel exports and printouts. */
{   int dummy;

    if (summary == NULL or *summary == '\0') {
        free(this->summary);
        this->summary = NULL;
        SummaryHeight = 0;
    }
    else {
        this->summary = strdup(summary);
        dummy = strlen(summary) - 1;
        if (this->summary[dummy] == '\n')
            this->summary[dummy] = '\0';
        TextDimensions(this->summary,-1,font, &dummy, &SummaryHeight);
    }
    summaryFg = fgcolour;
    summaryBg = bgcolour;
    TopHeight = SummaryHeight + HeadingHeight;
    CalcHeight();
	SetUnscrollableAreas();
}


static bool IsDigit(int ch)
{
    return (ch >= '0' and ch <= '9');
}


static bool IsAlpha(int ch)
{
    return (ch >= 'a' and ch <= 'z') or (ch >= 'A' and ch <= 'Z');
}


static bool LooksLikeDateOrTime(const char* s)
// If something like "MAR-06" is placed into excel.SetValue(), it will be interpreted
// as a date, unless we explicitly set the format.
{
	static str prefix[12] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	if (*s == '\0')
		return no;

	// Is it a number with a leading 0 (other than "0" itself of course)?
	if (*s == '0' and s[1] != '\0') {
		while (IsDigit(*s))
			s++;
		if (*s == '\0')
			return yes;
	}

	if (*s == ':') {
		s++;
		while (IsDigit(*s))
			s++;
		return (*s == '\0');
	}

	if (IsDigit(*s)) {
		const char* t = s;
		while (IsDigit(*t))
			t++;
		if (*t == ' ')
			t++;
		// Is it a 99-99 pattern?
		if (*t == '-') {
			t++;
			if (*t == ' ')
				t++;
			if (IsDigit(*t)) {
				t++;
				while (IsDigit(*t))
					t++;
				return not IsAlpha(*t);
			}
			else return no;
		}
		// Is it a 99MMM pattern?
		for (int i=0; i < 12; i++) {
			if (strbegins(t, prefix[i]))
				return yes;
		}
		return no;
	}
	else {
		// Is it a MMM-99 pattern?
		for (int i=0; i < 12; i++) {
			if (strbegins(s, prefix[i]))
				return yes;
		}
		return no;
	}
}


void RowColumnWin::Export(str filename, void ** headerRows, void ** bodyRows, void ** footerRows)
/* if filename == NULL opens Excel application with data */
/* if filename == "__clipboard__" copies data to clipboard */
/* otherwise saves data as .xls file */
{   
	char currDigit[4], fieldbuf[65536], *s, *t, *l;
    SAFEARRAYBOUND sab[2];
    int fgcolor,bgcolor;
    VARIANT var, arr;
    char alignment;
    int bookRow;
    int summaryRows = 0;
    void **UseRows = NULL;
    void *row = NULL;
    char * origsummary = NULL;
    int i;
    const int nNumRowSpaces = 2; // 1 row after summary + 1 row for header
	int* ExportIds;

	if (false) {
		// Call up the export field selector dialog box:
		if (NULL != mpExportDialog) {
			delete mpExportDialog;
			mpExportDialog = NULL;
		}
		mpExportDialog = new RowColExpDlg(this);
		mpExportDialog->ConstructUI();
		if (mpExportDialog->DisplayDialog() <= 0) // Cancelled
			return;
		ExportIds = mpExportDialog->GetExportOrderColumnIds();
	}
	else {
		// Export all columns without asking:
		ExportIds = (int*)ListCopy(ColumnIds);
	}

    // Create a new row set
    if (headerRows != NULL)
    {
        for ( each_aeli(row, headerRows) )
            ListAdd(UseRows, row);
    }

    if (bodyRows != NULL)
    {
        for ( each_aeli(row, bodyRows) )
            ListAdd(UseRows, row);
    }

    if (footerRows != NULL)
    {
        for ( each_aeli(row, footerRows) )
            ListAdd(UseRows, row);
    }

    // Can't export if there are more than 65535 rows:
    if (ListSize(UseRows) > 65535) {
        TfcMessage("Export to Excel", 'x', "There are too many rows to export\n"
            "(%d rows, and Excel's maximum is 65535). ",
            ListSize(UseRows));
        return;
    }
    else if (ListSize(UseRows) > 10000) {
        if (TfcChoose("Export to Excel", "~Export\0Cancel\0",
                    "You have %d rows. Are you sure you want to export?",
                    ListSize(UseRows)) != 1)
            return;
    }

    // Save default LocalInfo settings
    GetLocaleInfo(MAKELCID(MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),SORT_DEFAULT ),
                LOCALE_ICURRDIGITS ,
                currDigit,
                4);

    // Specify new currency digits after decimal point.
    SetLocaleInfo(MAKELCID(MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),SORT_DEFAULT ),
                LOCALE_ICURRDIGITS ,
                "4");

    // Initialize COM for this thread...
    CoInitialize(NULL);

    // Get CLSID for our server...
    CLSID clsid;
    HRESULT hr = CLSIDFromProgID(L"Excel.Application", &clsid);
    if (FAILED(hr)) {
        if (TfcChoose("Export to Excel", "~Save\0Cancel\0",
            "I couldn't start Excel (perhaps it's not installed). \n\n"
            "Do you want to save the data to a *.csv file?") == 1)
            SaveAsCsv();
        return;
    }

    // Start server and get IDispatch...
    IDispatch *pXlApp;
    hr = CoCreateInstance(clsid, NULL, CLSCTX_LOCAL_SERVER, IID_IDispatch, (void **)&pXlApp);
    if (FAILED(hr)) {
        if (TfcChoose("Export to Excel", "~Save\0Cancel\0",
            "I couldn't start Excel (perhaps it's not installed). \n\n"
            "Do you want to save the data to a *.csv file?") == 1)
            SaveAsCsv();
        return;
    }

    // Turn the cursor into a busy cursor:
    TfcBusyPush();

    /* Make it visible (i.e. app.visible = 1)
    {
        VARIANT x;
        x.vt = VT_I4;
        x.lVal = 1;
        //AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"Visible", 1, x);
    } */

    // Set ReferenceStyle R1C1 (-4150)
    VariantInit(&var);
    var.vt = VT_I4;
    var.lVal = -4150;
    AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"ReferenceStyle", 1, var);

    // Get Workbooks collection
    IDispatch *pXlBooks;
    AutoWrap(DISPATCH_PROPERTYGET, &var, pXlApp, L"Workbooks", 0);
    pXlBooks = var.pdispVal;

    // Call Workbooks.Add() to get a new workbook...
    IDispatch *pXlBook;
    AutoWrap(DISPATCH_PROPERTYGET, &var, pXlBooks, L"Add", 0);
    pXlBook = var.pdispVal;

    {
        // (mdo) Need the number of summary rows before creating the safe array OR need to
        // have a safe array that can be resized. Resizing only works on the least significant
        // dimension (no error given if it fails) but putting the row as the least significant
        // dimension is not possible as the array is 90 degrees rotated and I can't fix that.
        // This hack gets around this until someone with more tolerance for MS symbology
        // comes along.
        s = summary;
        while (s and *s) {
            s = strchr(s, '\n');
            if (s) 
			{
                summaryRows++;
                s++;
            }
        }
    }

    // Create a safearray of variants...
    arr.vt = VT_ARRAY | VT_VARIANT;
    sab[0].lLbound = 1; sab[0].cElements = ListSize(UseRows)+summaryRows+nNumRowSpaces;
    sab[1].lLbound = 1; sab[1].cElements = ListSize(ColumnIds);
    arr.parray = SafeArrayCreate(VT_VARIANT, 2, sab);

    /*
        sab[0].cElements++;
        if ((SafeArrayRedim(arr.parray, sab) != S_OK))
            assert(0);
    }*/

    // Export the summary section:
    bookRow = 1;
    origsummary = summary;
    while (summary and *summary) {
        s = strchr(summary, '\n');
        if (s == NULL)
            strcpy(fieldbuf, summary);
        else {
            memcpy(fieldbuf, summary, s-summary),
            fieldbuf[s-summary] = '\0';
            s++;
        }
        summary = s;
        l = fieldbuf;
        t = strchr(l, '\t');
        int n = 0;
        do {
            if (t)
                *t++ = '\0';            
            int nLen = lstrlen(l);
            BSTR bsRet = SysAllocStringLen(NULL, nLen+1);
            wchar_t *wcell = NULL;
            if (l)
            {
                wcell = new wchar_t [nLen+1];
                Utf8ToWide(l,wcell, (nLen+1)*2 );
                wcscpy(bsRet, wcell);
                delete [] wcell;
            }
            else
                mbstowcs(bsRet, l, nLen+1);
            var.vt = VT_BSTR;
            var.bstrVal = bsRet;
            long indicesName[] = {bookRow,n+1};
            SafeArrayPutElement(arr.parray, indicesName, (void*)&var);
            l = t;
            if (l)
                t = strchr(l, '\t');
            n++;
        } while (l);
        bookRow++;
    }
    summary = origsummary;

    
    // Export the column headings:
    ForExcel = yes;         // Instruct the application not to use formatting.
    //int N = ListSize(ColumnIds);
    int N = ListSize(ExportIds);
    for (int n=0; n < N; n++) 
    {
        char dummy2;
        kstr dummy1;
        //char* name = GetHeading(ColumnIds[n],&dummy1,&dummy2);
        kstr name = GetHeading(ExportIds[n],&dummy1,&dummy2);
        int nameLen = lstrlen(name);
        BSTR bsName = SysAllocStringLen(NULL, nameLen);
        mbstowcs(bsName, name, nameLen);
		for (wchar_t *s=bsName; *s; s++)
			if (*s == '\n')
				*s = ' ';
        var.vt = VT_BSTR;
        var.bstrVal = bsName;
        long indicesName[] = {bookRow,n+1};
        SafeArrayPutElement(arr.parray, indicesName, (void*)&var);
    }
    bookRow++;

    // Export the cells:
    for (int i=0; i < ListSize(UseRows); i++) 
    {
        paint_rownum = i;
        ForExcel = yes;         // Instruct the application not to use formatting.
        //for (int j=0; j < ListSize(ColumnIds); j++) 
        for (int j=0; j < ListSize(ExportIds); j++) 
        {
            ForExcel = yes;         // Instruct the application not to use formatting.
            // Create entry value for (i,j)
            number = NoNum;
            exceltype = '?';
            paint_x = j;
            char* cell = const_cast<char*>(GetField(UseRows[i], ExportIds[j], fieldbuf, 'F', &fgcolor, &bgcolor, &alignment));                
            if (number != NoNum and exceltype != 't') 
            {
                if (exceltype == '$' or
                        (exceltype == '?' and cell and *cell == '$')) 
                {
                   //var.vt = VT_CY;
                    var.vt = VT_R8;
                    //var.cyVal.int64 = number * 10000 + 0.5;
                    var.dblVal = number;
                }
                else if (exceltype == 'd') 
                {
                    var.vt = VT_DATE;
                    var.dblVal = number;        // days since 1/1/1900
                }
                else 
                {
                    var.vt = VT_R8;
                    var.dblVal = number;
                }
            }
            else 
            {
                if (!cell)
                    cell = "";
                while (*cell == ' ')
                    cell++;
                if (*cell == '$') 
                {
                    // If you want to export a currency value to Excel,
                    // then the 'ForExcel' version of GetField() should
                    // return a string starting with '$' and a '.'
                    // to denote decimals, with no other punctuation.

                    //++ (jla) VFIVE-346 Export as a double so currency symbol not exported --
                    var.vt = VT_R8;
                    //var.vt = VT_CY;
                    cell++;
                }
                else if (*cell == '\0')
                    var.vt = VT_VOID;
                else var.vt = VT_R8;

                const char* s = cell;
                if (*s == '-')
                {
                    //++ (jla) Some fields in report use '-' to denote its empty, so if it's only '-' allow it --
                    if (*(s + 1) == '\0')
                        var.vt = VT_BSTR;
                    else
                        s++;
                }
                if (*s == '=')
                    goto STRINGISE;  //Otherwise Excel will think it's
                    // a formula.
                for ( ; *s; s++) {
                    if (s - cell >= 19) {
                        STRINGISE:
                        var.vt = VT_BSTR;
                        memmove(fieldbuf+1, cell, strlen(cell)+1);
                        fieldbuf[0] = '"'; // VFIVE-1250
                        cell = fieldbuf;
                        // This special case is for the 20-digit Oslo
                        // order ID's.  They look like numbers but we'll
                        // lose accuracy if we treat them as numbers,
                        // instead they must be treated as strings.
                        break;
                    }
                    //else if (isdigit(*s))  // u can't use this function with Unicode chars
                    else if (IsUTF8Digit(*s))
                        continue;
					else if (*s == '.' and strchr(s+1, '.') == NULL) {
                        var.vt = VT_R8;
                    }
                    else {
                        var.vt = VT_BSTR;
                        break;
                    }
                }
                if (var.vt == VT_I4) {
                    var.lVal = atol(cell);
                    if (var.lVal == NoNum)
                        var.vt = VT_VOID;
                }
                else if (var.vt == VT_R8) {
                    var.dblVal = atof(cell);
                    if (var.dblVal == NoNum)
                        var.vt = VT_VOID;
                }
                else if (var.vt == VT_CY) {
                    var.cyVal.int64 = atof(cell) * 10000;
                    if (atof(cell) == NoNum)
                        var.vt = VT_VOID;
                }
                else if (LooksLikeDateOrTime(cell)) {
                    //++ (jla) Get around excel auto formatting strings to dates by prefixing space --
                    if (strlen(cell) > 910) {
                        cell[907] = '\0';
                        strcat(cell, "...");
                    }
					int nLen = lstrlen(cell) + 1;
                    BSTR bsRet = SysAllocStringLen(NULL, nLen+1);
                    wchar_t *wcell = ToWideStrdup(cell);
                    wcscpy(bsRet+1, wcell);
                    delete [] wcell;
                    bsRet[0] = ' ';
                    var.vt = VT_BSTR;
                    var.bstrVal = bsRet;
                }
                else {
                    if (strlen(cell) > 910) {
                        cell[907] = '\0';
                        strcat(cell, "...");
                    }
					int nLen = lstrlen(cell) + 1;
                    BSTR bsRet = SysAllocStringLen(NULL, nLen+1);
                    wchar_t *wcell = ToWideStrdup(cell);
                    wcscpy(bsRet, wcell);
                    delete [] wcell;
                    var.vt = VT_BSTR;
                    var.bstrVal = bsRet;
                }
            }

            // Add to safearray...
            long indices[] = {bookRow,j+1};
            SafeArrayPutElement(arr.parray, indices, (void*)&var);
        }
        bookRow++;
        if ((i&127) == 0 or (i == ListSize(UseRows) and i > 127))
            TfcWaitBox("%d/%d rows    ", i, ListSize(UseRows));
    }
    ForExcel = no;

    // Get ActiveSheet object
    IDispatch *pXlSheet;
    AutoWrap(DISPATCH_PROPERTYGET, &var, pXlApp, L"ActiveSheet", 0);
    pXlSheet = var.pdispVal;

    // Get Range object for the Range A1:O15...
    IDispatch *pXlRange;
    char* s0 = "AABCDEFGHIJKLMNOPQRSTUVWXYZ";
    /*if (ListSize(ColumnIds)<27)
        sprintf(fieldbuf,"A1:%c%d", s0[ListSize(ColumnIds)], ListSize(UseRows)+nNumRowSpaces+summaryRows);
    else 
    {
        int firstchar=0;
        int secondchar=ListSize(ColumnIds);
        while (secondchar>26) 
        {
            firstchar++;
            secondchar-=26;
        }
        sprintf(fieldbuf,"A1:%c%c%d", s0[firstchar], s0[secondchar], ListSize(UseRows)+nNumRowSpaces+summaryRows);
    }*/

    if (ListSize(ExportIds)<27)
        sprintf(fieldbuf,"A1:%c%d", s0[ListSize(ExportIds)], ListSize(UseRows)+nNumRowSpaces+summaryRows);
    else 
    {
        int firstchar=0;
        int secondchar=ListSize(ExportIds);
        while (secondchar>26) 
        {
            firstchar++;
            secondchar-=26;
        }
        sprintf(fieldbuf,"A1:%c%c%d", s0[firstchar], s0[secondchar], ListSize(UseRows)+nNumRowSpaces+summaryRows);
    }

    BSTR bsRet = SysAllocStringLen(NULL, lstrlen(fieldbuf));
    mbstowcs(bsRet, fieldbuf, lstrlen(fieldbuf));

    VARIANT parm;
    parm.vt = VT_BSTR;
    parm.bstrVal = bsRet;
    AutoWrap(DISPATCH_PROPERTYGET, &var, pXlSheet, L"Range", 1, parm);
    VariantClear(&parm);
    pXlRange = var.pdispVal;

    // Set range with our safearray...
    AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlRange, L"Value", 1, arr);

    // VFIVE-1376: resize the row height just in case the string contains \n
    var.vt = VT_I4;
    var.lVal = 12;
    AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlRange, L"RowHeight", 1, var);

    // Set ReferenceStyle A1 (1)
    var.vt = VT_I4;
    var.lVal = 1;
    AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"ReferenceStyle", 1, var);

    if (filename == NULL) {
        // Make it visible (i.e. app.visible = 1)
        var.vt = VT_I4;
        var.lVal = 1;
        AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"Visible", 1, var);
    }
    else if (streq(filename,"__clipboard__")) {
        var.vt = VT_BOOL;
        var.boolVal = 0;
        AutoWrap(DISPATCH_METHOD, NULL, pXlRange, L"Copy", 0);                // Copy range to clipboard
        AutoWrap(DISPATCH_PROPERTYPUT, NULL, pXlApp, L"DisplayAlerts", 1,var);// Turn alerts off
        AutoWrap(DISPATCH_METHOD, NULL, pXlApp, L"Quit", 0);                  // quit an application
    }
    else {
        // Save workbook to filename
        SetFileAttributes(filename, FILE_ATTRIBUTE_NORMAL);
        remove(filename);
        BSTR bsRet = SysAllocStringLen(NULL, lstrlen(filename));
        mbstowcs(bsRet, filename, lstrlen(filename));
        var.vt = VT_BSTR;
        var.bstrVal = bsRet;
        AutoWrap(DISPATCH_METHOD, NULL, pXlBook, L"SaveAs", 1, var);

        // Tell Excel to quit (i.e. App.Quit)
        AutoWrap(DISPATCH_METHOD, NULL, pXlApp, L"Quit", 0);
    }

    ListFree(UseRows);

    // Release references...
    pXlRange->Release();
    pXlSheet->Release();
    pXlBook->Release();
    pXlBooks->Release();
    pXlApp->Release();
    VariantClear(&arr);

    // Restore local settings
    SetLocaleInfo(MAKELCID(MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),SORT_DEFAULT),
                LOCALE_ICURRDIGITS,
                currDigit);


    // Uninitialize COM for this thread...
    CoUninitialize();
    TfcBusyPop();
    TfcWaitBox(NULL);
}


static int FileExists(char *fileName)
{   FILE *infile = fopen(fileName, "r");
    if (infile == NULL)
        return 0;
    fclose(infile);
    return 1;
}


void RowColumnWin::SaveAsCsv(str givenfilename)
/* NULL=ask the user what filename to use.  The separator used    */
/* will be derived from the filename: csv=','  sdv=';'  txt='\t', */
/* 'xml'=xml format.  Displays a 'was successful' dialog box.     */

/* The decimal point will be ',' in sdv files and '.' for all     */
/* other types. This gives the user control when selecting the    */
/* format, easing the process when an export should be sent from  */
/* a computer where the decimal point is ',' to a computer where  */
/* the decimal point is '.'. Furthermore we don't assume that     */
/* applications that will receive the data follows the computer   */
/* settings for decimal points.                                   */
{   char tmp[512], tmp2[80], separator;
    TfcRegistryKey RegKey;
    str ext,s,d;
    str filename;

    if (not givenfilename) {
        d = tmp;
        if (not windowTitle)
            *tmp = '\0';
        else {
            for (s=windowTitle; *s; s++) {
                if (*s == '/')
                    *d++ = '-';
                else if (*s >= ' ' and not strchr(":\\*?;\"", *s))
                    *d++ = *s;
                if (d - tmp > 45) {
                    d += sprintf(d, "...");
                    break;
                }
            }
            *d = '\0';
        }
        RegKey = TfcRegistryKey(yes, TFC_CURRENT_USER, "Software", "Tfc", NULL);
        ext = RegKey.Subkey(no,"DumpExtension").Get(tmp2, sizeof(tmp2));
        if (ext == NULL)
            ext = "txt";
		else
			*ext = tolower(*ext);

        str filter;
        
        if (*ext == 's')
            filter = "Semicolon-separated values\0*.sdv\0"
                        "Tab-separated values\0*.txt\0"
                        "Comma-separated values\0*.csv\0"
                        "XML\0*.xml\0";
        else if (*ext == 'c')
            filter =    "Comma-separated values\0*.csv\0"
                        "Tab-separated values\0*.txt\0"
                        "XML\0*.xml\0"
                        "Semicolon-separated values\0*.sdv\0";
        else if (*ext == 'x')
            filter =    "XML\0*.xml\0"
                        "Comma-separated values\0*.csv\0"
                        "Tab-separated values\0*.txt\0"
                        "Semicolon-separated values\0*.sdv\0";
        else filter =   "Tab-separated values\0*.txt\0"
                        "Comma-separated values\0*.csv\0"
                        "XML\0*.xml\0"
                        "Semicolon-separated values\0*.sdv\0";

        do {
            filename = TfcSelectFilename(yes, tmp, sizeof(tmp),
                filter, ext);   // <-- Unfortunately, this last parameter
                        // doesn't seem to work, hence we do the clumsy
                        // stuff above.
            if (not filename)
                return;
            if (FileExists(filename)) {
                if (DoDialog(
                    "File Exists",
                    StaticText("The file already exists. Overwrite?")
                                        -
                            (OkButton() | CancelButton())) == -1)
                    continue;
            }
            break;
        } forever;
    }
    else
        filename = givenfilename;
    ext = strrchr(filename, '.');
    separator = '\t';
    if (ext) {
        ext++;
        RegKey.Subkey(yes,"DumpExtension").Set(ext);
        separator = (*ext == 'c') ? ',' : (*ext == 's') ? ';' :
            (*ext == 'x') ? 'X' : '\t';
    }
    if (SaveAsCsvWithSeparator(separator, filename)) {
        if (givenfilename == NULL)
            TfcMessage(filename, 'i', "Saved to:\n\n%s", filename);
    }
}


static int GetNumDec(double f)
{
    if (fmod(floor(f*1000000+0.5), 10.0))
        return 6;
    else if (fmod(floor(f*100000+0.5), 10.0))
        return 5;
    else if (fmod(floor(f*10000+0.5), 10.0))
        return 4;
    else if (fmod(floor(f*1000+0.5), 10.0))
        return 3;
    else if (fmod(floor(f*100+0.5), 10.0))
        return 2;
    else if (fmod(floor(f*10+0.5), 10.0))
        return 1;
    else return 0;
}


static str SafeXML_tag(const char* s, str d)
/* We allow s==d. */
{   str d0=d;

    while (not IsUTF8AlphaNum(*s))
        s++;
    if (*s == '\0')
        *d++ = 'X';
    for (; *s; s++) {
        if (IsUTF8Alpha(*s))
            *d++ = *s;
        else if (IsUTF8Digit(*s)) {
            if (d > d0)
                *d++ = *s;
        }
        else if (*s == ' ')
            *d++ = '_';
        // What punctuation chars are allowed?
        else if (strchr("-_", *s))
            *d++ = *s;
    }
    *d = '\0';
    return d0;
}


static str SafeXML_value(const char* s, str d)
/* We allow s==d. */
{   str d0=d;

    for (; *s; s++) {
        if (*s == '&')
            strcpy(d, "&amp;"), d += 5;
        else if (*s < ' ' or *s >= 127)
            *d++ = '?';
        else
            *d++ = *s;
    }
    *d = '\0';
    return d0;
}


// As sprintf, but replaces '.' with ',' if the separator is ';'
static char *FormattedSPrintF(char *buf, int sizeofbuf, char separator, const char * fmt, ...)
{

    str s = buf;
    va_list argptr;
    va_start(argptr, fmt);
    vsnprintf(s, sizeofbuf, fmt, argptr);
    va_end(argptr);

    // Change the decimal point
    if (separator == ';') {
        while (*s) {
            if (*s == '.')
                *s = ',';
            ++s;
        }
    }

    return buf;
}


static void OutputToProperFormat(FILE *output, char *str, bool useUnicode)
{
    if (!useUnicode) {
        fputs(str, output);
    } else {
        wchar_t *owbuffer = (wchar_t *) malloc(sizeof(wchar_t) * (strlen(str) + 1));
        Utf8ToWide(str, owbuffer, strlen(str) + 1);
        fputws(owbuffer, output);
        free(owbuffer);
    }
}


bool RowColumnWin::SaveAsCsvWithSeparator(char separator, char* filename, bool useUnicode, bool printRaw)
{   char buf[65536], buf2[65536], colname[256], tmp2[256], alignment, status;
    int j,fgcolor,bgcolor,cid;
    FILE* output;
	kstr help;
    kstr s;

    if (useUnicode)
        output = fopen(filename, "w+b");
    else
        output = fopen(filename,"w+");
    if (not output) { // we can't open file for some reason (for example, it's already opened in Excel...
        TfcMessage("Error",'-', "Can't create file: %s\n\nReason: %s\n"
            "(Could it be open in another program?)",
                    filename, strerror(errno));
        return no;
    }

    // Output unicode header
    if (useUnicode)
        fprintf(output, "%c%c", 0xff, 0xfe);

    // Output the summary:
    if (summary and separator != 'X') {
        sprintf(buf, "%s\n", summary);
        OutputToProperFormat(output, buf, useUnicode);
    }

    // Outputting proper
    ForExcel = yes;         // Instruct the application not to use formatting.
    if (separator == 'X') {
        OutputToProperFormat(output, "<xml>\n", useUnicode);
        for (paint_rownum=0; paint_rownum < ListSize(Rows); paint_rownum++) {
            OutputToProperFormat(output, "<row>", useUnicode);
            for (j=0; j < ListSize(ColumnIds); j++) {
                cid = ColumnIds[j];
                s = GetHeading(cid,&help,&status);
                SafeXML_tag(s, colname);
                if (strieq(colname, "Row"))
                    strcpy(colname, "RowId");
                number = NoNum;
                paint_x = j;
                s = GetField(Rows[paint_rownum], cid, buf, 'F',
                            &fgcolor, &bgcolor, &alignment);
                if (*s == '\0')
                    continue;
                if (number == NoNum) {
                    sprintf(buf2, "<%s>%s</%s>",
                                    colname, SafeXML_value(s, tmp2), colname);
                } else {
                    sprintf(buf2, "<%s>%0.*lf</%s>", colname,
                                    GetNumDec(number), number, colname);
                }
                OutputToProperFormat(output, buf2, useUnicode);
                
            }
            OutputToProperFormat(output, "</row>\n", useUnicode);
        }
        OutputToProperFormat(output, "</xml>\n", useUnicode);
    }
    else {
        for (j=0; j < ListSize(ColumnIds); j++) {
            cid = ColumnIds[j];
            strcpy(colname, GetHeading(cid,&help,&status));
            for (str s=colname; *s; s++)
                if (*s == '\n' or *s == separator)
                    *s = ' ';
            sprintf(buf, "%s%c", colname, separator);
            OutputToProperFormat(output, buf, useUnicode);
        }
        OutputToProperFormat(output, "\n", useUnicode);
        for (paint_rownum=0; paint_rownum < ListSize(Rows); paint_rownum++) {
            for (j=0; j < ListSize(ColumnIds); j++) {
                number = NoNum;
                paint_x = j;
                s = const_cast<char*>(GetField(Rows[paint_rownum], ColumnIds[j], buf, 'F',
                                    &fgcolor, &bgcolor, &alignment));
                if (s == NULL)
                    s = "";
                char sep = (j < ListSize(ColumnIds)-1) ? separator : '\n';
                if (number == NoNum or exceltype == 'd' or printRaw) {
                    if (strchr(s, ' ') or strchr(s, separator))
                        sprintf(buf2,"\"%s\"%c", s, sep);
                    else
                        sprintf(buf2,"%s%c", s, sep);
                }
                else {
                    FormattedSPrintF(buf2, sizeof(buf2), separator, "%0.*lf%c",
                                    GetNumDec(number), number, sep);
                }
                OutputToProperFormat(output, buf2, useUnicode);
            }
        }
    }

    // Finishing up:
    ForExcel = no;         // Instruct the application to use formatting.
    fclose(output);
    return yes;
}


static void PaintRowIfRequired(RowColumnWin *win, void *row)
{
	int rowNum = win->RowToI(row);
	int y = win->TopHeight + rowNum * win->RowHeight;
	if (y < win->scrollY + win->clientHeight && y + win->RowHeight > win->scrollY)
		win->PaintRow(rowNum);
}



/*---------------------- RowColExpDlg: ------------------*/

RowColExpDlg::RowColExpDlg(RowColumnWin* win) : pColWin(win)
{   char status;
    kstr help;
    int n;

    mUnselectedHeadings = NULL;
    mSelectedHeadings = NULL;
    mUnselected = NULL;
    mSelected = NULL;
    mExportOrderColIds = NULL;
    int N = ListSize(win->ColumnIds);
    for (column_id cid=0; kstr heading = win->GetHeading(cid,&help,&status); cid++) {
        for (n=0; n < N; n++) {
            if (win->ColumnIds[n] == cid)
                break;
        }
        if (n == N) {
            ListAdd(mUnselectedHeadings, strdup(heading));   
            ListAdd(mUnselected, false);
        }
    }
    
    for (int n=0; n < N; n++) {
        kstr heading = win->GetHeading(win->ColumnIds[n],&help,&status);
        ListAdd(mSelectedHeadings, strdup(heading));
        ListAdd(mSelected, false);
    }
}


RowColExpDlg::~RowColExpDlg()
{
    ListFreeFree(mUnselectedHeadings, false);
    ListFreeFree(mSelectedHeadings, false);
    ListFree(mUnselected);
    ListFree(mSelected);
    ListFree(mExportOrderColIds);
}

void RowColExpDlg::ConstructUI()
{

    mUnselectedList = SetControl(mUnselected, mUnselectedHeadings, ListSize(mUnselectedHeadings), NULL, NULL, true);
    mSelectedList = SetControl(mSelected, mSelectedHeadings, ListSize(mSelectedHeadings), NULL, NULL, true);
    control cUnselectButton = Button("<-", TfcCallback(RowColExpDlg::UnselectClicked, this));
    control cSelectButton = Button("->", TfcCallback(RowColExpDlg::SelectClicked, this));
    control cUpButton = Button("Up", TfcCallback(RowColExpDlg::UpClicked, this));
    control cDownButton = Button("Down", TfcCallback(RowColExpDlg::DownClicked, this));
    mExportButton = Button("Export", TfcCallback(RowColExpDlg::ExportClicked, this));
    control cStatic1 = StaticText("Not Exported");
    control cStatic2 = StaticText("Exported");
    mMainControl = ((cStatic1 -  mUnselectedList) - mExportButton) |
                   (FillerControl(0, 30) - cUnselectButton - FillerControl(0, 5) - cSelectButton) |
                   ((cStatic2 - mSelectedList) - (cUpButton | cDownButton));
}

int RowColExpDlg::DisplayDialog()
{
    ListFree(mExportOrderColIds);
    mExportOrderColIds = NULL;
    int rv = DoDialog("Select Columns To Export", mMainControl, mExportButton, NULL, TFC_DLG_CHILD);
    return rv;
}

void RowColExpDlg::SelectButtonClicked()
{
    int nArraySize = ListSize(mUnselectedHeadings);
    for (int i = 0; i < nArraySize; i++)
    {
        if (mUnselected[i])
        {
            ListAdd(mSelectedHeadings, strdup(mUnselectedHeadings[i]));
            int nArray2Size = ListSize(mSelectedHeadings);
            for (int j = 0; j < nArray2Size; j++)
                mSelected[j] = false;
            ListAdd(mSelected, true);

            free(mUnselectedHeadings[i]);
            ListDelN(mUnselectedHeadings, i);
            ListDelN(mUnselected, i);
            mUnselectedList.SetList(mUnselectedHeadings, mUnselected);
            mSelectedList.SetList(mSelectedHeadings, mSelected);
            break;
        }
    }
}

void RowColExpDlg::UnselectButtonClicked()
{
    int nArraySize = ListSize(mSelectedHeadings);
    for (int i = 0; i < nArraySize; i++)
    {
        if (mSelected[i])
        {
            ListAdd(mUnselectedHeadings, strdup(mSelectedHeadings[i]));
            int nArray2Size = ListSize(mUnselectedHeadings);
            for (int j = 0; j < nArray2Size; j++)
                mUnselected[j] = false;
            ListAdd(mUnselected, true);

            free(mSelectedHeadings[i]);
            ListDelN(mSelectedHeadings, i);
            ListDelN(mSelected, i);
            mUnselectedList.SetList(mUnselectedHeadings, mUnselected);
            mSelectedList.SetList(mSelectedHeadings, mSelected);
            break;
        }
    }
}

void RowColExpDlg::UpButtonClicked()
{
    int nArraySize = ListSize(mSelectedHeadings);
    for (int i = 1; i < nArraySize; i++)
    {
        if (mSelected[i])
        {
            str temp = mSelectedHeadings[i-1];
            mSelectedHeadings[i-1] = mSelectedHeadings[i];
            mSelectedHeadings[i] = temp;
            mSelected[i-1] = true;
            mSelected[i] = false;
            mSelectedList.SetList(mSelectedHeadings, mSelected);
            break;
        }
    }
}
        
void RowColExpDlg::DownButtonClicked()
{
    int nMax = ListSize(mSelectedHeadings) - 1;
    for (int i = 0; i < nMax; i++)
    {
        if (mSelected[i])
        {
            str temp = mSelectedHeadings[i+1];
            mSelectedHeadings[i+1] = mSelectedHeadings[i];
            mSelectedHeadings[i] = temp;
            mSelected[i+1] = true;
            mSelected[i] = false;
            mSelectedList.SetList(mSelectedHeadings, mSelected);
            break;
        }
    }
}

void RowColExpDlg::ExportButtonClicked()
{
    int nArraySize = ListSize(mSelectedHeadings);
    for (int i = 0; i < nArraySize; i++) {
        str pSelected = mSelectedHeadings[i];

        kstr help;
        char status;
        for (column_id cid=0; kstr heading = pColWin->GetHeading(cid,&help,&status); cid++) {
            if (strieq(heading, pSelected)) {
                ListAdd(mExportOrderColIds, cid);
                break;
            }           
        }       

    }
   
    TfcEndDialog(1);
}

void RowColExpDlg::SelectClicked(RowColExpDlg* dlg)
{
    dlg->SelectButtonClicked();
}
       
void RowColExpDlg::UnselectClicked(RowColExpDlg* dlg)
{
    dlg->UnselectButtonClicked();
}

void RowColExpDlg::UpClicked(RowColExpDlg* dlg)
{
    dlg->UpButtonClicked();
}

void RowColExpDlg::DownClicked(RowColExpDlg* dlg)
{
    dlg->DownButtonClicked();
}

void RowColExpDlg::ExportClicked(RowColExpDlg* dlg)
{
    dlg->ExportButtonClicked();
}

