(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

UNSAFE MODULE Main;
IMPORT
  QOS, Data, Store, Code, Format, Value, Exec;

(* Runs a Quest Machine standalone executable file (boot file).
   The first program parameter is a search path for the Quest Machine and
   for the boot file.
   The second program parameter is the name of the boot file.
   The boot file should contain a function of no arguments which
   terminates by raising an exception. (If normal termination happens,
   via a Return instruction, there is nothing to return to and the
   machine crashes.) *)

VAR searchPath: Value.T;

PROCEDURE Search(fileName: Value.T; VAR (*out*) str: Value.T)
    : BOOLEAN =
  (* -- various Value.T may be invalidated by GC. *)
  CONST PathBuffSize = 256;
  VAR dirName, dName: Value.T;
    path: ARRAY[0..PathBuffSize-1] OF CHAR;
    length, dirNameStart, dirNameLength: Data.Int;
  BEGIN
    IF NOT (QOS.Lookup(LOOPHOLE(Store.heapAdr, ADDRESS), 
	  Value.ValStringStart(searchPath),
	  Value.ValStringLength(searchPath),
	  ADR(path[0]), 0, PathBuffSize, (*out*) length))
    THEN 
      RETURN FALSE
    END;
    IF NOT(QOS.FindAlongPath(ADR(path[0]), 0, length, 
          QOS.dirSeparator, QOS.pathSeparator, 
	  LOOPHOLE(Store.heapAdr, ADDRESS), Value.ValStringStart(fileName), 
	  Value.ValStringLength(fileName),
	  (*out*)dirNameStart, (*out*) dirNameLength))
    THEN 
      RETURN FALSE 
    END;
    IF NOT Value.ValStringOfBuff(ADR(path[0]), 
	dirNameStart, dirNameLength, (*out*)dirName) 
    THEN RETURN FALSE END;
    IF Value.ValStringIsEmpty(dirName) THEN
      dName := dirName;
    ELSE
      dName := Value.ValStringCat(dirName, Value.ValStringOfChar(QOS.dirSeparator));
    END;
    str := Value.ValStringCat(dName, fileName);
    RETURN TRUE;
  END Search;
  
PROCEDURE OpenRead(fileName: Value.T; VAR (*out*) rd: QOS.Reader)
    : BOOLEAN =
  VAR str: Value.T;
  BEGIN
    IF NOT Search(fileName, (*out*)str) THEN RETURN FALSE END;
    RETURN QOS.OpenRead(LOOPHOLE(Store.heapAdr, ADDRESS),
	  Value.ValStringStart(str),
	  Value.ValStringLength(str), 
	  (*out*) rd);
  END OpenRead;

(*
PROCEDURE OpenWrite(fileName: Value.T; VAR (*out*) wr: QOS.Writer)
    : BOOLEAN =
  VAR str: Value.T;
  BEGIN
    IF NOT Search(fileName, (*out*)str) THEN RETURN FALSE END;
    RETURN QOS.OpenWrite(LOOPHOLE(Store.heapAdr, ADDRESS),
	  Value.ValStringStart(str),
	  Value.ValStringLength(str), 
	  (*out*) wr);
  END OpenWrite;
*)

PROCEDURE BadFile(searchPath, fileName: Value.T) =
  BEGIN
      QOS.OutString(QOS.stdout, ": \'");
      IF QOS.PutSub(QOS.stdout, LOOPHOLE(Store.heapAdr, ADDRESS),
	Value.ValStringStart(fileName), 
 	Value.ValStringLength(fileName)) THEN END;
      QOS.OutString(QOS.stdout, "\' (along path \'");
      IF QOS.PutSub(QOS.stdout, LOOPHOLE(Store.heapAdr, ADDRESS),
	Value.ValStringStart(searchPath), 
 	Value.ValStringLength(searchPath)) THEN END;
      QOS.OutString(QOS.stdout, "\')");
      QOS.OutChar(QOS.stdout, QOS.newLine);
  END BadFile;

PROCEDURE DoIt() =
  CONST FileNameBuffSize = 256;
  VAR
    rd: QOS.Reader;
    dynamic, clos: Data.Pointer;
    stack: Value.T;
    TL: Data.Pointer;
    machineState: Value.T;
    outStack: Value.T;
    outTL, outTP, outFP: Data.Pointer;
    outCP: Value.T;
    outPC: Data.Pointer;
    outEX: Value.T;
    outHome: Value.T;
    outFault: Data.Bool;
    fileName: Value.T;
    excName: Value.T;
    fileNameBuff: ARRAY [0..FileNameBuffSize-1] OF CHAR;
    fileNameSize: Data.Int;
    param2Buff: ARRAY [0..1] OF CHAR;
  BEGIN
    stack := Value.NewValStack(Data.StackSize);
    TL := Data.TopLevelFP;
    searchPath := Value.ValStringOfChars("$1");
    param2Buff[0] := '$';
    param2Buff[1] := '2';
    IF NOT QOS.Lookup((*in*) ADR(param2Buff[0]), 0, 2, 
	  (*out*) ADR(fileNameBuff[0]), 0, FileNameBuffSize, (*out*) fileNameSize)
    OR NOT Value.ValStringOfBuff(ADR(fileNameBuff[0]), 0, 
	fileNameSize, (*out*) fileName) 
    THEN
      QOS.OutString(QOS.stdout, "Bad shell parameter: $2");
      QOS.OutChar(QOS.stdout, QOS.newLine);
      RETURN;
    END;
    IF NOT(OpenRead(fileName, (*out*) rd))
    THEN 
      QOS.OutString(QOS.stdout, "File not found");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF NOT(Format.Intern(rd, (*out*)dynamic)) THEN
      QOS.OutString(QOS.stdout, "Bad boot file");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF QOS.ReaderClose(rd) THEN END;
    clos := Value.ValDynamicValue(dynamic);
    (* Assumes clos is a function of no arguments, which terminates by raising
       an exception. (If normal termination happens, via a Return instruction,
       there is nothing to return to) *)
    machineState :=
      Value.NewValMachineState(stack, TL, 0, TL, clos, 0, Data.MinPointer,
        Data.MinPointer, FALSE);
    machineState := Exec.Exec(machineState);
    Value.ValMachineStateGet(machineState, (* out *) outStack,
      (* out *) outTL, (* out *) outTP, (* out *) outFP, (* out *) outCP,
      (* out *) outPC, (* out *) outEX, (* out *) outHome,
      (*out*) outFault);
    IF outFault = TRUE THEN
      QOS.OutString(QOS.stdout, "Exception: ");
      excName := Value.ValRaisePacketExc(outEX);
      IF QOS.PutSub(QOS.stdout, LOOPHOLE(Store.heapAdr, ADDRESS),
	Value.ValStringStart(excName),
	Value.ValStringLength(excName)) THEN END;
      (* -- exc value *)
      QOS.OutChar(QOS.stdout, QOS.newLine);
    END;
  END DoIt;

(* -- Dynamics test

PROCEDURE DoIt;
  CONST FileNameBuffSize = 256;
  VAR
    rd: QOS.Reader;
    wr: QOS.Writer;
    dynamic, clos: Data.Pointer;
    stack: Value.T;
    TL: Data.Pointer;
    machineState: Value.T;
    outStack: Value.T;
    outTL, outTP, outFP: Data.Pointer;
    outCP: Value.T;
    outPC: Data.Pointer;
    outEX: Value.T;
    outHome: Value.T;
    outFault: Data.Bool;
    fileName: Value.T;
    excName: Value.T;
    fileNameBuff: ARRAY [0..FileNameBuffSize-1] OF CHAR;
    fileNameSize: Data.Int;
    param2Buff: ARRAY [0..1] OF CHAR;
  BEGIN
    stack := Value.NewValStack(Data.StackSize);
    TL := Data.TopLevelFP;
    searchPath := Value.ValStringOfChars("$1");
    param2Buff := "$2";
    IF NOT QOS.Lookup((*in*) ADR(param2Buff[0]), 0, 2, 
	  (*out*) ADR(fileNameBuff[0]), 0, FileNameBuffSize, (*out*) fileNameSize)
    OR NOT Value.ValStringOfSubArray((*in*)fileNameBuff, 0, fileNameSize, 
	    (*out*) fileName) 
    THEN
      QOS.OutString(QOS.stdout, "Bad shell parameter: $2");
      QOS.OutChar(QOS.stdout, QOS.newLine);
      RETURN;
    END;
    IF NOT(OpenRead(fileName, (*out*) rd))
    THEN 
      QOS.OutString(QOS.stdout, "File not found");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF NOT(Format.Intern(rd, (*out*)dynamic)) THEN
      QOS.OutString(QOS.stdout, "Bad xtrn read");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF QOS.ReaderClose(rd) THEN END;
    IF NOT (OpenWrite(fileName, (*out*) wr)) THEN
      QOS.OutString(QOS.stdout, "Cannot write file");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF NOT(Format.Extern(wr, (*out*)dynamic)) THEN
      QOS.OutString(QOS.stdout, "Bad xtrn write");
      BadFile(searchPath, fileName);
      RETURN;
    END;
    IF QOS.WriterClose(wr) THEN END;
  END DoIt;
*)

(* <*EXTERNAL*> VAR collection_allowed: INTEGER; *)
BEGIN
(* --  collection_allowed:=0; *)
  QOS.Setup();
  Data.Setup();
  Store.Setup(8*Data.Mega);
  Code.Setup();
  Format.Setup();
  Value.Setup();
  Exec.Setup();
  DoIt();
END Main.
