(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)

INTERFACE Value;
IMPORT QOS, Data;

(*
|	FullSize =
|	  Data.PointeesPerPolymorph
|	SmallSize =
|	  Data.PointeesPerSmallInt
|	MiniSize =
|	  1
|	Bool =
|	  Immediate
|	Char =
|	  Immediate
|	Int =
|	  Immediate
|	Real =
|	  Immediate
|	String =
|	  Iter(length:FullSize, chars:Pointee)
|	Code =
|	  Iter(length:FullSize, codes:Pointee)
|	Badge =
|	  Seq(moduleName:Pointer, progName:Pointer, machineId:Immediate,
|	    processId:Immediate, timeSecs:Immediate, timeMicroSecs:Immediate,
|	    timeSerialNo:Immediate)
|	Literals =
|	  Iter(length:FullSize, literals:Pointer)
|	Prog =
|	  Seq(literals: Pointer, code: Code)
|	Closure[n] =
|	  Seq(prog: Pointer, globals: Polymorph[n])
|	Tuple[n] =
|	  Polymorph[n]
|	Array =
|	  Iter(length: FullSize, items :Polymorph)
|	RaisePacket =
|	  Seq(exc: Pointer, val: Polymorph)
|	Reader =
|	  Int
|	Writer =
|	  Int
|	Time =
|	  Seq(seconds: Int, microSeconds: Int)
|	Dynamic =
|	  Seq(type: Pointer, value: Polymorph)
|	MachineState =
|	  Seq(stack: Pointer, TL, TP, FP: Immediate,
|	      CP: Pointer, PC: Immediate, EX: Pointer,
|	      home: Polymorph, fault: Immediate)
*)

TYPE
  T = Data.Polymorph;

VAR
  typeNone: T;

PROCEDURE Setup();

CONST
  valOk = Data.OkImmediate;
  valFalse = Data.FalseImmediate;
  valTrue = Data.TrueImmediate;

PROCEDURE PlausiblePointer(ptr: Data.Pointer): BOOLEAN;

PROCEDURE NewValBool(bool: Data.Bool): T;

PROCEDURE NewValChar(char: Data.Char): T;

PROCEDURE NewValInt(int: Data.Int): T;

PROCEDURE NewValFloat(float: Data.Float): T;

PROCEDURE NewValString(
    size: Data.Int;
    init: Data.Pointee;
    VAR (*out*) string: T)
    : BOOLEAN;
PROCEDURE ValStringOfBuff(
    buff: ADDRESS;
    start, length: INTEGER;
    VAR (*out*) string: T)
    : BOOLEAN;
PROCEDURE ValStringStart(string: T): Data.Pointer;
PROCEDURE ValStringLength(string: T): Data.Int;
PROCEDURE ValStringIsEmpty(string: T): Data.Bool;
PROCEDURE ValStringGetPointee(
    string: T;
    i: Data.Int;
    VAR (*out*) pointee: Data.Pointee)
    : BOOLEAN;
PROCEDURE ValStringSetPointee(
    string: T;
    i: Data.Int;
    pointee: Data.Pointee)
    : BOOLEAN;
PROCEDURE ValStringGetSub(
    string: T;
    start, size: Data.Int;
    VAR (*out*) sub: T)
    : BOOLEAN;
PROCEDURE ValStringSetSub(
    string1: T;
    start1: Data.Int;
    string2: T;
    start2, size: Data.Int)
    : BOOLEAN;
PROCEDURE ValStringCat(string1, string2: T): T;
PROCEDURE ValStringCatSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) string3: T)
    : BOOLEAN;
PROCEDURE ValStringEqual(string1, string2: T): Data.Bool;
PROCEDURE ValStringEqualSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) eq: Data.Bool)
    : BOOLEAN;
PROCEDURE ValStringPrecedes(string1, string2: T): Data.Bool;
PROCEDURE ValStringPrecedesSub(
    string1: T;
    start1, size1: Data.Int;
    string2: T;
    start2, size2: Data.Int;
    VAR (*out*) prec: Data.Bool)
    : BOOLEAN;

PROCEDURE NewValTuple(size: Data.Int): T;
PROCEDURE ValTupleGet(tuple: T; i: Data.Card): T;
PROCEDURE ValTupleSet(tuple: T; i: Data.Card; item: T);

PROCEDURE NewValBadge(
    moduleName, progName: Data.Pointer;
    machineId, processId, timeSecs, timeMicroSecs, timeSerialNo: Data.Card)
    : T;
PROCEDURE ValBadgeName(badge: T; VAR (*out*) moduleName, progName: T);
PROCEDURE ValBadgeId(
    badge: T;
    VAR (*out*)
      machineId, processId, timeSecs, timeMicroSecs, timeSerialNo:
      Data.Card);

PROCEDURE NewValMachineState(
    stack: T;
    TL, TP, FP: Data.Pointer;
    CP: T;
    PC: Data.Pointer;
    EX: T;
    home: T;
    fault: Data.Bool)
    : T;
PROCEDURE ValMachineStateSet(
    machineState: T;
    stack: T;
    TL, TP, FP: Data.Pointer;
    CP: T;
    PC: Data.Pointer;
    EX: T;
    home: T;
    fault: Data.Bool);
PROCEDURE ValMachineStateGet(
    machineState: T;
    VAR (*out*) stack: T;
    VAR (*out*) TL, TP, FP: Data.Pointer;
    VAR (*out*) CP: T;
    VAR (*out*) PC: Data.Pointer;
    VAR (*out*) EX: T;
    VAR (*out*) home: T;
    VAR (*out*) fault: Data.Bool);

PROCEDURE NewValLiterals(size: Data.Int): T;
PROCEDURE ValLiteralsSize(literals: T): Data.Int;
PROCEDURE ValLiteralsGet(literals: T; i: Data.Card): T;
PROCEDURE ValLiteralsSet(literals: T; i: Data.Card; literal: T);

PROCEDURE NewValProg(codeSize: Data.Int): T;
PROCEDURE ValProgGetLiterals(prog: T): T;
PROCEDURE ValProgSetLiterals(prog: T; literals: T);
PROCEDURE ValProgCodeSize(prog: T): Data.Int;
PROCEDURE ValProgCodeStart(prog: T): Data.Pointer;

PROCEDURE NewValClosure(prog: T; globals: Data.Int): T;
PROCEDURE ValClosureProg(closure: T): T;
PROCEDURE ValClosureGetGlobal(closure: T; i: Data.Card): T;
PROCEDURE ValClosureSetGlobal(closure: T; i: Data.Card; global: T);
PROCEDURE ValClosureBreak(closure: T; break: BOOLEAN);
PROCEDURE ValClosureTrace(closure: T; trace: BOOLEAN);

PROCEDURE NewValArray(
    items: Data.Int;
    init: T;
    VAR (*out*) array: T)
    : BOOLEAN;
PROCEDURE ValArraySize(array: T): Data.Int;
PROCEDURE ValArrayGetItem(
    array: T;
    i: Data.Int;
    VAR (*out*) item: T)
    : BOOLEAN;
PROCEDURE ValArraySetItem(array: T; i: Data.Int; item: T): BOOLEAN;

PROCEDURE NewValRaisePacket(exc, val: T): T;
PROCEDURE ValRaisePacketExc(val: T): T;
PROCEDURE ValRaisePacketVal(val: T): T;

PROCEDURE ValReaderInput(): T;
PROCEDURE NewValReaderFile(fileName: T; VAR (*out*) reader: T): BOOLEAN;
PROCEDURE ValReaderClose(reader: T): BOOLEAN;
PROCEDURE ValReaderMore(reader: T; VAR (*out*) more: Data.Bool): BOOLEAN;
PROCEDURE ValReaderReady(reader: T; VAR (*out*) ready: Data.Int): BOOLEAN;
PROCEDURE ValReaderGetPointee(
    reader: T;
    VAR (*out*) pointee: Data.Pointee)
    : BOOLEAN;
PROCEDURE ValReaderGetString(
    reader: T;
    length: Data.Int;
    VAR (*out*) string: T)
    : BOOLEAN;
PROCEDURE ValReaderGetSubString(
    reader: T;
    string: T;
    start, length: Data.Int)
    : BOOLEAN;

PROCEDURE ValWriterOutput(): T;
PROCEDURE NewValWriterFile(fileName: T; VAR (*out*) writer: T): BOOLEAN;
PROCEDURE ValWriterClose(writer: T): BOOLEAN;
PROCEDURE ValWriterFlush(writer: T): BOOLEAN;
PROCEDURE ValWriterPutPointee(writer: T; pointee: Data.Pointee): BOOLEAN;
PROCEDURE ValWriterPutString(writer: T; string: T): BOOLEAN;
PROCEDURE ValWriterPutSubString(
    writer: T;
    string: T;
    start, length: Data.Int)
    : BOOLEAN;

VAR valTimeZero: T;
PROCEDURE ValTimeNow(): T;
PROCEDURE ValTimeSeconds(time: T): Data.Int;
PROCEDURE ValTimeMicroSeconds(time: T): Data.Int;
PROCEDURE ValTimeEqual(time1, time2: T): Data.Bool;
PROCEDURE ValTimeBefore(time1, time2: T): Data.Bool;
PROCEDURE ValTimeAfter(time1, time2: T): Data.Bool;

PROCEDURE NewValDynamic(type, value: T): T;
PROCEDURE ValDynamicType(dynamic: T): T;
PROCEDURE ValDynamicValue(dynamic: T): T;
PROCEDURE ValDynamicIntern(reader: T; VAR (*out*) dynamic: T): BOOLEAN;
PROCEDURE ValDynamicExtern(writer: T; dynamic: T): BOOLEAN;
PROCEDURE ValDynamicInternPortable(reader: T; VAR (*out*) dynamic: T): BOOLEAN;
PROCEDURE ValDynamicExternPortable(writer: T; dynamic: T): BOOLEAN;

(* RPC:
PROCEDURE ValRPCImport(
    instance, class, localOnly: T;
    VAR (*out*) rpc: T)
    : BOOLEAN;
PROCEDURE ValRPCClose(rpc: T);
PROCEDURE ValRPCInitCall(rpc: T): BOOLEAN;
PROCEDURE ValRPCSendCall(rpc: T): BOOLEAN;
PROCEDURE ValRPCEndCall(rpc: T): BOOLEAN;
PROCEDURE ValRPCExport(class, instance, localOnly: T): BOOLEAN;
PROCEDURE ValRPCGetRequest(): T;
PROCEDURE ValRPCStartResponse(rpc: T);
PROCEDURE ValRPCEndResponse(rpc: T);
PROCEDURE ValRPCPut(object: ADDRESS; length: INTEGER; rpc: T): BOOLEAN;
PROCEDURE ValRPCPutDynamic(rpc: T; dynamic: T): BOOLEAN;
PROCEDURE ValRPCPutLocalSpaceId(rpc: T): BOOLEAN;
PROCEDURE ValRPCGet(object: ADDRESS; length: INTEGER; rpc: T): BOOLEAN;
PROCEDURE ValRPCGetDynamic(rpc: T; VAR (*out*) dynamic: T): BOOLEAN;
*)

PROCEDURE NewValStack(sizeInPolymorphs: Data.Int): T;

PROCEDURE NewValLevelStack(sizeInPolymorphs: Data.Int): T;

PROCEDURE Same(val1, val2: T): BOOLEAN;

PROCEDURE ValStringOfChar(ch: CHAR): T;
PROCEDURE ValStringOfChars(chars: TEXT): T;

PROCEDURE PrintValString(wr: QOS.Writer; string: T);

PROCEDURE PrintValue(wr: QOS.Writer; value: T);

END Value.
