(* Copyright 1989 Digital Equipment Corporation.               *)
(* Distributed only by permission.                             *)
(* Last modified on Mon Jun  4 00:03:00 1990 by luca           *)

interface Code
import nester :Nester :Msg ident :Ident op :Op 
export

  (* Data structures representing the Quest Machine instructions.
     This is a potential back-end interface for code generators.
     Currently, these instructions are translated into portable
     bytecode; see Bytecode.spec for the actual bytecode layout. *)

  Def OpAddrSpecialClass =
    Option
      opAddrSpecialHomeCase opAddrSpecialStackCase
        opAddrSpecialPacketExcCase opAddrSpecialPacketValCase
    end
  (* Special addressing modes. Home is a general purpose global
     variable; Stack is the execution stack; Packet is where
     the current exception is kept during stack unwinding. *)

  Def Rec OpAddr =
    Option
      opAddrNoneCase
      opAddrDelayedCase with var delayed :OpAddr end
      opAddrImmedOkCase
      opAddrImmedBoolCase with bool :Bool end
      opAddrImmedCharCase with char :Char end
      opAddrImmedIntCase with int :Int end
      opAddrImmedRealCase with real :Real end
      opAddrLiteralCase with literalNo :Int end
      opAddrTemporaryCase with temporaryNo :Int var temporaryDispl :Int end
      opAddrArgumentCase with
          argumentNo: Int var argumentDispl :Int
          argumentByLoc :Bool
	end
      opAddrResultCase with resultNo: Int var resultDispl :Int end
      opAddrGlobalCase with globalNo :Int end
      opAddrIndexedCase
	with indexedAddr :OpAddr indexedIndex :Int end
      opAddrSpecialCase with special :OpAddrSpecialClass end
    end
  (* The addressing modes. Delayed is for aliasing addresses and avoiding
     unnecessary "move" operations. Immed's define immediate values. Literal 
     refers to the current program literal. Temporary, Argument, and Result 
     refer to the current frame (Argument also takes care of "var" arguments, 
     that are passed as pointer-dispacement pairs). Global refers to the current 
     closure. Indexed is to index off a data structure. Special is described 
     above. *)

  Def Rec OpAddrList =
    Option
      nil
      cons with first :OpAddr hand :Op_Hand var rest :OpAddrList end
    end
  (* A list of addresses. The hand distinguishes l-values from r-values. *)

  Def OpJumpClass =
    Option
      opJumpAlwaysCase opJumpEqCase opJumpNotEqCase opJumpLessCase
      opJumpLessEqCase opJumpMoreCase opJumpMoreEqCase
    end
  (* Ways to jump. *)

  Def Rec T =
    Option
      nil
      cons with first :Instr where: Msg_Where var rest :T end
    end
  (* A Code_T is a list of instructions and "where" pointers back 
     into the source code. *)

  and Instr =
    Option
      opNoneCase
      opMoveCase with moveSrc, moveDst :OpAddr end
      opFrameCase with var frameSize, frameTempNo: Int end
      opApplyCase with applyFun :OpAddr end
      opReturnCase with var returnSize :Int end
      opArgumentsCase
	with argumentsSrc :OpAddrList argumentsResNo :Int end
      opResultsCase with resultsDst :OpAddrList end
      opClosureCase
	with
	  closureProg :OpAddr 
	  closureGlobals :OpAddrList
          closureDst :OpAddr
	end
      opDumClosureCase
	with
          dumClosureSize :Int 
	  dumClosureProg :OpAddr
          dumClosureDst :OpAddr
	end
      opRecClosureCase
	with recClosureSrc :OpAddr recClosureGlobals :OpAddrList end
      opJumpCase
	with
          jumpClass :OpJumpClass jumpAddr1, jumpAddr2 :OpAddr
          jumpTarget :Patch
	end
      opLabelCase with labelNo :Int var labelAddress :Int end
      opStartCase
      opStopCase
      opEndCase (* -- obsolete? *)
      opDataCase with 
          dataMajOp :Op_MajOpClass dataMinOp :Op_MinOpClass 
          dataSrc :OpAddrList dataDst :OpAddr
        end
      opTupleCase with tupleSrc :OpAddrList tupleDst :OpAddr end
      opCaseCheckCase with optionSrc :OpAddr optionIndex :Int end
      opCaseCase with caseSrc :OpAddr caseBranchTargets :PatchTuple end
      opCaseFaultCase
      opArrayCase with arraySrc :OpAddrList arrayDst :OpAddr end
      opTrapCase with trapFrame :OpAddr trapTarget :Patch end
      opUntrapCase
      opRaiseCase with raiseSel :OpAddr raiseSrc :OpAddr end
      opUnwindCase
      opCrashCase
      (* optimizations; see also Op.spec *)
      opApplyArgumentsResults1Case
	with applyFun :OpAddr argumentsSrc :OpAddrList end
    end
  (* The Quest Machine instructions. *)

  and Patch = 
    Tuple var target :T end
    (* A jump target to be back-patched. *)
    (* -- should be Var(Instr) *)

  and PatchList =
    Option
      nil
      cons with first :Patch rest :PatchList end
    end
  (* A list of jump targets to back-patch. *)

  and PatchTuple =
    Array(Patch)
  (* A array of jump targets to back-patch. *)

  (* Allocation routines for the data structures above follow. *)

  addrOk, addrTrue, addrFalse :OpAddr

  newPatch(target :T) :Patch
  emptyPatchList :PatchList
  newPatchList(first :Patch rest :PatchList) :PatchList
  emptyPatchTuple :PatchTuple
  newPatchTuple(size :Int) :PatchTuple

  addrEmpty :OpAddr
  addrDelayed() :OpAddr
  addrImmedOk() :OpAddr
  addrImmedBool(bool :Bool) :OpAddr
  addrImmedChar(char :Char) :OpAddr
  addrImmedInt(int :Int) :OpAddr
  addrImmedReal(real :Real) :OpAddr
  addrLiteral(literalNo :Int) :OpAddr
  addrTemporary(temporaryNo :Int) :OpAddr
  addrArgument(argumentNo :Int argumentByLoc :Bool) :OpAddr
  addrResult(resultNo :Int) :OpAddr
  addrGlobal(globalNo :Int) :OpAddr
  addrIndexed(indexedAddr :OpAddr indexedIndex :Int) :OpAddr
  addrSpecial(special :OpAddrSpecialClass) :OpAddr

  addrSpecialHome, addrSpecialPacketExc, addrSpecialPacketVal 
    :OpAddrSpecialClass

  stripDelayed(addr :OpAddr) :OpAddr
  (* Remove unnecessary levels of delayed addresses. *)

  emptyOpAddrList :OpAddrList
  newOpAddrList(first :OpAddr hand :Op_Hand rest :OpAddrList) :OpAddrList
  opAddrListLength(list :OpAddrList) :Int
  opAddrListAppend(list1,list2 :OpAddrList) :OpAddrList

  emptyCode :T
  newCode(first :Instr where :Msg_Where rest :T) :T

  opJumpAlwaysCase, opJumpEqCase, opJumpNotEqCase, opJumpLessCase,
    opJumpLessEqCase, opJumpMoreCase, opJumpMoreEqCase: OpJumpClass

  opNone :Instr
  newOpMove(moveSrc, moveDst :OpAddr) :Instr
  (* Positively generates a move operation. *)
  move(moveSrc, moveDst :OpAddr) :Instr
  (* Aliases two addresses, so that a move operation is avoided
     if at least one of the two is a Delayed address. Otherwise
     generates a move. *)
  newOpFrame(frameSize, frameTempNo :Int) :Instr
  newOpApply(applyFun :OpAddr) :Instr
  newOpReturn(returnSize :Int) :Instr
  newOpArguments(argumentsSrc :OpAddrList argumentsResNo :Int) :Instr
  newOpResults(resultsDst :OpAddrList) :Instr
  newOpClosure(closureProg :OpAddr 
    closureGlobals :OpAddrList closureDst :OpAddr) :Instr
  newOpDumClosure(dumClosureProg :OpAddr 
    dumClosureSize :Int dumClosureDst :OpAddr) :Instr
  newOpRecClosure(recClosureSrc :OpAddr recClosureGlobals :OpAddrList) :Instr
  newOpJump(jumpClass :OpJumpClass jumpAddr1,jumpAddr2 :OpAddr
    jumpTarget :Patch) :Instr
  newOpLabel(labelNo :Int) :Instr
  newOpStart() :Instr
  newOpStop() :Instr
  newOpEnd() :Instr
  newOpData(dataMajOp :Op_MajOpClass dataMinOp :Op_MinOpClass 
    dataSrc :OpAddrList dataDst :OpAddr) :Instr
  newOpTuple(tupleSrc :OpAddrList tupleDst :OpAddr) :Instr
  newOpCaseCheck(optionSrc :OpAddr optionIndex :Int) :Instr
  newOpCase(caseSrc :OpAddr caseBranchTargets :PatchTuple) :Instr
  newOpCaseFault() :Instr
  newOpArray(arraySrc :OpAddrList arrayDst :OpAddr) :Instr
  newOpTrap(trapFrame :OpAddr trapTarget :Patch) :Instr
  newOpUntrap() :Instr
  newOpRaise(raiseSel :OpAddr raiseSrc :OpAddr) :Instr
  newOpUnwind() :Instr
  newOpCrash() :Instr
  (* optimizations *)
  newOpApplyArgumentsResults1(applFun :OpAddr argumentsSrc :OpAddrList) :Instr

  printCode(wr :nester.T pc :T indent :String) :Ok
  (* Print a piece of code (for debugging). *)
  printAddr(wr :nester.T addr :OpAddr) :Ok
  (* Print an address (for debugging). *)

end;
