(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: ModExpr.m3                                            *)
(* Last modified on Mon Oct 12 13:42:27 PDT 1992 by kalsow         *)
(*      modified on Thu Nov 29 03:31:49 1990 by muller         *)

MODULE ModExpr;

IMPORT Expr, ExprRep, Type, Int, Error, Emit, IntegerExpr, Temp, MBuf;
IMPORT Reel, LReel, EReel, ReelExpr;

TYPE
  Class = { cINT, cREAL, cLONG, cEXTND, cERR };

TYPE
  P = ExprRep.Tab BRANDED "ModExpr.P" OBJECT
        class: Class;
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := ExprRep.EqCheckAB;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	note_write   := ExprRep.NotWritable;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a, b: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a    := a;
    p.b    := b;
    p.type := Int.T;
    RETURN p;
  END New;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR ta, tb: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    Expr.TypeCheck (p.b, cs);
    ta := Type.Base (Expr.TypeOf (p.a));
    tb := Type.Base (Expr.TypeOf (p.b));
    IF    (ta = Int.T)   AND (tb = Int.T)   THEN p.class := Class.cINT;
    ELSIF (ta = Reel.T)  AND (tb = Reel.T)  THEN p.class := Class.cREAL;
    ELSIF (ta = LReel.T) AND (tb = LReel.T) THEN p.class := Class.cLONG;
    ELSIF (ta = EReel.T) AND (tb = EReel.T) THEN p.class := Class.cEXTND;
    ELSE p.class := Class.cERR;  ta := Int.T;
      Error.Msg ("illegal operands for MOD");
    END;
    p.type := ta;
  END Check;

PROCEDURE Compile (p: P): Temp.T =
  VAR t1, t2, t3: Temp.T; min, max: INTEGER;
  BEGIN
    t1 := Expr.Compile (p.a);
    t2 := Expr.Compile (p.b);
    t3 := Temp.Alloc (p);
    IF (p.class = Class.cINT) THEN
      Emit.OpT ("@ = _IMOD", t3);

      Expr.GetBounds (p.a, min, max);
      IF min >= 0 THEN    Emit.Op ("P");
      ELSIF max <= 0 THEN Emit.Op ("N");
      ELSE                Emit.Op ("X"); END;

      Expr.GetBounds (p.b, min, max);
      IF min >= 0 THEN    Emit.Op ("P");
      ELSIF max <= 0 THEN Emit.Op ("N");
      ELSE                Emit.Op ("X"); END;

      Emit.OpTT ("(@, @);\n", t1, t2);
    ELSE
      (* floating point: x MOD y == x - y * FLOOR (x / y)  *)
      Emit.OpTT ("@ = @ - ", t3, t1);
      Emit.OpT  ("(@ * ", t2);
      Emit.OpF  ("((@) ", p.type);
      Emit.OpTT ("_FLOOR (@ / @)));\n", t1, t2);
    END;
    Temp.Free (t1);
    Temp.Free (t2);
    RETURN t3;
  END Compile;

PROCEDURE Fold (p: P): Expr.T =
  VAR e1, e2, e3: Expr.T;
  BEGIN
    e1 := Expr.ConstValue (p.a);
    e2 := Expr.ConstValue (p.b);
    e3 := NIL;
    IF    (e1 = NIL) OR (e2 = NIL)     THEN
    ELSIF IntegerExpr.Mod (e1, e2, e3) THEN
    ELSIF ReelExpr.Mod    (e1, e2, e3) THEN
    END;
    RETURN e3;
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "MOD ");
    Expr.Fingerprint (p.a, map, wr);
    Expr.Fingerprint (p.b, map, wr);
  END FPrinter;

BEGIN
END ModExpr.
