(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Sep 22 14:32:23 PDT 1992 by meehan                   *)
(*      modified on Tue Jun 16 21:55:36 PDT 1992 by muller                   *)

MODULE Manpage;

IMPORT Fmt, FormsVBT, MTextRd, Range, Rd, RdUtils, Rsrc, Text,
       TextEditVBT, TextPort, Thread, VBT, VTDef, VText;

<* PRAGMA LL *>

<* FATAL FormsVBT.Unimplemented, Range.Error *>

TYPE
  T = FormsVBT.Closure OBJECT
        rd, revRd: MTextRd.T;
        textport : TextPort.T;
        length   : CARDINAL;
        er       : ErrorReporter;
        ready    : Thread.Condition;
        done                          := FALSE; <* LL = VBT.mu *>
        helpfindfirst, helpfindnext, helpfindprev, helpfindtext, notfound,
          helpcase: TEXT;
        caseSensitive := TRUE
      OVERRIDES
        apply := Help
      END;
  HelpThreadClosure = Thread.Closure OBJECT
                        t       : T;
                        resource: TEXT;
                        path    : Rsrc.Path
                      OVERRIDES
                        apply := ReadManpage
                      END;

PROCEDURE Init (fv           : FormsVBT.T;
                resource     : TEXT;
                er           : ErrorReporter;
                helpfindfirst                  := "helpfindfirst";
                helpfindnext                   := "helpfindnext";
                helpfindprev                   := "helpfindprev";
                helpfindtext                   := "helpfindtext";
                manpagetext                    := "manpagetext";
                notfound                       := "notfound";
                helpcase                       := "helpcase";
                path         : Rsrc.Path       := NIL              )
  RAISES {FormsVBT.Error} =
  VAR
    t := NEW (T, er := er, ready := NEW (Thread.Condition),
              helpfindfirst := helpfindfirst, helpfindnext := helpfindnext,
              helpfindprev := helpfindprev, helpfindtext := helpfindtext,
              notfound := notfound, helpcase := helpcase);
    htc := NEW (HelpThreadClosure, t := t, resource := resource, path := path);
  BEGIN
    TYPECASE FormsVBT.GetVBT (fv, manpagetext) OF
    | TextEditVBT.T (x) => t.textport := TextEditVBT.GetPort (x)
    ELSE
      RAISE FormsVBT.Error (
              "\"" & manpagetext & "\" is not the name of a TextEdit form")
    END;
    FormsVBT.Attach (fv, helpfindfirst, t);
    FormsVBT.Attach (fv, helpfindnext, t);
    FormsVBT.Attach (fv, helpfindprev, t);
    FormsVBT.Attach (fv, helpfindtext, t);
    FormsVBT.Attach (fv, helpcase, t);
    EVAL Thread.Fork (htc)
  END Init;

PROCEDURE ReadManpage (htc: HelpThreadClosure): REFANY =
  <* LL = 0 *>
  <* FATAL Range.Error *>
  VAR
    rd: Rd.T;
    t        := htc.t;
  PROCEDURE oops (msg: TEXT) =
    BEGIN
      LOCK VBT.mu DO
        t.er.apply (
          Fmt.F ("Sorry, couldn't read the manpage in %s\nError: %s\n",
                 htc.resource, msg))
      END
    END oops;
  BEGIN
    TRY                         (* Fetch the file in non-event time. *)
      rd := Rsrc.Open (htc.resource, htc.path);
      LOCK VBT.mu DO
        (* MText and (therefore) VText support a "text segment" that is
           actually just a reader, and they don't actually copy bytes until
           they're needed. *)
        WITH vtext = TextPort.GetVText (t.textport),
             mtext = vtext.mtext                     DO
          VText.ReplaceFile (
            vtext, begin := 0, end := LAST (CARDINAL), file := rd);
          VBT.Mark (t.textport);
          (* Create forward- and reverse-readers for searching. *)
          t.rd := MTextRd.New (mtext);
          t.revRd := MTextRd.New (mtext, reverse := TRUE);
          t.length := Rd.Length (rd)
        END
      END
    EXCEPT
    | Rd.Failure (ref) => oops (RdUtils.FailureText (ref))
    | Rd.EndOfFile => oops ("End of file")
    | VTDef.Error (code) => oops (VTDef.ErrorCodeTexts [code])
    | Thread.Alerted => oops ("interrupted (Thread.Alerted)")
    | Rsrc.NotFound => oops ("No such resource: " & htc.resource)
    END;
    LOCK VBT.mu DO t.done := TRUE END;
    Thread.Signal (t.ready);
    RETURN NIL
  END ReadManpage;

PROCEDURE Help (t: T; fv: FormsVBT.T; name: TEXT; time: VBT.TimeStamp) =
  <* LL = VBT.mu *>
  VAR
    pattern       : TEXT;
    pos           : INTEGER;
    n             : CARDINAL;
    indexL, indexR: CARDINAL;
  BEGIN
    (* Wait for ReadManpage to finish. *)
    WHILE NOT t.done DO Thread.Wait (VBT.mu, t.ready) END;
    TextPort.Selection (t.textport, indexL, indexR);
    TRY
      pattern := FormsVBT.GetText (fv, t.helpfindtext);
      n := Text.Length (pattern);
      IF n = 0 THEN
        RETURN
      ELSIF Text.Equal (name, t.helpfindfirst)
              OR Text.Equal (name, t.helpfindtext)
              OR Text.Equal (name, t.helpfindnext) THEN
        IF Text.Equal (name, t.helpfindnext) THEN
          Rd.Seek (t.rd, indexR)
        ELSE
          Rd.Seek (t.rd, 0)
        END;
        pos := RdUtils.Find (t.rd, pattern, NOT t.caseSensitive);
        IF pos = -1 THEN
          FormsVBT.PopUp (fv, t.notfound, time)
        ELSE
          TextPort.Select (
            t.textport, time, pos, pos + n, replaceMode := TRUE);
          TextPort.Normalize (t.textport, pos)
        END
      ELSIF Text.Equal (name, t.helpfindprev) THEN
        Rd.Seek (t.revRd, t.length - indexL);
        pos := RdUtils.Find (
                 t.revRd, TextReverse (pattern), NOT t.caseSensitive);
        IF pos = -1 THEN
          FormsVBT.PopUp (fv, t.notfound, time)
        ELSE
          TextPort.Select (t.textport, time, t.length - pos - n,
                           t.length - pos, replaceMode := TRUE);
          TextPort.Normalize (t.textport, t.length - pos - n)
        END
      ELSIF Text.Equal (name, t.helpcase) THEN
        t.caseSensitive := FormsVBT.GetBoolean (fv, name)
      END
    EXCEPT
    | VTDef.Error (e) => t.er.apply (VTDef.ErrorCodeTexts [e])
    | FormsVBT.Error (msg) => t.er.apply (msg)
    | Thread.Alerted =>
    | Rd.Failure (ref) => t.er.apply (RdUtils.FailureText (ref))
    END
  END Help;

PROCEDURE TextReverse (t: TEXT): TEXT =
  VAR
    len : CARDINAL          := Text.Length (t);
    buf : REF ARRAY OF CHAR;
    i, j: CARDINAL;
    c   : CHAR;
  BEGIN
    buf := NEW (REF ARRAY OF CHAR, len);
    Text.SetChars (buf^, t);
    i := 0;
    j := len - 1;
    WHILE i < j DO
      c := buf [i];
      buf [i] := buf [j];
      buf [j] := c;
      INC (i);
      DEC (j)
    END;
    RETURN Text.FromChars (buf^)
  END TextReverse;

BEGIN END Manpage.
