INTERFACE FileStamp;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

IMPORT Text;
IMPORT TimeDate, OSError;


(* Operations on file time stamps. The file time stamp of a file is the time
at which the file was last modified *)


TYPE
  T = TimeDate.Stamp; (* yet another alias for 'TimeDate.T' *)


CONST
  Bad = NIL;
(* 'Bad' is a system independent way of representing a bad time stamp on a
file. Stamping a file with the bad time stamp is a way of showing that the
file is invalid. There may be more than one value for a bad file time stamp
on some systems; this interface translates all such values to 'Bad' when the
file time stamp is fetched.
  'Bad' should not be given to any of the procedures in the 'Time' interface
or a checked runtime error will occur. Only some of the procedures in this
interface accept 'Bad'; those which do mention 'Bad' specifically in their
associated comments. If 'Bad' is not mentioned in the comment associated with
a procedure it is a checked runtime error to call that procedure with 'Bad' as
an argument. *)


PROCEDURE Get(name: Text.T): T RAISES {OSError.E};
(* Get the time stamp of the named file. If the stamp cannot be read (because
the file does not exist or access problems etc.) 'OSError.E' is raised. The
argument to this exception provides more information about the error (see the
'OSError' interface).
  Note that 'Get' may return 'Bad' *)

PROCEDURE Set(name: Text.T; t: T) RAISES {OSError.E};
(* Set the time stamp of the named file to 't'. If the stamp cannot be set
(because the file does not exist or access problems etc.) 'OSError.E' is
raised. The argument to this exception provides more information about the
error (see the 'OSError' interface). 't' may be 'Bad' *)

(* Note that the 'name' specified to 'Set' or 'Get' may be that of a directory.
If it is the effect is undefined - it depends whether the underlying system
supports the idea of a last modified stamp for directories. If stamps on
directories are supported the null text may be used as an abbreviation for the
current directory. 'name' must never be NIL or a checked runtime error will
occur *)


PROCEDURE Copy(t: T): T RAISES {};
(* Return a copy of 't'. 't' can be 'Bad' in which case the result is 'Bad'
also *)


EXCEPTION
  OutOfRange;

PROCEDURE Add(t: T; secs: INTEGER; uSecs: INTEGER := 0) RAISES {OutOfRange};
(* Add 'secs * 1000000 + uSecs' micro seconds to 't'. Note that both 'secs' and
'uSecs' are integers so 'Add' can be used to subtract as well.
  If the time resulting from the addition would be outside the range that the
system can represent, 'OutOfRange' is raised *)


PROCEDURE Compare(
   t1, t2: T;
   uSecs: CARDINAL := 0)
   : INTEGER
   RAISES {OutOfRange};
(* Compares two time stamps. The result is:
  < 0  if 't1 + uSecs < t2'
  > 0  if 't1 > t2 + uSecs'
  0 otherwise
Note that 'uSecs = 0', the default, gives an exact comparison.
  The 'OutOfRange' exception could occur due to the additions used in an
inexact comparison. It is, however, exceedingly unlikely and most users need
not worry about the possibility. If it ever does occur it is probably because:
1) 'uSecs' is inordinately big (who would ever want a comparison that inexact?)
2) the system clock is set incorrectly
3) your system clock is about to overflow (in which case you're in big trouble)
*)

PROCEDURE IsFuture(
    t: T;
    uSecs: CARDINAL := 0)
    : BOOLEAN
    RAISES {OutOfRange, OSError.E};
(* Is 't' in the future? 'uSecs' specifies a tolerance; if 't' is less than or
equal to the current time plus 'uSecs' micro seconds 'IsFuture' will return
FALSE. This tolerance is useful on distributed systems where files on a remote
machine may appear to be slightly in the future due to differing system clocks.
  'IsFuture' might conceivably raise the 'OutOfRange' exception because it is
just a veneer on 'Compare'. The chances of it actually happening, however, are
negligible.
  'IsFuture' raises 'OSError.E' if it cannot get the current time *)

END FileStamp.
