MODULE M3CBE_C_generic;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

IMPORT HashText, IO, Text, IOErr, CharsTo, PathNameStream, M3CBackEnd_C_cc;
IMPORT Err, Fmt;
IMPORT CharType; (* SRC compiler bug workaround *)

VAR
  table_g := HashText.New(20);

TYPE
    BE_name =  {a64, a32, a16, a8,
      minAlignment, recAlignment, arrayAlignment,
      ptrA, ptrS, realA, realS, longRealA, longRealS, 
      intA, intS, target};

  BE_info = REF RECORD be_name: BE_name; be_value: BE_value END;

  BE_value = ROOT;
  BE_int_value = BE_value OBJECT int: INTEGER END;
  BE_text_value = BE_value OBJECT text: TEXT END;

CONST
  BE_texts = ARRAY BE_name OF TEXT {"a64", "a32", "a16", "a8",
      "minAlignment", "recAlignment", "arrayAlignment",
      "ptrA", "ptrS", "realA", "realS", "longRealA", "longRealS", 
      "intA", "intS", "target"};

CONST
  BE_defaults = ARRAY BE_name OF INTEGER {16, 16, 16, 8,
      8, 16, 16,
      16, 32, 16, 32, 16, 64, 
      16, 32, 0};

  BE_target_default = "generic";

PROCEDURE InitGeneric() RAISES {}=
  BEGIN
    FOR i := FIRST(BE_name) TO LAST(BE_name) DO
      VAR 
        id: HashText.Id;
        be_info := NEW(BE_info, be_name := i);
        be_value: BE_value;
      BEGIN
        EVAL HashText.Enter(table_g, BE_texts[i], id);
        IF i = BE_name.target THEN
          be_value := NEW(BE_text_value, text := BE_target_default);
        ELSE
          be_value := NEW(BE_int_value, int := BE_defaults[i]);
        END;
        be_info.be_value := be_value;
        HashText.Associate(table_g, id, be_info);
      END;
    END; (* for *)
  END InitGeneric;

PROCEDURE Init() RAISES {}=
  VAR
    iter := HashText.NewIterator(table_g);
    key: TEXT;
    val: REFANY;
    be_info: BE_info;
  BEGIN
    WHILE HashText.Next(iter, key, val) DO
      be_info := val;
      TYPECASE be_info.be_value OF
      | BE_int_value(t) =>
          CASE be_info.be_name OF
          | BE_name.a64  => M3CBackEnd_C_cc.a64  := t.int;
          | BE_name.a32  => M3CBackEnd_C_cc.a32  := t.int;
          | BE_name.a16  => M3CBackEnd_C_cc.a16  := t.int;
          | BE_name.a8 => M3CBackEnd_C_cc.a8 := t.int;
          | BE_name.minAlignment => M3CBackEnd_C_cc.minAlignment := t.int;
          | BE_name.recAlignment => M3CBackEnd_C_cc.recAlignment := t.int;
          | BE_name.arrayAlignment => M3CBackEnd_C_cc.arrayAlignment := t.int;
          | BE_name.ptrA => M3CBackEnd_C_cc.ptrA := t.int;
          | BE_name.ptrS => M3CBackEnd_C_cc.ptrS := t.int;
          | BE_name.realA => M3CBackEnd_C_cc.realA := t.int;
          | BE_name.realS => M3CBackEnd_C_cc.realS := t.int;
          | BE_name.longRealA => M3CBackEnd_C_cc.longRealA := t.int;
          | BE_name.longRealS => M3CBackEnd_C_cc.longRealS := t.int;
          | BE_name.intA => M3CBackEnd_C_cc.intA := t.int;
          | BE_name.intS => M3CBackEnd_C_cc.intS := t.int;
          END;

      | BE_text_value(t) => M3CBackEnd_C_cc.target := t.text;
      END; (* typecase *)
    END; (* while *)
  END Init;

PROCEDURE ReadFile(name: TEXT): BOOLEAN RAISES {}=
  VAR
    id: HashText.Id;
    t: TEXT;
    s: IO.Stream := NIL;
    chars: ARRAY [0..255] OF CHAR;
  BEGIN
    TRY
      TRY
        s := PathNameStream.Open(name, IO.OpenMode.Read);
        LOOP
          EVAL IO.Skip(s);
          t := Text.FromChars(SUBARRAY(chars, 0, IO.GetUntil(s, chars)));
          IF HashText.Lookup(table_g, t, id)  THEN
            EVAL IO.Skip(s);
            VAR hack := NARROW(HashText.Value(table_g, id), BE_info);
              lval := IO.GetUntil(s, chars);
            BEGIN
            TYPECASE hack.be_value OF
            | BE_int_value(t) =>
                IF CharsTo.Int(SUBARRAY(chars, 0, lval), t.int) THEN
                ELSE
                  Err.Print(Fmt.F("illegal backend parameter value \'%s\'",
                      Text.FromChars(SUBARRAY(chars, 0, lval))), 
                      Err.Severity.Error);
                  RETURN FALSE;
                END; (* if *)
            | BE_text_value(t) => 
                t.text := Text.FromChars(SUBARRAY(chars, 0, lval));
            END;
            END;
          ELSE
            Err.Print(Fmt.F("unknown backend parameter \'%s\'", t),
                Err.Severity.Error);
            RETURN FALSE;
          END; (* if *)
        END;
      EXCEPT 
      | IO.EndOfStream =>  (* normal exit *)

      | IO.Error(errant) =>
          s := errant;
          RETURN FALSE;
      END;
      RETURN TRUE
    FINALLY
      IOErr.Close(s, Err.Severity.Error);
    END;
  END ReadFile;

BEGIN
  InitGeneric();
  M3CBackEnd_C_cc.RegisterTarget(BE_target_default, Init);
END M3CBE_C_generic.
