MODULE JoinScreen;

IMPORT VBT, Palette, ScrnCursor, ScrnPixmap, ScrnFont, ScrnPaintOp,
       ScreenType, PaintOp, Font, TrestleComm, Cursor,
       Pixmap, ScrnColorMap, VBTRep, PlttFrnds, Wr, Stdio, Fmt,
       JoinPaintOp, JoinCursor, JoinPaintOp, JoinPixmap, JoinCMap;

REVEAL
  T = Public BRANDED OBJECT
    (* sts contains the list of screens that this joined screen
       is shared over.  this might be helpful in making some of the
       oracles make intelligent decisions based on the screentypes
       involved in the join. *)
    sts : ScreenArray := NIL;
  OVERRIDES
    opApply := JoinPaintOp.Apply;
    pixmapApply := JoinPixmap.Apply;
    cursorApply := JoinCursor.Apply;
    fontApply := JoinFont.Apply;
    init := Init;
    addScreen := AddScreen;
    removeScreen := RemoveScreen;
  END; (* object *)

EXCEPTION FatalError; <* FATAL *>


TYPE 
  ScreenArray = REF ARRAY OF ScreenType.T;
  ColorMapOracle = ScrnColorMap.Oracle OBJECT
    st: T;
  OVERRIDES
    standard := ColorMapDefault;
    new := ColorMapNew;
    list := ColorMapList;
    lookup := ColorMapLookup
  END;

PROCEDURE ColorMapNew(orc: ColorMapOracle; 
                      nm: TEXT := NIL;
                      preLoaded := TRUE): ScrnColorMap.T
  RAISES {TrestleComm.Failure} = 
  BEGIN
    IF orc.st.sts = NIL THEN RAISE FatalError; END;
    RETURN orc.st.sts[0].cmap.new(nm, preLoaded);
  END ColorMapNew;
    
PROCEDURE ColorMapDefault(orc: ColorMapOracle): ScrnColorMap.T RAISES {}=
  BEGIN
    IF orc.st.sts = NIL THEN RAISE FatalError; END;
    RETURN orc.st.sts[0].cmap.standard();
  END ColorMapDefault;
  
PROCEDURE ColorMapList(orc: ColorMapOracle; 
                       pat: TEXT;
                       maxResults: CARDINAL): REF ARRAY OF TEXT  RAISES {} =
  BEGIN
    IF orc.st.sts = NIL THEN RAISE FatalError; END;
    RETURN orc.st.sts[0].cmap.list(pat, maxResults);
  END ColorMapList;
  
PROCEDURE ColorMapLookup(orc: ColorMapOracle;
                         pat: TEXT) : ScrnColorMap.T
    RAISES {} = 
  BEGIN
    IF orc.st.sts = NIL THEN RAISE FatalError; END;
    RETURN orc.st.sts[0].cmap.lookup(pat);
  END ColorMapLookup;

PROCEDURE RemoveScreen(self : T; st : VBT.ScreenType) RAISES {}=
  BEGIN
    IF self.sts = NIL THEN RAISE FatalError END;
  END RemoveScreen;

PROCEDURE AddScreen(self : T; st : VBT.ScreenType) RAISES {}=
  VAR
    empty : CARDINAL;
  BEGIN
    IF self.sts = NIL THEN
      self.sts := NEW(ScreenArray, 2);
      FOR i := 0 TO LAST(self.sts^) DO self.sts[i] := NIL END;
      self.sts[0] := st;

      self.depth := st.depth;
      self.color := st.color;
      self.res := st.res;
      self.bg := st.bg;
      self.fg := st.fg;
      self.bits := self;

      Palette.Init(self);
    ELSE
      empty := NUMBER(self.sts^);
      FOR i := 0 TO LAST(self.sts^) DO 
        IF st = self.sts[i] THEN RETURN END;
        IF i < empty AND self.sts[i] = NIL THEN empty := i END;
      END;
      IF empty = NUMBER(self.sts^) THEN
        VAR new := NEW(ScreenArray, 2 * empty);
        BEGIN
          SUBARRAY(new^, 0, NUMBER(self.sts^)) := self.sts^;
          FOR i := empty + 1 TO LAST(new^) DO 
            new[i] := NIL
          END;
          self.sts := new
        END;
      END;
      self.sts[empty] := st;
      RETURN;
    END; (* IF *)
  END AddScreen;

PROCEDURE PixmapApply(st: T;
                      cl: Palette.PixmapClosure;
                      pm: Pixmap.T): ScrnPixmap.T = 
  BEGIN
    IF st.sts = NIL THEN RAISE FatalError; END;
    LOOP
      WITH pix = st.sts[0].pixmaps[pm.pm] DO
        IF pix = NIL THEN EVAL Palette.ResolvePixmap(st.sts[0], pm);
        ELSIF pix # PlttFrnds.noPixmap THEN RETURN pix END;
      END; (* with *)
    END; (* loop *)
  END PixmapApply;
  
PROCEDURE New(): T =
  BEGIN
    RETURN NEW(T, bits := NIL).init();
  END New;

PROCEDURE Init (st: T): T =
  BEGIN
    st.op := JoinPaintOp.New(st);
    st.cursor := JoinCursor.New(st);
    st.pixmap := JoinPixmap.New(st);
    st.font := JoinFont.New(st);
    st.cmap := JoinCMap.New(st);
    st.depth := 1;
    st.color := FALSE;
    st.res := ARRAY Axis.T OF REAL{2.8, ..};
    st.bg := 0;
    st.fg := 1;
    IF st.bits = NIL THEN
      VAR bits := NEW(T);
      BEGIN
        bits.bits := bits;
        bits.init();
        st.bits := bits
      END
    END;
    RETURN st;
  END Init;

BEGIN END JoinScreen.
