(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Wed Jul 29 22:40:16 PDT 1992 by meehan    *)
(*      modified on Tue Jun 16 13:08:58 PDT 1992 by muller    *)
<* PRAGMA LL *>

INTERFACE EditCmd;

(* The long-term goal of the "EditCmd" interface is to export
   all the editing functions used by "TextPort", so that clients
   can specify arbitrary keybindings, extending or replacing 
   predefined keybindings for popular editors: Ivy, Emacs,
   Macintosh, etc.

   In the current version, we define routines for adjusting the
   current selection, for editing the selected text, and for
   managing the ``Undo'' stack.
*)

IMPORT TextPort, VBT;

(* \subsection {Adjust the current selection} *)

PROCEDURE ToNextWord (v: TextPort.T; time: VBT.TimeStamp);
(* Locate and select the ``next'' word.  From the current position,
   we scan right until we reach an alphanumeric character.  Then
   we continue scanning right until we reach the first
   non-alphanumeric character; that position defines the right
   end of the selection range.  Then we scan left until we find a
   non-alphanumeric character.  That position, plus 1, defines
   the left end of the selection range.

   If the initial position is in the middle of a word, then this
   actually selects the {\it current} word, but on successive
   calls, it selects each following word in turn. *)

PROCEDURE ToPrevWord (v: TextPort.T; time: VBT.TimeStamp);
(* Locate and select the ``previous'' word.  This is the same as
   "ToNextWord", except that all the scanning directions are
   reversed. *)

PROCEDURE DeleteToEndOfWord (v: TextPort.T; time: VBT.TimeStamp);
(* Delete from the current position to the end of the ``next''
   word (as defined in "ToNextWord"). *)

PROCEDURE DeleteToStartOfWord (v: TextPort.T; time: VBT.TimeStamp);
(* Delete from the current position to the beginning of the
   ``previous'' word (as defined in "ToPrevWord"). *)

(* \subsection {Edit the primary and secondary selections} *)

PROCEDURE Clear (v: TextPort.T; time: VBT.TimeStamp);
(* Delete the primary selection. *)

PROCEDURE Copy (v: TextPort.T; time: VBT.TimeStamp);
(* Copy the primary selection to the secondary selection
   (``Clipboard''). *)

PROCEDURE Paste (v: TextPort.T; time: VBT.TimeStamp);
(* Copy the secondary selection to the primary selection. *)

PROCEDURE Cut (v: TextPort.T; time: VBT.TimeStamp);
(* Call "Copy" then "Clear". *)

PROCEDURE Move (v: TextPort.T; time: VBT.TimeStamp);
(* Copy the secondary selection to the primary selection, then
   clear the secondary selection.  *)

PROCEDURE Swap (v: TextPort.T; time: VBT.TimeStamp);
(* Exchange the text at the primary and secondary selections. *)

(* \subsection {Managing the ``Undo'' stack} *)

(* The ``Undo'' stack records all the editing changes made to
   the "TextPort".  These changes can be undone; once undone, they
   can be redone.  There is no built-in limit to the number of
   changes that are recorded.  A sequence of insertions of graphic 
   characters (i.e., plain typing) counts as one ``edit''. *)
   
PROCEDURE Undo (v: TextPort.T); <* LL < v.mu *>
(* Reverse the effect of the last editing command. *)

PROCEDURE UndoUndo (v: TextPort.T); <* LL < v.mu *>
(* Reinstate the effect of the last editing command. *)

PROCEDURE ResetUndo (v: TextPort.T); <* LL < v.mu *>
(* Clear the ``Undo'' stack.  Nothing in the "TextPort" module
   calls this procedure. *)

PROCEDURE UndoCount (v: TextPort.T): CARDINAL;
(* Return the number of changes that can be undone. *)

PROCEDURE UndoUndoCount (v: TextPort.T): CARDINAL;
(* Return the number of undone changes that can be redone. *)

END EditCmd.
