(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Aug 11 20:29:25 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 13:08:50 PDT 1992 by muller *)
(*      modified on Mon Jun 15 21:49:40 1992 by mhb *)

(* The "FlexVBT.T" is a filter whose shape is based on a {\em
   natural} size with some {\em stretch} and {\em shrink}.  If a
   natural amount is left unspecified, the stretch and shrink are
   applied relative to the "FlexVBT"'s child's size.  If a
   stretch or shrink is left unspecified, 0 is assumed.  All
   units are specified in points (see the "Pts" interface).

   The actual data structure for specifying a flexible shape is
   defined in the "FlexShape" interface.  That interface also
   contains a number of common shape specifications, such as
   removing all stretch and shrink from a child.

   More precisely, the shape of a "FlexVBT.T" is computed as
   follows:

|       preferred := natural,
|       lo        := natural - shrink
|       hi        := natural + stretch + 1

   However, if any of these values are left unspecified, a size
   range is computed consistent with its child's shape according
   to the following rules:

   \begin{itemize}

   \item if natural is missing, use child's preferred;

   \item if shrink is missing, use 0 if natural is set, or
   child's lo (but no more than v's preferred) otherwise;

   \item if stretch is missing, use 0 if natural is set, or
   child's hi (but no less than v's preferred) otherwise.

   \end{itemize}

   This interface is similar to "RigidVBT", but more powerful in
   that one can specify a size based on a child's size and can
   dynamically change the size specification.  Also, it presents
   a slightly different model to the client: In "RigidVBT", one
   thinks in terms of the low and high bounds of some range.
   Here, one thinks in terms of the amount the a value can
   stretch and shrink.  Finally, don't forget that values in this
   interface are specified in points, whereas "RigidVBT" uses
   millimeters. *)


INTERFACE FlexVBT;

IMPORT Axis, Filter, FlexShape, VBT;

TYPE
  T <: Public;
  Public = Filter.T OBJECT
           METHODS
             init (ch: VBT.T; sh := FlexShape.DefaultShape): T
           END;

(* The call "v.init(...)" initializes "v" as a "FlexVBT" with
   child "ch" and shape specification "sh".  The default shape
   causes "v" to be a (costly) no-op: it will simply return the
   shape of its child as its own. *)

PROCEDURE New (ch: VBT.T; sh := FlexShape.DefaultShape): T;
(* "New(...)" is equivalent to "NEW(T).init(...)". *)

PROCEDURE FromAxis (ch: VBT.T;
                    ax: Axis.T;
                    sh           := FlexShape.Default): T;
(* Return a "FlexVBT" and whose shape specification in the "ax"
   dimension is "sh" and whose shape in the other dimension is
   that of "ch". *)

PROCEDURE Set (v: T; sh: FlexShape.Shape);
(* Change the shape of "v" to "sh", and notify "v"'s parent that
   "v"'s size has changed. *)

PROCEDURE SetRange (v: T; ax: Axis.T; sr: FlexShape.SizeRange);
(* Change the shape of "v" to "sr" along the "ax" axis, and
   notify "v"'s parent that "v"'s size has changed. *)

END FlexVBT.















