(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sun Aug  9 23:43:34 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 13:08:28 PDT 1992 by muller *)
(*      modified on Mon Jun 15 21:01:57 1992 by mhb *)

(* The "Radio" data structure supports mutually exclusive selections
   from a group of objects.

   This interface provides the notion of a group of "VBT"s.  After a
   group is created, "VBT"s can be added to and removed from the group.
   At most one member of the group is designated as the {\em current
   selection}.  The current selection may be set and retrieved; it may
   be "NIL".  All routines are self-locking.

   This interface could be used by non-"VBT" clients by recompiling
   with "MemberType" set to something other than "VBT.T".  It's
   probably a good candidate for Modula-3 generics. *)

INTERFACE Radio;

IMPORT List, VBT;

TYPE
  T <: REFANY;
  MemberType = VBT.T;

EXCEPTION NotAMember;

PROCEDURE New (): T;
(* Creates a new group. *)

PROCEDURE Add (group: T; member: MemberType);
(* Add "member" to "group".  This procedure is a no-op if "member"
   is "NIL" or if it's already a member of "group". *)

PROCEDURE Remove (group: T; member: MemberType)
  RAISES {NotAMember};
(* Remove "member" from "group".  If "member" was the selection,
   set the selection to "NIL".  Raise the exception if "member"
   is not a member of "group". *)

PROCEDURE Get (group: T): MemberType;
(* Return the currently selected member of "group".  If no
   member is selected, return "NIL". *)

PROCEDURE Put (group: T; member: MemberType) RAISES {NotAMember};
(* Make "member" the current selection of "group".  "member" may
   be "NIL".  The exception is raised if "member" is non-"NIL"
   and not a member of "group". *)

PROCEDURE IsMember (group: T; member: MemberType): BOOLEAN;
(* Return whether "member" belongs to "group" *)

PROCEDURE Members (group: T): List.T;
(* Return a list of members.  The elements will appear in the
   order in which they were added to "group". *)

END Radio.

