(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 16 13:08:27 PDT 1992 by muller     *)
(*      modified on Mon Jun 15 21:01:58 1992 by mhb        *)
(*      modified on Fri Mar 27 02:33:36 1992 by steveg     *)


MODULE Radio;

IMPORT List, Thread;

REVEAL T = BRANDED REF RECORD
    mu: Thread.Mutex; 
    selection: MemberType; 
    members: List.T 
    END;

PROCEDURE New (): T =
  VAR group := NEW (T);
  BEGIN
    group.mu := Thread.NewMutex();
    group.selection := NIL;
    RETURN group;
  END New;

PROCEDURE Add (group: T; member: MemberType) =
  BEGIN
    LOCK group.mu DO
      IF member # NIL AND NOT List.MemberQ(group.members, member) THEN
        List.Push(group.members, member)
      END
    END
  END Add;

PROCEDURE Remove (group: T; member: MemberType)
  RAISES {NotAMember} =
  BEGIN
    LOCK group.mu DO
      IF group.selection = member THEN
        group.selection := NIL
      ELSE
        IF NOT List.MemberQ(group.members, member) THEN
          RAISE NotAMember
        END
      END;
      group.members := List.Delete(group.members, member);
    END
  END Remove;

PROCEDURE Get (group: T): MemberType =
  BEGIN
    LOCK group.mu DO RETURN group.selection END
  END Get;

PROCEDURE Put (group: T; member: MemberType)
  RAISES {NotAMember} =
  BEGIN
    LOCK group.mu DO
      IF member # NIL AND NOT List.MemberQ(group.members, member) THEN
        RAISE NotAMember
      ELSE
        group.selection := member
      END
    END
  END Put;

PROCEDURE IsMember (group: T; member: MemberType): BOOLEAN =
  BEGIN
    LOCK group.mu DO
      RETURN List.MemberQ(group.members, member)
    END
  END IsMember;

PROCEDURE Members (group: T): List.T =
  BEGIN
    LOCK group.mu DO
      RETURN List.ReverseD(List.Copy(group.members))
    END
  END Members;

BEGIN 
END Radio.

