(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 16 13:08:15 PDT 1992 by muller     *)
(*      modified on Sat Jun 13 08:34:18 1992 by mhb        *)
(*      modified on Wed Mar  4 00:22:33 1992 by steveg     *)
(*      modified on Sat Nov 30  0:32:40 PST 1991 by meehan     *)


MODULE ShadowedVBT;

IMPORT Axis, Filter, MouseSplit, Point, Pts, Rect, Region, Shadow,
       ShadowPaint, VBT, VBTClass;

REVEAL
  T = Public BRANDED "ShadowedVBT.T" OBJECT
        shadow: Shadow.T;
        style : Shadow.Style;
        bSize : ARRAY Axis.T OF CARDINAL;  (* shadow's size in pixels *)
      OVERRIDES
        init      := Init;
        reshape   := Reshape;
        rescreen  := Rescreen;
        repaint   := Repaint;
        shape     := Shape;
        locate    := Locate;
      END;

PROCEDURE Init (v     : T;
                ch    : VBT.T;
                shadow: Shadow.T     := NIL;
                style : Shadow.Style := Shadow.Style.Flat): T =
  VAR size: REAL;
  BEGIN
    IF shadow = NIL THEN shadow := Shadow.None END;
    v.shadow := shadow;
    v.style := style;
    size := ABS(shadow.size);
    IF NOT Shadow.Supported(shadow, v) THEN size := size / 2.0 END;
    FOR ax := FIRST(Axis.T) TO LAST(Axis.T) DO
      v.bSize[ax] := Pts.ToScreenPixels(v, size, ax);
    END;
    EVAL Filter.Replace(v, ch);
    RETURN v
  END Init;

PROCEDURE New (ch    : VBT.T;
               shadow: Shadow.T     := NIL;
               style : Shadow.Style := Shadow.Style.Flat): T =
  BEGIN
    RETURN NEW(T).init(ch, shadow, style)
  END New;

PROCEDURE Shape (v: T; ax: Axis.T; n: CARDINAL): VBT.SizeRange =
  VAR
    temp: VBT.SizeRange;
    ch                  := Filter.Child (v);
  BEGIN
    IF ch = NIL THEN
      temp := VBT.DefaultShape  (* was VBTDotLeaf.shape(v, ax, n) *)
    ELSE
      temp := VBTClass.GetShape(ch, ax, n)
    END;
    RETURN VBT.SizeRange {lo := temp.lo + 2 * v.bSize [ax], hi :=
                          temp.hi + 2 * v.bSize [ax], pref :=
                          temp.pref + 2 * v.bSize [ax]}
  END Shape;

PROCEDURE Set (v: T; shadow: Shadow.T) =
  BEGIN
    IF v.shadow.size # shadow.size THEN VBT.NewShape (v); END;
    v.shadow := shadow;
    VBT.Mark (v);
  END Set;

PROCEDURE SetStyle (v: T; style: Shadow.Style) =
  BEGIN
    IF v.style # style THEN v.style := style; VBT.Mark(v); END;
  END SetStyle;

PROCEDURE Get (v: T): Shadow.T =
  BEGIN
    RETURN v.shadow;
  END Get;

PROCEDURE GetStyle (v: T): Shadow.Style =
  BEGIN
    RETURN v.style;
  END GetStyle;

PROCEDURE ChDom (v: T): Rect.T =
  (* Compute child domain from v's domain and border sizes *)
  BEGIN
    WITH dh = v.bSize [Axis.T.Hor], dv = v.bSize [Axis.T.Ver] DO
      RETURN Rect.Change (VBT.Domain (v), dh, -dh, dv, -dv)
    END
  END ChDom;

PROCEDURE Reshape (v: T; READONLY cd: VBT.ReshapeRec) RAISES {} =
  VAR ch := Filter.Child (v);
  BEGIN
    MouseSplit.InvalidateCache (v);
    IF ch # NIL THEN
      VBTClass.Reshape (ch, new := ChDom (v), saved := cd.saved)
    END;
    RepaintBorder (v, Region.Full)
  END Reshape;

PROCEDURE Rescreen (v: T; READONLY cd: VBT.RescreenRec) =
  VAR size: REAL;
  BEGIN
    size := ABS(v.shadow.size);
    IF NOT Shadow.Supported(v.shadow, v) THEN size := size / 2.0 END;
    FOR ax := FIRST(Axis.T) TO LAST(Axis.T) DO
      v.bSize[ax] := Pts.ToScreenPixels(v, size, ax);
    END;
    VBT.NewShape(v);
    Filter.T.rescreen (v, cd)
  END Rescreen;

PROCEDURE Repaint (v: T; READONLY badR: Region.T) =
  BEGIN
    Filter.T.repaint (v, badR);
    RepaintBorder(v, badR)
  END Repaint;


PROCEDURE RepaintBorder (v: T; READONLY clip: Region.T) =
  (* repaint the part of v's border that lies within clip. LL = VBT.mu. *)
  BEGIN
    ShadowPaint.Border(v, clip, v.shadow, v.style, ChDom(v), VBT.Domain (v));
  END RepaintBorder;

PROCEDURE Locate (v: T; READONLY pt: Point.T; VAR (* OUT *) r: Rect.T):
  VBT.T =
  BEGIN
    RETURN VBT.Split.locate (v, pt, r)
  END Locate;
  
BEGIN
END ShadowedVBT.
