(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sun Aug  9 23:55:56 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 21:18:24 PDT 1992 by muller *)
(*      modified on Mon Jun 15 21:14:18 1992 by mhb *)

(* A "SourceVBT" is a button that looks for special "VBT"s marked
   as {\em targets} while the mouse button remains down.  When
   the mouse is over a target, a procedure posted on the active
   "SourceVBT" is invoked to see if the target is acceptable for
   the source.  If so, an ``excited'' method on the target is
   invoked to give feedback.  Eventually, a target's ``normal''
   method is called to remove the feedback. *)

INTERFACE SourceVBT;

IMPORT DragSwitchVBT, Feedback, FeedbackClass, HighlightVBT, VBT;

TYPE
  T <: Public;
  Public = DragSwitchVBT.T OBJECT
           METHODS
             init (f: Feedback.T): T;
             hit (target: VBT.T; READONLY cd: VBT.PositionRec):
                  BOOLEAN := NeverHit
           END;

(* The call "v.init(...)" initializes "v" as a "SourceVBT".
   The type "SourceVBT.T" overrides the "pre", "post", "during",
   and "cancel" methods of its supertype.

   The default "pre" method changes the cursor to a starburst.
   The default "during" method calls the "hit" method whenever it
   is on a point controlled by a "VBT" that is a target.  If the
   "hit" method returns "TRUE", the target's "excited" and
   "normal" methods are called appropriately.  The "post" method
   invokes the target's "normal" method and restores the original
   cursor.  It's guaranteed that a target's "excited" and
   "normal" methods are called in non-nested pairs. *)

PROCEDURE GetHighlighter (v: T): HighlightVBT.T;
(* Returns the "HighlightVBT" above the nearest Trestle-installed
   ancestor of "v". *)

PROCEDURE GetTarget (v: T): Target;
(* Called by a "callback" method to find out the current target.
   That is, returns "w" for which the most recent call to
   "HitProc(v,w)" returned "TRUE", or "NIL" if\/ "HitProc" returned
   "FALSE" or if the mouse is not over a valid target. *)

PROCEDURE AlwaysHit (         v     : Public;
                              target: VBT.T;
                     READONLY cd    : VBT.PositionRec): BOOLEAN;
(* Always return "TRUE". *)

PROCEDURE NeverHit (         v     : Public;
                             target: VBT.T;
                    READONLY cd    : VBT.PositionRec): BOOLEAN;
(* Always returns "FALSE". *)


(* \subsubsection{Targets} *)

TYPE Target = VBT.T;
(* A target is a "VBT" on which "BeTarget" has been invoked. *)

TYPE TargetClass <: FeedbackClass.T;
(* A target class determines the feedback when a target's "excited"
   method is called. *)
   
PROCEDURE BeTarget (w: VBT.T; class: TargetClass);
(* Make "w" into a target for a "SourceVBT".  As a target, "w" may
   be passed to some "SourceVBT"'s "hit" method. *)

PROCEDURE GetSource (w: Target): T;
(* Called by a target's "normal" or "excited" methods to find out
   the "SourceVBT" causing the method to be invoked. *)

PROCEDURE NewInserterTarget (): TargetClass;
(* Displays a grid over itself when excited.  Appropriate for an
   adjusting bar in a tiling window manager, like the Trestle
   window manager on SRC's Firefly.  The parent of the target
   must be an "HVSplit", and grid has a minimun size in the
   "HVSplit"'s axis. *)

PROCEDURE NewSwapTarget (): TargetClass;
(* Displays a grid over itself when excited.  This target is
   appropriate for a non-adjusting bar in a tiling window
   manager. *)

PROCEDURE NewTarget (): TargetClass;
(* Inverts itself when excited.  This target class is a
   general-purpose target. *)

PROCEDURE TargetClassOf (w: Target): TargetClass;
(* Return the "class" argument for which there was a previous call
   to "BeTarget(w, class)", or "NIL" if there was no such call. *)
   
END SourceVBT.








