(* Copyright (C) 1989-1992, Digital Equipment Corporation       *)
(* All rights reserved.                                         *)
(* See the file COPYRIGHT for a full description.               *)
(*                                                              *)
(* Last modified on Sun Aug 16 23:22:04 PDT 1992 by meehan      *)
(*      modified on Mon Jun 15 22:38:47 1992 by mhb             *)
(*      modified on Fri Mar 20 22:34:09 1992 by steveg          *)
(*      modified on Fri Feb 8 15:36:58 PST 1991 by brooks       *)
(*      modified on Wed May 17 17:06:31 PDT 1989 by gidi        *)
<* PRAGMA LL *>

(* A "TextPort" is a "VBT" that allows the user to type and edit
   single- or multi-line text.  It provides a default set of
   keybindings, described in Appendix~\ref{TextPortUI}, for the
   editing commands.  The client can effectively change the
   keybindings by overriding the "filter" method.
   
   The implementation provides a two-selection model, ``primary''
   and ``secondary'', corresponding to "VBT.Target" and
   "VBT.Source", respectively.  For a description of the
   selection model, see Appendix~\ref{TextPortSelection}.

   The implementation also supports a simple copy/cut/paste
   selection model.  See section \ref{Clipboard} for details.
   
   "TextPort" is internally synchronized, so clients do not need
   to worry about concurrent calls.  The locking level for all
   procedures is "LL <= VBT.mu" except as noted.
   *)

INTERFACE TextPort;

IMPORT Font, PaintOp, ScrollerVBT, VBT, VText;

TYPE
  Points = REAL;
  T <: Public;
  Public =
    VBT.Leaf OBJECT
    METHODS
      init (singleLine                          := FALSE;
            hMargin, vMargin : Points           := 1.5;
            font                                := Font.BuiltIn;
            colorScheme   : PaintOp.ColorScheme := NIL;
            expandOnDemand                      := FALSE;
            wrap                                := TRUE;
            readOnly                            := FALSE;
            turnMargin       : Points           := 2.0    ): T;
      returnAction   (READONLY cd: VBT.KeyRec);
      tabAction      (READONLY cd: VBT.KeyRec);
      defaultAction  (READONLY cd: VBT.KeyRec);
      scrollUpdate   (viewStart, viewEnd, length: INTEGER);
      getFont        (): Font.T;
      setFont        (font: Font.T); <* LL = VBT.mu *>
      getColorScheme (): PaintOp.ColorScheme;
      setColorScheme (c: PaintOp.ColorScheme); <* LL = VBT.mu *>
      focus          (gaining: BOOLEAN; time: VBT.TimeStamp);
      filter         (VAR (* inOut *) cd: VBT.KeyRec);
      modified       ();
    END;

(* The call "v.init(...)" initializes "v" as a "TextPort" and
   returns "v".

   If "singleLine" is "TRUE", then the "TextPort"'s height is set
   to the height of the tallest character in the current font,
   and it does not expand vertically.  Its default width is 30
   times the width of the widest character in the current font;
   characters. This is intended for ``type-in'' boxes.

   The parameters "hMargin" and "vMargin" indicate how much
   whitespace, measured in points, to leave around the text.

   If "colorScheme" is "NIL", then "PaintOp.bgFg" will be used.

   If "expandOnDemand" is "TRUE", then multi-line and scrollable
   "TextPort"s will expand (and contract) vertically as the text
   requires, so that the entire text is visible in the window.
   This parameter is ignored for single-line or nonscrollable
   "TextPort"s.

   If "wrap" is "TRUE" and "singleLine" is "False", then text
   will be wrapped across line boundaries; otherwise it will
   be clipped.  If it is wrapped, then "turnMargin" specifies
   the width of the gray bar placed at the end of the first line
   and the beginning of the second, indicating that the text
   has been wrapped.

   If "readOnly" is "TRUE", then the text cannot be changed
   through the user interface (keyboard and mouse).  Primary and
   secondary selections are allowed, and the secondary selection
   can be used for copying (but not removing) text.

   The implementation calls "v.returnAction(cd)" when user presses 
   the Return key.  (Shift-return calls "Newline(v)", and option-Return 
   inserts a newline {\em after} the cursor.) The default method 
   calls "v.defaultAction(cd)" method if this is a ``singleLine'' 
   "TextPort"; otherwise it calls "NewlineAndIndent(v, vd)". 

   The implementation calls "v.tabAction(cd)" when the user
   presses the Tab key, regardless of modifiers.  The default
   method inserts 4 spaces.

   The implementation calls "v.defaultAction(cd)" when user types
   a non-printing key or a key/modifier combination that has no
   predefined function.  The default is a no-op.

   The implementation calls "v.scrollUpdate" whenever "v"'s state
   has changed in a way that might change the associated
   scrollbar, if any.  The default calls "ScrollerVBT.Update"
   with the "TextPort"'s scrollbar, if it has one.

   The call "v.setFont(font)" changes the font used for displaying the
   text.

   The call "v.setColorScheme(c)" changes the colors used for displaying
   the text.

   The implementation calls "v.focus(gaining, ...)" whenever "v" gains
   or loses the keyboard focus.  "gaining" is "TRUE" if "v" is
   gaining the keyboard focus, "FALSE" otherwise.  When gaining
   the focus, this method is called {\em before} the selection
   feedback is established, so it is reasonable to call "Select"
   (below).  It is not within the power of the "focus" method to
   prevent "v" from gaining or losing the focus.  The default is
   a no-op.

   \subsection {Keybindings}

   "TextPort" has a predefined set of ``keybindings'', mappings
   from keystrokes to procedures.  A keystroke for which there is
   a mapping is called a ``bound key''.  "v.key(cd)" starts by
   calling "v.filter(cd)".  Then, if the key is bound, it applies
   the associated procedure; otherwise it calls
   "v.defaultAction(cd)".  The default "filter" and
   "defaultAction" methods are no-ops, but by overriding these
   methods, the client can effectively reprogram the behavior of
   the editor.

   Note that "cd" is a "VAR" parameter to the "filter" method.
   In addition to calling other procedures in the "TextPort"
   interface, a "filter" method may:

   \begin{itemize}
   \item{leave "cd" untouched,}

   \item{translate it into some other keystroke, by changing
   "cd.whatChanged" and/or "cd.modifiers", or}

   \item{suppress it entirely by setting "cd.whatChanged" to
   "VBT.NoKey", which is a bound key that has no effect.}
   \end{itemize}

   If the client's "filter" method handles a key directly, by
   taking some action, then it should usually set
   "cd.whatChanged" to "VBT.NoKey".  Since unbound keys will be
   passed to both the "filter" and "defaultAction" methods, it is
   usually a mistake for both methods to handle an unbound key.

   \subsection{Other state information}

   A "TextPort" maintains some internal information that is
   accessed indirectly:

   \begin{itemize}
   \item a ``modified'' flag.  Any operation that
   changes the text sets this flag to "TRUE".  If it was
   previously "FALSE", then the implementation calls
   "v.modified()", {\bf after} the change has already happened to
   "v".  The default is a no-op.  The "SetModified" and
   "IsModified" procedures set and test this flag, respectively.

   \item a scrollbar (optional).  See the "TextEditVBT" interface
   in Section~\ref{TextEditVBTSection}.
   \end{itemize}

   *)
   
TYPE Scroller <: ScrollerVBT.T;
(* A scrollbar that is to be used with a "TextPort" must be
   one of these.  *)

PROCEDURE SetScrollBar (v: T; scrollBar: Scroller);
(* Set the scrollbar associated with "v". *)

PROCEDURE SetReadOnly (v: T; readOnly: BOOLEAN);
(* Set the ``read-only'' property of "v".  *)

PROCEDURE SetWrap (v: T; wrap: BOOLEAN);
(* Set the ``wrap'' property of "v", and redisplay it. *)

PROCEDURE Length (v: T): CARDINAL;
(* Return the number of characters in "v". *)

PROCEDURE Width (v: T): CARDINAL;
(* Return the number of characters that will fit on a line, given
   the current size of "v".  If "v" has no width (because
   its window is iconic, for example), return 0. If
   "v"'s font is not fixed-pitch, the result will be computed
   in terms of the width of the widest character in the font.
   *)

PROCEDURE VertSize (v: T): CARDINAL;
(* Equivalent to
| v.shape(Axis.T.Ver, 0)
*)

PROCEDURE ShapeInfo (v: T; VAR lineCount, lineLength: INTEGER);
(* Return the number of lines in the text and the number of
   characters in the longest line (excluding the newline).  A
   client may use this information in conjunction with
   "VBTShape.Set" to shape "v" to fit its text.  Having done so,
   it should be prepared to do it again on a rescreen to a screen
   of different density. *)

PROCEDURE Newline (v: T; READONLY event: VBT.KeyRec);
(* Insert a newline character at the type-in point, using the
   semantics of "StandardInsert", below. *)

PROCEDURE NewlineAndIndent (v: T; READONLY event: VBT.KeyRec);
(* Insert a newline character and enough spaces to match the
   indentation of the previous line.  As it leaves a blank line,
   it will delete all spaces from that line so as to leave it
   truly empty. *)

(* \subsubsection{Editing commands} *)

PROCEDURE GetText (v    : T;
                   begin: CARDINAL := 0;
                   end  : CARDINAL := LAST (CARDINAL)): TEXT;
<* LL = VBT.mu *>
(* Returns a sub-sequence of the text in "v".  The result will be
   empty if "begin >= Length(v)"; otherwise the range of indexes
   of the subsequence is
| [begin .. MIN (end, Length (v)) - 1]
   *)

PROCEDURE SetText (v: T; t: TEXT); <* LL <= VBT.mu *>
(* Replace the current contents of "v" with "t".  *)

PROCEDURE PutText (v: T; t: TEXT); <* LL <= VBT.mu *> 
(* Append "t" to the current contents of "v". *)

PROCEDURE Replace (v: T; begin, end: CARDINAL; newText: TEXT);
(* Replace the text between positions "begin" and "end" in "v"
   with "newText".  If "begin" and "end" are beyond the end of
   the text, they are taken to refer to the end of the text. *)

PROCEDURE StandardInsert (v: T; text: TEXT);
(* Insert "text" at the current type-in point in "v" in standard
   fashion: if there is a replace-mode selection (see
   Section~\ref{ReplaceMode}, page~\pageref{ReplaceMode}),
   replace it with "text"; otherwise simply insert "text" at
   the type-in point.  *)

(* \subsubsection{Selection} *)

(* Each "TextPort" maintains two text-selections, referred to as
   ``primary'' and ``secondary'', corresponding to "VBT.Target"
   and "VBT.Source", respectively.  User actions that establish
   the primary selection cause "VBT.Target" to be acquired;
   likewise for the secondary selection and "VBT.Soure". *)
   
TYPE
  SelectionType = {Primary, Secondary};
  WhichEnd = VText.WhichEnd;    (* Right, Left *)

(* As described in detail in Section \ref{TextPortSelection}, "TextPort" 
   uses a two-selection model.  "Primary" corresponds to "VBT.Target", 
   and "Secondary" corresponds to "VBT.Source", each represented 
   by a pair of indexes ("begin" and "end") into the text.  This 
   range-information is retained even after the "TextPort" has lost 
   the "VBT.Selection"; it represents the {\em most recent} selection 
   for this "TextPort" (see Section \ref{RestoringPrimary}).  and 
   can be retrieved via the "Selection" procedure.  The procedure 
   "TryFocus" tries to acquire the keyboard focus; if it succeeds, 
   it also tries to re-acquire the primary selection, using the retained 
   information.  *) 

PROCEDURE Select (v         : T;
                  time      : VBT.TimeStamp;
                  begin, end: CARDINAL;
                  sel         := SelectionType.Primary;
                  replaceMode := FALSE;
                  caretEnd    := WhichEnd.Right         );
(* Make a selection in "v", at event-time "time".  If\/ "begin"\/
   and/or\/ "end"\/ are beyond the end of the text, they will be
   clipped to the end of the text.  Take ownership of the
   corresponding "VBT.Selection", and, in case of
   "SelectionType.Primary", of the keyboard focus as well.

   The values "replaceMode" and "caretEnd" are relevant only if
   "sel = SelectionType.Primary".  If "replaceMode" is "TRUE",
   then "Insert" and "VBT.Write" will {\em replace} the selected
   text; otherwise, they cause the new text to be {\em inserted}
   at whichever end of the primary selection is specified by
   "caretEnd". *)

PROCEDURE Unselect (v: T; t: VBT.TimeStamp);
(* Remove any caret and primary selection that may have been
   present. *)

PROCEDURE HasSelection (v: T; sel := SelectionType.Primary):
  BOOLEAN; <* LL.sup = VBT.mu *>
(* Determine whether "v" currently owns a given selection. *)

PROCEDURE Selection (              v         : T;
                     VAR (* out *) begin, end: VText.Index;
                     sel := SelectionType.Primary);
(* Determine the extent of the most recent primary or secondary
   selection in "v". *)

PROCEDURE CaretPosition (v: T): CARDINAL;
(* Return the current caret (``type-in'') position. *)

PROCEDURE IsReplaceMode (v: T): BOOLEAN;
(* Return "TRUE" if "replaceMode" was "TRUE" in the most recent
   call to "Select(v, ...)". *)

PROCEDURE TryFocus (v: T; t: VBT.TimeStamp): BOOLEAN;
(* Try to acquire the keyboard focus and the primary selection,
   and report whether it succeeded. *)

PROCEDURE HasFocus (v: T): BOOLEAN; <* LL.sup = VBT.mu *>
(* Test whether "v" has the keyboard focus. *)

PROCEDURE Normalize (v: T; to := -1);
(* Scroll "v" if necessary to ensure that position "to" is
   visible.  If "to" is defaulted, it refers to the current
   type-in point.  If "to" is larger than the length of the text,
   normalizes to the end of the text. *)

PROCEDURE IsVisible (v: T; pos: CARDINAL): BOOLEAN;
(* Test whether character position "pos" is visible. *)

PROCEDURE IsModified (v: T): BOOLEAN;
(* Return the value of the ``modified'' flag for "v". Any change to
   the text will cause this flag to be set to "TRUE". *)

PROCEDURE SetModified (v: T; modified: BOOLEAN);
(* Set the value of the ``modified'' flag for "v". *)

PROCEDURE GetVText (v: T): VText.T;
(* For wizards only: extract the underlying "vtext".  It is legal
   to create and manipulate highlighting intervals on it.  It is
   legal to run readers on it, provided you can be sure that you
   are locking out concurrent change (for example, by holding
   "VBT.mu").  It is not legal to modify it directly.  It
   is not legal to scroll it directly either, because that will
   leave the scrollbar incorrect. *)

END TextPort.

