(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Thu Jul 30 10:23:51 PDT 1992 by meehan     *)
(*      modified on Tue Jun 16 13:08:05 PDT 1992 by muller     *)
(*      modified on Thu Jul 11 12:55:24 PDT 1991 by mhb        *)
(*      modified on Thu Apr  4 15:55:03 PST 1991 by brooks     *)
(*      modified on Thu May 17  9:54:12 PDT 1990 by mcjones    *)

MODULE TypescriptVBT;

(** Here's how the typescript works:

    The vtext holds the underlying text.
    0 <= outputEnd <= typeinStart <= len(vtext).

    vtext [0 .. outputEnd-1] is the "history".  It is accessible to
    neither the reader nor the writer.  ClearHistory "erases" this, i.e.,
    deletes that section of the vtext and decrements outputEnd and
    typeinStart accordingly.

    Wr.Flush inserts characters at outputEnd, in the "middle" of the
    vtext.  After the insertion, outputEnd and typeinStart are
    incremented by the number of inserted characters.

    vtext [outputEnd ..  typeinStart-1] is the section that's accessible
    to the reader.  RSeek copies characters from this part of the vtext.
    If outputEnd = typeinStart (i.e., if there are no characters
    available) and dontBlock is false, then RSeek calls Wr.Flush and
    waits for inputReady to be signaled.

    vtext [typeinStart ..  len(vtext) - 1] contains typed-in characters.
    That is, keyboard input is appended to the end of the vtext.  This
    segment is editable.  When Return is typed, a Newline is appended,
    typeinStart is set to len(vtext), and inputReady is signaled, thus
    making the input line accessible to the reader.

**)

IMPORT Font, MText, PaintOp, Rd, RdClass, Shadow, Text, TextEditVBT,
       TextPort, TextPortPrivate, Thread, VBT, VTDef, VText, Wr, WrClass;

<* FATAL VTDef.Error, Rd.Failure, Rd.EndOfFile, Thread.Alerted, Wr.Failure *>

REVEAL
  T = Public BRANDED "TypescriptVBT.T" OBJECT
        rd        : Reader;
        wr        : Writer;
        lastReader: Thread.T;          (* whom to alert on ^C *)
        inputReady: Thread.Condition;
        terminated: BOOLEAN;    (* This record is protected by v.mu *)
        outputEnd : CARDINAL;
      OVERRIDES
        init            := Init;
        interrupt       := Interrupt;
        handleInterrupt := HandleInterrupt;
        setThread       := SetThread;
        terminate       := Terminate;
      END;

REVEAL
  Port = PublicPort BRANDED "TypescriptVBT.Port" OBJECT
           v: T
         OVERRIDES
           init          := InitPort;
           returnAction  := ReturnAction;
         END;
        
CONST TerminalReaderBuffSize = 4096;

REVEAL
  Reader = PublicReader BRANDED "Typescript.Reader" OBJECT
             v: T
           OVERRIDES
             seek       := RSeek;
             typescript := RdTypescript
           END;


CONST TerminalWriterBuffSize = 4096;

REVEAL
  Writer = PublicWriter BRANDED OBJECT
             v: T
           OVERRIDES
             seek       := WSeek;
             flush      := WFlush;
             typescript := WrTypescript
           END;

PROCEDURE Init (v     : T;
                shadow: Shadow.T     := NIL (* Shadow.None *);
                style : Shadow.Style := Shadow.Style.Flat      ):
  T =
  VAR port: Port := v.port;
  BEGIN
    IF shadow = NIL THEN shadow := Shadow.None END;
    IF port = NIL THEN
      port := NEW (Port).init (colorScheme := shadow);
      v.port := port
    END;
    port.v := v;
    EVAL
      TextEditVBT.T.init (v, shadow := shadow, style := style);
    v.inputReady := NEW (Thread.Condition);
    v.rd := NEW (Reader, v := v, lo := 0, cur := 0, hi := 0,
                 st := 0, buff := NEW (REF ARRAY OF CHAR,
                                       TerminalReaderBuffSize),
                 closed := FALSE, seekable := FALSE,
                 intermittent := TRUE);
    v.wr :=
      NEW (
        Writer, v := v, lo := 0, cur := 0,
        hi := TerminalWriterBuffSize, st := 0,
        buff := NEW (REF ARRAY OF CHAR, TerminalWriterBuffSize),
        closed := FALSE, seekable := FALSE, buffered := TRUE);
    v.terminated := FALSE;
    v.outputEnd := 0;
    RETURN v
  END Init;

PROCEDURE InitPort (p               : Port;
                    hMargin, vMargin                      := 1.5;
                    font                                  := Font.BuiltIn;
                    colorScheme     : PaintOp.ColorScheme := NIL;
                    wrap                                  := TRUE;
                    readOnly                              := FALSE;
                    turnMargin                            := 2.0           ):
  Port =
  BEGIN
    EVAL TextPort.T.init (p, FALSE, hMargin, vMargin, font, colorScheme,
                          FALSE, wrap, readOnly, turnMargin);
    RETURN p
  END InitPort;

(***********************  Typescript-specific code  ***********************)

PROCEDURE WSeek (wr: Writer; <* UNUSED *> n: CARDINAL)
  RAISES {Wr.Failure, Thread.Alerted} =
  BEGIN
    wr.flush ()
  END WSeek;

PROCEDURE WFlush (wr: Writer) RAISES {Wr.Failure, Thread.Alerted} =
  VAR
    v    := wr.v;
    port := v.port;
  BEGIN
    LOCK port.mu DO
      WITH normP = VText.InRegion (
                     port.vtext, 0, VText.CaretIndex (port.vtext)),
           nchars = wr.cur - wr.lo DO
        VText.ReplaceSub (
          port.vtext, v.outputEnd, v.outputEnd, wr.buff^, 0, nchars);
        INC (v.outputEnd, nchars);
        INC (port.typeinStart, nchars);
        wr.lo := wr.cur;
        wr.hi := wr.lo + NUMBER (wr.buff^);
        IF normP THEN
          TextPort.Normalize (port)
        ELSE
          TextPortPrivate.MarkAndUpdate (port)
        END
      END
    END;
    IF Thread.TestAlert () THEN RAISE Thread.Alerted END
  END WFlush;


PROCEDURE RSeek (rd: Reader; dontBlock: BOOLEAN): RdClass.SeekResult
  RAISES {Rd.Failure, Thread.Alerted} =
  VAR nchars: CARDINAL;
  BEGIN
    WITH v = rd.v, p = v.port DO
      TRY
        LOCK p.mu DO
          v.lastReader := Thread.Self();
          nchars := p.typeinStart - v.outputEnd;
          IF nchars > 0 THEN
          ELSIF v.terminated THEN
            rd.buff := NIL;
            RETURN RdClass.SeekResult.Eof
          ELSIF dontBlock THEN
            RETURN RdClass.SeekResult.WouldBlock
          ELSE
            REPEAT
              Thread.Release(p.mu);
              TRY Wr.Flush(v.wr) FINALLY Thread.Acquire(p.mu) END;
              Thread.AlertWait(p.mu, v.inputReady);
              nchars := p.typeinStart - v.outputEnd
            UNTIL nchars > 0
          END;
          WITH
            n = MIN(nchars, NUMBER (rd.buff^)),
            txt = MText.GetText(
                    p.vtext.mtext, v.outputEnd, v.outputEnd + n) DO
            Text.SetChars(rd.buff^, txt);
            INC(v.outputEnd, n);
            rd.lo := rd.cur;
            rd.hi := rd.lo + n; (* NOT v.outputEnd! *)
            RETURN RdClass.SeekResult.Ready
          END                   (* WITH n *)
        END                     (* LOCK *)
      EXCEPT
        Thread.Alerted =>
          (* Since thread may end up in debugger, we must raise Alerted
             *without* holding p.mu. *)
          RAISE Thread.Alerted
      END                       (* TRY *)
    END                         (* WITH v, p *)
  END RSeek;

PROCEDURE ReturnAction (p: Port; READONLY event: VBT.KeyRec) =
  BEGIN
    IF event.modifiers = VBT.Modifiers {} AND NOT p.readOnly THEN
      (* Input action, called when the user presses Return in the input
         area. Unblocks RSeek if it was blocked. *)
      LOCK p.mu DO
        VText.MoveCaret (p.vtext, MText.Length (p.vtext.mtext));
        TextPortPrivate.InsertLocked (p, "\n");
        p.typeinStart := MText.Length (p.vtext.mtext)
      END;
      (* activate the reading client *)
      Thread.Signal (p.v.inputReady);
      TextPort.Normalize (p)
    END
  END ReturnAction;

PROCEDURE Interrupt (v: T; time: VBT.TimeStamp) =
  (* Interrupt, called when the user types option-C. It flushes (ignores)
     all pending typein, then calls the interrupt handler. *)
  VAR
    length: INTEGER;
    p               := v.port;
  BEGIN
    LOCK p.mu DO
      length := MText.Length (p.vtext.mtext);
      VText.MoveCaret (p.vtext, length);
      TextPortPrivate.InsertLocked (p, "^C");
      length := MText.Length (p.vtext.mtext);
      v.outputEnd := length;    (* flush all pending typein *)
      p.typeinStart := length
    END;
    v.handleInterrupt (time)
  END Interrupt;

PROCEDURE HandleInterrupt (v: T; <* UNUSED *> time: VBT.TimeStamp) =
  VAR reader: Thread.T;
  BEGIN
    LOCK v.port.mu DO reader := v.lastReader END;
    IF reader # NIL THEN Thread.Alert (reader) END
  END HandleInterrupt;

PROCEDURE GetRd (v: T): Reader =
  BEGIN
    RETURN v.rd
  END GetRd;

PROCEDURE GetWr (v: T): Writer =
  BEGIN
    RETURN v.wr
  END GetWr;

PROCEDURE RdTypescript (r: Reader): T =
  BEGIN 
    RETURN r.v
  END RdTypescript;
  
PROCEDURE WrTypescript (r: Writer): T =
  BEGIN 
    RETURN r.v
  END WrTypescript;
  
PROCEDURE GetHistory (v: T): TEXT =
  BEGIN
    WITH port = v.port DO
      LOCK port.mu DO
        RETURN MText.GetText (port.vtext.mtext, 0, v.outputEnd)
      END
    END
  END GetHistory;

PROCEDURE ClearHistory (v: T) =
  BEGIN
    WITH port = v.port DO
      LOCK port.mu DO
        VText.Replace (port.vtext, 0, v.outputEnd, "");
        DEC (port.typeinStart, v.outputEnd);
        v.outputEnd := 0;
        TextPortPrivate.MarkAndUpdate (port)
      END
    END
  END ClearHistory;

  
(**************************  Special controls  **************************)

PROCEDURE SetThread (v: T; thread: Thread.T := NIL) =
  BEGIN
    LOCK v.port.mu DO
      IF thread = NIL THEN
        v.lastReader := Thread.Self ()
      ELSE
        v.lastReader := thread
      END
    END
  END SetThread;

PROCEDURE Terminate (v: T) =
  BEGIN
    LOCK v.port.mu DO v.terminated := TRUE; v.port.readOnly := TRUE  END;
    Thread.Signal (v.inputReady)
  END Terminate;

BEGIN
END TypescriptVBT.
