(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sun Jul  5 20:07:25 PDT 1992 by meehan *)
(*      modified on Tue Jun 30 16:01:44 PDT 1992 by mhb    *)
(*      modified on Tue Jun 16 13:16:22 PDT 1992 by muller *)
(*      modified on Sat Feb 29 11:46:47 PST 1992 by kalsow *)
(*      modified on Mon May 7 8:49:45 PDT 1990 by mcjones *)
(*      modified on Wed Nov 15 10:29:45 1989 by chan *)
(*      modified on Thu May 4 9:27:56 PDT 1989 by mbrown *)

MODULE MTextRd;

(* Note on readers: 'mtext' always refers to the MText in which this reader
   reads. 'node' is the node which HAS BEEN GIVEN to the reader most
   recently. 'nodeIndex' is the index of the first character of that node,
   relative to the start of the MText. 'limit' is the Head which limits
   this reader, or NIL if this reader has no limit. A reader will NEVER be
   found beyond its limit. atEnd is true IFF node = limit.node or
   node.type = anchor. When 'atEnd' is true, the Get procedure will return
   -1 without attempting to advance. A Get that gets the node containing
   the limit will return the part of that node that is before the limit,
   and set atEnd to TRUE. A Seek may set atEnd true or false, but it will
   never seek past the limit. *)

IMPORT RdClass, MText, MTextDs, MTextPrivate, Rd, Text, TextF, Thread;
FROM MTextPrivate IMPORT Node, NodeType, Check;

CONST BufferSize = 512;

REVEAL
  T = Rd.T BRANDED "MTextRd.T.1" OBJECT
        m                   : MText.T;
        rangeStart, rangeEnd: CARDINAL;
        version             : INTEGER;
        alertable           : BOOLEAN;
      OVERRIDES
        length  := ReaderLength;
        close   := Close;
      END;

TYPE
  ForwardReader =
    T BRANDED "ForwardReader" OBJECT OVERRIDES seek := Seek END;
  ReverseReader =
    T BRANDED "ReverseReader" OBJECT OVERRIDES seek := RevSeek END;


PROCEDURE New (m         : MText.T;
               start     : CARDINAL   := 0;
               rangeStart: CARDINAL   := 0;
               rangeEnd  : CARDINAL   := LAST(CARDINAL);
               reverse   : BOOLEAN    := FALSE;
               alertable : BOOLEAN    := FALSE           ): T =
  <* FATAL Rd.Failure, Thread.Alerted *>
  VAR t: T;
  BEGIN 
    Check (rangeStart, rangeEnd, m.length);
    start := MIN (MAX (start, rangeStart), rangeEnd);
    IF reverse THEN
      t := NEW(ReverseReader, m := m, alertable := alertable,
               version := m.version, closed := FALSE, seekable := TRUE,
               intermittent := FALSE, st := 0, lo := 0, hi := 0,
               rangeStart := rangeStart, rangeEnd := rangeEnd,
               cur := rangeEnd - start,
               buff := NEW(REF ARRAY OF CHAR, BufferSize));
      Rd.Seek(t, rangeEnd - start)
    ELSE
      t := NEW(ForwardReader, m := m, alertable := alertable,
               version := m.version, closed := FALSE, seekable := TRUE,
               intermittent := FALSE, st := 0, lo := 0, hi := 0,
               rangeStart := rangeStart, rangeEnd := rangeEnd,
               cur := start - rangeStart,
               buff := NEW(REF ARRAY OF CHAR, BufferSize));
      Rd.Seek(t, start - rangeStart)
    END;
    RETURN t
  END New;

PROCEDURE Seek (rd: ForwardReader;
                <* UNUSED *>
                dontBlock: BOOLEAN): RdClass.SeekResult
  RAISES {Rd.Failure, Thread.Alerted} =
  VAR
    beginN          : Node;
    beginI, count, i: CARDINAL;
    result                     := RdClass.SeekResult.Ready;
  BEGIN
    IF rd.cur + rd.rangeStart >= rd.rangeEnd THEN
      RETURN RdClass.SeekResult.Eof
    END;
    MTextDs.LocateB (rd.m, rd.cur + rd.rangeStart, beginN, beginI);
    count := MIN (MIN (beginN.length - beginI, BufferSize),
                  rd.rangeEnd - rd.rangeStart - rd.cur);
    CASE beginN.type OF
    | NodeType.text =>
        SUBARRAY (rd.buff^, 0, count) :=
          SUBARRAY (beginN.text^, beginI, count);
    | NodeType.buf =>
        SUBARRAY (rd.buff^, 0, count) :=
          SUBARRAY (beginN.buffer^, beginI, count);
    | NodeType.file =>
        Rd.Seek (beginN.file, beginI + beginN.start);
        i := 0;
        TRY
          WHILE i < count DO
            rd.buff [i] := Rd.GetChar (beginN.file);
            INC (i)
          END
        EXCEPT
          Rd.EndOfFile =>
        END;
        count := i;
    ELSE <* ASSERT FALSE *>
    END;
    rd.lo := rd.cur;
    rd.hi := rd.lo + count;
    RETURN result
  END Seek;

PROCEDURE RevSeek (rd: ReverseReader; <* UNUSED *> dontBlock: BOOLEAN):
  RdClass.SeekResult RAISES {Rd.Failure, Thread.Alerted} =
  VAR
    beginN                 : Node;
    beginI, count, first, i: CARDINAL;
  PROCEDURE RevCopy (READONLY v: ARRAY OF CHAR) =
    BEGIN
      FOR i := 1 TO count DO rd.buff [count - i] := v [first + i - 1] END
    END RevCopy;
  BEGIN
    IF rd.cur + rd.rangeStart >= rd.rangeEnd THEN
      RETURN RdClass.SeekResult.Eof
    END;
    MTextDs.Locate (rd.m, rd.rangeEnd - rd.cur, beginN, beginI);
    count :=
      MIN (MIN (beginI, BufferSize), rd.rangeEnd - rd.rangeStart - rd.cur);
    first := beginI - count;
    CASE beginN.type OF
    | NodeType.text =>
        RevCopy (SUBARRAY (beginN.text^, 0, Text.Length (beginN.text)))
    | NodeType.file =>
        Rd.Seek (beginN.file, beginN.start + first);
        i := 0;
        TRY
          WHILE i < count DO
            rd.buff [count - 1 - i] := Rd.GetChar (beginN.file);
            INC (i)
          END
        EXCEPT
          Rd.EndOfFile =>
        END;
        count := i
    | NodeType.buf => RevCopy (beginN.buffer^)
    ELSE <* ASSERT FALSE *>
    END;
    rd.lo := rd.cur;
    rd.hi := rd.lo + count;
    RETURN RdClass.SeekResult.Ready
  END RevSeek;


PROCEDURE ReaderLength (t: T): CARDINAL =
  BEGIN
    LOCK t.m.lock DO RETURN t.rangeEnd - t.rangeStart END
  END ReaderLength;

PROCEDURE Close (t: T) RAISES {Rd.Failure, Thread.Alerted} =
  BEGIN
    t.m := NIL;                 (* release the mtext *)
    Rd.T.close (t)
  END Close;

BEGIN
END MTextRd.



