(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(* Last modified on Fri Jun 12 18:03:57 1992 by mhb                          *)

MODULE PixmapFromAscii;

IMPORT AsciiUtil, Pixmap, Point, Rd, Rect, Scan, ScrnPixmap, Text, Thread,
         Word;

EXCEPTION Unimplemented; 

<* FATAL Unimplemented *>
<* FATAL Rd.Failure *>

PROCEDURE NextWord (rd: Rd.T): Word.T RAISES {Thread.Alerted, Error} =
  VAR res: INTEGER; ch: CHAR;
  BEGIN
    TRY
      ch := Rd.GetChar(rd);
      WHILE ch = ' ' OR ch = '\n' OR ch = '\t' OR ch = ',' DO
        ch := Rd.GetChar(rd)
      END;
      res := 0;
      CASE ch OF
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' =>
          res := ORD(ch) - ORD('0')
      | 'a', 'b', 'c', 'd', 'e', 'f' => res := ORD(ch) - ORD('a') + 10
      | 'A', 'B', 'C', 'D', 'E', 'F' => res := ORD(ch) - ORD('A') + 10
      ELSE RAISE Error
      END;
    EXCEPT
    | Rd.EndOfFile => RAISE Error
    END;
    TRY
      LOOP
        ch := Rd.GetChar(rd);
        CASE ch OF
          '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' =>
            res := 16 * res + ORD(ch) - ORD('0')
        | 'a', 'b', 'c', 'd', 'e', 'f' =>
            res := 16 * res + ORD(ch) - ORD('a') + 10
        | 'A', 'B', 'C', 'D', 'E', 'F' =>
            res := 16 * res + ORD(ch) - ORD('A') + 10
        | 'x', 'X' => res := 0
        ELSE EXIT
        END
      END;
    EXCEPT
    | Rd.EndOfFile =>
    END;
    RETURN res
  END NextWord;

PROCEDURE Read (rd: Rd.T): Pixmap.T RAISES {Thread.Alerted, Error} =
  BEGIN
    RETURN Pixmap.FromBitmap(ReadRaw(rd))
  END Read;

PROCEDURE ReadRaw (rd: Rd.T): Pixmap.Raw RAISES {Thread.Alerted, Error} =
  VAR ch: CHAR;
  BEGIN
    TRY
      ch := Rd.GetChar(rd)
    EXCEPT
      Rd.EndOfFile => RAISE Error
    END;
    Rd.UnGetChar(rd);
    IF ch = '#' THEN RETURN ReadRawX(rd) ELSE RETURN ReadRawT(rd) END
  END ReadRaw;

PROCEDURE ReadRawT (rd: Rd.T): Pixmap.Raw RAISES {Thread.Alerted, Error} =
  VAR
    depth, west, east, north, south, offset, bitsPerPixel, wordsPerRow,
    bytesPerHexValue, pixelsLength: INTEGER;
    pixelOrder: Text.T;
    word      : Word.T;
    lsb       : BOOLEAN;
  BEGIN
    TRY
      AsciiUtil.GetInt(rd, "depth:", depth);
      AsciiUtil.Get4Int(rd, "bounds:", west, east, north, south);
      AsciiUtil.GetInt(rd, "offset:", offset);
      AsciiUtil.GetInt(rd, "bitsPerPixel:", bitsPerPixel);
      AsciiUtil.GetInt(rd, "wordsPerRow:", wordsPerRow);
      AsciiUtil.GetText(rd, "pixelOrder:", pixelOrder);
      AsciiUtil.GetInt(rd, "bytesPerHexValue:", bytesPerHexValue);
      AsciiUtil.GetInt(rd, "pixelsLength:", pixelsLength);
    EXCEPT
      Scan.BadFormat, AsciiUtil.Error => RAISE Error
    END;
    IF Text.Equal(pixelOrder, "LSBFirst") THEN
      lsb := TRUE
    ELSIF Text.Equal(pixelOrder, "MSBFirst") THEN
      lsb := FALSE
    ELSE
      RAISE Error
    END;

    IF (depth # 1) OR (west # 0) OR (north # 0) OR (offset # 0)
         OR (bitsPerPixel # 1) THEN
      RAISE Unimplemented
    END;

    VAR r := ScrnPixmap.NewRaw(1, Rect.FromSize(east, south));
    BEGIN
      FOR v := 0 TO south - 1 DO
        FOR h := 0 TO east - 1 DO
          IF h MOD 16 = 0 THEN word := NextWord(rd); END;
          VAR bit: INTEGER;
          BEGIN
            IF lsb THEN
              bit := Word.And(word, 1);
              word := Word.RightShift(word, 1)
            ELSE
              IF Word.And(word, 16_8000) = 0 THEN
                bit := 0
              ELSE
                bit := 1
              END;
              word := Word.LeftShift(word, 1)
            END;
            r.set(Point.T{h, v}, bit);
          END;
        END
      END;
      RETURN r
    END

  END ReadRawT;

PROCEDURE ReadRawX (rd: Rd.T): Pixmap.Raw RAISES {Thread.Alerted, Error} =
  VAR
    width, height: INTEGER;
    word         : Word.T;
  BEGIN
    TRY
      width := AsciiUtil.ScanRInt(rd);
      height := AsciiUtil.ScanRInt(rd);
      EVAL AsciiUtil.ScanLine(rd);
    EXCEPT
      Scan.BadFormat, AsciiUtil.Error => RAISE Error
    END;
    VAR r := ScrnPixmap.NewRaw(1, Rect.FromSize(width, height)); BEGIN
      FOR v := 0 TO height - 1 DO
        FOR h := 0 TO width - 1 DO
          IF h MOD 8 = 0 THEN word := NextWord(rd); END;
          r.set(Point.T{h, v}, Word.And(word, 1));
          word := Word.RightShift(word, 1)
        END
      END;
      RETURN r
    END
  END ReadRawX;

BEGIN
END PixmapFromAscii.
