(* Copyright 1992 Digital Equipment Corporation.           *)
(* Distributed only by permission.                         *)
(* Last modified on Mon Jul 27 20:35:51 PDT 1992 by johnh*)
(*      modified on Tue May 12 08:50:59 1992 by mhb    *)


MODULE Classes;

IMPORT Algorithm, AlgorithmClass, List, Text, View, ViewClass, ZeusPanel;

TYPE
  AlgInfo = REF RECORD
    proc: ZeusPanel.NewAlgProc;
    name: TEXT;
  END;

VAR mu := NEW(MUTEX);    (* protect classes against simultaneous access *)

VAR
  algs: List.T (* of AlgInfo *);

PROCEDURE RegisterAlg (proc: ZeusPanel.NewAlgProc; name: TEXT) =
  VAR algInfo := NEW(AlgInfo, name := name, proc := proc);
  BEGIN
    LOCK mu DO algs := List.Append(algs, List.List1(algInfo)) END;
  END RegisterAlg;

PROCEDURE FindAlg (name: TEXT): INTEGER RAISES {NotFound} =
  VAR
    which := 0;
    rest  := algs;
  BEGIN
    LOCK mu DO
      WHILE rest # NIL DO
        IF Text.Equal(name, NARROW(rest.first, AlgInfo).name) THEN
          RETURN which
        END;
        INC(which);
        rest := rest.tail
      END;
    END;
    RAISE NotFound;
  END FindAlg;

PROCEDURE NewAlg (which: INTEGER): Algorithm.T =
  VAR alg: Algorithm.T;
  BEGIN
    LOCK mu DO
      WITH algInfo = NARROW(List.Nth(algs, which), AlgInfo) DO
        alg := algInfo.proc();
        alg.name := algInfo.name;
        RETURN alg;
      END
    END;
  END NewAlg;

PROCEDURE AlgCount (): INTEGER =
  BEGIN
    LOCK mu DO RETURN List.Length(algs) END
  END AlgCount;


TYPE
  ViewInfo = REF RECORD
    proc: ZeusPanel.NewViewProc;
    name: TEXT;
  END;

VAR
  views: List.T (* of ViewInfo *);

PROCEDURE RegisterView (proc: ZeusPanel.NewViewProc; name: TEXT) =
  VAR viewInfo := NEW(ViewInfo, name := name, proc := proc);
  BEGIN
    LOCK mu DO views := List.Append(views, List.List1(viewInfo)) END
  END RegisterView;

PROCEDURE FindView (name: TEXT): INTEGER RAISES {NotFound} =
  VAR
    which := 0;
    rest  := views;
  BEGIN
    LOCK mu DO
      WHILE rest # NIL DO
        IF Text.Equal(name, NARROW(rest.first, ViewInfo).name) THEN
          RETURN which
        END;
        INC(which);
        rest := rest.tail
      END;
    END;
    RAISE NotFound;
  END FindView;

PROCEDURE NewView (which: INTEGER): View.T =
  VAR view: View.T;
  BEGIN
    LOCK mu DO
      WITH viewInfo = NARROW(List.Nth(views, which), ViewInfo) DO
        view := viewInfo.proc();
        view.name := viewInfo.name;
        RETURN view;
      END
    END;
  END NewView;

PROCEDURE ViewCount (): INTEGER =
  BEGIN
    LOCK mu DO RETURN List.Length(views) END
  END ViewCount;


BEGIN
END Classes.
