(* Copyright 1992 Digital Equipment Corporation. *)
(* Distributed only by permission. *)
(* Last modified on Mon Oct  5 13:32:51 PDT 1992 by steveg *)
(*      modified on Wed Sep 23 13:17:13 PDT 1992 by mhb *)
(*      modified on Wed Jul 29 18:49:24 PDT 1992 by johnh*)

(* The Zeus control panel provides a simple programmer interface for
   building algorithm animations, and provides a user interface for
   controlling the the selection of algorithms and views, and the execution
   of algorithms.  There is one control panel per Zeus address space, and
   the control panel itself is a Zeus "Session". *)

INTERFACE ZeusPanel;

IMPORT Algorithm, FormsVBT, Rsrc, Thread, View;

CONST
  StateDir = "~/.zeusState";    (* directory in which state is saved
                                   between invocations *)
  FinalState = "Final_State";   (* filename for state at "Quit" *)

<* PRAGMA LL *>

PROCEDURE Interact (title: TEXT      := "ZEUS Control Panel";
                                                                                                                                                                                                           path : Rsrc.Path := NIL                   );
<* LL = 0 *>
(* Called once; doesn't return until the user deletes the control
   panel.  After installing a Zeus control panel in Trestle,
   tries to restore the state to the last time Zeus was exited
   (stored in StateDir) and then awaits user commands.  When the
   user deletes the control panel, either by a WM gesture or
   using the Quit button from the Zeus menu, Zeus tries to
   snapshot the state into StateDir before returning. *)

PROCEDURE GetPath (): Rsrc.Path;
<* LL = 0 *>
(* Return the "path" that was specified when "Interact" was
   called.  There is no way to change the path dynamically.  The
   path is mostly part of ZeusPanel as a convenience for
   application-writers to share a single path among multiple
   modules.  Also, it's used within ZeusPanel to open forms when
   "GetForm" is called. *)

PROCEDURE SetTitle (title: TEXT);
<* LL = 0 *>
(* Use "title" in the control panel's chassis.  Typically not
   called, since a title can be set with a parameter to
   "Interact".  However, the title can be changed dynamically
   (for example, to reflect the name of the current algorithm or
   input). *)

PROCEDURE ReportError (text: TEXT := NIL); <* LL = VBT.mu *>
(* Display the specified text as an error message in the control panel. *)

PROCEDURE NewForm (name: TEXT; path: Rsrc.Path := NIL):
  FormsVBT.T;
(* Returns a form stored in the resource "name" using the
   resource "path".  However, if "path" is "NIL", then it uses
   the value returned by "GetPath".  It's a runtime error if
   there are any problems reading the form.  Most causal clients
   will read .fv files using this procedure. *)

TYPE
  NewAlgProc = PROCEDURE (): Algorithm.T;

PROCEDURE RegisterAlg (proc: NewAlgProc; name, sessName: TEXT); <* LL = {} *>
(* Register an algorithm.  "name" is the name of the algorithm.  "sessName"
   is the name of the session to which the algorithm belongs, that is, the
   basename of the .evt file.  "proc" is a NewAlgProc, a procedure that
   returns an initialized instance of the algorithm.  This means that
   "proc" must call the init() method of the algorithm. *)


TYPE
  NewViewProc = PROCEDURE (): View.T;

PROCEDURE RegisterView (proc: NewViewProc; name, sessName: TEXT); <* LL = {} *>
(* Register a view.  "name" is the name of the view.  "sessName" is the
   name of the session to which the algorithm belongs, that is, the
   basename of the .evt file.  "proc" is a NewViewProc, a procedure that
   returns an initialized instance of the view.  This means that "proc"
   must call the init() method of the view. *)

PROCEDURE SetSessTitle (sessName, sessTitle: TEXT); <* LL = {} *>
(* The default title under which a session is listed in the "Sessions" menu
   is its name, that is, the basename of the .evt file.  Use this procedure
   to change the title of session "sessName" to "sessTitle." This procedure
   creates a session named "sessName," if none existed previously. *)


PROCEDURE Pause (alg: Algorithm.T; msg: TEXT := NIL)
  RAISES {Thread.Alerted};
<* LL=0,S=Running *>
(* This procedure may only be called from "alg"s run method.  It
   returns after the user issues a "Resume" or "Step" command, or
   it may raise Thread.Alerted (for instance, if the algorithm is
   aborted).  The "msg", if non-"NIL", is displayed in the
   control panel's status area. *)

PROCEDURE StartFeedback (alg: Algorithm.T) RAISES {Thread.Alerted};
<* LL=0,S=Running *>
(* This procedure may only be called from "alg"s run method.  It returns
   after "alg" has called EndFeedback or it may raise Thread.Alerted (for
   instance, if the algorithm is aborted).

   The effect of this procedure is to suspend the algorithm and allow
   feedback events (as if the user had clicked Pause).  When this procedure
   returns, the session continues under interpreter control (returns to the
   Running state).  This procedure is a noop if there already is a
   'pending' StartFeedback for this alg.

   StartFeedback calls the reactivity methods of the algorithm and views to
   enable and disable feedback events. *)


PROCEDURE EndFeedback(alg: Algorithm.T) RAISES {Thread.Alerted};
<* LL=VBT.mu,S=Paused *>

(* This procedure signals a previous call to StartFeedback to return. 
   This procedure is typically called from an algorithm's Feedback 
   method. *) 

PROCEDURE PhotographViews(alg: Algorithm.T) RAISES {Thread.Alerted};
<* LL=VBT.mu, s=Any *>
(* This procedure takes a "photograph" (captures a miniture pixmap)
   of all active views and enters them into an "photo album".  It
   creates the album if none exists.  All views will get redisplayed
   (and maybe reshaped) when the photograph is taken. *)

PROCEDURE ClearPhotoAlbum(alg: Algorithm.T) RAISES {Thread.Alerted};
<* LL=VBT.mu, s=Any *>
(* This procedure removes any "photographs" from the "photo album"
   (see PhotographViews, above). *)

END ZeusPanel.
