(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: CopyExpr.m3                                           *)
(* Last Modified On Mon Mar  2 10:43:02 PST 1992 By kalsow     *)
(*      Modified On Sun Dec 23 08:07:22 1990 By muller         *)

MODULE CopyExpr;

IMPORT Expr, ExprRep, Type, Emit, Temp, MBuf;

TYPE
  P = Expr.T BRANDED "CopyExpr" OBJECT
        expr : Expr.T;
	tipe : Type.T;
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := EqCheck;
        getBounds    := Bounder;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsAlways;
	isZeroes     := ExprRep.IsNever;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a: Expr.T; t: Type.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.expr := a;
    p.tipe := t;
    p.type := t;
    RETURN p;
  END New;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  BEGIN
    Expr.TypeCheck (p.expr, cs);
    Type.Check (p.tipe);
  END Check;

PROCEDURE EqCheck (a: P;  e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(b) => RETURN Type.IsEqual (a.tipe, b.tipe, NIL)
                 AND Expr.IsEqual (a.expr, b.expr);
    ELSE      RETURN FALSE;
    END;
  END EqCheck;

PROCEDURE Compile (p: P): Temp.T =
  VAR e: Expr.T;  t1, t2: Temp.T;
  BEGIN
    e  := p.expr;
    t2 := Expr.Compile (e);
    t1 := Temp.AllocEmpty (p.tipe, TRUE);
    IF Type.IsEqual (p.tipe, Expr.TypeOf (e), NIL) THEN
      Emit.OpTT ("@ = @;\n", t1, t2);
    ELSE
      Emit.OpT  ("@ = ", t1);
      Emit.OpFT ("(@) @;\n", p.tipe, t2);
    END;
    Temp.Free (t2);
    RETURN t1;
  END Compile;

PROCEDURE Fold (p: P): Expr.T =
  BEGIN
    RETURN Expr.ConstValue (p.expr);
    (********* WKK 11/21/91 ??? **********************
    e := Expr.ConstValue (p.expr);
    IF (e = NIL) THEN RETURN NIL END;
    p.expr := e;
    RETURN p;
    *****************************************************)
  END Fold;

PROCEDURE Bounder (p: P;  VAR min, max: INTEGER) =
  VAR min1, max1: INTEGER;
  BEGIN
    Expr.GetBounds (p.expr, min, max);
    EVAL Type.GetBounds (p.tipe, min1, max1);
    min := MAX (min, min1);
    max := MIN (max, max1);
  END Bounder;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "COPY ");
    Expr.Fingerprint (p.expr, map, wr);
    Type.Fingerprint (p.tipe, map, wr);
  END FPrinter;

BEGIN
END CopyExpr.
