(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: DivExpr.m3                                            *)
(* Last modified on Wed Jun  5 01:17:22 1991 by kalsow         *)
(*      modified on Thu Nov 29 03:31:09 1990 by muller         *)

MODULE DivExpr;

IMPORT Expr, ExprRep, Type, Int, Error, Emit, IntegerExpr, Temp, MBuf;

TYPE
  P = ExprRep.Tab BRANDED "DivExpr.P" OBJECT
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := ExprRep.EqCheckAB;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a, b: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a := a;
    p.b := b;
    p.type := Int.T;
    RETURN p;
  END New;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR ta, tb: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    Expr.TypeCheck (p.b, cs);
    ta := Type.Base (Expr.TypeOf (p.a));
    tb := Type.Base (Expr.TypeOf (p.b));
    IF (ta # Int.T) OR (tb # Int.T) THEN
      Error.Msg ("illegal operands for DIV");
    END;
  END Check;

PROCEDURE Compile (p: P): Temp.T =
  VAR t1, t2, t3: Temp.T; min, max: INTEGER;
  BEGIN
    t1 := Expr.Compile (p.a);
    t2 := Expr.Compile (p.b);
    t3 := Temp.Alloc (p);
    Emit.OpT ("@ = _IDIV", t3);

    Expr.GetBounds (p.a, min, max);
    IF min >= 0 THEN    Emit.Op ("P");
    ELSIF max <= 0 THEN Emit.Op ("N");
    ELSE                Emit.Op ("X"); END;

    Expr.GetBounds (p.b, min, max);
    IF min >= 0 THEN    Emit.Op ("P");
    ELSIF max <= 0 THEN Emit.Op ("N");
    ELSE                Emit.Op ("X"); END;

    Emit.OpTT ("(@, @);\n", t1, t2);
    Temp.Free (t1);
    Temp.Free (t2);
    RETURN t3;
  END Compile;

PROCEDURE Fold (p: P): Expr.T =
  VAR e1, e2, e3: Expr.T;
  BEGIN
    e1 := Expr.ConstValue (p.a);
    e2 := Expr.ConstValue (p.b);
    e3 := NIL;
    IF IntegerExpr.Div (e1, e2, e3) THEN END;
    RETURN e3;
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "DIV ");
    Expr.Fingerprint (p.a, map, wr);
    Expr.Fingerprint (p.b, map, wr);
  END FPrinter;

BEGIN
END DivExpr.
