(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Wed Jul  1 21:19:07 1992 by rustan         *)

(* new interface KRML *)
UNSAFE INTERFACE RTStack;
(* This interface is marked as UNSAFE because of the untraced references
   used in the implementation.  Unintended calls to, for example,
   Dispose or GetBottom, may cause trouble. *)

IMPORT Word, Thread;

TYPE
  T <: ADDRESS;
  ResumeKey = RECORD c: INTEGER := 0; t: T END;

PROCEDURE New( t: Thread.T; size: CARDINAL;
               VAR stackLow, stackHigh: ADDRESS ): T;
(* REQUIRES Thread.inSystemCritical AND size # 0 *)
(* Return a new stack.  't' is the thread that is getting a stack.
   'size' indicates the number of Word.T that the stack will consist
   of.
   The OUT parameters stackLow and stackHigh return
   as values that can be passed to SetCurrentStackLimits. *)

PROCEDURE Dispose( VAR stack: T );
(* REQUIRES Thread.inSystemCritical *)
(* Dispose of a stack.  't' returns as NIL, and its previous value
   should not be deferenced after a call to this procedure.
   Note, a thread can never dispose of its own stack, since it needs
   its stack to call this function. *)

PROCEDURE GetBounds( VAR resumeKey: ResumeKey;
                     VAR low, high: UNTRACED REF Word.T;
                     VAR regLow, regHigh: UNTRACED REF Word.T ): BOOLEAN;
(* Returns the inclusive stack bounds 'low' and 'high' and register
   bounds 'regLow' and 'regHigh' of the "next" (explained below) thread.
   This procedure is intended to be called from the garbage collector thread.

   This procedure iterates over the threads in the system, skipping
   the the garbage collecting thread (which is intended to
   be the calling thread).  An iteration of all such threads is
   started with a call with resumeKey set to its default value.  The
   value returned in 'resumeKey' is then used as the IN-parameter of
   the next call.  The procedure returns TRUE as long as there was
   another thread to be queried, and returns FALSE when all threads have
   been queried.  When the procedure returns FALSE, 'resumeKey', 'low',
   'high', 'regLow', and 'regHigh' are undefined and should not be used.

   Once a described iteration begins, it is assumed that it will
   be completed before the next iteration begins.  Moreover, it
   is assumed that no context switch or other call to the interface
   is made during a described iteration. *)

<* EXTERNAL "RTStack__SetCurrentStackLimits" *>
PROCEDURE SetCurrentStackLimits( low, high: ADDRESS );
(* Sets the stack limits to be used for checking stack overflows. *)

<* EXTERNAL "RTStack__GetCurrentStackLimits" *>
PROCEDURE GetCurrentStackLimits( VAR low, high: ADDRESS );
(* Gets the stack limits currently used for checking stack overflows. *)

END RTStack.
