/* Copyright (C) 1994, Klaus Preschern.                        */
/* All rights reserved.                                        */
/* See the file COPYRIGHT.KP for a full description.           */

#include <std.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include "unassmbl.h"
#include "syms.h"
#include "/ex32/include/sys/message.h"
#include "/ex32/include/sys/debug.h"
#include "/ex32/include/sys/syscalls.h"
#include "/ex32/include/signal.h"

#define TRUE			1
#define FALSE			0
#undef  DEBUG

#define MAX_ARGS		30
#define ARG_LEN			40
#define STACK_DEPTH		256
#define RET_INSTR		0xc3

extern int __pid;
static int go_out = 0, go_in = 1;

REGS	global_regs;

static int	db_sp = 0;
static char	* db_stack [STACK_DEPTH];
static void    	sigsegv_handler (void);
static int     	start_child (char **);
static int	db_pid, db_start = 1;

static panic (int proc, char * string)
  {
    printf ("db panic (%d): %s\n", proc, string);
    kill (proc, SIGKILL);
    exit (-1);
  }


static char * st_tos ()
  {
    if (db_sp > 0) {
      return db_stack [db_sp - 1];
    }
    return 0;
  }

static void st_push (char * symbol)
  {
    if (db_sp < STACK_DEPTH) {
      db_stack [db_sp] = symbol;
      db_sp++;
    } else {
      panic (db_pid, "stack overflow");
    }
  }
  
static char * st_pop (void)
  {
    if (db_sp > 0) {
      db_sp--;
      return db_stack [db_sp];
    } else {
      panic (db_pid, "stack underflow");
    }
  }  

static void ident (void)
  {
    int i;
    for (i = 1; i < db_sp; i++) {
      printf ("  ");
    }
  }

int memget (int addr, void * buf, int size)
  {
    return _proccpy (db_pid, __pid, (void *) addr, buf, size);
  }

static struct {
  char command;
  char argument [40];
} db_command;

static int stepcount = 0;
static int list_addr = 0;

#define BUFLEN			80

static void print_regs ()
  {
    printf ("eax = 0x%x, ebx = 0x%x, ecx = 0x%x, edx = 0x%x\n",
      global_regs.eax, global_regs.ebx, global_regs.ecx, global_regs.edx);
    printf ("esi = 0x%x, edi = 0x%x, ebp = 0x%x, esp = 0x%x\n",
      global_regs.esi, global_regs.edi, global_regs.ebp, global_regs.esp);
    printf ("eip = 0x%x, eflags = 0x%x\n", 
      global_regs.eip, global_regs.eflags);
  }

static void list_instr ()
  { 
    int i; char ch = 'y';
    
    while ((ch == 'y') || (ch == 'Y')) {
      for (i = 0; i < 10; i++) {
        list_addr = unassemble (list_addr, 0);
      }
      printf ("continue list ? (y/n) ");
      do {
        ch = getchar ();
      } while (ch == '\n');
    }
  }

static void dump (char * str_addr)
  {
    int addr, i, j, k; char dump_buffer [64]; int * iptr; char * cptr;
    if (sscanf (str_addr, "%x", &addr) != 1) {
      return;
    }
    if (memget (addr, dump_buffer, sizeof (dump_buffer)) < 0) {
      return;
    }
    iptr = (int *) &dump_buffer [0];
    cptr = &dump_buffer [0];
    
    for (i = 0; i < 4; i++) {
      for (j = 0; j < 4; j++) {
        printf ("0x%8x ", *iptr);
	iptr++;
      }
      for (k = 0; k < 16; k++) {
        if (isprint (*cptr)) {
	  printf ("%c", *cptr);
	} else {
	  printf (".");
	}
	cptr++;
      }
      printf ("\n");
    }
  }

static int label2addr (char * label)
  {
    int val;
    if (label [0] == '0') {
      if (sscanf (label, "%x", &val) != 1) {
        return -1;
      }
    } else {
      if ((val = syms_name2val (db_command.argument)) == 0) {
        return -1;
      }
    }
    return val;
  }

static void go (char * label)
  {
    int val;
    if ((val = label2addr (label)) < 0) {
      return;
    }
    _breakpoint (db_pid, D_INSTR_BRK, val);
    go_out = 1;
  }

static void data_brk (char * label)
  {
    int val;
    if ((val = label2addr (label)) < 0) {
      return;
    }
    _breakpoint (db_pid, D_WR_DATA_BRK_4, val);
    go_out = 1;
  }

static void whereis (char * label)
  {
    int val; char * name = label;
    if ((val = label2addr (label)) < 0) {
      return;
    }
    if (label [0] == '0') {
      name = syms_val2name (val, 0);
    }
    printf ("%s = 0x%x (%u)\n", name, val, val);
  }

static int skip_blanks (char * buffer, int i)
  {
    while ((buffer [i] != '\0') && (buffer [i] == ' ')) {
      i++;
    }
    return i;
  }

static int check_command (char * buffer)
  {
    int i = 0, len = strlen (buffer), j = 0;
    if (len == 0) {
      return 0;
    }
    i = skip_blanks (buffer, i);
    db_command.command = buffer [i++];
    i = skip_blanks (buffer, i);
    if (buffer [i] == '\0') {
      return 1;
    }
    while ((buffer [i] != '\0') && (buffer [i] != ' ')) {
      db_command.argument [j++] = buffer [i++];
      db_command.argument [j] = '\0';
    }
    return 2;
  }

static void print_help ()
  {
    printf ("a <addr>       set data breakpoint at address <addr>\n");
    printf ("d <addr>       dump memory at address <addr>\n");
    printf ("g <addr>/<sym> goto <addr> / <sym>\n");
    printf ("h              this message\n");
    printf ("l              list instructions\n");
    printf ("l <sym>        list instructions from symbol <sym>\n");
    printf ("q              quit\n");
    printf ("r              print registers\n");
    printf ("s              step 20 instructions\n");
    printf ("w <sym>        print address of symbol <sym>\n");
    printf ("\n");
  }

static int get_command ()
  {
    int args = 0, count = 0; char buffer [BUFLEN];
    printf ("db>"); fflush (stdout);
    count = read (0, buffer, BUFLEN);
    buffer [count - 1] = '\0';
    if (count == 1) {
      return 0;
    }
    args = check_command (buffer);
    switch (db_command.command) {
    case 'a': if (args != 2) {
                printf ("db: a <address>\n");
		db_command.command = 0;
              }
	      break;
    case 'd': if (args != 2) {
                printf ("db: d <address>\n");
		db_command.command = 0;
              }
              break;
    case 'g': if (args != 2) {
                printf ("db: g <symbol name>\n");
		db_command.command = 0;
              }
              break;
    case 'h': break;
    case 'l': break;
    case 'q': kill (db_pid, SIGKILL);
              exit (0);
    case 'r': break;
    case 's': break;
    case 'w': if (args != 2) {
                printf ("db: w <symbol name/address>\n");
		db_command.command = 0;
              }
              break;
    default : printf ("db: unknown command\n"); 
              db_command.command = 0;
    }
    return args;
  }

static void execute_command (int argc)
  {
    int i = 0, val;
    switch (argc) {
    case 0:  go_out = 1;
             break;
    case 1:  switch (db_command.command) {
             case 'h': print_help ();
	               break;
             case 'l': list_addr = global_regs.eip;
	               list_instr ();
	               break;
             case 'r': print_regs ();
	               break;
             case 's': stepcount = 20;
	               go_out = 1;
	               break;
             }
             break;
    case 2:  printf ("db:%c %s\n", db_command.command, db_command.argument);
             switch (db_command.command) {
	     case 'a': data_brk (db_command.argument);
	               break;
	     case 'd': dump (db_command.argument);
	               break;
             case 'g': go (db_command.argument);
		       break;
	     case 'l': list_addr = syms_name2val (db_command.argument);
	               list_instr ();
		       break;
             case 'w': whereis (db_command.argument);
		       break;
             }
             break;
    default: printf ("db: wrong number of arguments\n");
    }
  }

static void do_debugging (REGS regs)
  {
    unassemble (regs.eip, 0);
    go_in = 1; go_out = 0; 
    if (stepcount != 0) {
      go_in = 0;
      stepcount--;
    }
    
    if (go_in) {
      do {
        execute_command (get_command());
      } while (! go_out);
    }
  }

static void sigquit_handler ()
  {
    REGS regs; 
    printf ("\ndb: quit signal received\n");

    _breakpoint (db_pid, D_DELBRK, 0);
    _procget (db_pid, &regs);
    global_regs = regs;
    do_debugging (regs);
    
    signal (SIGQUIT, (SignalHandler) &sigquit_handler);
    kill (db_pid, SIGUSR1);			 /* restart proc  */
  }

static void sigsegv_handler ()
  {
    REGS regs; char * symbol;
    _procget (db_pid, &regs);
    global_regs = regs;
    symbol = syms_val2name (regs.eip, 0);
    printf ("db: segmentation violation %d %s: eip = %d = 0x%x\n", 
             db_pid, symbol, regs.eip, regs.eip);
    do_debugging (regs);
    
    signal (SIGSEGV, (SignalHandler) &sigsegv_handler);
    kill (db_pid, SIGUSR1);			 /* restart proc  */
  } 

static void sigtrap_handler ()
  {
    REGS regs; 

    _procget (db_pid, &regs);
    global_regs = regs;
    do_debugging (regs);
    
    signal (SIGTRAP, (SignalHandler) &sigtrap_handler);
    kill (db_pid, SIGUSR1);			 /* restart proc  */
  }

static int start_child (char ** argv)
  {
     int child; int i = 0;
     
#ifdef DEBUG
     printf ("exec (");
     while (argv [i] != 0) {
       printf ("%s ", argv [i]); i++;
     }
     printf (")\n");
     fflush (stdout);
#endif

     syms_init (argv [0]);
     
     if ((child = fork ()) == 0) {
       /* child */
       _setexec (EXEC_EMU | EXEC_DEBUG);
       if (execve (argv [0], (const char **) argv, environ) < 0) {
         panic (child, "cannot exec");
	 exit (-1);
       }
     }
     /* parent */
     return child;
  }

static void say_hello ()
  {
    printf ("\n");
    printf ("\t\tCopyright (C) 1994, Klaus Preschern.\n");
    printf ("\t\tAll rights reserved.\n");
    printf ("\t\tSee the file COPYRIGHT.KP for a full description.\n");
    printf ("\n\n");
    printf ("\t\tHi, this is db.\n");
    printf ("\t\tType 'h' for help.\n");
    printf ("\n\n");
  }

void main (int argc, char ** argv, char ** envp)
  {
    int status;
    
    if (argc < 2) {
      printf ("usage: %s program arguments ...\n", argv [0]);
      exit (-1);
    }
    
    say_hello ();
    signal (SIGSEGV,	(SignalHandler) &sigsegv_handler);
    signal (SIGTRAP,	(SignalHandler) &sigtrap_handler);
    signal (SIGQUIT,	(SignalHandler) &sigquit_handler);
    
    db_pid = start_child (&argv [1]);
    /* parent */
    if (db_pid > 0) {
      go ("start");  
    }
    while (wait (&status) != db_pid) ;
    printf ("\ndb: process %d has said good bye with %d\n", db_pid, status);
  }

