/* This is file SYMS.C */
/*
** Copyright (C) 1991 DJ Delorie, 24 Kirsten Ave, Rochester NH 03867-2954
**
** This file is distributed under the terms listed in the document
** "copying.dj", available from DJ Delorie at the address above.
** A copy of "copying.dj" should accompany this file; if not, a copy
** should be available from where this file was obtained.  This file
** may not be distributed without a verbatim copy of "copying.dj".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include "\ex32\include\sys\message.h"

/* typedef unsigned long word32; */
/* typedef unsigned short word16; */
typedef unsigned char word8;
typedef signed long int32;

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <io.h>
#include <unistd.h>
#include "aout.h"
#include "stab.h"
#include "syms.h"

typedef struct SYMTREE {
  unsigned val;
  unsigned type;
  unsigned name_len;
  char *   name;
  struct SYMTREE * nleft, * nright, *ndown;
  struct SYMTREE * vleft, * vright, *vdown;
} SYMTREE;

typedef struct LINENODE {
  struct LINENODE * next;
  unsigned num;
  unsigned addr;
} LINENODE;

typedef struct FILENODE {
  struct FILENODE * next;
  LINENODE * line_list;
  unsigned first_addr, last_addr;
  unsigned name_len;
  char * name;
} FILENODE;

static SYMTREE  * symtree_root=0;
static FILENODE * file_list=0;
static unsigned nsyms=0;

static char tmps[256];
static char tmps2[256];

static SYMTREE * symtree_add(unsigned val, unsigned type, char *name)
{
  SYMTREE * s, * sv, * temp;
  int cval;
  int32 cval32;

  temp = (SYMTREE *) malloc (sizeof (SYMTREE));
  memset(temp, 0, sizeof(temp));
  temp->name_len = strlen(name) + 1;
  temp->name = (char *) malloc (temp->name_len);
  strncpy (temp->name, name, temp->name_len);
  temp->val = val;
  temp->type = type;

  if (symtree_root == 0)
  {
    symtree_root = temp;
    return temp;
  }
  s = symtree_root;
  while (1)
  {
    cval = strcmp(name, s->name);
    if ((cval == 0) &&
        ( (s->val == 0)
       || (val == 0)
       || (s->val == val)))
    {
      if (val)
      {
        s->val = val;
        s->type = type;
      }
      return temp;
    }
    else if (cval == 0)
    {
      if (s->ndown == 0)
      {
        s->ndown = temp;
        break;
      }
      s = s->ndown;
    }
    else if (cval < 0)
    {
      if (s->nleft == 0)
      {
        s->nleft = temp;
        break;
      }
      s = s->nleft;
    }
    else
    {
      if (s->nright == 0)
      {
        s->nright = temp;
        break;
      }
      s = s->nright;
    }
  }
  nsyms++;

  sv = symtree_root;
  while (1)
  {
    if (sv == temp)
      return temp;
    cval32 = val - sv->val;
    if (cval32 == 0)
    {
      if (sv->vdown == 0)
      {
        sv->vdown = temp;
        break;
      }
      sv = sv->vdown;
    }
    else if (cval32 < 0)
    {
      if (sv->vleft == 0)
      {
        sv->vleft = temp;
        break;
      }
      sv = sv->vleft;
    }
    else
    {
      if (sv->vright == 0)
      {
        sv->vright = temp;
        break;
      }
      sv = sv->vright;
    }
  }
  return temp;
}

static symtree_print(SYMTREE * s, int byval, int level)
{
  SYMTREE * tmp;
  if (s == 0)
    return;
  tmp = s;
  if (byval)
    symtree_print(tmp->vleft, byval, level+1);
  else
    symtree_print(tmp->nleft, byval, level+1);

  printf("0x%08lx 0x%08lx %*s%s\n", tmp->val, tmp->type, level, "", tmp->name);

  if (byval)
    symtree_print(tmp->vdown, byval, level);
  else
    symtree_print(tmp->ndown, byval, level);

  if (byval)
    symtree_print(tmp->vright, byval, level+1);
  else
    symtree_print(tmp->nright, byval, level+1);
}

void syms_list(int v)
{
  symtree_print(symtree_root, v, 0);
}

typedef struct SYM_ENTRY {
  unsigned string_off;
  word8 type;
  word8 other;
  word16 desc;
  unsigned val;
} SYM_ENTRY;

get_string(FILE *f, unsigned where)
{
  int i = 0;
  if (where != ftell(f)) {
    fseek(f, where, 0);
  }
  do {
    tmps2 [i++] = fgetc(f);
  } while (tmps2[i-1]);
}

static long tell (int fd)
  {
    return lseek (fd, 0, SEEK_CUR);
  }

void syms_init(char *fname)
{
  GNU_AOUT header;
  int fd, j, per, oldper=-1;
  unsigned nsyms, i;
  FILE *sfd;
  unsigned sfd_base;
  SYM_ENTRY sym;
  char *strings, *cp;
  unsigned string_size;
  unsigned last_dot_o=0;
  FILENODE * filen = 0;
  LINENODE * linen = 0;

  fd = open(fname, O_RDONLY | O_BINARY);
  sfd = fopen(fname, "rb");

  read(fd, &header, sizeof(header));
  if ((header.info & 0xffff) == 0x14c)
  {
    lseek(fd, 0xa8L, 0);
    read(fd, &header, sizeof(header));
    lseek(fd, 0xa8+sizeof(header) + header.tsize + header.dsize, 0);
  }
  else if (((header.info & 0xffff) == 0x10b) ||
     ((header.info & 0xffff) == 0x0107))
  {
    lseek(fd, sizeof(header) + header.tsize + header.dsize
      + header.txrel + header.dtrel, 0);
  }
  else
  {
    nsyms = 0;
    return;
  }

  nsyms = header.symsize / sizeof(SYM_ENTRY);

  printf("Reading %d symbols...   0%%", nsyms);
  fflush(stdout);

  fseek(sfd, tell(fd)+header.symsize, 0);
  sfd_base = ftell(sfd);

  for (i=0; i<nsyms; i++)
  {
    if (nsyms > 1)
      per = i*100L/(nsyms-1);
    else
      per = 100;
    if (per != oldper)
    {
      printf("\b\b\b\b%3d%%", per);
      fflush(stdout);
      oldper = per;
    }
    read(fd, &sym, sizeof(sym));
    if (sym.string_off) {
      get_string(sfd, sfd_base+sym.string_off);
    }
    switch (sym.type)
    {
      case N_TEXT:
      case N_TEXT | N_EXT:
        cp = tmps2;
        cp += strlen(cp) - 2;
        if (strcmp(cp, ".o") == 0)
        {
          last_dot_o = sym.val;
          if (filen && (filen->last_addr == 0))
          {
            filen->last_addr = last_dot_o - 1;
          }
          break;
        }
        if (strcmp(cp, "d.") == 0) /* as in gcc_compiled. */
          break;
      case N_DATA:
      case N_DATA | N_EXT:
      case N_ABS:
      case N_ABS | N_EXT:
      case N_BSS:
      case N_BSS | N_EXT:
      case N_FN:
      case N_SETV:
      case N_SETV | N_EXT:
      case N_SETA:
      case N_SETA | N_EXT:
      case N_SETT:
      case N_SETT | N_EXT:
      case N_SETD:
      case N_SETD | N_EXT:
      case N_SETB:
      case N_SETB | N_EXT:
      case N_INDR:
      case N_INDR | N_EXT:
        if (sym.string_off)
          symtree_add(sym.val, sym.type, tmps2);
        break;
      case N_SO:
        filen = (FILENODE *) malloc (sizeof (FILENODE));
        memset(filen, 0, sizeof(FILENODE));
        filen->name_len = strlen(tmps2)+1;
	filen->name = (char *) malloc (filen->name_len);
	strncpy (filen->name, tmps2, filen->name_len);
        filen->next = file_list;
        file_list = filen;
        filen->first_addr = last_dot_o;
        break;
      case N_SLINE:
        linen = (LINENODE *) malloc (sizeof (LINENODE));
        memset(linen, 0, sizeof(LINENODE));
        linen->next = filen->line_list;
        filen->line_list = linen;
        linen->num = sym.desc;
        linen->addr = sym.val;
        break;
    }
  }
  printf(" %ld symbol%s read\n", nsyms, nsyms==1 ? "" : "s");
  return;
}

extern REGS	global_regs;

#define Ofs(n)		(n * sizeof (word32))

struct {
  char *name;
  int size;
  int ofs;
  } regs[] = {
  "%eip",	4,	Ofs(8),
  "%eflags",	4,	Ofs(9),
  "%eax",	4,	Ofs(0),
  "%ebx",	4,	Ofs(1),
  "%ecx",	4,	Ofs(2),
  "%edx",	4,	Ofs(3),
  "%esp",	4,	Ofs(7),
  "%ebp",	4,	Ofs(6),
  "%esi",	4,	Ofs(4),
  "%edi",	4,	Ofs(5),
  "%ax",	2,	Ofs(0),
  "%bx",	2,	Ofs(1),
  "%cx",	2,	Ofs(2),
  "%dx",	2,	Ofs(3),
  "%ah",	1,	Ofs(0)+1,
  "%bh",	1,	Ofs(1)+1,
  "%ch",	1,	Ofs(2)+1,
  "%dh",	1,	Ofs(3)+1,
  "%al",	1,	Ofs(0),
  "%bl",	1,	Ofs(1),
  "%cl",	1,	Ofs(2),
  "%dl",	1,	Ofs(3),
  0,		0, 	0
};

int undefined_symbol=0;

unsigned syms_name2val(char *name)
{
  SYMTREE * s;
  int cval, idx, sign=1, i;
  unsigned v;
  char *cp;

  undefined_symbol = 0;

  idx = 0;
  sscanf(name, "%s", name);

  if (name[0] == 0)
    return 0;

  if (name[0] == '-')
  {
    sign = -1;
    name++;
  }
  else if (name[0] == '+')
  {
    name++;
  }
  if (isdigit(name[0]))
  {
    if (sign == -1)
      return -strtol(name, 0, 16);
    return strtol(name, 0, 16);
  }

  cp = strpbrk(name, "+-");
  if (cp)
    idx = cp-name;
  else
    idx = strlen(name);

  if (name[0] == '%') /* register */
  {
    for (i=0; regs[i].name; i++)
      if (strncmp(name, regs[i].name, idx) == 0)
      {
        switch (regs[i].size)
        {
          case 1:
            v = *(word8 *) ((word8 *)(&global_regs + regs[i].ofs));
            break;
          case 2:
            v = *(word16 *) ((word8 *)(&global_regs + regs[i].ofs));
            break;
          case 4:
            v = *(word32 *) ((word8 *)(&global_regs + regs[i].ofs));
            break;
        }
        return v + syms_name2val(name+idx);
      }
  }

  for (i=0; i<idx; i++)
    if (name[i] == '#')
    {
      FILENODE * f;
      LINENODE * l;
      int lnum;
      sscanf(name+i+1, "%d", &lnum);
      for (f=file_list; f; f=f->next)
      {
        if ((strncmp(name, f->name, i) == 0) && (f->name[i] == 0))
        {
          for (l=f->line_list; l; l=l->next)
          {
            if (l->num == lnum)
              return l->addr + syms_name2val(name+idx);
          }
          printf("undefined line number %.*s\n", idx, name);
          undefined_symbol = 1;
          return 0;
        }
      }
      printf("Undefined file name %.*s\n", i, name);
      undefined_symbol = 1;
      return 0;
    }

  s = symtree_root;
  while (s)
  {
    cval = strncmp(name, s->name, idx);
    if ((cval == 0) && s->name[idx])
      cval = -1;
    if (cval == 0)
      return s->val*sign + syms_name2val(name+idx);
    else if (cval < 0)
      s = s->nleft;
    else
      s = s->nright;
  }
  s = symtree_root;
  while (s)
  {
    if (s->name[0] == '_')
      cval = strncmp(name, (s->name)+1, idx);
    else
      cval = '_' - s->name[0];
    if ((cval == 0) && s->name[idx+1])
      cval = -1;
    if (cval == 0)
      return s->val*sign + syms_name2val(name+idx);
    else if (cval < 0)
      s = s->nleft;
    else
      s = s->nright;
  }
  printf("Undefined symbol %.*s\n", idx, name);
  undefined_symbol = 1;
  return 0;
}

static char noname_buf[11];

char *syms_val2name(unsigned val, unsigned *delta)
{
  SYMTREE * s, * lasts;

  if (delta)
    *delta = 0;
  lasts = 0;
  s = symtree_root;
  while (s)
  {
    if (s->val <= val)
      lasts = s;
    if (val == s->val)
    {
      while (s->vdown)
        s = s->vdown;
      return s->name;
    }
    else if (val < s->val)
      s = s->vleft;
    else
      s = s->vright;
  }
  if (lasts)
  {
    while (lasts->vdown)
      lasts = lasts->vdown;
    if (strcmp(lasts->name, "_etext") == 0)
      goto noname;
    if (strcmp(lasts->name, "_end") == 0)
      goto noname;
    if (delta)
      *delta = val - lasts->val;
    return lasts->name;
  }
noname:
  sprintf(noname_buf, "%#lx", val);
  return noname_buf;
}

char *syms_val2line(unsigned val, int *lineret, int exact)
{
  FILENODE * filen;
  LINENODE * linen, * closest;
  closest = 0;
  for (filen = file_list; filen; filen=filen->next)
  {
    if ((val <= filen->last_addr) && (val >= filen->first_addr))
    {
      for (linen=filen->line_list; linen; linen = linen->next)
      {
        if (val == linen->addr)
        {
          *lineret = linen->num;
          return filen->name;
        }
        if (val > linen->addr)
        {
          if (!closest)
            closest = linen;
          else if (closest->addr < linen->addr)
            closest = linen;
        }
      }
      if (closest && !exact)
      {
        *lineret = closest->num;
        return filen->name;
      }
    }
  }
  return 0;
}

static char type2char(type)
{
  switch (type)
  {
    case N_TEXT:
      return 't';
    case N_TEXT | N_EXT:
      return 'T';
    case N_DATA:
      return 'd';
    case N_DATA | N_EXT:
      return 'D';
    case N_BSS:
      return 'b';
    case N_BSS | N_EXT:
      return 'B';
    default:
      return ' ';
  }
}

static int linecnt, quit_list;

static syms_listwild2(SYMTREE * s_pos, char *pattern)
{
  SYMTREE * s;
  char *name;
  int lnum;
  s = s_pos;
  if ((s == 0) || quit_list)
    return;
  syms_listwild2(s->vleft, pattern);
  if (quit_list)
    return;
  if (wild(pattern, s->name))
  {
    if (++linecnt > 20)
    {
      printf("--- More ---");
      switch (getchar ())
      {
        case ' ':
          linecnt = 0;
          break;
        case 13:
          linecnt--;
          break;
        case 'q':
          quit_list = 1;
          return;
      }
      printf("\r            \r");
    }
    printf("0x%08lx %c %s", s->val, type2char(s->type), s->name);
    name = syms_val2line(s->val, &lnum, 0);
    if (name)
      printf(", line %d of %s", lnum, name);
    printf ("\n");
    /* mputchar('\n'); */
  }
  syms_listwild2(s->vdown, pattern);
  if (quit_list)
    return;
  syms_listwild2(s->vright, pattern);
}

void syms_listwild(char *pattern)
{
  quit_list = linecnt = 0;
  syms_listwild2(symtree_root, pattern);
}

