/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: error.h,v 1.4 1994/04/08 02:52:01 janssen Exp $ */
/***********************************************************************
Error signalling in the ILU kernel.

Each routine that can raise errors does so through a designated
parameter, normally the last.  This parameter is typed "ilu_Error*".
A routine that returns successfully will set the ilu_Error* parameter
to NULL.  An exceptional return involves creating an ilu_Error, and
returning it through the ilu_Error* parameter.
*/

typedef struct _ilu_ErrorBase_s *ilu_Error;

typedef struct {
  char *name;
  char *description;
  void (*freeproc) (ilu_Error);
} ilu_ErrorTypeStruct, *ilu_ErrorType;

struct _ilu_ErrorBase_s {
  ilu_ErrorType type;
  void *data;
};

/*
Each error is an instance of some `type' of errors.  Each type of
error is `declared' once, in a header file that is #included into all
the modules that raise or handle errors of that type (or, if there's
only one such module, the error type can be declared in that modue).
An error type is declared using the following syntax:

	ILU_DECL_ERR(<id>) {
		<memberlist>
	};

Different error types must use different <id>s; there are no other
restrictions on the <id>s.  The <memberlist> is as in a structure
definition.
*/

#define ILU_DECL_ERR(x)		extern ilu_ErrorTypeStruct ilu_ETS_##x ; \
typedef struct ilu_EVS_##x##_s ilu_EVS_##x , *ilu_EV_##x ; \
extern ilu_EV_##x ilu_ErrQua_##x (ilu_Error e); \
typedef struct ilu_EVS2_##x##_s ilu_EVS2_##x, *ilu_EV2_##x; \
struct ilu_EVS_##x##_s { \
  ilu_ErrorType type; \
  ilu_EV2_##x data; \
  }; \
struct ilu_EVS2_##x##_s

/*
Each error is represented by a pointer to an error-type-specific
structure; such a pointer type can be expressed as

	ILU_ERR_TYPE(<id>)

The values are kept in a struct of a different type, a pointer to
which may be named with the expression

	ILU_ERR_SLOTS_TYPE(<id>)

A programmer that correctly uses the ILU error handling macros should
never have to name this second type.
*/

#define ILU_ERR_TYPE(x)		ilu_EV_##x

#define ILU_ERR_SLOTS_TYPE(x)	ilu_EV2_##x

/*
Each error type is `defined' exactly once, using the following syntax:

	ILU_DEF_ERR(<xid>, "<descriptive string>")
		<statements>
	ILU_END_DEF_ERR

The <xid> is the same one used in the declaration of the error.  The
<descriptive string> is an arbitrary C string (no embedded \000's)
that describes the error, and is used only for purposes of debugging
the error system itself.  The <statements> form the body of the
procedure used to free errors of this type; they free the members of
the error structure, but not the error structure itself.  These
statements may assume the existence of a variable named `e' containing
the pointer to the struct defined by the ILU_DECL_ERR declaration.
*/

#define ILU_DEF_ERR(xid,xdescrip)	static void ilu_EF_##xid (ilu_Error); \
ilu_ErrorTypeStruct ilu_ETS_##xid = { # xid , xdescrip, ilu_EF_##xid }; \
static void ilu_EF2_##xid (ilu_EV2_##xid); \
static void ilu_EF_##xid (ilu_Error ee) \
{ \
  ilu_EF2_##xid (((ilu_EV_##xid) ee)->data); \
} \
static void ilu_EF2_##xid (ilu_EV2_##xid e)

/*
An error is raised by creating an appropriate error instance and
storing it through the ilu_Error* parameter.  The following
pseudo-code describes the macrology used to create error values:

	ILU_ERR_TYPE(<id>) ILU_ERR_CONS(<id>)

The result points to freshly allocated memory containing the
appropriate structure, initialized such that directly contained
pointers are NULL.  The structure's members are then set to
appropriate values.  Next, the error pointer is stored through the
ilu_Error* parameter; the following syntax is used to cast the error
pointer to the generic ilu_Error type:

	ILU_QUA_ERR(<expr>)

where <expr> evaluates to the pointer returned by ILU_ERR_CONS(<id>).
Both the creating and assignment can be done with

        {
	  ILU_ERR_BIND(<ptr>, <id>, <ev>)
	  {
	    <statements>
	  }
	}

which, if <ptr> is non-nil, creates new error value of type <id>, and
binds the variable name <ev> to a value of the type
ILU_ERR_VAL_SLOTS(<id>), and executes the statements, which should
assign values to particular error slots.  Note that ILU_RAISE_ERROR
must occur within its own block.

Finally, the erring procedure returns.
*/

#define ILU_ERR_CONS(x)		((ilu_EV_##x ) _ilu_ErrCons(&ilu_ETS_##x , sizeof(struct ilu_EVS2_##x##_s)))

#define ILU_QUA_ERR(x)		((ilu_Error) (x))

#define ILU_ERR_BIND(ptr,id,ev)   ILU_ERR_SLOTS_TYPE(id) ev; if (ptr == NULL) {} else { *ptr = ILU_QUA_ERR(ILU_ERR_CONS(id)); ev = ((ilu_EV_##id) *ptr)->data; } if (ptr != NULL)

/*
An error is handled by testing whether the ilu_Error* parameter was
set to NULL or not.  If not, the error is decoded with the following
syntax:

	ILU_ERR_SWITCH(<expr>)
	ILU_ERR_CASE(<tid-1>, <vid-1>)
		<statements-1>
	ILU_ERR_CASE(<tid-2>, <vid-2>)
		<statements-2>
	...
	ILU_ERR_ELSE
		<statements-E>
	ILU_ERR_ENDSWITCH

where: <expr> evaluates to the ilu_Error that was stored through the
ilu_Error* parameter; for each i>0, <statements-i> are to be executed
in the extent of <vid-i>, which will be a variable initialized to the
pointer to the corresponding error structure, iff the error is of the
type identified by <tid-i>; and <statements-E> are to be executed if
the error is of some other type.  The "ILU_ERR_ELSE <statements-E>"
part can be omitted when the earlier cases are expected to handle all
posibilities.

The name and description of an ilu_Error value may be accessed with
the macros defined by the following pseudocode:

	const char *ILU_ERR_NAME(ilu_Error err);
	const char *ILU_ERR_DESCRIPTION(ilu_Error err);
*/

#define ILU_ERR_SWITCH(x)	{ ilu_Error __ilu_CurrentError = ((ilu_Error) (x)); \
if ((x) == NULL) { { ilu_Assert(0, #x " == NULL");

#define ILU_ERR_CASE(tid,vid)		} } else if (__ilu_CurrentError->type == &ilu_ETS_##tid ) { \
struct ilu_EVS2_##tid##_s *vid ; \
vid = ((ilu_EV_##tid ) __ilu_CurrentError)->data; \
{

#define ILU_ERR_ELSE		} } else if ( __ilu_CurrentError != NULL ) { {

#define ILU_ERR_ENDSWITCH	} } else { ilu_Assert (0, "unhandled error"); } }

#define ILU_ERR_NAME(err)	 ((err)->type->name)
#define ILU_ERR_DESCRIPTION(err) ((err)->type->description)

/*
After an error is handled, it must be freed by a call on the following routine:

	void ilu_FreeError (ilu_Error e);

which does not itself raise any errors.
*/

extern void ilu_FreeError (ilu_Error);	

/* C code that uses this error system in the way prescribed above ---
using only the defined macro syntax, not any knowledge about how the
macros expand --- effectively achieves type safety through runtime
typing of errors.  An error value can only be created by use of the
ILU_ERR_CONS macro, which always tags the error value consistently
with the structure allocated and the C type of the value produced by
the ILU_ERR_CONS invocation.  Members of error structures are only
accessible: (1) through the (statically typed) result of ILU_ERR_CONS,
or (2) within ILU_ERR_SWITCH constructs, which invoke
error-type-specific code only on errors tagged with that error type.

For non-C code (ie, non-C language-specific runtimes), achieving type
safety is a little harder.  Of course, we can't even consider it
unless the language's type system can describe the C data types used
in the errors --- so let's assume it can.  This error system affords
two levels of safety.  The first is provided by the ilu_ErrQua_<id>
procedures declared by the ILU_[EN]DECL_ERR(<id>) macros.  These
procedure take non-NULL arguments, and return either NULL or an
appropriately-typed non-NULL pointer, depending on whether the given
ilu_Error is of the type appropriate for the procedure (and thus its
result).  If the non-C code accesses error parameters using only these
procedures, it will be type-safe --- provided the ilu_ErrQua_<id>
procedures are declared correctly in the non-C language.

The ilu_ErrQua_<id> procedures need to be defined as well as declared.
To minimize kernel size in some configurations (eg, C LS runtime
only), we put those definitions in a separate file.  For each error
type <id>, the macro

	ILU_QUADEF(<id>)

is invoked in that file.
*/

#define ILU_QUADEF(x)		ilu_EV_##x ilu_ErrQua_##x (ilu_Error e) \
{ \
  if (e->type == &ilu_ETS_##x ) \
	return ((ilu_EV_##x ) e); \
  else return (NULL); \
}

/*
The second, higher, level of type-safety is built on the first, by
automatically transliterating the C header (".h") files into the non-C
language of interest.  This guarantees that the ilu_ErrQua_<id>
procedures are declared correctly.  Sadly, I don't know of any program
that does such automatic transliteration to the non-C languages of
interest to us.
*/
