/*
We use mutual exclusion.
There are mutexes at two levels of abstraction, discussed with L1 and L2.

At L2, there is a call mutex per connection and an I/O mutex per connection.  The call and I/O mutexes are held during I/O of a request or reply message on the connection; the I/O mutex is also held while closing a connection.  For a connection using a concurrent protocol, the call mutex is released while waiting for a reply; for a connection using a non-concurrent protocol, the call mutex is held from the start of the request to the end of the reply.  When starting a call over a non-concurrent protocol, one of the server's available connections is used, or a new one is created and used if all the existing ones have their call mutexes held.  When starting a type or GC callback over a concurrent protocol, the secondary connection is used if the primary one's call mutex is held.  When starting an ordinary call over a concurrent protocol, we block until the primary connection's call mutex is available.  conn.callmu < conn.iomu.

At L1, there are a several mutexes:
smu:	global mutex for the server table;
otmu:	global mutex for object type data structures;
cmu:	global LRU list of connections
prmu:	global mutex for protocol registry
trmu:	global mutex for transport registry
gcmu:	global mutex for GC data structures
timu:	global mutex for alarm implementation.
server:	one mutex per server.

There can be no deadlocks that involve only L1 mutexes because threads acquire L1 mutexes in an order consistent with a fixed partial order.  Here is the fixed partial order:

cmu < server
smu < server
server < prmu
server < trmu
gcmu < server
gcmu < timu
cmu < smu		(for ??)
gcmu < cmu		(for gc.c:gcaInvoke)
cmu < timu		(for call.c:GR{Set,Cancel})
prmu < otmu		(for Protocol->interpret_request)

There is a common locking invariant, called the "Main Invariant":
	L1 = {}, and
	forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})
This is exactly what's guaranteed to hold while an application's service routines are called.  This could be among the things guaranteed to hold while a stub is marshalling or unmarshalling (and will be when we no longer buffer entire messages).  This is exactly what's guaranteed to hold while waiting for I/O on a File Descriptor to be enabled.  This is among the things guaranteed to hold while doing I/O on a File Descriptor.

There is another common requirement, which is with respect to some server s and object type cl, is called Inside(s, cl), and is this:
  ~GC(cl)	      => L1 >= {      cmu, s};
   GC(cl) &&  true(s) => L1 >= {gcmu, cmu, s};
   GC(cl) && ~true(s) => L1  = {      cmu, s};
   GC(cl) && ~true(s) => forall conn: (L2 >= {conn.iomu})
					 => (L2 >= {conn.callmu}
  Main unconstrained.
Note that this invariant has the property that if C2 is a subtype of C1, then Inside(s, C1) => Inside(s, C2).  This property is used in situations where we know some static type of an object, but not necessarily the most specific type (yet).

There is only one kind of mutex held while waiting for a reply: a connection's call mutex.

Theorem: There can be no deadlocks involving L2 mutexes.
Proof: There are two cases when a thread blocks trying to acquire a call mutex: (A) when making an ordinary call (not a type or GC callback) on a server with a concurrent connection, and (B) when making a type or GC callback on a server with a concurrent connection.  In case A, the call mutex in question is held only during the processing of a message (not between request and reply messages), which will eventually finish without trying to acquire any mutex held by the thread blocked on acquiring the call mutex.  At worst the message processing may involve a type or GC callback, but these will acquire a different call mutex (and maybe a server's tabmu, which won't be held by the thread trying acquire the call mutex in question), and the server can implement the calls using only tightly held locks.  Case B is like case A, but the message processing in progress will not itself initiate nested type or GC callbacks.  Similarly, a thread blocks trying to acquire an I/O mutex while another thread completes some I/O task that won't try to acquire any mutexes held by the first thread.

We document locking requirements with comments about the symbols "L1" and "L2", which stand for the sets of mutexes held by a thread; "L1_sup" stands for the maximum member of L1 (but we suppose that "L1_sup < XXX" is satisfied by a thread holding an empty set of L1 mutexes); "L2_sup" is not meaningful because we don't impose a partial order on L2 mutexes.
For data, the comments say what mutexes must be held to access the datum.
For procedures, the comments say what mutexes must be held to call
the procedure, and, if the procedure changes the set of held mutex, how.

We have three sorts of locking comments: those about L1, those about L2, and those about whether the Main Invariant applies.  Locking comments come in blocks.  There are two kinds of blocks of locking comments: a "sticky" block is followed by a blank line; a "one-shot" is not.  A locking comment is also called "sticky" or "one-shot", depending on the kind of the comment block in which the comment is contained.  A one-shot comment applies only to the immediately following item.  A sticky comment of a certain sort applies to all items between it and the next sticky comment of the same sort, except those items to which a one-shot comment of the same sort applies.  Another exception is this: where the Main Invariant holds, we needn't explicitly override comments of the L1 sort.


Sadly, we need condition variables to get high-performance multithreaded operation.  A thread can wait on a condition variable.  Another thread can "notify" that condition variable.  This causes all threads currently waiting on the condition variable to return from the wait operation.  To prevent timing splinters, decisions about waiting and notifying should be made inside a mutex.  This means the mutex must be released while waiting on a condition variable, and there must be no possibilty of a thread switch between the release of the mutex and the start of the wait; the wait operation takes the mutex as an argument, because in a pre-emptive threads environment (eg, PCR) the release and the wait must be an atomic thread operation.

Some runtimes (eg, a single-threaded one) cannot support condition variables; these runtimes supply NULL for all the condition variable operations.
*/
/* Last tweaked by Mike Spreitzer March 17, 1994 10:52 pm PST */

