/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: object.c,v 1.54 1994/04/13 00:27:51 janssen Exp $ */
/* Last tweaked by Mike Spreitzer April 4, 1994 2:42 pm PDT */

#define _POSIX_SOURCE

#ifdef MACOS
#pragma segment ilu
#endif

#include <string.h>
#include <time.h>

#include "ilu.h"
#include "iluntrnl.h"

#include "object.h"
#include "server.h"
#include "type.h"
#include "protocol.h"
#include "port.h"
#include "vector.h"

#include "version.h"	/* defines ILU_VERSION */

/*L1, L2, Main unconstrained*/

static const ilu_string _ilu_ilu_version = ILU_VERSION_STRING ;

/*L1 >= {cmu}*/
static ilu_ObjectNoter theNoter = NULL;
/* The proc to call when the "very interested" predicate changes. */

/*L1_sup < cmu*/
void ilu_SetNoter(ilu_ObjectNoter n)
{
  _ilu_Assert(n!=NULL, "SetNoter: given NULL noter");
  _ilu_Assert(theNoter==NULL, "SetNoter: already set");
  _ilu_AcquireMutex(ilu_cmu);
  theNoter = n;
  _ilu_ReleaseMutex(ilu_cmu);
  return;
}

/*L1, L2, Main unconstrained*/

static ilu_Server DefaultServer = NULL;

ilu_string ilu_GetILUVersion (void)
{
  return (_ilu_ilu_version);
}

void ilu_SetDefaultServer(ilu_Server s)
{ DefaultServer = s;
}

ilu_Server ilu_GetDefaultServer(void)
{ return DefaultServer;
}

ilu_boolean ilu_ParseStringBindingHandle (ilu_string sbh,
					  ilu_string *oid,
					  ilu_string *connectInfo)
{
  ilu_integer count;

/*
  DEBUG(OBJECT_DEBUG,
	(stderr, "ilu_ParseStringBindingHandle:  SBH <%s>.\n", sbh));
*/

  if (sbh == NULL)
    return (FALSE);

  for (count = 0;  sbh[count] != '\0';  count += 1) {
      if (sbh[count] == '@') {
	  if (sbh[count+1] == '@')
	    count += 1;
	  else
	    break;
	}
    }

  if (sbh[count] == '@')
    count += 1;

  for (;  sbh[count] != '\0';  count += 1) {
      if (sbh[count] == '@') {
	  if (sbh[count + 1] == '@')
	    count += 1;
	  else
	    break;
	}
    }

  if (oid != NULL) {
      if (count > 0) {
	  *oid = (ilu_string) malloc (count + 1);
	  strncpy (*oid, sbh, count);
	  (*oid)[count] = '\0';
	}
      else
	*oid = NULL;
    }

  if (sbh[count] == '@')
    count += 1;

  if (connectInfo != NULL) {
      if (sbh[count] != '\0') {
	  *connectInfo = (ilu_string) malloc (strlen (sbh + count) + 1);
	  strcpy(*connectInfo, sbh + count);
	}
      else
        *connectInfo = NULL;
    }


  DEBUG(OBJECT_DEBUG,
	(stderr,
	 "ilu_ParseStringBindingHandle:  parts are <%s> and <%s>\n",
	 (oid == NULL) ? "*" : *oid,
	 (connectInfo == NULL) ? "*" : *connectInfo));

  return (TRUE);
}

ilu_boolean ilu_ParseOID (ilu_string oid, ilu_string *instH,
					  ilu_string *serverID)
{
  ilu_integer count;
/*
  DEBUG(OBJECT_DEBUG, (stderr, "ilu_ParseOID:  OID <%s>.\n", oid));
*/
  if (oid == NULL)
    return (FALSE);

  for (count = 0;  oid[count] != '\0';  count += 1) {
      if (oid[count] == '@') {
	  if (oid[count+1] == '@')
	    count += 1;
	  else
	    break;
	}
    }

  if (instH != NULL) {
      if (count > 0) {
	  *instH = (ilu_string) malloc (count + 1);
	  strncpy (*instH, oid, count);
	  (*instH)[count] = '\0';
	}
      else
	*instH = NULL;
    }

  if (oid[count] == '@')
    count += 1;

  if (serverID != NULL) {
      if (oid[count] != '\0') {
	  *serverID = (ilu_string) malloc (strlen (oid + count) + 1);
	  strcpy(*serverID, oid + count);
	}
      else
        *serverID = NULL;
    }

  DEBUG(OBJECT_DEBUG, (stderr, "ilu_ParseOID:  parts are <%s> and <%s>\n",
		       (  instH  == NULL) ? "*" : *instH,
		       (serverID == NULL) ? "*" : *serverID));

  return (TRUE);
}

/*L1, L2, Main unconstrained*/

/* This procedure is used internally for creating both true and surrogate objects.  For true objects, this procedure is called with non-NULL ih, server, cl, and lspo; for surrogate objects, this procedure is called with non-NULL ih, server, and sbh.  Storage for ih and sbh is owned by the object, for mstid is owned by the caller. */
static ilu_Object CreateObject (ilu_string ih, ilu_Server server,
				ilu_Class class, ilu_refany lspo,
				ilu_string sbh, ilu_string mstid)
{
  ilu_Object new = (ilu_Object) malloc(sizeof(struct _ilu_Object_s));
  ilu_Alarmette_s gco = {NULL, NULL, FALSE, {0, 0}};
  
  object_oid(new)	= _ilu_Strcat3(ih, "@", server_id(server));
  object_ih(new)	= ih;
  object_server(new)	= server;
  object_timeout(new)	= 1200;
  object_class(new)	= class;
  object_mstid(new)	= _ilu_Strdup(mstid);
  object_sbh(new)	= sbh;
  object_lspo(new)	= lspo;
  object_holds(new)	= 0;
  object_intNoted(new)	= FALSE;
  object_notifying(new)	= FALSE;
  object_known(new)	= FALSE;
  object_gco(new)	= gco;
  object_lastRemote(new) = 0;
  object_gclist(new)	= NULL;
  return (new);
}

/*Destroy an object created but not yet put into the server's object table. */
static void UncreateObject(ilu_Object o)
{
  free(o->ob_oid);
  free(o->ob_ih);
  if (o->ob_sbh != NULL)
      free(o->ob_sbh);
  free(o);
}

/*L1 >= {s}*/

ilu_boolean _ilu_Addable(ilu_Server s, ilu_Class t, ilu_Object *h)
{
  int i, l;
  if (!class_singleton(t))
      return TRUE;
  if (  (*h = (ilu_Object) _ilu_hash_FindInTable(server_singles(s), t))
	!= NULL)
      return FALSE;
  l = class_superclass_count(t);
  for (i = 0; i < l; i++)
      if (!_ilu_Addable(s, class_superclass(t, i), h))
          return FALSE;
  return TRUE;
}

ilu_boolean _ilu_AddSingleton(ilu_Server s, ilu_Class t, ilu_Object o)
{
  int i, l;
  if (!class_singleton(t))
      return ilu_FALSE;
  _ilu_Assert(_ilu_hash_AddToTable(server_singles(s), t, o),
	      "AddSingleton");
  l = class_superclass_count(t);
  for (i = 0; i < l; i++)
      _ilu_AddSingleton(s, class_superclass(t, i), o);
  return ilu_TRUE;
}

/*L1 >= {the object's server};
  L1 >= {gcmu} if cl collectible*/
ilu_Object ilu_FindOrCreateTrueObject (ilu_string ih, ilu_Server server,
					ilu_Class cl, ilu_refany lspo)
{ 
  ilu_Object new;
  if (server == NULL) {
      server = DefaultServer;
      if (server == NULL)
        return (NULL);
    }
  _ilu_Assert(ih != NULL, "CreateTrueObject: ih==NULL");
  _ilu_Assert(lspo != NULL, "CreateTrueObject: lspo==NULL");
  _ilu_Assert(cl != NULL, "CreateTrueObject: class==NULL");
  _ilu_HoldMutex(server_lock(server));
  new = (ilu_Object) _ilu_hash_FindInTable(server_objs(server), ih);
  if (new != NULL) {
      if (new->ob_class != cl) {
          DEBUG(OBJECT_DEBUG,
		(stderr,
		 "FindOrCreateTrueObject: type of %s is %s, not %s",
		 new->ob_oid, new->ob_class->cl_unique_id, cl->cl_unique_id));
          return (NULL);
        }
      return (new);
    }
  if (server->sr_closing == TRUE) {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "ilu_CreateTrueObject: invoked on closing server %s.\n",
	     server->sr_id));
      return (NULL);
    }
  if (!_ilu_Addable(server, cl, &new)) {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "%s %s %s of type %s in server %s because %s %s.\n",
	     "ilu_CreateTrueObject: can't create another",
	     "singleton object", ih, cl->cl_unique_id, server->sr_id, new->ob_ih,
	     "already exists"));
      return (NULL);
    }
  new = CreateObject(ih, server, cl, lspo, NULL, cl->cl_unique_id);
  _ilu_Assert(_ilu_hash_AddToTable(server_objs(server), ih, new),
	      "FindOrCreateTrueObject: AddToTable failed");
  _ilu_AddSingleton(server, cl, new);
  if (class_collectible(cl))
    _ilu_StartGCingTrueObj(new);
  return new;
}

/*before: L1 = {};
  after:  result!=NULL => Inside(result's server, cl);
  after:  result==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  Main otherwise unconstrained*/
ilu_Object ilu_ObjectOfSBH (ilu_string sbh, ilu_string mstid,
			    ilu_Class static_type)
{
  ilu_string connectInfo = NULL, oid = NULL;
  ilu_string ih = NULL, serverID = NULL;
  ilu_Object o = NULL;
  ilu_boolean tidfound = FALSE;
  ilu_Class c2;
  ilu_Server s;

  _ilu_AutoSetDebugLevel();
  
  _ilu_Assert(sbh != NULL, "ObjectOfSBH: sbh==NULL");
  ASSERT(static_type!=NULL, buf,
	 (buf, "ObjectOfSBH: static_type==NULL, sbh=%s", sbh));

  if (   !ilu_ParseStringBindingHandle (sbh, &oid, &connectInfo)
      || oid == NULL || connectInfo == NULL) {
      DEBUG(OBJECT_DEBUG,
	    (stderr, "ObjectOfSBH: SBH parse of %s unsatisfactory.\n",
	     sbh));
      return o;
    }
  if (   !ilu_ParseOID (oid, &ih, &serverID)
      || ih == NULL || serverID == NULL) {
      DEBUG(OBJECT_DEBUG,
	    (stderr, "ObjectOfSBH: OID parse of %s unsatisfactory.\n",
	     oid));
      return o;
    }

  if (mstid != NULL) {
      c2 = ilu_FindClassFromID(mstid);
      if (c2 == NULL)
          c2 = _ilu_FindMSKA(mstid);
      c2 = (tidfound = (c2 != NULL)) ? c2 : static_type;
    }
  else c2 = static_type;
  s = _ilu_FindServer(serverID, TRUE, connectInfo);
  ASSERT(ih != NULL, buf, (buf, "ObjectOfSBH: ih==NULL, sbh=%s", sbh));
  ASSERT(s != NULL, buf, (buf, "ObjectOfSBH: s==NULL, sid=%s", serverID));
  ilu_EnterServer(s, static_type);
      
  if (s->sr_objs == NULL) {
      DEBUG(CONNECTION_DEBUG,
	    (stderr, "ilu_ObjectOfSBH:  %s (id=%s, true=%d).\n",
	     "asked for object in closed server", serverID,
	     server_is_true(s)));
      ilu_ExitServer(s, static_type);
      free(oid);
      free(ih);
      free(serverID);
      free(connectInfo);
      return NULL;
    }
      
  DEBUG(CONNECTION_DEBUG,
	    (stderr,
	     "ilu_ObjectOfSBH:  Server for id %s, ci %s is %p.\n",
	     serverID, connectInfo, s));

  free (connectInfo);
  free (serverID);
  o = _ilu_FindObjectInServer(ih, s);

  if (o == NULL)
    {
      ilu_Object o2;
      if (server_is_true(s)) {
          ilu_ExitServer(s, static_type);
          free (ih);
        }
      else if (s->sr_closing == TRUE) {
          ilu_ExitServer(s, static_type);
          DEBUG(OBJECT_DEBUG,
		(stderr, "%s (oid %s) for closing server %s.\n",
		 "ilu_ObjectOfSBH:  refusing to create new surrogate",
		 oid, s->sr_id));
          free (ih);
        }
      else if (!_ilu_Addable(s, c2, &o2)) {
          ilu_ExitServer(s, static_type);
          DEBUG(OBJECT_DEBUG,
		(stderr,
		 "%s %s of type %s in server %s because %s %s.\n",
		 "ilu_ObjectOfSBH:  won't create new singleton", ih,
		 c2->cl_unique_id, s->sr_id, o2->ob_ih, "already exists"));
          free (ih);
        }
      else
	{
          o = CreateObject (ih, s, c2, NULL, _ilu_Strdup(sbh), mstid);
          DEBUG(OBJECT_DEBUG,
		    (stderr,
		     "ilu_ObjectOfSBH:  Couldn't find obj for oid <%s> on"
		     " server %p, created new surrogate object %p.\n",
		     oid, s, o));
          if (mstid==NULL || !tidfound) {
              ilu_ExitServer(s, static_type);
              c2 = _ilu_FindClassViaRPC(o);
              if (c2 != NULL) {
                  object_class(o) = c2;
                  ilu_EnterServer(s, static_type);
                }
              else {
                  DEBUG(OBJECT_DEBUG,
			(stderr,
			 "ilu_ObjectOfSBH:  Couldn't determine type for"
			 " object %p, given mstid \"%s\".\n", o, mstid));
                  UncreateObject(o);
                  o = NULL;
                }
            }
          if (o != NULL) {
              if (s->sr_closing == TRUE) {
                  DEBUG(OBJECT_DEBUG,
			(stderr,
			 "ilu_ObjectOfSBH:  %s (oid %s) %s %s.\n",
			 "refusing to create new surrogate",
			 oid, "for closing server", s->sr_id));
                  UncreateObject(o);
                  ilu_ExitServer(s, static_type);
                  o = NULL;
                }
              else if (NULL != (o2 = (ilu_Object) _ilu_hash_FindInTable
						  (server_objs(s), ih) ) )
                {
                  UncreateObject(o);
                  o = o2;
                }
              else if (!_ilu_Addable(s, object_class(o), &o2)) {
                  DEBUG(OBJECT_DEBUG,
			(stderr,
			 "%s %s of type %s in server %s because %s %s.\n",
			 "ilu_ObjectOfSBH:  won't create new singleton",
			 ih, object_class(o)->cl_unique_id, s->sr_id, o2->ob_ih,
			 "already exists"));
                  UncreateObject(o);
                  ilu_ExitServer(s, static_type);
                  o = NULL;
                }
              else {
                  _ilu_Assert(_ilu_hash_AddToTable(server_objs(s), ih, o),
			      "ObjectOfSBH: add to cache (2) failed");
                  _ilu_AddSingleton(s, object_class(o), o);
                }
            }
	}
    }
  else
      free (ih);
      
  free (oid);

  DEBUG(OBJECT_DEBUG, (stderr,
		       "ilu_ObjectOfSBH:  Object of <%s> is %p.\n",
		       sbh, o));
  if (o==NULL)
      {}
  else if (o->ob_mstid==NULL && mstid!=NULL)
      o->ob_mstid = _ilu_Strdup(mstid);
  else if (o->ob_mstid==NULL || mstid==NULL)
      {}
  else if (strcmp(o->ob_mstid, mstid) != 0) {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "ObjectOfSBH: Existing object %s has mstid %s, not %s.\n",
	     o->ob_oid, o->ob_mstid, mstid));
    }
  if (o!=NULL && !ilu_IsSubObjectType(o->ob_class, static_type)) {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "%s %s has type %s:%s, not a subtype of %s:%s.\n",
	     "ilu_ObjectOfSBH: Existing object", o->ob_oid,
	     o->ob_class->cl_unique_id, o->ob_class->cl_name,
	     static_type->cl_unique_id, static_type->cl_name));
    }

  return (o);
}

/*L1 >= {obj's server}; L1_sup < prmu*/
ilu_string ilu_SBHOfObject (ilu_Object obj)
{
  ilu_string p;
  ilu_Server s;

  if (obj == NULL)
    return (NULL);

  s = object_server(obj);
  _ilu_HoldMutex(server_lock(s));
  if (object_sbh(obj) == NULL)
    {
      _ilu_Assert(object_is_true(obj),"SBHOfObject: surrogate lacks SBH");
      if (server_default_port(s) != NULL)
	{ 
	  ilu_Port port = server_default_port(s);
	  ilu_string pinfo = protocol_form_handle(port_protocol(port),
						  obj);
	  if (pinfo != NULL) {
	      object_sbh(obj) = _ilu_Strcat5(object_oid(obj), "@",
					     pinfo, "|",
					     port_tinfo(port));
	      free(pinfo);
	    }
	}
    }
  p = object_sbh(obj);
  return (p);
}

/*L1, L2, Main unconstrained*/

ilu_string ilu_MstidOfObject (ilu_Object obj)
{
  return (obj->ob_mstid);
}

ilu_Class ilu_ClassOfObject (ilu_Object obj)
{
  if (obj == NULL)
    return (NULL);
  else
    return (object_class(obj));
}

ilu_boolean ilu_TrueInstanceP (ilu_Object obj)
{
  ilu_boolean s = FALSE;

  if (obj != NULL)
    {
      s = object_is_true(obj);
    }
  return (s);
}

ilu_Server ilu_ServerOfObject (ilu_Object obj)
{
  if (obj == NULL)
    return (NULL);
  else
    return (object_server(obj));
}

ilu_string ilu_IhOfObject (ilu_Object obj)
{
  if (obj == NULL)
    return (NULL);
  else
    return (object_ih(obj));
}

/*L1 >= {cmu, s}*/
void _ilu_DestroyObject (ilu_Server s, ilu_Object obj)
{
  ilu_Class c = object_class(obj);
  ASSERT(obj->ob_oid != NULL, buf,
	 (buf, "object.c:DestroyObject: %p->ob_oid == NULL", obj));
  ASSERT(c != NULL, buf,
	 (buf, "object.c:DestroyObject: class(%s) == NULL", obj->ob_oid));
  ASSERT(obj->ob_ih != NULL, buf,
	 (buf, "object.c:DestroyObject: ih(%s) == NULL", obj->ob_oid));
  _ilu_HoldMutex(ilu_cmu);
  _ilu_HoldMutex(server_lock(s));
  if (object_collectible(obj) && server_is_true(s))
      _ilu_StopGCingTrueObj(obj);
  _ilu_ServerRemoveObject(s, obj);
  FREETOKEN (object_ih(obj));
  FREETOKEN (object_oid(obj));
  if (object_sbh(obj) != NULL)
      FREETOKEN (object_sbh(obj));
  free (obj);
  return;
}

/*L1 >= {s}*/
ilu_refany ilu_GetLanguageSpecificObject (ilu_Object obj)
{
  ilu_refany s = NULL;
  if (obj != NULL)
    {
      _ilu_HoldMutex(server_lock(object_server(obj)));
      s = object_lspo(obj);
    }
  return (s);
}

/*Inside(obj's server, obj's type)*/

static void IUpdate(ilu_Object obj)
{
  ilu_Server s = object_server(obj);
  if (obj->ob_notifying)
      return;
  if (object_collectible(obj) && !object_is_true(obj)) {
      int should = (obj->ob_lspo != NULL) || (obj->ob_holds != 0);
      obj->ob_notifying = TRUE;
      while (should != obj->ob_known) {
          ilu_ExitServer(s, obj->ob_class);
          if (should)
               _ilu_RegisterGCInterest(obj);
          else _ilu_UnregisterGCInterest(obj);
          ilu_EnterServer(s, obj->ob_class);
          obj->ob_known = should;
          should = (obj->ob_lspo != NULL) || (obj->ob_holds != 0);
        }
      obj->ob_notifying = FALSE;
    }
  if (obj->ob_lspo != NULL || obj->ob_holds != 0)
      return;
  if (object_is_true(obj) && object_collectible(obj)) {
      if (obj->ob_gclist!=NULL && _ilu_vector_size(obj->ob_gclist)!=0)
          return;
      if (time(NULL) < obj->ob_lastRemote + obj->ob_timeout)
          return;
    }
  DEBUG(OBJECT_DEBUG,
        (stderr, "IUpdate:  deleting uninteresting object %s (%p).\n",
         obj->ob_oid, obj));
  _ilu_DestroyObject(object_server(obj), obj);
  return;
}

void _ilu_VIUpdate(ilu_Object obj)
{
  int vi =   (obj->ob_holds != 0)
	  || (object_collectible(obj) &&
	      (object_is_true(obj)
	      ? (   (obj->ob_gclist != NULL &&
	             _ilu_vector_size(obj->ob_gclist) != 0)
	         || (time(NULL) < obj->ob_lastRemote + obj->ob_timeout) )
	      : (obj->ob_notifying && !obj->ob_known) ));
  if (vi != obj->ob_intNoted) {
      obj->ob_intNoted = vi;
      if (theNoter != NULL)
        (*theNoter)(obj, vi);
    }
  IUpdate(obj);
}

void ilu_RegisterLanguageSpecificObject (ilu_Object obj, ilu_refany lso)
{
  obj->ob_lspo = lso;
  IUpdate(obj);
}

void _ilu_DeltaHolds(ilu_Object obj, ilu_integer dholds)
{
  obj->ob_holds += dholds;
  _ilu_VIUpdate(obj);
}

/* a bit of scaffolding code for debugging the class record */
/*L1, L2, Main unconstrained; for calling from debugger*/
#ifdef NOPE
static void dump_class_record(ilu_Class class)
{
	printf("dumping class record:\n");
	printf("class name: %s\n",class->cl_name);
	printf("class brand: %s\n",class->cl_brand ? class->cl_brand : "");
	printf("class id: %s\n",class->cl_unique_id ? class->cl_unique_id : "");
	printf("class singleton == 0x%x\n",class->cl_singleton);
	printf("class collectible == 0x%x\n",class->cl_collectible);
	printf("class authentication: %s\n",
	       class->cl_authentication ? class->cl_authentication : "");
	printf("class methods table: 0x%x\n",class->cl_methods);
	printf("class method count: %d\n",class->cl_method_count);
	printf("class superclass record: 0x%x\n",class->superclass);
	printf("class superclass name: 0x%x (%s)\n",class->superclassname,
	       class->superclassname ? class->superclassname : "");
}
#endif

/*L1, L2, Main unconstrained*/

ilu_string _ilu_Strdup (const ilu_string str)
{
  ilu_string p;

  if (str == NULL)
    return (NULL);
  else if ((p = malloc(strlen(str) + 1)) == NULL)
    return (NULL);
  else
    {
      strcpy(p, str);
      return(p);
    }
}

ilu_string _ilu_Strcat3(const ilu_string s1, const ilu_string s2,
			const ilu_string s3)
{
  int l1 = (s1==NULL) ? 0 : strlen(s1);
  int l2 = (s2==NULL) ? 0 : strlen(s2);
  int l3 = (s3==NULL) ? 0 : strlen(s3);
  ilu_string t = malloc(1 + l1 + l2 + l3);
  ilu_string u = t;
  if (t==NULL)
    return t;
  if (s1 != NULL) {
    strcpy(u, s1);
    u += l1;
  }
  if (s2 != NULL) {
    strcpy(u, s2);
    u += l2;
  }
  if (s3 != NULL) {
    strcpy(u, s3);
    u += l3;
  }
  *u = 0;
  return t;
}

ilu_string _ilu_Strcat5(const ilu_string s1, const ilu_string s2,
			const ilu_string s3, const ilu_string s4,
			const ilu_string s5)
{
  int l1 = (s1==NULL) ? 0 : strlen(s1);
  int l2 = (s2==NULL) ? 0 : strlen(s2);
  int l3 = (s3==NULL) ? 0 : strlen(s3);
  int l4 = (s4==NULL) ? 0 : strlen(s4);
  int l5 = (s5==NULL) ? 0 : strlen(s5);
  ilu_string t = malloc(1 + l1 + l2 + l3 + l4 + l5);
  ilu_string u = t;
  if (t==NULL)
    return t;
  if (s1 != NULL) {
    strcpy(u, s1);
    u += l1;
  }
  if (s2 != NULL) {
    strcpy(u, s2);
    u += l2;
  }
  if (s3 != NULL) {
    strcpy(u, s3);
    u += l3;
  }
  if (s4 != NULL) {
    strcpy(u, s4);
    u += l4;
  }
  if (s5 != NULL) {
    strcpy(u, s5);
    u += l5;
  }
  *u = 0;
  return t;
}

/*Main Invariant holds; L2 otherwise unconstrained*/
ilu_boolean ilu_PingObject (ilu_Object o)
{
  ilu_Call call;
  ilu_cardinal estatus = 0;
  ilu_ProtocolException internal;
  ilu_cardinal reqSize;
  ilu_Class pclass = object_class(o);
  ilu_Server s = object_server(o);
  ilu_boolean status;

  DEBUG(OBJECT_DEBUG,
        (stderr, "_ilu_FindClassViaRPC:  object %p...\n", o));

  if (class_singleton(pclass))
    {
      DEBUG(OBJECT_DEBUG,
	    (stderr,
	     "%s %s is singleton, not attempting GetTypes RPC call.\n",
	     "_ilu_FindClassViaRPC:  pclass", class_name(pclass)));
      return (ilu_FALSE);
    }
  if ((call = ilu_BeginCall(s)) == NULL)
    return (ilu_FALSE);
  _ilu_AcquireMutex(server_lock(s));
  reqSize = ilu_SizeOfObjectID(call, o, TRUE, _ilu_rootClass);
  _ilu_ReleaseMutex(server_lock(s));
  if (ilu_BeginRequest(call, _ilu_rootClass, _ilu_PingMethod, reqSize))
    {
      ilu_EnterServer(s, object_class(o));
      ilu_OutputObjectID(call, o, TRUE, _ilu_rootClass);
      ilu_FinishRequest(call);
      internal = ilu_GetReply(call, &estatus);
      status = (internal == ilu_ProtocolException_Success AND estatus == 0);
    }
  else
    status = ilu_FALSE;
  ilu_FinishCall(call);
  return status;
}

/*L2    >=    {conn's callmu, iomu} before,
  L2 disjoint {conn's callmu, iomu} after*/
void _ilu_HandlePing (ilu_Call call)
{
  ilu_Object disc;
  ilu_string names = NULL;
  ilu_boolean true = FALSE;

  ilu_InputObjectID (call, &disc, TRUE, _ilu_rootClass);
  if (disc != NULL) {
      _ilu_DeltaHolds(disc, 1);
      ilu_ExitServer(object_server(disc), _ilu_rootClass);
    }

  ilu_RequestRead(call);
  
  if ((disc != NULL) AND ilu_TrueInstanceP(disc))
    {
      if (ilu_BeginReply(call, FALSE, 0))
	ilu_FinishReply (call);
    }
  else
    {
      if (ilu_BeginException(call, 0, ilu_ProtocolException_GarbageArguments))
	ilu_FinishException (call);
    }

  if (disc != NULL) {
      ilu_Server s = object_server(disc);
      ilu_Class cl = object_class(disc);
      ilu_EnterServer(s, cl);
      _ilu_DeltaHolds(disc, -1);
      ilu_ExitServer(s, cl);
    }
  return;
}
