/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: text.c,v 1.5 1994/02/15 20:33:29 janssen Exp $ */
/* Last tweaked by Mike Spreitzer January 13, 1994 5:20 pm PST */

#ifdef MACOS
#pragma segment ilu
#endif

#include <string.h>
#include "ilu.h"
#include "iluntrnl.h"
#include "packet.h"
#include "text.h"

#include "call.h"
#include "protocol.h"
#include "connect.h"
#include "pipe.h"
#include "transprt.h"
#include "port.h"
#include "object.h"
#include "type.h"
#include "method.h"
#include "mooring.h"
#include "debug.h"

extern ilu_cardinal _ilu_SafeStrlen(ilu_string );

/*************************************************************/
/*		Implement XDR PACKET			     */
/*************************************************************/

#define ODD(x)		(((x)&0x1)!=0)
#define EVEN(x)		(((x)&0x1)==0)  
#define PADDED_SIZE(x) ((((unsigned) (x))+3) & (~0x3))
#define ABS(x)		(((x)<0)?(-(x)):(x))
#define HEXVALUE(x)	((((x)>='0')&&((x)<='9'))?((x)-'0'):((((x)>='a')&&((x)<='f'))?(((x)-'a')+10):((((x)>='A')&&((x)<='F'))?(((x)-'A')+10):0)))

ilu_cardinal digits(ilu_cardinal l)
{
  ilu_cardinal dig = 0;
  while (l > 0)
    {
      dig++;
      l /= 10;
    }
  return(dig);
}

static ilu_boolean _textxdr_put_num(PACKET p, ilu_boolean neg, ilu_cardinal l)
{
  ilu_cardinal d;
  ilu_cardinal needed;
  ilu_cardinal absl = l;
  ilu_boolean rneg = (neg && (l != 0));

  d = digits(absl);
  needed = 1+(rneg ? 1 : 0)+((d < 1) ? 1 : d);
  if (packet_allocation(p) - packet_size(p) >= needed)
    {
      *p->next++ = ' ';
      if (rneg)
	*p->next++ = '-';
      if (d == 0)
	*p->next++ = '0';
      else
	{
	  while(d-- > 0)
	    *p->next++ = ((absl / (10 * d)) % 10) + '0';
	}
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _textxdr_put_u_short (PACKET p,ilu_shortcardinal l)
{
  return (_textxdr_put_num (p, FALSE, l));
}

static ilu_boolean _textxdr_put_short (PACKET p, short l)
{
  return (_textxdr_put_num (p, (l < 0), ABS(l)));
}

static ilu_boolean _textxdr_put_long (PACKET p, ilu_integer l)
{
  return (_textxdr_put_num (p, (l < 0), ABS(l)));
}

static ilu_boolean _textxdr_put_u_long (PACKET p, ilu_cardinal l)
{
  return (_textxdr_put_num (p, FALSE, l));
}

static ilu_boolean _textxdr_put_byte (PACKET p, char l)
{
  return (_textxdr_put_num (p, (l < 0), ABS(l)));
}

static ilu_boolean _textxdr_put_u_byte (PACKET p, ilu_byte l)
{
  return (_textxdr_put_num (p, FALSE, l));
}

static ilu_boolean _textxdr_put_double (PACKET p, ilu_real l)
{
  char buf[1000];
  ilu_cardinal len;

  sprintf (buf, " %.10e", l);
  if (packet_allocation(p) - packet_size(p) >= (len = strlen(buf)))
    {
      strncpy (p->next, buf, len);
      p->next += len;
      return (TRUE);
    }
  else
    return(FALSE);
}

static ilu_boolean _textxdr_put_float (PACKET p, ilu_shortreal l)
{
  return (_textxdr_put_double (p, (ilu_real) l));
}

static ilu_cardinal TextMaxStringSize = 65535;

static char hexdigit[] = "0123456789abcdef";

static ilu_boolean _textxdr_put_bytes (PACKET p, ilu_byte *b, ilu_cardinal l)
{
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

/*
  DEBUG(TEXT_DEBUG, (stderr, "_textxdr_put_bytes:  allocation 0x%x, size 0x%x (diff %d), needed %u\n",
			packet_allocation(p), packet_size(p), packet_allocation(p) - packet_size(p),
			(4 + PADDED_SIZE(l))));
*/
			
  if (l > TextMaxStringSize)
    {
      fprintf (stderr, "Attempt to pass byte sequence of length %u, which exceeds TextMaxStringSize value of %u.\n",
	       l, TextMaxStringSize);
      return (FALSE);
    }

  if (packet_allocation(p) - packet_size(p) >= ((2 * l) + 2 + digits(l)))
    {
      _textxdr_put_num (p, FALSE, l);
      if (l > 0)
	{
	  *p->next++ = ' ';
	  for (p1 = p->next, c = l, p2 = b;  c > 0;  c--)
	    {
	      *p1++ = hexdigit[((*p2 >> 4) & 0xF)];
	      *p1++ = hexdigit[(*p2++) & 0xF];
	    }
	}
      p->next = p1;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_cardinal countQuotes (ilu_string p, ilu_cardinal l)
{
  register ilu_string p2;
  register ilu_cardinal len;
  ilu_cardinal count = 0;

  for (p2 = p, len = l;  l > 0;  p2++)
    {
      if (*p2 == '\\' OR (*p2 == '"'))
	count++;
      else if ((*p < ' ') OR (*p > 0x7e))
	count += 3;
    }
  return (count);
}

static ilu_boolean _textxdr_put_string (PACKET p, ilu_string s, ilu_cardinal l)
{
  register ilu_string p1, p2;
  register ilu_cardinal c;

/*
  DEBUG(TEXT_DEBUG, (stderr, "_textxdr_put_bytes:  allocation 0x%x, size 0x%x (diff %d), needed %u\n",
			packet_allocation(p), packet_size(p), packet_allocation(p) - packet_size(p),
			(4 + PADDED_SIZE(l))));
*/
			
  if (l > TextMaxStringSize)
    {
      fprintf (stderr, "Attempt to pass byte sequence of length %u, which exceeds TextMaxStringSize value of %u.\n",
	       l, TextMaxStringSize);
      return (FALSE);
    }

  if (packet_allocation(p) - packet_size(p) >= (l + 4 + digits(l) + countQuotes(s, l)))
    {
      _textxdr_put_num (p, FALSE, l);
      if (l > 0)
	{
	  *p->next++ = ' ';
	  *p->next++ = '"';
	  for (p1 = p->next, c = l, p2 = s;  c > 0;  c--, p2++)
	    {
	      if ((*p2 == '"') OR (*p2 == '\\'))
		{
		  *p1++ = '\\';
		  *p1++ = *p2;
		}
	      else if ((*p2 < ' ') OR (*p2 > 0x7e))
		{
		  *p1++ = '\\';
		  *p1++ = (*((ilu_byte *)p2) >> 6);
		  *p1++ = ((*((ilu_byte *)p2) >> 3) & 0x7);
		  *p1++ = (*((ilu_byte *)p2) & 0x7);
		}
	      else
		*p1++ = *p2;
	    }
	  p->next = p1;
	  *p->next++ = '"';
	}
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean _textxdr_get_num (PACKET p, ilu_integer *neg, ilu_cardinal *val)
{
  ilu_cardinal v = 0;
  *neg = 1;

  if (packet_allocation(p) - packet_size(p) < 2)
    return(FALSE);

  p->next++;	/* skip leading space */
  if (*p->next == '-')
    {
      p->next++;
      *neg = -1;
    }
  while (isdigit(*p->next))
    v = (v * 10) + ((*p->next++) - '0');
  *val = v;
  return (TRUE);
}

static ilu_boolean _textxdr_get_u_short (PACKET p, ilu_shortcardinal *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = num;
  return(stat);
}

static ilu_boolean _textxdr_get_long (PACKET p, long *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = (neg * num);
  return(stat);
}

static ilu_boolean _textxdr_get_short (PACKET p, short *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = (neg * num);
  return(stat);
}

static ilu_boolean _textxdr_get_u_long (PACKET p, ilu_cardinal *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = num;
  return(stat);
}

static ilu_boolean _textxdr_get_byte (PACKET p, char *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = (char) (neg * num);
  return(stat);
}

static ilu_boolean _textxdr_get_u_byte (PACKET p, ilu_byte *l)
{
  ilu_cardinal num;
  ilu_integer neg;
  ilu_boolean stat;

  if (stat = _textxdr_get_num (p, &neg, &num))
    *l = num;
  return(stat);
}

static ilu_boolean _textxdr_get_double (PACKET p, double *l)
{
  if (packet_allocation(p) - packet_size(p) < 2)
    return(FALSE);

  p->next++;	/* skip leading space */
  if (sscanf (p->next, "%e", l) != 1)
    return (FALSE);
  while (*p->next++ != ' ')
    ;
  return(TRUE);      
}

static ilu_boolean _textxdr_get_float (PACKET p, float *l)
{
  double d;
  ilu_boolean stat;

  if (stat = _textxdr_get_double (p, &d))
    *l = d;
  return(stat);
}

static ilu_boolean _textxdr_get_bytes (PACKET p, ilu_byte **b, ilu_cardinal *l)
{
  ilu_cardinal l2;
  ilu_integer  neg;
  ilu_boolean stat;
  register ilu_byte *p1, *p2;
  register ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) < 4
      || (!_textxdr_get_num (p, &neg, &l2))
      || (packet_allocation(p) - packet_size(p)) < (2 * l2))
    return (FALSE);

  if (l2 > TextMaxStringSize)
    {
      fprintf (stderr, "Attempt to read string of length %u, which exceeds TextMaxStringSize value of %u.\n",
	       l2, TextMaxStringSize);
      return (FALSE);
    }

  if (l2 > 0)
    {
      if (*b == NULL)
	*b = (ilu_byte *) malloc(l2);
      for (p1 = *b, p2 = p->next, c = l2;  c > 0;  c--)
	*p1++ = (HEXVALUE(*p2++) << 4) + HEXVALUE(*p2++);
      p->next = p2;
      *l = l2;
    }
  else
    {
      *b = NULL;
      *l = 0;
    }
  return (TRUE);
}

static ilu_boolean _textxdr_get_string (PACKET p, ilu_string *s, ilu_cardinal *l)
{
  ilu_cardinal l2;
  ilu_boolean stat;
  register ilu_byte *p1, *p2;
  ilu_cardinal c;

  if (packet_allocation(p) - packet_size(p) < 4)
    return (FALSE);

  if (!_textxdr_get_num (p, &c, &l2))
    return (FALSE);
  if (l2 > 0)
    {
      p->next++;	/* skip leading space */
      {
	if (*s == NULL)
	  *s = (ilu_string) malloc(l2);
	for (p1 = *s, p2 = p->next, c = l2;  c > 0;  c--, p2++)
	  {
	    if (*p2 == '\\')
	      {
		p2++;
		if (isdigit(*p2) && ((*p2 - '0') < 8))
		  {
		    *p1++ = ((p2[0] - '0') * 64) + (p2[1] - '0') * 8 + p2[3];
		    p2 += 2;
		  }
		else
		  *p1++ = *p2;
	      }
	    else
	      *p1++ = *p2;
	  }
	p->next = p2;
      }
    }
  else
    *s = NULL;
  *l = l2;
  return (TRUE);
}

static ilu_boolean _textxdr_get_opaque (PACKET p, ilu_bytes *b, ilu_cardinal len)
{
  ilu_cardinal len2;

  return (_textxdr_get_bytes(p, b, &len2) AND (len2 == len));
}

static void _textxdr_destroy (PACKET p)
{
  if (p->buf != NULL && p->count > 0)
    free (p->buf);
  free (p);
}

static PACKET _textxdr_CreatePacket (ilu_bytes buf, ilu_cardinal size)
{
  PACKET p = (PACKET) malloc (sizeof(struct ilu_packet_s));
  if (buf != NULL)
    p->buf = buf;
  else if (size > 0)
    p->buf = (ilu_byte *) malloc(size);
  else
    p->buf = NULL;
  p->count = size;
  p->next = p->buf;

  p->put_long = _textxdr_put_long;
  p->put_byte = _textxdr_put_byte;
  p->put_short = _textxdr_put_short;
  p->put_u_long = _textxdr_put_u_long;
  p->put_u_byte = _textxdr_put_u_byte;
  p->put_u_short = _textxdr_put_u_short;
  p->put_float = _textxdr_put_float;
  p->put_double = _textxdr_put_double;
  p->put_bytes = _textxdr_put_bytes;
  p->put_opaque = _textxdr_put_bytes;

  p->get_long = _textxdr_get_long;
  p->get_byte = _textxdr_get_byte;
  p->get_short = _textxdr_get_short;
  p->get_u_long = _textxdr_get_u_long;
  p->get_u_byte = _textxdr_get_u_byte;
  p->get_u_short = _textxdr_get_u_short;
  p->get_float = _textxdr_get_float;
  p->get_double = _textxdr_get_double;
  p->get_bytes = _textxdr_get_bytes;
  p->get_opaque =_textxdr_get_opaque;

  p->destroy = _textxdr_destroy;

  return (p);
}

/**********************************************************************
  End of packet implementation
/**********************************************************************/

static ilu_refany _text_CreateDataBlock (void)
{
  struct Text *new = (struct Text *) malloc(sizeof(struct Text));

  new->in = NULL;
  new->out = NULL;
  return ((ilu_refany) new);
}

static void _text_FreeDataBlock (struct Text *d)
{
  if (d->in != NULL)
    {
      packet_destroy (d->in);
      d->in = NULL;
    }
  if (d->out != NULL)
    {
      packet_destroy (d->out);
      d->out= NULL;
    }
  free (d);
}

static void formRegistryName (ilu_string buf)
{
  ilu_string registryhome = (ilu_string ) getenv("ILUREGISTRY");
  ilu_string iluhome = (ilu_string ) getenv ("ILUHOME");

#ifdef MACOS
  if (registryhome != NULL)
    sprintf (buf, "%s:%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s:lib:%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s:%s", REGISTRY_LAST_RESORT, PROGRAM_NUMBER_REGISTRY_NAME);
#else
  if (registryhome != NULL)
    sprintf (buf, "%s/%s", registryhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else if (iluhome != NULL)
    sprintf (buf, "%s/lib/%s", iluhome, PROGRAM_NUMBER_REGISTRY_NAME);
  else
    sprintf (buf, "%s/%s", REGISTRY_LAST_RESORT, PROGRAM_NUMBER_REGISTRY_NAME);
#endif
}

static HashTable ProgramNumberHashTable = NULL;
static HashTable RegistryHashTable = NULL;

static void _text_EnsureRegistries()
{
  struct textinfo *s = NULL;

  if (RegistryHashTable == NULL)
    {
      char buf[1000], line[1000], name[1000], id[100];
      FILE *f;
      long lineno;

      RegistryHashTable = _ilu_hash_MakeNewTable (137, NULL, NULL);
      ProgramNumberHashTable = _ilu_hash_MakeNewTable (137, _ilu_hash_HashPointer, _ilu_hash_PointerCompare);

      formRegistryName (buf);
      if ((f = fopen(buf, "r")) == NULL)
	{
	  fprintf (stderr, "_text_EnsureRegistries:  Can't access program number registry file %s.\n",
		   buf);
	  return;
	}
      lineno = 0;
      while (fgets(line, sizeof(line), f) != NULL)
	{
	  lineno++;
     	  s = (struct textinfo *) malloc(sizeof(struct textinfo));
	  s->class = NULL;
	  if (strncmp(line, ".LastAssignedProgramNumber ", 27) == 0)
	    ; /* skip this record */
	  else {
#ifdef MACOS
            /* AC: all the following code 'fixed' to do a hand-parse due to brok
en sscanf() */
                /* NB: code now does a return if any line from registry is malfo
rmed */
		/* NB: We also don't use isspace() to advance over whitespace,
		 * because isspace() is also broken on the mac...*/
            if ((tmp=strchr(line,':'))==NULL) {
                fprintf (stderr, "_text_EnsureRegistries:  Invalid record (1)
in program number registry file \"%s\":  %s", buf, line);
                        return;
                }
                strncpy(name,line,tmp-line);
                name[tmp-line]='\0';
                tmp++;
                start=tmp;
                ch=*tmp;
                while ((ch!=' ') && (ch!='\0')) {
                        tmp++;
                        ch=*tmp;
                }
                if (ch=='\0') {
                        fprintf (stderr, "_text_EnsureRegistries:  Invalid rec
ord (2) in program number registry file \"%s\":  %s", buf, line);
                        return;
                }
                strncpy(id,start,tmp-start);
                id[tmp-start]='\0';
                tmp++;
                /* advance past white-space since strtoul is too stupid to do it
 for us... */
                while (*tmp==' ')
                        tmp++;
                s->pnumber=_ilu_strtoul(tmp,&tmp,16);
                if (tmp==NULL) {
                    fprintf (stderr, "_text_EnsureRegistries:  Invalid record
(3) in program number registry file \"%s\":  %s", buf, line);
                        return;
                }
                while (*tmp==' ')
                        tmp++;
                s->version=_ilu_strtoul(tmp,NULL,10);
#else /* MACOS */
if (sscanf (line, "%[^:]:%s %x %u", &name, &id, &s->pnumber, &s->version) != 4)
	    fprintf (stderr, "_text_EnsureRegistries:  Invalid record at line %u in program number registry file \"%s\":  %s", lineno, buf, line);
	  else
#endif	/* MACOS */
	       {
	         if (_ilu_hash_FindInTable (RegistryHashTable, id)
	   	  OR _ilu_hash_FindInTable (ProgramNumberHashTable, (void *) s->pnumber))
	   	{
		  fprintf (stderr, "_text_EnsureRegistries:  Duplicate type id or program number at line %u in program number registry \"%s\":  <%s> (ignoring)\n", lineno, buf, line);
		  continue;
		}
	      _ilu_hash_AddToTable (RegistryHashTable, s->type_id = _ilu_Strdup(id), s);
	      _ilu_hash_AddToTable (ProgramNumberHashTable, (void *) s->pnumber, s);
	     }
          }
	}
      fclose (f);
    }
}

static struct textinfo *_text_FindTextInformationForClass (ilu_Class class)
{
  struct textinfo *s = NULL;

  _text_EnsureRegistries();

  if ((s = (struct textinfo *) _ilu_hash_FindInTable (RegistryHashTable, class->cl_unique_id)) != NULL)
    s->class = class;

  DEBUG(TEXT_DEBUG,
	(stderr, "_text_FindTextInformationForClass:  Class \"%s:%s\", pnumber is 0x%x, version is %d.\n",
	 class->cl_name, class->cl_unique_id, s == NULL ? 0 : s->pnumber, s == NULL ? 0 : s->version));

  return (s);
}

static struct textinfo *_text_FindClassFromProgramNumber (ilu_cardinal pnumber, ilu_cardinal version)
{
  struct textinfo *s = NULL;

  _text_EnsureRegistries();

  s = (struct textinfo *) _ilu_hash_FindInTable (ProgramNumberHashTable, (void *) pnumber);

  DEBUG(TEXT_DEBUG,
	(stderr, "_text_FindClassFromProgramNumber:  pnumber is 0x%x, version is %u(%u), class is \"%s\" (\"%s:%s\").\n",
	 pnumber, version, (s == NULL) ? 0 : s->version,
	 (s == NULL) ? "unknown" : s->type_id,
	 (s == NULL || s->class == NULL) ? "" : s->class->cl_name,
	 (s == NULL || s->class == NULL) ? "" : s->class->cl_unique_id));

  return (s);
}

/*========================================================================*/
/*============================ Basic I/O code ==================================*/
/*========================================================================*/

#define INPUT_ERROR		1
#define OUTPUT_ERROR		2

ilu_cardinal ilu_text_SetMaxStringSize (ilu_cardinal size)
{
  ilu_cardinal old_size = TextMaxStringSize;
  if (size > 0)
    TextMaxStringSize = size;
  return (old_size);  
}

/* ==================== integer ==================== */

static ilu_boolean _text_OutputInteger (ilu_Call call, ilu_integer i)
{
  return (packet_put_long (TEXT(call_connection(call))->out, i));
}

static ilu_boolean _text_InputInteger (ilu_Call call, ilu_integer *i)
{
  return (packet_get_long (TEXT(call_connection(call))->in, i));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfInteger (ilu_Call call, ilu_integer i)
{
  return (4);
}

/* ==================== cardinal ==================== */

static ilu_boolean _text_OutputCardinal (ilu_Call call, ilu_cardinal i)
{
  return (packet_put_u_long (TEXT(call_connection(call))->out, i));
}

static ilu_boolean _text_InputCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (packet_get_u_long (TEXT(call_connection(call))->in, i));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfCardinal (ilu_Call call, ilu_cardinal i)
{
  return (4);
}

/* ==================== short integer ==================== */

static ilu_boolean _text_OutputShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (packet_put_short (TEXT(call_connection(call))->out, i));
}

static ilu_boolean _text_InputShortInteger (ilu_Call call, ilu_shortinteger *i)
{
  return (packet_get_short (TEXT(call_connection(call))->in, i));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfShortInteger (ilu_Call call, ilu_shortinteger i)
{
  return (4);
}

/* ==================== long integer ==================== */

static ilu_boolean _text_OutputLongInteger (ilu_Call call, ilu_integer *i)
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_put_u_long (TEXT(call_connection(call))->out, i[1]));
  return(stat && (packet_put_u_long (TEXT(call_connection(call))->out, i[0])));
}

static ilu_boolean _text_InputLongInteger (ilu_Call call, ilu_longinteger (*i))
{
  ilu_boolean stat = ilu_TRUE;
  stat = (packet_get_long (TEXT(call_connection(call))->in, &(*i)[1]));
  return (stat && (packet_get_long (TEXT(call_connection(call))->in, &(*i)[0])));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfLongInteger (ilu_Call call, ilu_integer *i)
{
  return (8);
}

/* ==================== short cardinal ==================== */

static ilu_boolean _text_OutputShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (packet_put_u_short (TEXT(call_connection(call))->out, i));
}

static ilu_boolean _text_InputShortCardinal (ilu_Call call, ilu_shortcardinal *i)
{
  return (packet_get_u_short (TEXT(call_connection(call))->in, i));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfShortCardinal (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== long cardinal ==================== */

static ilu_boolean _text_OutputLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  ilu_boolean stat;

  stat = (packet_put_u_long (TEXT(call_connection(call))->out, i[1]));
  return (stat && (packet_put_u_long (TEXT(call_connection(call))->out, i[0])));
}

static ilu_boolean _text_InputLongCardinal (ilu_Call call, ilu_longcardinal (*i))
{
  ilu_boolean stat;

  stat = (packet_get_u_long (TEXT(call_connection(call))->in, &(*i)[1]));
  return (stat && (packet_get_u_long (TEXT(call_connection(call))->in, &(*i)[0])));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfLongCardinal (ilu_Call call, ilu_cardinal *i)
{
  return (8);
}

/* ==================== enumeration ==================== */

static ilu_boolean _text_OutputEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  ilu_cardinal i2 = i;
  return (packet_put_u_long (TEXT(call_connection(call))->out, i2));
}

static ilu_boolean _text_InputEnumeration (ilu_Call call, ilu_shortcardinal *i)
{
  ilu_cardinal i2;
  ilu_cardinal stat;

  stat = packet_get_u_long (TEXT(call_connection(call))->in, &i2);
  *i = i2;
  return(stat);
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfEnumeration (ilu_Call call, ilu_shortcardinal i)
{
  return (4);
}

/* ==================== real ==================== */

static ilu_boolean _text_OutputReal (ilu_Call call, double d)
{
  return (packet_put_double (TEXT(call_connection(call))->out, d));
}

static ilu_boolean _text_InputReal (ilu_Call call, double *d)
{
  return (packet_get_double (TEXT(call_connection(call))->in, d));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfReal (ilu_Call call, double d)
{
  return (8);
}

/* ==================== short real ==================== */

static ilu_boolean _text_OutputShortReal (ilu_Call call, float f)
{
  float f2;

  f2 = f;
  return (packet_put_float (TEXT(call_connection(call))->out, f2));
}

static ilu_boolean _text_InputShortReal (ilu_Call call, float *f)
{
  return (packet_get_float (TEXT(call_connection(call))->in, f));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfShortReal (ilu_Call call, float d)
{
  return (4);
}

/* ==================== string ==================== */

static ilu_boolean _text_OutputString (ilu_Call call, ilu_string s, ilu_cardinal len, ilu_cardinal limit)
{
  ilu_cardinal len2 = len;

  return (_textxdr_put_string (TEXT(call_connection(call))->out, s, len2));
}

static ilu_boolean _text_InputString (ilu_Call call, ilu_string *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  if (!_textxdr_get_bytes (TEXT(call_connection(call))->in,
	(ilu_byte **) s, len))
    return (FALSE);
  return (TRUE);
}

  /*ARGSUSED*/
  ilu_cardinal _text_SizeOfString (ilu_Call call, ilu_string i, ilu_cardinal l, ilu_cardinal limit)
{
  if (l > TextMaxStringSize)
    {
      fprintf (stderr, "_text_SizeOfString:  Attempt to pass string of length %u, which exceeds either TextMaxStringSize value of %u, or limit on this sequence type of %u.\n",
	       l, TextMaxStringSize, limit);
      return (0);
    }

  return (4 + digits(l) + l + countQuotes(i, l));
}

/* ==================== bytes ==================== */

static ilu_boolean _text_OutputBytes (ilu_Call call, ilu_bytes s, ilu_cardinal len, ilu_cardinal limit)
{
  return (packet_put_bytes (TEXT(call_connection(call))->out, s, len));
}

static ilu_boolean _text_InputBytes (ilu_Call call, ilu_bytes *s, ilu_cardinal *len, ilu_cardinal limit)
{
  *s = NULL;
  return (packet_get_bytes (TEXT(call_connection(call))->in, s, len));
}

  /*ARGSUSED*/
static ilu_cardinal _text_SizeOfBytes (ilu_Call call, ilu_bytes i, ilu_cardinal l, ilu_cardinal limit)
{
  if (((l > 0) && (l > limit)) || l > TextMaxStringSize)
    {
      fprintf (stderr, "_text_SizeOfBytes:  Attempt to pass bytestring of length %u, which exceeds either TextMaxStringSize of %u, or limit on this sequence type of %u.\n",
	       l, TextMaxStringSize, limit);
      return (0);
    }
  return (4 + PADDED_SIZE(l));
}

/* ==================== byte ==================== */

static ilu_boolean _text_OutputByte (ilu_Call call, ilu_byte b)
{
  return (packet_put_u_byte (TEXT(call_connection(call))->out, b));
}

static ilu_boolean _text_InputByte (ilu_Call call, ilu_byte *b)
{
  return (packet_get_u_byte (TEXT(call_connection(call))->in, b));
}

/*ARGSUSED*/
ilu_cardinal _text_SizeOfByte (ilu_Call call, ilu_byte i)
{
  return (4);
}

/* ==================== opaque ==================== */

static ilu_boolean _text_OutputOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (packet_put_bytes (TEXT(call_connection(call))->out, o, len));
}

static ilu_boolean _text_InputOpaque (ilu_Call call, ilu_bytes *o, ilu_cardinal len)
{
  ilu_cardinal len2;

  *o = NULL;
  return(packet_get_bytes (TEXT(call_connection(call))->in, o, &len2));
}

ilu_cardinal _text_SizeOfOpaque (ilu_Call call, ilu_bytes o, ilu_cardinal len)
{
  return (PADDED_SIZE(len));
}

/* ==================== pipe ==================== */

static ilu_boolean _text_OutputPipe (ilu_Call call, ilu_Pipe pipe)
{
}

static ilu_boolean _text_InputPipe (ilu_Call call, ilu_Pipe *pipe)
{
}

ilu_cardinal _text_SizeOfPipe (ilu_Call call, ilu_Pipe pipe)
{
}

ilu_boolean _text_OutputSequence (ilu_Call c, ilu_cardinal sequenceLength, ilu_cardinal limit)
{
  return (_text_OutputCardinal (c, sequenceLength));
}

ilu_boolean _text_OutputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

ilu_boolean _text_InputSequenceMark (ilu_Call c, ilu_cardinal extent)
{
  return (TRUE);
}

ilu_boolean _text_InputSequence (ilu_Call c, ilu_cardinal *sequenceLength, ilu_cardinal limit)
{
  ilu_boolean stat;
  ilu_cardinal len;

  stat = _text_InputCardinal (c, &len);
  *sequenceLength = len;
  return(stat);
}

ilu_boolean _text_EndSequence (ilu_Call c)
{
  return (TRUE);
}

ilu_cardinal _text_SizeOfSequence (ilu_Call c, ilu_cardinal length, ilu_cardinal limit)
{
  return (_text_SizeOfCardinal(c, length));
}

ilu_boolean _text_OutputUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_text_OutputCardinal (c, typeIndex));
}

ilu_boolean _text_InputUnion (ilu_Call c, ilu_shortcardinal *typeIndex)
{
  ilu_boolean stat;
  ilu_cardinal i;

  stat = _text_InputCardinal (c, &i);
  *typeIndex = i;
  return(stat);
}

ilu_boolean _text_EndUnion (ilu_Call c)
{
  return (TRUE);
}

ilu_cardinal _text_SizeOfUnion (ilu_Call c, ilu_shortcardinal typeIndex)
{
  return (_text_SizeOfCardinal(c, typeIndex));
}

ilu_boolean _text_OutputArray (ilu_Call c, ilu_cardinal len)
{
  return(TRUE);
}

ilu_boolean _text_InputArray (ilu_Call c)
{
  return(TRUE);
}

ilu_boolean _text_EndArray (ilu_Call c)
{
  return (TRUE);
}

ilu_cardinal _text_SizeOfArray (ilu_Call c, ilu_cardinal len)
{
  return(0);
}

ilu_boolean _text_OutputRecord (ilu_Call c)
{
  return(TRUE);
}

ilu_boolean _text_InputRecord (ilu_Call c)
{
  return(TRUE);
}

ilu_boolean _text_EndRecord (ilu_Call c)
{
  return (TRUE);
}

ilu_cardinal _text_SizeOfRecord (ilu_Call c)
{
  return(0);
}

/*======================================================================
/*======================================================================
/*====================  Non-I/O code ===================================
/*======================================================================
/*====================================================================*/

ilu_string _text_FormProtocolHandle (ilu_Object obj)
{
  char buf[100];
  ilu_string p;
  struct textinfo *s = NULL;
  ilu_Class class = object_class(obj);

  if ((s = _text_FindTextInformationForClass (class)) == NULL)
    {
      fprintf (stderr, "_text_FormProtocolHandle:  Can't figure program#/version for class %s.\n",
	       class->cl_name);
      return (NULL);
    }
  else
    {
      sprintf (buf, "text_2_0x%x_%d", s->pnumber, s->version);
      return (_ilu_Strdup(buf));
    }
}
     
ilu_bytes _text_ReadPacket (ilu_Transport bs, ilu_cardinal *packetLength, ilu_PacketType *type, ilu_cardinal *sn)
{
  ilu_bytes packet;
  register ilu_bytes p;
  ilu_cardinal serialNumber, packetType, size;

  if (!transport_read_message(bs, &packet, &size)) {
    DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	  (stderr, "_text_ReadPacket:  error on transport_read_message\n"));
    return (NULL);
  };

  for (serialNumber = 0, p = packet+2;  isdigit(*p);  p++)
    serialNumber = (serialNumber * 10) + (*p - '0');

  if (*packet == 'Q')
    packetType = ilu_PacketType_Request;
  else if (*packet == 'P')
    packetType = ilu_PacketType_Reply;

  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_ReadPacket:  reading packet of length %d, SN %d, type %d (%s).\n",
	 size, serialNumber, packetType,
	 ((packetType == ilu_PacketType_Request) ? "request"
	  : (packetType == ilu_PacketType_Reply) ? "reply"
	  : "unknown type")));

  *packetLength = size;
  *type = packetType;
  *sn = serialNumber;
  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_ReadPacket:  returning packet SN %d\n", serialNumber));
  return (packet);  
}

ilu_boolean _text_InterpretPipeRequest (ilu_Call call, ilu_bytes packet, ilu_cardinal len)
{
  struct Text *c = TEXT(call_connection(call));
  ilu_cardinal rpcVersion, credSize, verifierSize, programNumber, programVersion, methodID, i;
  ilu_cardinal verifierType, credType, serialNumber, packetType;
  ilu_bytes credBuf, verifierBuf;
  ilu_Class class;

  c->in = _textxdr_CreatePacket (packet, len);

  _text_InputCardinal (call, &serialNumber);
  _text_InputCardinal (call, &packetType);
  _text_InputCardinal (call, &rpcVersion);
  _text_InputCardinal (call, &programNumber);
  _text_InputCardinal (call, &programVersion);
  _text_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretPipeRequest:  (call SN %d) class ID = %x (version %u), method ID = %u\n",
	 call->ca_SN, programNumber, programVersion, methodID));

  _text_InputCardinal (call, &credType);
  _text_InputBytes (call, &credBuf, &credSize, 0);
  _text_InputCardinal (call, &verifierType);
  _text_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG, (stderr,
			       "_text_InterpretPipeRequest:  (call SN %d) credBuf is %x, credSize is %d, verifierBuf is %x, verifierSize is %d\n",
			       call->ca_SN, credBuf, credSize, verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);
  return (TRUE);
}

ilu_boolean _text_InterpretRequest (ilu_Call call, ilu_bytes packet, ilu_cardinal len)
{
  struct Text *c = TEXT(call_connection(call));
  struct textinfo *s;
  ilu_cardinal rpcVersion, credSize, verifierSize, programNumber, programVersion, methodID, i;
  ilu_cardinal verifierType, credType, serialNumber, packetType;
  ilu_bytes credBuf, verifierBuf;
  ilu_Class class;

  c->in = _textxdr_CreatePacket (packet+1, len-1);

  packetType = (*packet == 'Q') ? ilu_PacketType_Request : ((*packet == 'R') ? ilu_PacketType_Reply : 0);
  _text_InputCardinal (call, &serialNumber);
  _text_InputCardinal (call, &rpcVersion);
  _text_InputCardinal (call, &programNumber);
  _text_InputCardinal (call, &programVersion);
  _text_InputCardinal (call, &methodID);
  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretRequest:  (call SN %d) class ID = %x (version %u), method ID = %u\n",
	 call->ca_SN, programNumber, programVersion, methodID));

  _text_InputCardinal (call, &credType);
  _text_InputBytes (call, &credBuf, &credSize, 0);
  _text_InputCardinal (call, &verifierType);
  _text_InputBytes (call, &verifierBuf, &verifierSize, 0);
  DEBUG(AUTHENTICATION_DEBUG, (stderr,
			       "_text_InterpretRequest:  (call SN %d) credBuf is %x, credSize is %d, verifierBuf is %x, verifierSize is %d\n",
			       call->ca_SN, credBuf, credSize, verifierBuf, verifierSize));
  
  /* not used at the moment */
  FREETOKEN(credBuf);
  FREETOKEN(verifierBuf);

    {
      if (port_singleton_p(connection_port(call_connection(call))))
	call_object(call) = _ilu_GetServerSingleton( port_server( connection_port( call_connection(call))));
      else
	ilu_InputObjectID (call, &call->object, TRUE, NULL);
    }

  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretRequest:  (call SN %d) object is 0x%x\n",
	 call->ca_SN, call->object));

  if (call->object == NULL)
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "_text_InterpretRequest:  (call SN %d) Can't find descriminator object port 0x%x.\n",
	     call->ca_SN, call->ca_connection->co_port));
      return (FALSE);
    }
  else if (call->object->ob_class == NULL)
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "_text_InterpretRequest:  (call SN %d) Class of object 0x%x not bound.\n",
	     call->ca_SN, call->object));
      return (FALSE);
    }

  s = _text_FindClassFromProgramNumber(programNumber, programVersion);
  class = (s == NULL) ? NULL
        : (s->class == NULL) ? ilu_FindClassFromID(s->type_id) : s->class;
  if (s == NULL OR class == NULL)
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "%s  (call %d) %s 0x%x, version %u\n",
	     "_text_InterpretRequest:", call->ca_SN,
	     "Can't find ilu_Class with pn",
	     programNumber, programVersion));
      return (FALSE);
    }

  call->ca_method = ilu_FindMethodByID (call->object, class, methodID);
  
  if (call->ca_method == NULL)
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr,
	     "%s  (call %d) %s \"%s\" (pn %x) with methodID = %u.\n",
	     "_text_InterpretRequest:", call->ca_SN,
	     "Can't find method on class", call->object->ob_class->cl_name,
	     programNumber, methodID));
      return (FALSE);
    }
  else
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "%s %d is 0x%x (%s).\n",
	     "_text_InterpretRequest:  record for method", 
	     methodID, call->ca_method, call->ca_method->me_name));
    }
  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretRequest:  returning TRUE\n"));
  return (TRUE);
}

enum ilu_ProtocolExceptions _text_InterpretReply (ilu_Call call, ilu_bytes packet, ilu_cardinal len, ilu_cardinal *estatus)
{
  struct Text *c = TEXT(call->ca_connection);
  ilu_cardinal verifierSize, replyStatus, serialNumber, packetType;
  ilu_bytes verifierBuf;
  ilu_cardinal authenticationType;
  static ilu_ProtocolException replyStatusExceptions[] = {
    ilu_ProtocolException_Success,
    ilu_ProtocolException_NoSuchClassAtServer,
    ilu_ProtocolException_ClassVersionMismatch,
    ilu_ProtocolException_NoSuchMethodOnClass,
    ilu_ProtocolException_GarbageArguments
    };
  static ilu_string acceptErrors[] = {
    "Success",
    "program unavailable",
    "program version mismatch",
    "procedure unavailable",
    "garbage arguments",
  };

  c->in = _textxdr_CreatePacket (packet+1, len);

  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretReply:  SN %d, %d bytes\n", call->ca_SN, len));

  _text_InputCardinal (call, &serialNumber);
  _text_InputCardinal (call, &replyStatus);

  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	(stderr, "_text_InterpretReply:  replyStatus is %d\n", replyStatus));

  if (replyStatus == 0)		/* MSG_ACCEPTED */
    {
      _text_InputCardinal (call, &authenticationType);
      _text_InputBytes (call, &verifierBuf, &verifierSize, 0);
      FREETOKEN(verifierBuf);
      _text_InputCardinal (call, &replyStatus);

      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "_text_InterpretReply:  *real* replyStatus is %d\n", replyStatus));

      if (replyStatus > 0)		/* *really* accepted */
	{
	  DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
		(stderr, "Sun RPC call %d signals protocol error %d (%s).\n",
		 call->ca_SN, replyStatus,
		 replyStatus < 5 ? acceptErrors[replyStatus] : "unknown error"));
	}
      else if (call->ca_method->exceptionCount > 0)
	_text_InputCardinal (call, estatus);
      else
	*estatus = 0;

      if (replyStatus < 5)
	return (replyStatusExceptions[replyStatus]);
      else
	return(ilu_ProtocolException_Unknown);
    }
  else if (replyStatus == 1)	/* MSG_REJECTED */
    {
      DEBUG((INCOMING_DEBUG | TEXT_DEBUG),
	    (stderr, "Sun RPC call %d signals \"Message Rejected\".\n",
	     call->ca_SN));
      return(ilu_ProtocolException_RequestRejected);
    }
}

static ilu_boolean GetTextProgramNumberAndVersion (ilu_Class pclass, ilu_cardinal *pnumber, ilu_cardinal *version)
{
  struct textinfo *s = NULL;

  if ((s = _text_FindTextInformationForClass (pclass)) != NULL)		/* probably consult external DB */
    {
      *pnumber = s->pnumber;
      *version = s->version;
    }
  return (s != NULL);
}

static ilu_boolean _text_BeginRequest (ilu_Call call, ilu_Class pclass, ilu_Method method, ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  ilu_cardinal pnumber=0, version=0;
  struct Text *c = TEXT(call->ca_connection);

  call->ca_method = method;

  if (call_object(call)==NULL) { /* no call object, must be pipe request */

  DEBUG(CALL_DEBUG, (stderr, "_text_BeginRequest:  pipe call 0x%x (sn %d), argSize %d, class %s (%s), method %s (%d)\n",
		     call, call_serial_number(call), argSize,
		     class_name(pclass), class_unique_id(pclass), method_name(method), method_id(method)));
  } else {
  DEBUG(CALL_DEBUG, (stderr, "_text_BeginRequest:  call 0x%x (sn %d), argSize %d, class %s (%s), method %s (%d)\n",
		     call, call_serial_number(call), argSize,
		     class_name(pclass), class_unique_id(pclass), method_name(method), method_id(method)));

  if (!GetTextProgramNumberAndVersion (pclass, &pnumber, &version))
    {
      DEBUG(CALL_DEBUG, (stderr, "_text_BeginRequest:  Can't determine program-number/version of object 0x%x (%s) of class \"%s\" on call 0x%x.\n", call->object, ilu_SBHOfObject(call->object), call->object->ob_class->cl_name, call));
      return (FALSE);
    }
  }
  DEBUG(CALL_DEBUG, (stderr, "_text_BeginRequest:  call 0x%x (sn %d), argSize %d, pnum/vers 0x%x/%d, method id %d\n",
		     call, call_serial_number(call), argSize, pnumber, version, method_id(method)));

  packetSize = argSize
    /* plus extra for header */
    + 2 + 9 + 5 + digits(call->ca_SN)
      + digits(pnumber) + digits(version)
	+ digits(method->me_id);

  c->out = _textxdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  *c->out->next++ = 'Q';
  *c->out->next++ = ' ';

  _text_OutputCardinal (call, call->ca_SN);
  _text_OutputCardinal (call, 1);		/* Text version 1 */
  _text_OutputCardinal (call, pnumber);		/* use Sun RPC program #'s */
  _text_OutputCardinal (call, version);
  _text_OutputCardinal (call, method->me_id);
  _text_OutputCardinal (call, 0);		/* credentials:  AUTH_NULL */
  _text_OutputCardinal (call, 0);		/* 		 0 bytes */
  _text_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _text_OutputCardinal (call, 0);		/*            0 bytes */
  
  DEBUG(CALL_DEBUG, (stderr, "_text_BeginRequest:  request %d begun (size %d).\n",
		       call->ca_SN, packetSize));
  return (TRUE);
}

static void _text_FlushBuffer (ilu_Call call)
{
  transport_flush_output(connection_transport(call_connection(call)));
}

static void _text_DropRequest (ilu_Call call)
{
  ilu_Transport bs = call_transport(call);
  struct Text *s = TEXT(call_connection(call));
  if (s->out != NULL)
      packet_destroy(s->out);
  s->out = NULL;
}

static ilu_boolean _text_SendPacket (ilu_Call call, ilu_boolean isReply)
{
  ilu_Transport bs = call_transport(call);
  struct Text *s = TEXT(call_connection(call));

  transport_send_message (bs, packet_buffer(s->out), packet_size(s->out));
  if (!bs->tr_class->tc_timesout)
    packet_buffer(s->out) = NULL;
  if (!(method_asynchronous(call_method(call))))
    transport_flush_output(bs);
  if (isReply) {
      if (bs->tr_class->tc_timesout) {
          _ilu_CacheCall(call, packet_buffer(s->out),
                         packet_size(s->out));
          packet_buffer(s->out) = NULL;
        }
      _text_DropRequest(call);
    }
  return (TRUE);
}

static ilu_boolean _text_FinishRequest (ilu_Call call)
{
  return (_text_SendPacket(call, FALSE));
}

static ilu_boolean _text_BeginReply (ilu_Call call, ilu_boolean exceptions, ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct Text *c = TEXT(call->ca_connection);

  DEBUG(PACKET_DEBUG, (stderr, "_text_BeginReply:  SN %d, argSize %d, exceptions %s, fd %d.\n",
		       call->ca_SN, argSize, exceptions ? "TRUE" : "FALSE",
		       transport_file_descriptor(connection_transport(call_connection(call)))));

  packetSize = argSize
    + 2			/* for packet type */
      + 9		/* fixed size fields */
	+ (exceptions ? 2 : 0)
	  + digits(call->ca_SN);

  c->out = _textxdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  *c->out->next++ = 'P';
  *c->out->next++ = ' ';

  _text_OutputCardinal (call, call->ca_SN);
  _text_OutputCardinal (call, 0);		/* message accepted */
  _text_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _text_OutputCardinal (call, 0);		/*            0 bytes */
  _text_OutputCardinal (call, 0);		/* successful execution */
  if (exceptions)
    _text_OutputCardinal (call, 0);		/* return code is Success */

  DEBUG(PACKET_DEBUG, (stderr, "_text_BeginReply:  started reply %d (size %d).\n", call->ca_SN, packetSize));
  return (TRUE);
}

static ilu_boolean _text_FinishReply (ilu_Call call)
{
  return (_text_SendPacket (call, TRUE));
}

static ilu_boolean _text_BeginException (ilu_Call call, ilu_cardinal evalue, ilu_cardinal argSize)
{
  ilu_cardinal packetSize;
  struct Text *c = TEXT(call->ca_connection);

  /* if "evalue" == 0, then argSize contains a protocol exception detail code. */

  if (evalue == 0)	/* signal protocol error */
    packetSize = (4*6);
  else
    packetSize = (4*7) + argSize;

  c->out = _textxdr_CreatePacket ((ilu_bytes) malloc(packetSize), packetSize);

  *c->out->next++ = 'P';
  *c->out->next++ = ' ';

  packetSize = argSize
    + 2			/* for packet type */
      + 8		/* fixed size fields */
	+ ((evalue == 0) ? digits(argSize) : (2 + digits(evalue)))
	  + digits(call->ca_SN);

  _text_OutputCardinal (call, call->ca_SN);
  _text_OutputCardinal (call, 0);		/* message accepted */
  _text_OutputCardinal (call, 0);		/* verifier:  AUTH_NULL */
  _text_OutputCardinal (call, 0);		/*            0 bytes */
  _text_OutputCardinal (call, (evalue == 0) ? argSize : 0);		/* successful execution */
  if (evalue > 0)
    _text_OutputCardinal (call, evalue);	/* exception value */

  DEBUG(PACKET_DEBUG, (stderr, "_text_BeginException:  exception started to peer %d:  SN %d, size %d, evalue %d.\n",
		       transport_file_descriptor(connection_transport(call_connection(call))),
		       call_serial_number(call), packetSize, evalue));
  return (TRUE);
}

static ilu_boolean _text_FinishException (ilu_Call call)
{
  return (_text_SendPacket(call, TRUE));
}

static ilu_Protocol _text_NewText (void)
{
  ilu_Protocol new = (ilu_Protocol) malloc(sizeof(struct _ilu_Protocol_s));

  new->pr_type = ilu_ProtocolType_Text;
  new->pr_concurrent_requests = ilu_FALSE;

  new->pr_begin_request = _text_BeginRequest;
  new->pr_finish_request = _text_FinishRequest;
  new->pr_resend_request = _text_FinishRequest;
  new->pr_drop_request = _text_DropRequest;
  new->pr_begin_reply = _text_BeginReply;
  new->pr_finish_reply = _text_FinishReply;
  new->pr_begin_exception = _text_BeginException;
  new->pr_finish_exception = _text_FinishException;

  new->pr_output_integer = _text_OutputInteger;
  new->pr_input_integer = _text_InputInteger;
  new->pr_size_of_integer = _text_SizeOfInteger;

  new->pr_output_shortinteger = _text_OutputShortInteger;
  new->pr_input_shortinteger = _text_InputShortInteger;
  new->pr_size_of_shortinteger = _text_SizeOfShortInteger;

  new->pr_output_longinteger = _text_OutputLongInteger;
  new->pr_input_longinteger = _text_InputLongInteger;
  new->pr_size_of_longinteger = _text_SizeOfLongInteger;

  new->pr_output_cardinal = _text_OutputCardinal;
  new->pr_input_cardinal = _text_InputCardinal;
  new->pr_size_of_cardinal = _text_SizeOfCardinal;

  new->pr_output_shortcardinal = _text_OutputShortCardinal;
  new->pr_input_shortcardinal = _text_InputShortCardinal;
  new->pr_size_of_shortcardinal = _text_SizeOfShortCardinal;

  new->pr_output_longcardinal = _text_OutputLongCardinal;
  new->pr_input_longcardinal = _text_InputLongCardinal;
  new->pr_size_of_longcardinal = _text_SizeOfLongCardinal;

  new->pr_output_real = _text_OutputReal;
  new->pr_input_real = _text_InputReal;
  new->pr_size_of_real = _text_SizeOfReal;

  new->pr_output_shortreal = _text_OutputShortReal;
  new->pr_input_shortreal = _text_InputShortReal;
  new->pr_size_of_shortreal = _text_SizeOfShortReal;

  new->pr_output_longreal = _text_OutputReal;
  new->pr_input_longreal = _text_InputReal;
  new->pr_size_of_longreal = _text_SizeOfReal;

  new->pr_output_optional = (ilu_boolean (*)(ilu_Call,ilu_boolean)) _text_OutputCardinal;
  new->pr_input_optional = (ilu_boolean (*)(ilu_Call,ilu_boolean *)) _text_InputCardinal;
  new->pr_size_of_optional = (ilu_cardinal (*)(ilu_Call,ilu_boolean)) _text_SizeOfCardinal;

  new->pr_output_enum_code = _text_OutputEnumeration;
  new->pr_input_enum_code = _text_InputEnumeration;
  new->pr_size_of_enum_code = _text_SizeOfEnumeration;

  new->pr_output_byte = _text_OutputByte;
  new->pr_input_byte = _text_InputByte;
  new->pr_size_of_byte = _text_SizeOfByte;

  new->pr_output_character = _text_OutputShortCardinal;
  new->pr_input_character = _text_InputShortCardinal;
  new->pr_size_of_character = _text_SizeOfShortCardinal;

  new->pr_output_boolean = (ilu_boolean (*)(ilu_Call,ilu_boolean)) _text_OutputCardinal;
  new->pr_input_boolean = (ilu_boolean (*)(ilu_Call,ilu_boolean *)) _text_InputCardinal;
  new->pr_size_of_boolean = (ilu_cardinal (*)(ilu_Call,ilu_boolean)) _text_SizeOfCardinal;

  new->pr_output_shortchar = _text_OutputByte;
  new->pr_input_shortchar = _text_InputByte;
  new->pr_size_of_shortchar = _text_SizeOfByte;

  new->pr_output_string = _text_OutputString;
  new->pr_input_string = _text_InputString;
  new->pr_size_of_string = _text_SizeOfString;

  new->pr_output_bytes = _text_OutputBytes;
  new->pr_input_bytes = _text_InputBytes;
  new->pr_size_of_bytes = _text_SizeOfBytes;

  new->pr_output_opaque = _text_OutputOpaque;
  new->pr_input_opaque = _text_InputOpaque;
  new->pr_size_of_opaque = _text_SizeOfOpaque;

  new->pr_output_stringvec = (ilu_boolean (*)(ilu_Call,ilu_string,ilu_cardinal)) _text_OutputOpaque;
  new->pr_input_stringvec = (ilu_boolean (*)(ilu_Call,ilu_string *,ilu_cardinal)) _text_InputOpaque;
  new->pr_size_of_stringvec = (ilu_cardinal (*)(ilu_Call,ilu_string,ilu_cardinal)) _text_SizeOfOpaque;

  new->pr_output_pipe = _text_OutputPipe;
  new->pr_input_pipe = _text_InputPipe;
  new->pr_size_of_pipe = _text_SizeOfPipe;

  new->pr_output_sequence = _text_OutputSequence;
  new->pr_output_sequence_mark = _text_OutputSequenceMark;
  new->pr_input_sequence = _text_InputSequence;
  new->pr_input_sequence_mark = _text_InputSequenceMark;
  new->pr_end_sequence = _text_EndSequence;
  new->pr_size_of_sequence = _text_SizeOfSequence;

  new->pr_output_record = _text_OutputRecord;
  new->pr_input_record = _text_InputRecord;
  new->pr_end_record = _text_EndRecord;
  new->pr_size_of_record = _text_SizeOfRecord;

  new->pr_output_array = _text_OutputArray;
  new->pr_input_array = _text_InputArray;
  new->pr_end_array = _text_EndArray;
  new->pr_size_of_array = _text_SizeOfArray;

  new->pr_output_union = _text_OutputUnion;
  new->pr_input_union = _text_InputUnion;
  new->pr_end_union = _text_EndUnion;
  new->pr_size_of_union = _text_SizeOfUnion;

  new->pr_form_handle = _text_FormProtocolHandle;

  new->pr_read_packet = _text_ReadPacket;
  new->pr_interpret_pipe_request = _text_InterpretPipeRequest;
  new->pr_interpret_request = _text_InterpretRequest;
  new->pr_interpret_reply = _text_InterpretReply;

  new->pr_create_data_block = _text_CreateDataBlock;
  new->pr_free_data_block = (void (*)(void *)) _text_FreeDataBlock;
  new->pr_flush = _text_FlushBuffer;

  return (new);
}

ilu_Protocol _ilu_text_Protocol(void)
{
  static ilu_Protocol StandardText = NULL;
  if (StandardText == NULL)
    StandardText = _text_NewText();
  return (StandardText);
}
