/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: udp.c,v 1.13 1994/05/13 00:27:14 janssen Exp $ */
/* Last tweaked by Mike Spreitzer April 22, 1994 3:08 pm PDT */

#define _BSD_SOURCE

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/param.h>		/* for MAXHOSTNAMELEN */
#include <netdb.h>		/* for gethostbyname() */
#include <arpa/inet.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/filio.h>		/* for FIONBIO */
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ilu.h"
#include "iluntrnl.h"

#include "transprt.h"
#include "mooring.h"

#define max_udp_recv 8200

/*L1_sup < trmu; L2, Main unconstrained*/
ilu_TransportClass _ilu_udp_TransportClass(void);

typedef struct udpparms {
  /*L1, L2, Main unconstrained*/
  
  ilu_string hostname;
  struct sockaddr_in addr;	/* For a Transport, the other side;
				   For a Mooring, this side */
  ilu_boolean incoming;		/* for a Transport: into server,
				   or going out from client? */
  /*L1 >= {connection's iomu}*/

  ilu_bytes buf;			/* for incoming; the packet */
  ilu_integer len;		/* for incoming; the pkt len */
} *UDPParms;
/* What goes in the data field of a UDP ilu_Transport or ilu_Mooring.
   The FD (socket) of an outgoing UDP Transport is `bound' to its
   destination; the FD of an incoming UDP Mooring or Transport is
   not bound. */

/*L1, L2, Main unconstrained*/

#define UDP_HOSTNAME(a) (((UDPParms)(a))->hostname)
#define UDP_PORT(a) (((UDPParms)(a))->addr.sin_port)

#define MAXDUMP		10000

/**********************************************************************
***********************************************************************
***********************************************************************
***** First, the methods for the UDP Transport ************************
***********************************************************************
***********************************************************************
**********************************************************************/

/*L2, Main unconstrained*/

/*L1 unconstrained*/

static ilu_private InterpretInfo (ilu_string info)
{
  char hostname[1000];
  ilu_cardinal port;
  struct hostent *hp;

  if ((sscanf (info, "udp_%[^_]_%u", hostname, &port)) == 2)
    {
      UDPParms new = (UDPParms)
			     malloc(sizeof(struct udpparms));
      memset((ilu_string ) &new->addr, 0, sizeof(new->addr));
      new->addr.sin_family = AF_INET;
      new->addr.sin_port = htons((ilu_shortcardinal) port);

      new->addr.sin_addr.s_addr = OS_INETADDR(hostname);
      if (   new->addr.sin_addr.s_addr == -1
          || new->addr.sin_addr.s_addr == 0)
	if ((hp = OS_GETHOSTBYNAME(hostname)) != NULL)
	  memcpy ((ilu_string) &new->addr.sin_addr, hp->h_addr, hp->h_length);

      if (   new->addr.sin_addr.s_addr == -1
          || new->addr.sin_addr.s_addr == 0)
	{
	  DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		(stderr, "udp InterpretInfo:  Invalid host name (%s).\n",
		 hostname));
	  free(new);
	  return (NULL);
	}

      new->hostname = _ilu_Strdup(hostname);
      new->incoming = FALSE;
      new->buf = NULL;
      new->len = 0;
      return ((ilu_private) new);
    }
  else
    return (NULL);
}

static void DumpPacket (ilu_byte *packet, ilu_shortcardinal length)
{
  ilu_cardinal dumplength;
  int n;
  ilu_byte c;
  register ilu_integer i, j;

  if (length > MAXDUMP)
    {
      fprintf (stderr, "Request to dump packet of %u bytes.  Only %u bytes being dumped.\n",
	       length, MAXDUMP);
      dumplength = MAXDUMP;
    }
  else
    dumplength = length;
  if (packet == NULL)
    {
      fprintf (stderr, "Attempt to dump NULL packet.\n");
      return;
    }
  fprintf (stderr, "DumpPacket of packet 0x%x, length ", (unsigned long) packet);
  fprintf (stderr, "%u bytes, dumping %u bytes:\n", length, dumplength);
  for (i = 0;  i < dumplength;  i += 16)
    {
      fprintf (stderr, "%6u:  ", i);
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	fprintf (stderr, "%02x%s ", packet[i + j],
		 ((j % 4) == 3) ? " " : "");
      n = (((16-j)/4)*(13)+((16-j)%4)*3)+1; /* padding before ascii */
      fprintf (stderr, "%*.*s", n, n, "");
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	{
	  c = packet[i + j];
	  fprintf (stderr, "%c", ((c >= ' ') && (c <= '~')) ? (char) c
							    : '.');
	}
      fprintf (stderr, "\n");
    }
}

/*L2 >= {conn's iomu}*/
/*L1, Main unconstrained*/
static ilu_integer NDataPresent (ilu_Transport t)
{
  UDPParms parms = (UDPParms) transport_data(t);
  long count;
  ilu_boolean status;
  if (parms->incoming)
    return 1;
  status = ioctl (t->tr_fd, FIONREAD, &count);
  if (status==0 && count==0) {
      int errnum = 0;
      int errsize = sizeof(errnum);
      status = getsockopt(t->tr_fd, SOL_SOCKET, SO_ERROR,
                          (char *) &errnum, &errsize);
      if (status < 0 || errnum != 0)
           return -1;
      else return 0;
    }
  return ( (status != 0) ? -1 : (count > 0) ? 1 : 0 );
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_boolean WaitForInput (ilu_Transport t, ilu_boolean *sure,
				 ilu_FineTime *limit)
{
  UDPParms parms = (UDPParms) transport_data(t);
  if (parms->incoming) {
      *sure = TRUE;
      return (parms->buf != NULL);
    }
  else {
      _ilu_WaitForInputOnFD(t->tr_fd, sure, limit);
      return (TRUE);
    }
}

static ilu_boolean FlushOutput (ilu_Transport self)
{
  return (TRUE);	/* Never buffered */
}

/*Main Invariant holds*/
/*L2 >= {conn's iomu}*/

static ilu_boolean SendMessage (ilu_Transport conn, ilu_byte *buffer,
				ilu_cardinal count)
{
  ilu_boolean status = TRUE;
  ilu_integer sent;
  UDPParms parms = (UDPParms) conn->tr_data;

  if (conn == NULL)
    return (FALSE);

  DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
	(stderr,
	 "udp.c:SendMessage:  writing %d bytes from 0x%x to fd %d.\n",
	 count, (unsigned long) buffer, conn->tr_fd));

  if ((_ilu_DebugLevel & PACKET_DEBUG) != 0)
      DumpPacket(buffer, count);

  if (parms->incoming)
       sent = sendto(conn->tr_fd, (char *) buffer, count, 0,
		     (struct sockaddr *) &parms->addr, sizeof(parms->addr));
       /* Even though the SunOS man page says sendto can be used on
          connected sockets, in fact that raises errno 56 */
  else sent = send(conn->tr_fd, (char *) buffer, count, 0);
  _ilu_Assert(sent <= (ilu_integer) count,
              "udp SendMessage: excess bytes sent!");
  if (sent < count) {
      fprintf(stderr, "%s %u of %d bytes signalled error \"%s\".\n",
	      "udp.c:SendMessage:  output to fd", conn->tr_fd, count,
	      (sent < 0) ? ANSI_STRERROR(errno)
		         : ((sent == 0) ? "Connection lost"
				        : "not enough bytes written"));
      status = FALSE;
    }
  else
      status = TRUE;
  return (status);
}

/*L1, L2, Main unconstrained*/

/*Main Invariant holds*/
static ilu_boolean Connect (ilu_Transport self)
{
  UDPParms parms = (UDPParms) transport_data(self);
  ilu_integer fd;
  ilu_boolean status = TRUE;
  int one = 1;

  _ilu_AutoSetDebugLevel();

  if (self == NULL)
    return (FALSE);

  DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
	(stderr, "udp.c:Connect:  connecting to host %s, port %d...\n",
	 UDP_HOSTNAME(transport_data(self)),
	 UDP_PORT(transport_data(self))));

  if ((fd = OS_SOCKET(AF_INET, SOCK_DGRAM, 0)) < 0)
    {
      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
	    (stderr, "udp.c:Connect:  socket call failed:  %s.\n",
	     ANSI_STRERROR(errno)));
      status = FALSE;
    }
  else
    {
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_REUSEADDR,
			   (ilu_string ) NULL, 0);
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_USELOOPBACK,
			   (ilu_string ) NULL, 0);

    

      if (ioctl(fd, FIONBIO, &one) != 0) {
        DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		(stderr, "%s (descriptor 0x%x, host %s) non-blocking.\n",
		 "udp.c:Connect:  Failed to set socket",
		 fd, parms->hostname));;
	  status = FALSE;
	}
      else
	{
	  if (OS_CONNECT(fd, (struct sockaddr *) &parms->addr,
	                 sizeof(parms->addr)) == 0)
	    {
	      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		    (stderr, "udp.c:Connect:  connected to %s:%u\n",
		     inet_ntoa(parms->addr.sin_addr),
		     ntohs(parms->addr.sin_port)));
	      self->tr_fd = fd;
	      status = TRUE;
	    }
	  else if (errno == EINPROGRESS) {
	    struct sockaddr peername;
	    int pnlen = sizeof(peername);
	    ilu_boolean sure;
	    _ilu_WaitForOutputOnFD(fd, &sure, NULL);
	    if (getpeername(fd, &peername, &pnlen) == 0) {
	      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		    (stderr,
		     "udp.c:Connect:  eventually connected to %s[%u]\n",
		     inet_ntoa(parms->addr.sin_addr),
		     ntohs(parms->addr.sin_port)));
	      self->tr_fd = fd;
	      status = TRUE;
	      }
	    else {
	      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		    (stderr, "%s %s:%u failed (%s).\n",
		     "udp.c:Connect: connect to",
		     OS_INETNTOA(parms->addr.sin_addr),
		     ntohs(parms->addr.sin_port),
		     "no meaningful errno available" ));
	      close(fd);
	      status = FALSE;
	      }
	    }
	  else
	    {
	      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
		    (stderr, "%s %s[%u] failed, error %d (%s)\n",
		     "udp.c:Connect: connect to",
		     OS_INETNTOA(parms->addr.sin_addr),
		     ntohs(parms->addr.sin_port), errno,
		     ANSI_STRERROR(errno)));
	      OS_CLOSE(fd);
	      status = FALSE;
	    }
	}
    }
  return (status);
}

/*L2 >= {conn's iomu}*/

static void Close (ilu_Transport self)
{
  UDPParms parms = (UDPParms) transport_data(self);
  if (self == NULL)
    return;
  if (parms->incoming && (parms->buf != NULL))
      free(parms->buf);
  if (!parms->incoming)
      OS_CLOSE (self->tr_fd);
  return;
}

/*Main Invariant holds*/
static ilu_boolean ReadMessage (ilu_Transport self,
				ilu_bytes *buffer,
				ilu_cardinal *size)
{
  UDPParms parms = (UDPParms) transport_data(self);
  ilu_bytes packet;
  int len;

  if (self == NULL)
    return(FALSE);

  if (! parms->incoming) {
      packet = (unsigned char *) malloc(max_udp_recv);
      _ilu_Assert(packet != NULL, "udp.c:ReadMessage: malloc failed");
      len = recv(self->tr_fd, (char *) packet, max_udp_recv, 0);
      if (len == 0) {
          DEBUG(UDP_DEBUG, (stderr, "udp.c:ReadMessage: got length 0"));
          free(packet);
          return FALSE;
        }
      if (len < 0) {
          DEBUG(UDP_DEBUG,
                (stderr,
                 "udp.c:ReadMessage: recv(%s:%d) failed, ernno=%d(%s).\n",
                 UDP_HOSTNAME(transport_data(self)),
                 UDP_PORT(transport_data(self)),
                 errno, ANSI_STRERROR(errno) ));
          free(packet);
          return FALSE;
        }

      *buffer = packet;
      *size = len;
    }
  else {
      _ilu_Assert(parms->buf != NULL,
                  "udp ReadMessage: incoming buf is NULL");
      *buffer = parms->buf;
      *size = parms->len;
      parms->buf = NULL;
      parms->len = 0;
    }
  if (_ilu_DebugLevel & PACKET_DEBUG != 0)
    DumpPacket(*buffer, *size);

  return (TRUE);
}

/**********************************************************************
***********************************************************************
***********************************************************************
**** Now the methods for the UDP Mooring ******************************
***********************************************************************
***********************************************************************
**********************************************************************/

/*L1, L2, Main unconstrained*/

static ilu_string FormHandle (ilu_refany parameterblock)
{
  UDPParms parms = (UDPParms) parameterblock;
  char buf[11];
  if (parms == NULL)
    return (NULL);
  sprintf (buf, "%u", UDP_PORT(parms));
  return (_ilu_Strcat5("udp", "_", UDP_HOSTNAME(parms), "_", buf));
}

/*L1_sup < trmu*/
static ilu_Transport AcceptClient (ilu_Mooring self)
{
  int buflen = max_udp_recv;
  struct sockaddr_in from;
  int fromlen = sizeof(from);
  UDPParms mparms = self->mo_data;
  UDPParms cparms = (UDPParms) malloc(sizeof(struct udpparms));
  ilu_Transport new = NULL;

  _ilu_AutoSetDebugLevel();

  cparms->buf = (unsigned char *) malloc(buflen);
  _ilu_Assert(cparms->buf != NULL, "udp_AcceptClient: malloc failed");
  
  cparms->len = recvfrom(self->mo_fd, (char *) cparms->buf, buflen, 0,
                         (struct sockaddr *) &from, &fromlen);
  
  if (cparms->len < 0) {
      DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
            (stderr,
             "_udp_AcceptClient: recvfrom port %u failed, errno=%d.\n",
             ntohs(mparms->addr.sin_port), errno));
      free(cparms->buf);
      free(cparms);
      return (NULL);
    }

  cparms->hostname = _ilu_Strdup(OS_INETNTOA(from.sin_addr));
  cparms->addr = from;
  cparms->incoming = TRUE;
  new = (ilu_Transport) malloc (sizeof(struct _ilu_Transport_s));
  new->tr_class = _ilu_udp_TransportClass();
  new->tr_fd = self->mo_fd;
  new->tr_to1 = ilu_FineTime_FromDouble( 0.7);
  new->tr_toN = ilu_FineTime_FromDouble( 5.0);
  new->tr_tto = ilu_FineTime_FromDouble(20.0);
  new->tr_data = cparms;

  DEBUG((CONNECTION_DEBUG | UDP_DEBUG),
	(stderr,
	 "udp AcceptClient: request of len %u from %s:%u on port %u.\n",
	 cparms->len, cparms->hostname, ntohs(cparms->addr.sin_port),
	 ntohs(mparms->addr.sin_port)));

  return (new);
}

/*L1_sup < trmu*/
static ilu_Mooring CreateMooring (ilu_private mooringInfo)
{
  UDPParms parms = (UDPParms) mooringInfo;
  struct sockaddr_in sin = ((UDPParms) mooringInfo)->addr;
  ilu_integer skt;
  int namelen;
  ilu_Mooring self;

  _ilu_AutoSetDebugLevel();

  if ((skt = OS_SOCKET(AF_INET, SOCK_DGRAM, 0)) < 0) {
    DEBUG((EXPORT_DEBUG | UDP_DEBUG),
	  (stderr, "_udp_CreateMooring: socket failed:  %s.\n",
	   ANSI_STRERROR(errno)));
    return (NULL);
  }
    
  sin.sin_addr.s_addr = INADDR_ANY;	/* me, me! */

  if (OS_BIND(skt, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
    DEBUG((EXPORT_DEBUG | UDP_DEBUG),
	  (stderr, "_udp_CreateMooring: bind to port %u failed:  %s.\n",
	   ntohs(parms->addr.sin_port), ANSI_STRERROR(errno)));
    OS_CLOSE (skt);
    return (NULL);
  }

  if (sin.sin_port == 0) {
    namelen = sizeof(sin);
    if (OS_GETSOCKNAME(skt, (struct sockaddr *) &sin, &namelen) < 0)
      {
	DEBUG((EXPORT_DEBUG | UDP_DEBUG),
	      (stderr, "_udp_CreateMooring: getsockname failed:  %s.\n",
	       ANSI_STRERROR(errno)));
	OS_CLOSE (skt);
	return (NULL);
      }
    parms->addr = sin;
  }
  
  DEBUG((EXPORT_DEBUG | UDP_DEBUG),
        (stderr, "udp CreateMooring: new mooring on FD %d, port %u.\n",
         skt, ntohs(sin.sin_port)));

  self = (ilu_Mooring) malloc (sizeof(struct _ilu_Mooring_s));

  self->mo_fd = skt;
  self->mo_accept_connection = AcceptClient;
  if (UDP_HOSTNAME(parms) != NULL)
    free(UDP_HOSTNAME(parms));
  _ilu_AcquireMutex(ilu_trmu);
  UDP_HOSTNAME(parms) = _ilu_Strdup(_ilu_tcp_CurrentHostInetName());
  _ilu_ReleaseMutex(ilu_trmu);
  self->mo_transportClass = _ilu_udp_TransportClass();
  self->mo_data = (ilu_private) parms;

  return (self);
}

static void CloseMooring(ilu_Mooring m)
{
  int res;
  res = close(m->mo_fd);
  ASSERT(res==0, buf,
	 (buf, "udp_CloseMooring: res=%d, errno=%s", res,
	  ANSI_STRERROR(errno) ));
}

/*L1_sup < trmu*/
ilu_TransportClass _ilu_udp_TransportClass (void)
{
  static ilu_TransportClass m = NULL;
  _ilu_AcquireMutex(ilu_trmu);
  if (m == NULL)
    {
      m = (ilu_TransportClass)
	  malloc(sizeof(struct _ilu_TransportClass_s));
      m->tc_type		= ilu_TransportType_UDP;
      m->tc_pFD		= 1;
      m->tc_cFD		= 1;	/* Ack!  Half untrue! */
      m->tc_timesout	= TRUE;
      m->tc_interpret_info	= InterpretInfo;
      m->tc_form_info	= FormHandle;
      m->tc_connect	= Connect;
      m->tc_wait_for_input	= WaitForInput;
      m->tc_create_mooring	= CreateMooring;
      m->tc_close_mooring	= CloseMooring;
      m->tc_close		= Close;
      m->tc_n_data_present	= NDataPresent;
      m->tc_send_message	= SendMessage;
      m->tc_read_message	= ReadMessage;
      m->tc_flush_output	= FlushOutput;
    }
  _ilu_ReleaseMutex(ilu_trmu);
  return (m);
}

