(* Ilu.i3 *)
(* Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights
   Reserved.

   Unlimited use, reproduction, and distribution of this software is
   permitted.  Any copy of this software must include both the above
   copyright notice of Xerox Corporation and this paragraph.  Any
   distribution of this software must comply with all applicable United
   States export control laws.  This software is made available AS IS,
   and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
   NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY
   FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY
   DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE)
   OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGES. *)
(* Edited by Mike Spreitzer at January 11, 1994 1:29 pm PST *)

INTERFACE Ilu;

(* What clients and server applications need to know about ILU.  See
   IluRuntime for the interface between stubs and the ILU runtime. *)

IMPORT IluKernel, Word;
FROM IluBasics IMPORT Failed, Failure;

<*PRAGMA lL, Ll, Main*>


(* Concurrency and locking:

   As in iluExports.h.  The ILU/Modula-3 runtime adds the folloing
   mutexes:
| ssMu		global mutex for server registry;
| srmu		global mutex for StrongRef implementation;
| ocMu		global mutex for ObjectCreator registry;
| Ilu.Server	each one is a mutex;

   and the following ordering constraints:
|  IluKernel.Server < ssMu < Ilu.Server
|  IluKernel.Server < srmu
|  IluKernel.Server < ocMu

   *)

(* RPC protocol failures *)

TYPE
  ProtocolFailure =
    Failure BRANDED OBJECT case: ProtocolFailureCase; END;


  ProtocolResultCode =
    {Success, NoSuchTypeAtServer, TypeVersionMismatch,
     NoSuchMethodOnType, GarbageArguments, Unknown, LostConnection,
     RequestRejected, RequestTimeout};

  ProtocolFailureCase = [ProtocolResultCode.NoSuchTypeAtServer ..
                          ProtocolResultCode.RequestTimeout];


(* Datatypes defined in ISL. *)

TYPE Byte = [0 .. 255];
TYPE PackedByte = BITS 8 FOR Byte;
TYPE ShortInt = [-32768 .. 32767];
TYPE Integer = INTEGER;
TYPE LongInt = ARRAY [0 .. 1] OF INTEGER;
TYPE ShortCard = [0 .. 65535];
TYPE Cardinal = Word.T;
TYPE LongCard = ARRAY [0 .. 1] OF Word.T;
TYPE ShortReal = REAL;
TYPE Real = LONGREAL;
TYPE LongReal = EXTENDED;
TYPE ShortChar = ['\000' .. '\377'];
TYPE PackedShortChar = BITS 8 FOR ShortChar;
TYPE Character = ShortCard; (* In Unicode. *)
TYPE String = TEXT; (* With no embedded '\000'. *)
TYPE WString = REF ARRAY OF Character; (* With no embedded 0. *)
TYPE Bytes = REF ARRAY OF PackedByte;


(* The String Binding Handle. *)

TYPE
  SBH = TEXT;
  (* A string that includes an instance ID and a contact-info *)

TYPE
  InstanceId = TEXT;
  (* A unique identifier for an object; it is factored into a ServerId
     and an ObjectHandle. *)

TYPE
  ServerId = TEXT;
  (* A unique identifier for a server *)

TYPE
  ObjectHandle = TEXT;
  (* A server-relative identifier for an object *)

TYPE
  ContactInfo = TEXT;
  (* An encoding of how to reach a server *)


(* ================ Server stuff ================ *)

TYPE
  ServerPrivate <: ROOT;
  Server = ServerPrivate OBJECT
             <*lL, Ll, Main unconstrained*>
             id: ServerId; (*READONLY*)
           METHODS
           END;
  (* A data structure that represents a server, either local to this
     program or remote.  Each server is actually one of the following
     two types. *)

TYPE SurrogateServer <: Server;

TYPE
  TrueServer <:
    Server OBJECT
    METHODS
      <*Main Invariant holds; Ll otherwise unconstrained*>

      HandleListenerFailure (f: Failure): FailureAction;
      (* When there's a failure in a listener for this server, this
         procedure is notified, and the result indicates whether the
         listener is abandoned or continues listening. *)
      HandleWorkerFailure (f: Failure): FailureAction;
      (* When there's a failure in a worker for this server, this
         procedure is notified, and the result indicates whether the
         connection is abandoned or continues listening. *)
    END;
  (* A server local to this program. *)

TYPE FailureAction = {Quit, Continue};

<*lL, Ll = {}*>
PROCEDURE DefaultHandleListenerFailure (self: TrueServer; f: Failure):
  FailureAction;

<*lL, Ll = {}*>
PROCEDURE DefaultHandleWorkerFailure (self: TrueServer; f: Failure):
  FailureAction;

<*Main Invariant holds; Ll otherwise unconstrained*>

PROCEDURE InitTrueServer (self  : TrueServer;
                          id    : ServerId := NIL;
                          objtab: ObjectTable := NIL ): TrueServer
  RAISES {Failed};
  (* A true server is created by calling InitTrueServer(NEW(TrueServer),
     ...).  The caller may specify the ILU "server-id", or let ILU
     generate one.  The caller may specify an ObjectTable; if NIL is
     given, a new standard one is used.  Failed is raised if the
     server-id is discovered to have been used before.  TrueServer may
     be sub-classed, overriding any of the public methods. *)

TYPE
  ObjectTable =
    OBJECT
    METHODS
      <*lL >= {the kernel server}*>
      <*lL >= {gcmu} if the object is collectible*>
      <*Ll, Main unconstrained*>

      ObjectToHandle (o: Object): ObjectHandle;
      (* Returns the handle associated with the given object, inventing
         and recording a new handle if necessary. *)
      HandleToObject (h: ObjectHandle): Object;
      (* Returns the Object associated with the given handle, or NIL if
         no such Object. *)
    END;
  (* An one-to-one association between Objects and ObjectHandles, such
     as a server might maintain. *)

PROCEDURE Export_Server (server: TrueServer;
                         p     : ProtocolInfo;
                         t     : TransportInfo ) RAISES {Failed};
  (* Make this server reachable via the given protocol and transport.
     The transport and protocol may use field values that mean
     "unspecified", in which case this procedure will choose suitable
     values.  This procedure must be called before SbhFromObject is
     invoked on any object of the server. *)

TYPE ProtocolInfo = BRANDED OBJECT END;
TYPE SunRpc2 = ProtocolInfo BRANDED OBJECT prognum, version := 0 END;
TYPE Courier = ProtocolInfo BRANDED OBJECT prognum, version := 0 END;

TYPE TransportInfo = BRANDED OBJECT END;
TYPE
  TCP = TransportInfo BRANDED OBJECT host, port := 0 END;
  UDP = TransportInfo BRANDED OBJECT host, port := 0 END;
  (* host and port are in host, not network, byte order. *)
TYPE SPP = TransportInfo BRANDED OBJECT addr := AnyXnsAddr END;

TYPE
  XnsAddr = RECORD
              net   : XnsNet;
              host  : XnsHost;
              socket: XnsSocket
            END;
  XnsNet = Cardinal;
  XnsHost = ARRAY [0 .. 5] OF PackedByte;
  XnsSocket = ShortCard;
CONST AnyXnsAddr = XnsAddr{0, XnsHost{0, ..}, 0};

TYPE
  Root <: ROOT; (* A hook for applications to jack up the base of the
                   world; look for some extra-lingual mechanism to
                   supply the concrete type. *)

  Object <: ObjectPublic;
  ObjectPublic =
    Root OBJECT
      <*lL, Ll, Main unconstrained*>
      ilu_is_surrogate: BOOLEAN := FALSE;
    METHODS
      <*lL, Ll, Main unconstrained*>

      ILU_Get_Server (): Server;
      ILU_Get_Type (): ObjectType;
      (* Returns the most specific ILU type known to this program for
         the ILU object represented by this Modula-3 object. *)
      ILU_Qua_Type (ot: ObjectType): Object;

      <*Main Invariant holds; Ll otherwise unconstrained*>

      ILU_Close           () RAISES {};
      ILU_Close_Surrogate () RAISES {};
    END;
  (* An ILU object, either true or a surrogate.  The implementor of a
     true object is responsible for ensuring that ilu_is_surrogate is
     FALSE and that ILU_Get_Server and ILU_Get_Type return appropriate,
     non-NIL values.  Surrogate objects will be implemented by
     ILU-generated stubs.

     ILU_Close is called on an object by the application program when it
     is done using the object; this enables the kernel to reclaim
     resources used by the object.  It is always safe (but maybe
     expensive to recover from if the object needs to be resurrected) to
     call ILU_Close on a surrogate object or a true object that will be
     found by the HandleToObject method of its server's ObjectTable.
     When Modula-3 gets WeakRefs, the runtime will automatically call
     ILU_Close when an object becomes globally inaccessible.
     ILU_Close_Surrogate is an obsolete version of ILU_Close that is
     applicable only to surrogate objects.

     ILU_Qua_Type facilitates embedding a multiple-inheritance structure
     in Modula-3.  If this object has, in the MI structure, the given
     type, ILU_Qua_Type returns an M3 object having an appropriate M3
     type; otherwise, it returns NIL.  For surrogate objects, the stubs
     take care of implementing this.  The implementor of a true object
     must override ILU_Qua_Type. *)

TYPE ObjectType = IluKernel.ObjectType;

PROCEDURE SbhFromObject (o: Object): SBH RAISES {Failed};
  (* May be applied to any Object; returns a reference that can be
     passed to other programs.  Export_Server must have been called on
     the object's server. *)

<*lL, Ll, Main unconstrained*>
PROCEDURE IdOfObjectType (ot: ObjectType): TEXT;
  (* Returns a shortish string that identifies this object type. *)


END Ilu.
