/*
 ** Copyright (c) 1991, 1992, 1993, 1994 Xerox Corporation.  All Rights Reserved.  
 **
 ** Unlimited use, reproduction, and distribution of this software is
 ** permitted.  Any copy of this software must include both the above
 ** copyright notice of Xerox Corporation and this paragraph.  Any
 ** distribution of this software must comply with all applicable United
 ** States export control laws.  This software is made available AS IS,
 ** and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 ** INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 ** AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 ** PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 ** THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 ** CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 ** XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */

#include "cstubber.h"

extern void generateClassCode ();
static void generateClassTable ();
extern void generateDispatchClassMethod ();
static void generateMethodsTable ();
void generateExceptionLinking ();
extern void generateProcHeader (Procedure m, Context c, boolean external);
extern int methodNdxOf (Procedure p, Type t);
extern void FreeValue (Type t, char *name, Context, int indent);

struct double_s {
  Context   c;
  Type   t;
  unsigned int  id;
  int   useId;
  int   valuesUsed;
};

static boolean  IncludeComma = FALSE;
extern cardinal MethodRecordID;
static int mCount;
extern boolean useParserIds;

static char   *ArrayName = NULL;
static unsigned  ExceptionIndex = 0;
static unsigned  MethodIndex = 0;

static void nMethods (Type t)
{
  Class c;

  if (t == NULL ||
      type_basic_type (t) != object_Type ||
      (c = class_object (t)) == NULL)
    return;
  if (c->superclasses != NULL)
    list_enumerate (c->superclasses, (void (*)(refany, refany)) nMethods, NULL);
  mCount += (list_size (c->methods));
}

int countMethods (Class o)
{
  if (o == NULL)
    return (0);
  mCount = 0;
  if (o->superclasses != NULL)
    list_enumerate (o->superclasses, (void (*)(refany, refany)) nMethods, NULL);
  return (list_size (o->methods) + mCount);
}

static void mkExtern (Type class, Context context)
{
  fprintf (context->file, "extern struct _ilu_Class_s _%s__ILUClassRecord;\n",
	   c_type_name (class));
}

static void superClsOf (Type class,
			Context context)
{
  fprintf (context->file,
	   "  &_%s__ILUClassRecord,\n",
	   c_type_name (class));
}

static void classIDs (Type class,
		      Context context)
{
  Type  st;
  
  for (st = class;  st->supertype != NULL;  st = st->supertype)
    ;
  fprintf (context->file, 
	   "  \"%s\",\n", 
	   st->uid);
}

static void generateClassTable (Type        class,
				Context     context)
{
  char        *tn = c_type_name (class);
  Class       o = class_object (class);
  
  if (o->superclasses && o->superclasses->count > 0) {
    list_enumerate (o->superclasses, 
		    (void (*)(refany, refany)) mkExtern, 
		    context);
    fprintf (context->file,
	     "static ilu_Class SuperclassesOf_%s[ %lu ] = {\n",
	     tn,
	     list_size (o->superclasses));
    list_enumerate (o->superclasses, 
		    (void (*)(refany, refany)) superClsOf, 
		    context);
    fprintf (context->file, "};\n");
    fprintf (context->file,
	     "static ilu_CString SuperclassIDsOf_%s[ %lu ] = {\n",
	     tn,
	     list_size (o->superclasses));
    list_enumerate (o->superclasses, 
		    (void (*)(refany, refany)) classIDs, 
		    context);
    fprintf (context->file, "};\n");
  }
  fprintf (context->file,
	   "struct _ilu_Class_s _%s__ILUClassRecord = {\n",
	   tn);
  fprintf (context->file, "  \"%s.%s\",\t\t/* ILU name */\n  \"%s\",\t\t/* Brand */\n",
	   name_base_name (context->interface->name), name_base_name (class->name),
	   o->brand == NULL ? "" : o->brand);
  fprintf (context->file,
	   "  \"%s\",\t\t/* unique ID */\n  %s,\t\t/* singleton? */\n  %s,\t\t/* collectible? */\n",
	   class->uid, o->singleton ? "ilu_TRUE" : "ilu_FALSE", o->collectible ? "ilu_TRUE" : "ilu_FALSE");
  fprintf (context->file, "  %s%s%s,\t\t/* authentication method */\n",
	   o->authentication == NULL ? "" : "\"",
	   o->authentication == NULL ? "NULL" : o->authentication,
	   o->authentication == NULL ? "" : "\"");
  if (countMethods (o) < 1)
    fprintf (context->file, "  NULL,\t\t/* no methods */\n  0,\t\t/* zero methods */\n");
  else {
    fprintf (context->file, "  MethodsOf%s,\t\t/* methods table */\n", tn);
    fprintf (context->file, "  sizeof (MethodsOf%s)/sizeof (struct _ilu_Method_s),\n", tn);
  }
  if (o->superclasses == NULL || list_size (o->superclasses) <= 0)
    fprintf (context->file,
	     "  0,\t\t/* # superclasses */\n  NULL,\t\t/* superclass IDs */\n  NULL,\t\t/* superclass pointers */\n  0};\n\n");
  else {
    fprintf (context->file, 
	     "  %lu,\t\t/* # of superclasses */\n", 
	     list_size (o->superclasses));
    fprintf (context->file, 
	     "  SuperclassIDsOf_%s,\t\t/* superclass IDs */\n",
	     tn);
    fprintf (context->file, 
	     "  SuperclassesOf_%s,\t\t/* superclass pointers */\n", 
	     tn);
    fprintf (context->file, "  0\n};\n\n");
  }
}

static void generateExceptionEntry (Exception  e,
				    Context  context)
{
  if (e->interface == context->interface && e->import == NULL)
    {
      fprintf (context->file, "ILU_C_ExceptionCode _%s__Exception_%s = \"%s: %s\";\n",
	       c_interface_name(e->interface), c_simple_name(e->name),
	       name_base_name (e->interface->name), name_base_name (e->name));
    }
}

static void countNonImported (Exception e, unsigned int *count)
{
  if (e->import == NULL)
    *count = *count + 1;
}

static void generateExceptionTable (Interface  interface,
				    Context  context)
{
  unsigned int count = 0;

  list_enumerate (interface->exceptions, (void (*)(refany, refany)) countNonImported, &count);
  if (count > 0) {
    list_enumerate (interface->exceptions, (void (*)(void *, void *)) generateExceptionEntry, context);
  }
}

static void catchException (Exception e, Context context)
{
  character buf[ 1000 ];
  enum PrimitiveTypes t = type_basic_type (e->type);

  if (t == void_Type) 
    return;
  fprintf (context->file, "    else if (_val->returnCode == ex_%s) {\n", 
	   c_exception_name (e));
  fprintf (context->file, "      _val->ptr = (void *) malloc (sizeof (%s));\n",
	   c_type_name (e->type));
  sprintf (buf, "%s((%s) _val->ptr)",
	   (t == object_Type) ? "*" : "",
	   (t == array_Type) ? c_return_type(e->type) : c_parameter_type (e->type, Out));
  UnmarshallValue (context, e->type, e->def, buf, 6, TRUE, FALSE);
  if (HasFreeRoutine(e->type))
    fprintf (context->file, "      _val->freeRoutine = (void (*)(void *)) %s__Free;\n",
	     c_type_name(e->type));
  else
    fprintf (context->file, "      _val->freeRoutine = NULL;\n");
  fprintf (context->file, "    }\n");
}

static void generateCatchException (Context  context)
{
  fprintf (context->file, "void _%s_CatchException (ilu_Call _call, ILU_C_ENVIRONMENT *_val, ",
	   c_interface_name (context->interface));
  fprintf (context->file, "  ilu_cardinal _ecode)\n{\n");
  fprintf (context->file, "  if ((_ecode > _call->ca_method->me_exceptionCount) || _ecode == 0) {\n");
  fprintf (context->file, "#ifdef CORBA11\n");
  fprintf (context->file, "    _val->_major = StExcep_SYSTEM_EXCEPTION;\n");
  fprintf (context->file, "    _val->returnCode = ex_StExcep_UNKNOWN;\n");
  fprintf (context->file, "#endif /* def CORBA11 */\n");
  fprintf (context->file, "#ifdef CORBA12\n");
  fprintf (context->file, "    _val->_major = CORBA_SYSTEM_EXCEPTION;\n");
  fprintf (context->file, "    _val->returnCode = ex_CORBA_UNKNOWN;\n");
  fprintf (context->file, "#endif /* def CORBA12 */\n");
  fprintf (context->file, "    _val->ptr = NULL;\n  }\n");
  fprintf (context->file, "  else {\n");
  fprintf (context->file, "#ifdef CORBA11\n    _val->_major = StExcep_USER_EXCEPTION;\n#endif /* def CORBA11 */\n");
  fprintf (context->file, "#ifdef CORBA12\n    _val->_major = CORBA_USER_EXCEPTION;\n#endif /* def CORBA12 */\n");
  fprintf (context->file, "    _val->returnCode = (ILU_C_ExceptionCode)_call->ca_method->me_exceptionVector[_ecode-1];\n");
  fprintf (context->file, "    if (_val->returnCode == NULL)\n      return;\n");
  list_enumerate (context->interface->exceptions, (void (*)(void *, void *)) catchException, context);
  fprintf (context->file, "  }\n  return;\n}\n\n");
}

static void sendExceptionValue (Exception  e,
				Context  context)
{
  char   buf[ 1000 ];
  enum PrimitiveTypes t = type_basic_type (e->type);

  if (e->type == NULL)
    return;
  fprintf (context->file, "  else if (stat->returnCode == ex_%s)\n", c_exception_name (e));
  sprintf (buf, "%s (%s) stat->ptr",
	   (t == array_Type OR
	    t == sequence_Type OR
	    t == record_Type OR
	    t == union_Type) ? "" : "*",
	   (t == array_Type) ? c_return_type(e->type) : (TypeIsString(e->type) ? c_type_name(e->type) : c_parameter_type (e->type, Out)));
  MarshallValue (context, e->type, buf, 4);
}

static void sizeException (Exception  e,
			   Context  context)
{
  char   buf[ 1000 ];
  enum PrimitiveTypes t = type_basic_type (e->type);

  if (e->type == NULL)
    return;
  fprintf (context->file, "  else if (stat->returnCode == ex_%s)\n", c_exception_name (e));
  fprintf (context->file, "    argSize = ");
  sprintf (buf, "%s((%s) stat->ptr)",
	   (t == array_Type OR
	    t == sequence_Type OR
	    t == record_Type OR
	    t == union_Type) ? "" : "*",
	   (t == array_Type) ? c_return_type(e->type) : (TypeIsString(e->type) ? c_type_name(e->type) : c_parameter_type (e->type, Out)));
  SizeValue (context, e->type, buf);
  fprintf (context->file, ";\n");
}

void generateSendException (Context  context)
{
  fprintf (context->file, "#include <stdarg.h>\n\n");
  fprintf (context->file, "void _%s_SendException (ilu_Call _call, ILU_C_ENVIRONMENT *stat)\n",
	   c_interface_name (context->interface));
  fprintf (context->file, "{\n  ilu_cardinal argSize = 0;\n");
  fprintf (context->file, "  ilu_cardinal eIndex = 0, i, limit;\n");
  fprintf (context->file, "  for (i = 1, limit = _call->ca_method->me_exceptionCount; i <= limit;  i += 1)\n");
  fprintf (context->file, "    if (_call->ca_method->me_exceptionVector[i-1] == stat->returnCode)\n");
  fprintf (context->file, "       eIndex = i;\n");
  fprintf (context->file, "  if (stat->returnCode == NULL)\n    /* do nothing */;\n");
  list_enumerate (context->interface->exceptions, (void (*)(void *, void *)) sizeException, context);
  fprintf (context->file, "  ilu_BeginException (_call, eIndex, argSize);\n");
  fprintf (context->file, "  if (stat->returnCode == NULL)\n    /* send nothing */;\n" );
  list_enumerate (context->interface->exceptions, (void (*)(void *, void *)) sendExceptionValue, context);
  fprintf (context->file, "  ilu_FinishException (_call);\n  return;\n}\n\n");
}

static void setExceptionValue (Exception  e,
			       Context  context)
{
  enum PrimitiveTypes t;

  if (e->type == NULL)
    return;
  t = type_basic_type (e->type);
  fprintf (context->file, "  else if (stat->returnCode == ex_%s) {\n", 
	   c_exception_name (e));
  fprintf (context->file, "    stat->ptr = (void *) malloc (sizeof (%s));\n", c_type_name(e->type));
  if (t == array_Type)
    {
      char *rtn = c_return_type(e->type);
      fprintf (context->file, "    memcpy ((%s)stat->ptr, va_arg(ap, %s), sizeof(%s));\n", rtn, rtn,
	       c_type_name(e->type));
    }
  else
    fprintf (context->file, "    *(%s)stat->ptr = %sva_arg (ap, %s);\n",
	     c_parameter_type (e->type, Out),
	     (t == record_Type OR t == union_Type OR (t == sequence_Type AND NOT (TypeIsString(e->type))))
	     ? "*" : "", c_parameter_type (e->type, In));
  if (HasFreeRoutine(e->type))
    fprintf (context->file, "    stat->freeRoutine = (void (*) (void *)) %s__Free;\n", c_type_name(e->type));
  else
    fprintf (context->file, "    stat->freeRoutine = NULL;\n");
  fprintf (context->file, "  }\n");
}     

static void generateSignalException (Context  context)
{
  fprintf (context->file, "#include <stdarg.h>\n\n");
  fprintf (context->file, "void %s_RaiseException (ILU_C_ENVIRONMENT *stat, ilu_Exception exception, ...)\n",
	   c_interface_name (context->interface));
  fprintf (context->file, "{\n  va_list ap;\n  va_start (ap, exception);\n");
  fprintf (context->file, "#ifdef CORBA11\n  stat->_major = StExcep_USER_EXCEPTION;\n#endif /* def CORBA11 */\n");
  fprintf (context->file, "#ifdef CORBA12\n  stat->_major = CORBA_USER_EXCEPTION;\n#endif /* def CORBA12 */\n");
  fprintf (context->file, "  stat->returnCode = exception;\n");
  fprintf (context->file, "  if (exception == NULL)\n    /* no exception */;\n");
  list_enumerate (context->interface->exceptions, (void (*)(void *, void *)) setExceptionValue, context);
  fprintf (context->file, "}\n\n");
}

static void generateExceptionProcs (Context  context)
{
  if (list_size(context->interface->exceptions) > 0)
    {
      generateCatchException (context);
      generateSignalException (context);
      generateSendException (context);
    }
}

static int DimStop;
static int DimCount;

static void dimHeader (long d, Context context)
{
  if (DimStop >= 0 AND DimCount >= DimStop)
    return;

  fprintf (context->file, "%*.*s{\n%*.*sregister int _i%u;\n%*.*sfor (_i%u = 0;  _i%u < %lu;  _i%u += 1)\n",
	   DimCount*2+2, DimCount*2+2, "", DimCount*2+4, DimCount*2+4, "", DimCount,
	   DimCount*2+4, DimCount*2+4, "", DimCount, DimCount, d, DimCount);
  DimCount += 1;
}

static void dimRef (long d, char * buf)
{
  if (DimStop >= 0 AND DimCount >= DimStop)
    return;

  sprintf (buf + strlen (buf), "[_i%u]", DimCount);
  DimCount += 1;
}

static void dimFooter (long d, Context context)
{
  DimCount -= 1;
  if (DimStop >= 0 AND DimCount < DimStop)
    return;

  fprintf (context->file, "%*.*s}\n", (DimCount - DimStop)*2, (DimCount - DimStop)*2, "");
}

static void outputRecordArg (Argument arg, Context context)
{
  char buf[1000];
  enum PrimitiveTypes t;

  t = type_basic_type (arg->type);
  sprintf (buf, "(%s_val->%s)",
	   (t == record_Type OR
	    (t == sequence_Type AND !TypeIsString(arg->type)) OR
	    t == union_Type) ? "&" : "", c_argument_name(arg));
  MarshallValue (context, arg->type, buf, 2);
}

static void caseConst (ConstantValue val, struct double_s *s)
{
  switch (val->type)
    {
    case integer_Type:
    case shortinteger_Type:
    case cardinal_Type:
    case shortcardinal_Type:
    case byte_Type:
      fprintf (s->c->file, "    case %s%ld:\n    ", (val->val.i.sign < 0) ? "-" : "", val->val.i.value);
      break;
    case shortcharacter_Type:
      fprintf (s->c->file, "    case %s_%s:\n    ", c_interface_name (s->c->interface), val->val.s);
      break;
    default:
      error ("illegal discriminator type\n");
    }
}

static void outputUnionType (Argument a, struct double_s * s)
{
  char buffer[ 1000 ];
  int mkReferent;
  Type type = a->type;
  enum PrimitiveTypes t = type_basic_type (type);
  char *name;
  
  if (a->name)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));
  mkReferent = !TypeIsString (type) && (t == record_Type OR t == union_Type OR t == sequence_Type);
  sprintf (buffer, "%s_val->_u.%s", (mkReferent) ? "&" : "", name);
  if (s->useId)
    {
      if (a->values)
	{
	  list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
	  s->valuesUsed = True;
	}
      else if (s->valuesUsed)
	fprintf (s->c->file, "    default:\n");
      else
	fprintf (s->c->file, "    case %d:\n", s->id++);
    }
  else
    fprintf (s->c->file, "    case %s_%s:\n", c_type_name (s->t), c_string (c_simple_name (type->name)));
  MarshallValue (s->c, type, buffer, 6);
  fprintf (s->c->file, "      break;\n");
}

static void ComputeTotalNumberOfElements (unsigned long dim, unsigned long *total)
{
  *total = *total * dim;
}

static void FindZero (long unsigned int val, boolean *z)
{
  if (val == 0)
    *z = TRUE;
}

static void generateOutputCode (Type type, enum PrimitiveTypes t, Context context)
{
  TypeDescription d = type_description (type);
  char buf[ 1000 ];

  fprintf (context->file, "void _%s_Output_%s (ilu_Call _call, %s _val)\n{\n",
	   c_interface_name (context->interface), c_simple_name (type->name), c_parameter_type (type, In));
  fprintf (context->file, "  if (_val == NULL)\n    return;\n\n");

  if (t == array_Type)
    {
      unsigned long size = 1;
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);
      
      list_enumerate (d->structuredDes.array.dimensions, (void (*)(refany, refany)) ComputeTotalNumberOfElements, &size);
      fprintf (context->file, "  ilu_OutputArray (_call, %lu, ilu_FALSE, ilu_TRUE);\n", size);

      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*silu_Output%s (_call, %s, %lu, ilu_FALSE);\n", dims * 2 + 2, dims*2+2, "",
		   (t2 == byte_Type) ? "Opaque" : "StringVec", buf,
		   (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims - 1;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*silu_OutputWStringVec (_call, _val, %lu, ilu_FALSE, NULL, NULL);\n",
		   dims * 2 + 2, dims*2+2, "", (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims - 1;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  DimStop = -1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  MarshallValue (context, d->structuredDes.array.type, buf, 2*dims+4);
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	}

      fprintf (context->file, "  ilu_EndArray (_call);\n");
    } 

  else if (t == sequence_Type)
    {
      Type et = d->structuredDes.sequence.type;
      enum PrimitiveTypes st = type_basic_type (d->structuredDes.sequence.type);
      
      fprintf (context->file, "  ilu_OutputSequence (_call, _val->_length, %lu, ilu_FALSE, ilu_TRUE);\n",
	       d->structuredDes.sequence.limit);
      fprintf (context->file, "  {\n    %s *p;  int i;\n\n", c_type_name (et));
      fprintf (context->file, "    for (p = _val->_buffer, i = 0;  i < _val->_length;  p++, i++)\n");
      sprintf (buf, "%sp",
	       (st == record_Type OR (st == sequence_Type AND NOT TypeIsString(et)) OR st == union_Type) ? "" : "*");
      MarshallValue (context, d->structuredDes.sequence.type, buf, 6);
      fprintf (context->file, "  }\n  ilu_EndSequence (_call);\n");
    }

  else if (t == union_Type)
    {
      struct double_s  s;

      s.c = context;
      s.t = type;
      s.id = 0;
      s.valuesUsed = False;
      if (d->structuredDes.uniond.discriminator_type  &&
	  type_basic_type (d->structuredDes.uniond.discriminator_type) != 
	  shortcardinal_Type)
	s.useId = True;
      else
	s.useId = False;

      fprintf (context->file, "  ilu_OutputUnion (_call, _val->_d, ilu_FALSE, ilu_TRUE);\n");
      fprintf (context->file, "  switch (_val->_d) {\n");
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) outputUnionType, &s);
      fprintf (context->file, "  };\n  ilu_EndUnion (_call);\n");
    }

  else if (t == record_Type)
    {
      fprintf (context->file, "  ilu_OutputRecord (_call, ilu_FALSE, ilu_TRUE);\n");
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) outputRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call);\n");
    }

  fprintf (context->file, "}\n\n");
}

static void inputRecordArg (Argument  arg,
			    Context  context)
{
  char  buf[ 1000 ];

  sprintf (buf, "%s_val->%s", (type_basic_type(arg->type) == array_Type) ? "" : "&", c_argument_name (arg));
  UnmarshallValue (context, arg->type, 0, buf, 2, TRUE, FALSE);
}

static void inputUnionType (Argument   a,
			    struct double_s  *s)
{
  char   buffer[ 1000 ];
  ConstantValue val;
  char  *name;
  Type  type = a->type;

  if (a->name)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));

  /*
   ** what this glop is all about is to generate the appropriate
   ** case values for the type of the discriminator.
   */

  if (s->useId)
    {
      if (a->values)
	{
	  list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
	  s->valuesUsed = True;
	}
      else if (s->valuesUsed)
	{
	  fprintf (s->c->file, "    default:\n");
	}
      else
	{
	  fprintf (s->c->file, "    case %d:\n", s->id);
	}
    }
  else
    {
      fprintf (s->c->file, "    case %s_%s:\n", c_type_name (s->t), c_string (c_simple_name (type->name)));
    }
  sprintf (buffer, "%s_val->_u.%s",
	   (type_basic_type(a->type) == array_Type OR type_basic_type(a->type) == object_Type) ? "" : "&", name);
  UnmarshallValue (s->c, a->type, 0, buffer, 6, TRUE, FALSE);

  /*
   ** now do it again to assign the discriminator
   */

  if (s->useId)
    {
      if (a->values)
	{
	  /*
	   ** there is a many to one relationship here -
	   ** many disciminator values to one type. find
	   ** the first value and assign it
	   */

	  val = (ConstantValue) a->values->head->data;
	  switch (val->type)
	    {
	    case integer_Type:
	    case shortinteger_Type:
	    case cardinal_Type:
	    case shortcardinal_Type:
	    case byte_Type:
	      fprintf (s->c->file, "      _val->_d = %s%ld;\n", (val->val.i.sign < 0) ? "-" : "", val->val.i.value);
	      break;
	    case shortcharacter_Type:
	      fprintf (s->c->file, "      _val->_d = %s_%s;\n", c_interface_name (s->c->interface), val->val.s);
	      break;
	    default:
	      error ("Cannot determine discriminator value for union\n");
	    }
	}
      else if (s->valuesUsed)
	{
	  /*
	   ** not sure what value should be assigned
	   ** - one thats not in the list already i guess.
	   ** rather than scan the list again, just use
	   ** this funky value.
	   */
	  fprintf (s->c->file, "      _val->_d = 0xdeadcafe;\n");
	}
      else
	{
	  fprintf (s->c->file, "      _val->_d = %d;\n", s->id);
	}
    }
  else
    {
      fprintf (s->c->file, "      _val->_d= %s_%s;\n", c_type_name (s->t), c_string (c_simple_name (type->name)));
    }
  fprintf (s->c->file, "      break;\n");
  s->id++;
}

static void generateInputCode (Type type, enum PrimitiveTypes t, Context context)
{
  char *ret = (char *) c_return_type (type);
  char *parm = c_parameter_type (type, InOut);
  TypeDescription d = type_description (type);
  char *name = c_type_name (type);

  context->class = type;

  fprintf (context->file, "%s _%s_Input_%s (ilu_Call _call, %s _ref)\n{\n  %s%s _val;\n\n",
	   (t == array_Type OR t == object_Type) ? ret : parm,
	   c_interface_name (context->interface), c_simple_name (type->name),
	   (t == object_Type) ? ret : parm,
	   (t == array_Type) ? name : ((t == object_Type) ? ret : parm), (t == array_Type) ? " *" : "");
  if (t == array_Type)
    {
      char buf[ 1000 ];
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      dims = list_size(d->structuredDes.array.dimensions);
      zeroarray = FALSE;
      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);

      fprintf (context->file, "  ilu_InputArray (_call, ilu_FALSE, NULL);\n");
      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  buf[0] = '\0';
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file,
		   "  if (_ref != NULL)\n    _val = (%s *) _ref;\n  else\n    _val = (%s *) malloc(sizeof(%s));\n",
		   c_type_name(type), c_type_name(type), c_type_name(type));
	  fprintf (context->file, "%*.*s{\n%*.*s%s _tmp = &(*_val)%s[0];\n",
		   dims*2, dims*2, "", dims*2+2, dims*2+2, "", (t2 == byte_Type) ? "unsigned char *" : "char *", buf);
	  fprintf (context->file, "%*.*silu_Input%s (_call, &_tmp, %lu, ilu_FALSE);\n", dims*2+2, dims*2+2, "",
		   (t2 == byte_Type) ? "Opaque" : "StringVec",
		   (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  fprintf (context->file, "%*.*s}\n", dims*2, dims*2, "");
	  DimCount = dims - 1;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  buf[0] = '\0';
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file,
		   "  if (_ref != NULL)\n    _val = (%s *) _ref;\n  else\n    _val = (%s *) malloc(sizeof(%s));\n",
		   c_type_name(type), c_type_name(type), c_type_name(type));
	  fprintf (context->file, "%*.*s{\n%*.*silu_character *_tmp = &(*_val)%s[0];\n",
		   dims*2, dims*2, "", dims*2+2, dims*2+2, "", buf);
	  fprintf (context->file, "%*.*silu_InputWStringVec (_call, &_tmp, %lu, ilu_FALSE);\n", dims*2+2, dims*2+2, "",
		   (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  fprintf (context->file, "%*.*s}\n", dims*2, dims*2, "");
	  DimCount = dims - 1;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  fprintf (context->file,
		   "  if (_ref != NULL)\n    _val = (%s *) _ref;\n  else\n    _val = (%s *) malloc(sizeof(%s));\n",
		   c_type_name(type), c_type_name(type), c_type_name(type));
	  DimCount = 0;
	  DimStop = -1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  sprintf (buf, "&(*_val)");
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  UnmarshallValue (context, d->structuredDes.array.type, type->def, buf, dims*2+2, TRUE, FALSE);
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  fprintf (context->file, "  ilu_EndArray (_call);\n");
	}
    }
  else if (t == sequence_Type)
    {
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.sequence.type);

      if (t2 == byte_Type)
	{
	  fprintf (context->file, "  ilu_cardinal len;\n  ilu_bytes b = NULL;\n\n");
	  fprintf (context->file, "  if (_ref == NULL)\n    _val = (%s *) malloc(sizeof(%s));\n", name, name);
	  fprintf (context->file, "  else\n    _val = _ref;\n");
	  fprintf (context->file, "  ilu_InputBytes (_call, &b, &len, %lu, ilu_FALSE);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "  _val->_maximum = len;\n");
	  fprintf (context->file, "  _val->_length = len;\n");
	  fprintf (context->file, "  _val->_buffer = b;\n");
	}
      else
	{
	  fprintf (context->file, "  ilu_cardinal _count, _index;\n");
	  fprintf (context->file, "  %s _tmp;\n\n", c_type_name (d->structuredDes.sequence.type));
	  fprintf (context->file, "  ilu_InputSequence (_call, &_count, %lu, ilu_FALSE, NULL);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "  if (_ref != NULL) {\n    _val = _ref;\n    %s_Init(_val, 0, NULL);\n    }\n", name);
	  fprintf (context->file, "  else {\n    _val = (%s) calloc (1, sizeof (%s));\n", parm, name);
	  fprintf (context->file, "    _val->_maximum = 0;\n");
	  fprintf (context->file, "    _val->_length = 0;\n");
	  fprintf (context->file, "    _val->_buffer = NULL;\n  }\n");
	  fprintf (context->file, "  for (_index = 0;  _index < _count;  _index++)\n  {\n");
	  UnmarshallValue (context, d->structuredDes.sequence.type, type->def,
			   (t2 == array_Type) ? "_tmp" : "&_tmp", 4, TRUE, FALSE);
	  fprintf (context->file, "    %s_Append (_val, %s_tmp);\n", c_type_name(type),
		   (t2 == record_Type OR
		    (t2 == sequence_Type AND NOT TypeIsString(d->structuredDes.sequence.type)) OR
		    t2 == union_Type) ? "&" : "");
	  fprintf (context->file, "  }\n");
	  fprintf (context->file, "  ilu_EndSequence (_call);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s s;

      fprintf (context->file, "  ilu_shortcardinal discriminator;\n\n");
      fprintf (context->file, "  ilu_InputUnion (_call, &discriminator, ilu_FALSE, NULL);\n");
      fprintf (context->file, "  if (_ref != NULL)\n    _val = _ref; \n");
      fprintf (context->file, "  else\n    _val = (%s) calloc (1, sizeof (%s));\n",
	       parm, name);
      fprintf (context->file, "  switch (discriminator) {\n");
      s.c = context;
      s.t = type;
      s.id = 0;
      s.valuesUsed = False;
      if (d->structuredDes.uniond.discriminator_type &&
	  type_basic_type (d->structuredDes.uniond.discriminator_type) != 
	  shortcardinal_Type)
	s.useId = True;
      else
	s.useId = False;
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) inputUnionType, &s);
      fprintf (context->file, "  }\n");
      fprintf (context->file, "  ilu_EndUnion (_call);\n");
    }
  else if (t == record_Type)
    {
      fprintf (context->file, "  ilu_InputRecord (_call, ilu_FALSE, NULL);\n");
      fprintf (context->file, "  if (_ref != NULL)\n");
      fprintf (context->file, "    _val = _ref;\n  else\n    _val = (%s) calloc (1, sizeof (%s));\n", parm, name);
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) inputRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call);\n");
    }
  else if (t == object_Type)
    {} /* done elsewhere */
  else
    {}

  if (t == array_Type)
    fprintf (context->file, "  return ((%s) _val);\n}\n\n", c_return_type(type));
  else
    fprintf (context->file, "  return (_val);\n}\n\n");
}

static void sizeUnionType (Argument a, struct double_s *s)
{
  char   buffer[1000];
  int  mkReferent;
  char  *name;
  enum PrimitiveTypes t = type_basic_type (a->type);

  if (a->name)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));

  if (s->useId)
    {
      if (a->values) {
	list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
	s->valuesUsed = True;
      }
      else if (s->valuesUsed)
	fprintf (s->c->file, "    default:\n");
      else
	fprintf (s->c->file, "    case %d:\n", s->id++);
    }
  else
    fprintf (s->c->file, "    case %s_%s:\n", c_type_name (s->t), c_string (c_simple_name (a->type->name)));

  mkReferent = (!TypeIsString (a->type) AND
		(t == record_Type OR t == union_Type OR t == sequence_Type));
  sprintf (buffer, "%s_val->_u.%s", (mkReferent) ? "&" : "", name);
  fprintf (s->c->file, "      size += ");
  SizeValue (s->c, a->type, buffer);
  fprintf (s->c->file, ";\n      break;\n");
}

static void sizeRecordArg (Argument arg, Context context)
{
  char buf[ 1000 ];
  enum PrimitiveTypes t;

  t = type_basic_type (arg->type);
  sprintf (buf, "(%s_val->%s)", 
	   (TypeIsNonObjectStruct (arg->type) OR t == sequence_Type) ? "&" : "", 
	   c_argument_name (arg));
  fprintf (context->file, "  size += ");
  SizeValue (context, arg->type, buf);
  fprintf (context->file, ";\n");
}

static void generateSizeofCode (Type type, enum PrimitiveTypes t, Context context)
{
  char buf[1000];
  TypeDescription d = type_description (type);

  fprintf (context->file, "ilu_cardinal _%s_SizeOf_%s (ilu_Call _call, %s _val)\n{\n",
	   c_interface_name (context->interface), c_simple_name (type->name), 
	   c_parameter_type (type, In));
  fprintf (context->file, "  ilu_cardinal size = 0;\n\n");
  if (t == array_Type)
    {
      unsigned long size = 1;
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);
      list_enumerate (d->structuredDes.array.dimensions, (void (*)(refany, refany)) ComputeTotalNumberOfElements, &size);

      fprintf (context->file, "  size = ilu_SizeOfArray (_call, %lu, ilu_FALSE, ilu_TRUE);\n", size);

      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ilu_SizeOf%s (_call, %s, %lu, ilu_FALSE);\n", dims*2+2, dims*2+2, "",
		   (t2 == byte_Type) ? "Opaque" : "StringVec", buf, (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ilu_SizeOfStringVec (_call, _val, %lu, ilu_FALSE, NULL, NULL);\n",
		   dims * 2 + 2, dims*2+2, "", (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  DimStop = -1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ", dims*2+4, dims*2+4, "");
	  SizeValue (context, d->structuredDes.array.type, buf);
	  fprintf (context->file, ";\n");
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	}

      fprintf (context->file, "  ilu_EndArray (_call);\n");
    } 

  else if (t == sequence_Type)
    {
      Type et = d->structuredDes.sequence.type;
      enum PrimitiveTypes st = type_basic_type (d->structuredDes.sequence.type);
      
      fprintf (context->file, "  size = ilu_SizeOfSequence (_call, _val->_length, %lu, ilu_FALSE, ilu_TRUE);\n",
	       d->structuredDes.sequence.limit);
      fprintf (context->file, "  {\n    %s *p;  int i;\n\n", c_type_name (et));
      fprintf (context->file, "    for (p = _val->_buffer, i = 0;  i < _val->_length;  p++, i++)\n      size += ");
      sprintf (buf, "%sp",
	       (st == record_Type OR (st == sequence_Type AND NOT TypeIsString(et)) OR st == union_Type) ? "" : "*");
      SizeValue (context, d->structuredDes.sequence.type, buf);
      fprintf (context->file, ";\n  }\n  ilu_EndSequence (_call);\n");
    }

  else if (t == union_Type)
    {
      struct double_s  s;

      s.c = context;
      s.t = type;
      s.id = 0;
      s.valuesUsed = False;
      if (d->structuredDes.uniond.discriminator_type  &&
	  type_basic_type (d->structuredDes.uniond.discriminator_type) != 
	  shortcardinal_Type)
	s.useId = True;
      else
	s.useId = False;

      fprintf (context->file, "  size = ilu_SizeOfUnion (_call, _val->_d, ilu_FALSE, ilu_TRUE);\n");
      fprintf (context->file, "  switch (_val->_d) {\n");
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) sizeUnionType, &s);
      fprintf (context->file, "  };\n  ilu_EndUnion (_call);\n");
    }

  else if (t == record_Type)
    {
      fprintf (context->file, "  size = ilu_SizeOfRecord (_call, ilu_FALSE, ilu_TRUE);\n");
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) sizeRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call);\n");
    }
  else if (t == object_Type)
    ; /* done elsewhere */
  else
    {};

  fprintf (context->file, "  return size;\n}\n\n");
}

static void generateSequenceCreateCode (Type  type,
					Context  context)
{
  string  name = c_type_name (type);

  fprintf (context->file, 
	   "%s %s_%s_Create (unsigned long count, unsigned char *bytes)\n",
	   c_return_type (type),
	   c_interface_name (type->interface), 
	   c_simple_name (type->name));
  fprintf (context->file, "{\n  %s n;\n", name);
  fprintf (context->file, 
	   "  n._length = count;\n");
  fprintf (context->file, 
	   "  n._maximum = count;\n");
  fprintf (context->file, 
	   "  n._buffer = bytes;\n  return (n);\n}\n\n");
}

static void FreeRecordField (Argument field, Context context)
{
  char buf[1000];

  sprintf (buf, "_val->%s", c_argument_name(field));
  FreeValue (field->type, buf, context, 2);
}

static void ListUnionValue (Constant c, Context context)
{
  switch (c->value->type)
    {
    case boolean_Type:
      fprintf (context->file, "    case %s:\n", c->value->val.b ? "ilu_TRUE" : "ilu_FALSE");
      break;

    case integer_Type:
    case shortinteger_Type:
      fprintf (context->file, "    case %ld:\n", c->value->val.i.value * c->value->val.i.sign);
      break;

    case cardinal_Type:
    case shortcardinal_Type:
    case byte_Type:
      fprintf (context->file, "    case %lu:\n", c->value->val.i.value);
      break;

    case shortcharacter_Type:  /* must be an enumeration value */
      fprintf (context->file, "    case %s_%s:\n", c_interface_name(context->interface),
	       c_string(c->value->val.s));
      break;

    default:
      fprintf (stderr, "Bad constant specified in union.\n");
      exit(1);
    }
}

static void FreeUnionField (Argument field, Context context)
{
  char buf[1000];

  list_enumerate (field->values, (void (*) (void *, void *)) ListUnionValue, context);
  sprintf (buf, "_val->%s", c_argument_name(field));
  FreeValue (field->type, buf, context, 6);
  fprintf (context->file, "      break;\n\n");
}

static void generateFreeCode (Type type, enum PrimitiveTypes t, Context context)
{
  /* generate code to free any allocated storage occupied by this object */

  fprintf (context->file, "void %s__Free (%s _val)\n{\n", c_type_name(type), c_parameter_type (type, In));
  fprintf (context->file, "  /* frees allocated storage inside _val (if any), but does not free _val */\n\n");

  /* null for now */

  if (t == record_Type)

    list_enumerate (type_description(type)->structuredDes.record, (void (*)(void *, void *)) FreeRecordField, context);

  else if (t == union_Type)
    {
      fprintf (context->file, "  switch (_val->_d) {\n");
      list_enumerate (type_description(type)->structuredDes.record, (void (*)(void *, void *)) FreeUnionField, context);
      fprintf (context->file, "    default:\n      break;\n  }\n");
    }

  else if (t == sequence_Type)
    {
      Type st = type_description(type)->structuredDes.sequence.type;
      enum PrimitiveTypes t2 = type_basic_type(type_description(type)->structuredDes.array.type);
      if (t2 == shortcharacter_Type)
	{
	  fprintf (context->file, "  free(_val);\n");
	}
      else
	{
	  fprintf (context->file, "  if (_val->_buffer != NULL) {\n");
	  if (HasFreeRoutine(st))
	    {
	      fprintf (context->file, "    int i;\n\n    for (i = 0;  i < _val->_length;  i++)\n");
	      fprintf (context->file, "      %s__Free (%s_val->_buffer[i]);\n", c_type_name(st),
		       (t2 == array_Type || t2 == optional_Type || TypeIsString(st)) ? "" : "&");
	    }
	  fprintf (context->file, "    free(_val->_buffer);\n  };\n");
	}
    }

  else if (t == array_Type)
    {
      TypeDescription d = type_description(type);

      if (HasFreeRoutine(d->structuredDes.array.type))
	{
	  char buf[1000];
	  int dims = list_size(d->structuredDes.array.dimensions);

	  DimStop = -1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  FreeValue (d->structuredDes.array.type, buf, context, 2 * dims + 4);
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	}
      else
	{
	  fprintf (context->file, "  /* simple type, nothing to free */\n");
	}
    }

  fprintf (context->file, "}\n\n");
}

static void generateTypeIoCode (Type type, Context context)
{
  enum PrimitiveTypes t = type_basic_type (type);

  if (type->importInterfaceName == NULL AND
      (t == union_Type OR
       t == record_Type OR
       (t == sequence_Type AND
	NOT (type_basic_type(type_description(type)->structuredDes.sequence.type) == shortcharacter_Type OR
	     (type_basic_type(type_description(type)->structuredDes.sequence.type) == byte_Type) OR
	     (type_basic_type(type_description(type)->structuredDes.sequence.type) == character_Type))) OR 
       (t == array_Type AND
	NOT (list_size(type_description(type)->structuredDes.array.dimensions) == 1 AND
	     (type_basic_type(type_description(type)->structuredDes.array.type) == byte_Type OR
	      type_basic_type(type_description(type)->structuredDes.array.type) == shortcharacter_Type OR
	      type_basic_type(type_description(type)->structuredDes.array.type) == character_Type)))
       ))
    generateOutputCode (type, t, context);
  
  if (type->importInterfaceName == NULL AND
      (t == union_Type OR
       t == record_Type OR
       (t == sequence_Type AND
	NOT (type_basic_type(type_description(type)->structuredDes.sequence.type) == shortcharacter_Type OR
	     (type_basic_type(type_description(type)->structuredDes.sequence.type) == byte_Type) OR
	     (type_basic_type(type_description(type)->structuredDes.sequence.type) == character_Type))) OR 
       (t == array_Type AND
	NOT (list_size(type_description(type)->structuredDes.array.dimensions) == 1 AND
	     (type_basic_type(type_description(type)->structuredDes.array.type) == byte_Type OR
	      type_basic_type(type_description(type)->structuredDes.array.type) == shortcharacter_Type OR
	      type_basic_type(type_description(type)->structuredDes.array.type) == character_Type)))
       ))
    generateSizeofCode (type, t, context);
  
  if (type->importInterfaceName == NULL AND
      (t == union_Type OR
       t == object_Type OR
       t == record_Type OR
       (t == sequence_Type AND (!TypeIsString(type))) OR
       (t == array_Type)
       ))
    generateInputCode (type, t, context);
  
  if (HasFreeRoutine(type))
    generateFreeCode (type, t, context);
  
  if (t == sequence_Type &&
      type_basic_type (type_description (type)->structuredDes.sequence.type) == byte_Type)
    {
      generateSequenceCreateCode (type, context);
    }
}

static void generateIosCode (Interface  interface,
			     Context  context)
{
  list_enumerate (interface->types, 
		  (void (*)(void *,void *)) generateTypeIoCode, 
		  context);
}

static void generateGlobalCode (Interface  interface,
				Context  context)
{
  generateExceptionTable (interface, context);
  generateExceptionProcs (context);
  generateIosCode (interface, context);
}

static void RegisterClass (Type  class,
			   Context  context)
{
  if (class->builtIn && class->importInterfaceName == NULL) 
    return;
  fprintf (context->file, 
	   "  SetupExceptionVectors_%s ();\n", 
	   c_type_name (class));
  fprintf (context->file, 
	   "  ilu_RegisterClass (&_%s__ILUClassRecord);\n",
	   c_type_name (class));
}

static void generateClientRegistrationCode (Interface  interface,
					    Context  context)
{
  char *interface_name = (char *) c_interface_name (interface);

  fprintf (context->file, "void _%s__GeneralInitialization (void)\n{\n", interface_name);
  fprintf (context->file, "  static int initialized = 0;\n");
  fprintf (context->file, "  if (initialized)\n    return;\n");
  fprintf (context->file, "  initialized = 1;\n");
  list_enumerate (interface->classes, (void (*)(void *,void *)) RegisterClass, context);
  fprintf (context->file, "}\n\n");
}

static void generateSequenceCode (Type seq, Context context)
{
  boolean hasReferent;
  char *rtn;
  Type sequenceType;
  char *st;
  enum PrimitiveTypes t;
  char *tn;

  if (type_basic_type (seq) != sequence_Type)
    return;
  sequenceType = type_description (seq)->structuredDes.sequence.type;
  t = type_basic_type (sequenceType);
  if (t == shortcharacter_Type OR t == character_Type OR t == byte_Type)
    {
      /* no declaration, already handled in typedef */
    }
  else
    {
      tn = c_type_name (seq);
      st = c_parameter_type (sequenceType, In);
      rtn = c_type_name (sequenceType);
      hasReferent = (t == record_Type OR (t == sequence_Type AND NOT TypeIsString(sequenceType)) OR t == union_Type);
      fprintf (context->file, "void %s_Every (%s *h, void (*f)(%s%s, void *), void * data)\n{\n", tn, tn, st,
	       hasReferent ? "" : " *");
      fprintf (context->file, "  _ILU_C_EveryElement ((ILU_C_Sequence) h, (void (*)(void *, void *)) f, sizeof(%s), (void *) data);\n}\n\n", rtn);
      fprintf (context->file, "void %s_Append (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_AppendGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",
	       (hasReferent) ? "" : "&", rtn);
      fprintf (context->file, "void %s_Push (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_PushGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",
	       (hasReferent) ? "" : "&", rtn);
      fprintf (context->file, "void %s_Pop (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_PopGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",
	       hasReferent ? "" : "&", rtn);
      fprintf (context->file, "%s *%s_%s_Create (unsigned long sz, %s %sp)\n", tn, c_interface_name (context->interface), 
	       c_simple_name (seq->name), st, hasReferent ? "" : "*");
      fprintf (context->file, "{\n  %s *s;\n", tn);
      fprintf (context->file, "  if (sz == 0 && p != NULL)\n    return (NULL);\n");
      fprintf (context->file, "  s = (%s *) malloc (sizeof (%s));\n", tn, tn);
      fprintf (context->file, "  if (sz > 0)\n    s->_maximum = sz;\n");
      fprintf (context->file, "  else\n    s->_maximum = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p != NULL)\n    s->_length = sz;\n");
      fprintf (context->file, "  else\n    s->_length = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p == NULL)\n");
      fprintf (context->file, "    s->_buffer = (%s *) malloc (sz * sizeof (%s));\n", rtn, rtn);
      fprintf (context->file, "  else\n    s->_buffer = p;\n");
      fprintf (context->file, "  return (s);\n}\n\n");
      fprintf (context->file, "void %s_%s_Init (%s *s, unsigned long sz, %s %sp)\n",
	       c_interface_name (context->interface), c_simple_name (seq->name), tn, st, (hasReferent ? "" : "*"));
      fprintf (context->file, "{\n");
      fprintf (context->file, "  if (sz == 0 && p != NULL)\n    return;\n");
      fprintf (context->file, "  if (sz > 0)\n    s->_maximum = sz;\n");
      fprintf (context->file, "  else\n    s->_maximum = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p != NULL)\n    s->_length = sz;\n");
      fprintf (context->file, "  else\n    s->_length = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p == NULL)\n");
      fprintf (context->file, "    s->_buffer = (%s *) malloc (sz * sizeof (%s));\n", rtn, rtn);
      fprintf (context->file, "  else\n    s->_buffer = p;\n");
      fprintf (context->file, "  return;\n}\n\n");
    }
}

static void fillException(
			  Exception  e,
			  Context  context )
{
  fprintf (context->file, "  %s.me_exceptionVector[%u] = ex_%s;\n", ArrayName, ExceptionIndex, c_exception_name(e));
  ExceptionIndex += 1;
}

static void setupMethodExceptions(
				  Procedure  m,
				  Context  context )
{
  if ( m->exceptions != NULL && list_size( m->exceptions ) > 0 )
    {
      char buf[ 1000 ];

      sprintf (buf, "MethodsOf%s[%d]", c_type_name( context->class ), MethodIndex);
      fprintf (context->file, "  %s.me_exceptionVector = (ilu_Exception *) malloc(sizeof(ilu_Exception) * %lu);\n",
	       buf, list_size( m->exceptions ));
      fprintf (context->file, "  %s.me_exceptionCount = %lu;\n", buf, list_size(m->exceptions));
      ExceptionIndex = 0;
      ArrayName = buf;
      list_enumerate (m->exceptions, (void ( * )( )) fillException, context);

      fprintf (context->file, "\n" );
    }
  MethodIndex += 1;
}

static void generateMethodExceptions(
				     Type  t,
				     Context  context )
{
  Class  od;

  if ( t == NULL || type_basic_type(t) != object_Type || 
      (od = class_object(t)) == NULL)
    return;
  list_enumerate (od->methods, (void (*)(void *, void *)) setupMethodExceptions, context);
}

void generateExceptionLinking(
			      Type        type,
			      Context     context )
{
  char *tn = c_type_name( type );
  
  fprintf (context->file, "static void SetupExceptionVectors_%s(void)\n{\n", tn );
  MethodIndex = 0;
  context->class = type;
  generateMethodExceptions( type, context );
  fprintf (context->file, "}\n\n" );
}

static void generateDispatchMethods (Type        class,
				     Context     context)
{
  Class  od = class_object (class);
  static int recursing = 0;
  
  /*
   ** here's some wild stuff. whats happening
   ** is that we always want to retain the
   ** highest class level as we recurse through
   ** the class tree thats why we need to
   ** retain this nasty recursing flag so we
   ** can know when to reset the context->class
   */

  if (od->superclasses) {
    if (!recursing)
      context->class = class;
    recursing++;
    list_enumerate (od->superclasses, (void (*)(refany, refany)) generateDispatchMethods, context);
    recursing--;
  }
  if (!recursing)
    context->class = class;
  if (countMethods (od) > 0) {
    list_enumerate (od->methods, 
		    (void (*)(void *,void *)) generateDispatchClassMethod,
		    context);
  }
}

static void generateDispatch (Type        class,
			      Context     context)
{

  MethodRecordID = 0;
  clearMethodList ();
  generateDispatchMethods (class, context);
}

static void generateMethodLine (Procedure   m,
				Context     context)
{
  fprintf (context->file,
	   "%s\n  { \"%s\", %u, %u, %u,  NULL, 0, NULL }",
	   IncludeComma ? "," : "",
	   procedure_name (m),
	   m->id,
	   m->functional ? 1 : 0,
	   m->asynch ? 1 : 0);
}

static void generateMethods (Type        t,
			     Context     context)
{
  Class od;
  
  if (t == NULL OR type_basic_type (t) != object_Type OR
      (od = class_object (t)) == NULL)
    return;
  if (list_size (od->methods) > 0) {
    generateMethodLine (list_car (od->methods), context);
    IncludeComma = True;
    if (list_size (od->methods) > 1)
      list_enumerate (list_cdr (od->methods), 
		      (void (*)(void *,void *)) generateMethodLine, 
		      context);
  }
}

static void generateMethodsTable (Type        class,
				  Context     context)
{
  Class od = class_object (class);
  
  if (countMethods (od) > 0) {
    fprintf (context->file, "static struct _ilu_Method_s MethodsOf%s[] = {\n",
	     c_type_name (class));
    fprintf (context->file, "/* name, id, cacheable, asynchronous, evector, ecount, stubproc */\n");
    IncludeComma = False;
    MethodRecordID = 1;
    generateMethods (class, context);
    fprintf (context->file, "\n};\n\n");
  }
}

static void listArgumentNames (Argument arg, Context context)
{
  fprintf (context->file, ", %s", c_argument_name(arg));
}

void generateDispatchClassMethod(
				 Procedure   m,
				 Context     c )
{
  void listArgumentTypes(void *, void *);
  int ndx;
  enum PrimitiveTypes t = type_basic_type (m->returnType);

  /*
   ** this stuff is trying to ensure that we dont
   ** generate the same method more than once.
   ** this can happen if the current class has
   ** more than one super classes and each of
   ** those is a subclass of the same superclass.
   ** got it!
   */

  if ( methodInList( c_simple_name( m->name ))) {
    MethodRecordID++;
    return;
  }

  addMethodToList( c_simple_name( m->name ));
  generateProcHeader (m, c, TRUE);
  fprintf (c->file, "  %s (*_f)(%s, ILU_C_ENVIRONMENT *", (t == void_Type) ? "void" : c_return_type(m->returnType),
	   c_type_name(c->class));
  list_enumerate (m->arguments, (void ( * )( )) listArgumentTypes, c);
  fprintf (c->file, ");\n" );
  fprintf (c->file, "  _f = (%s (*)(%s, ILU_C_ENVIRONMENT *",
	   (t == void_Type) ? "void" : c_return_type(m->returnType), c_type_name( c->class ));
  list_enumerate (m->arguments, (void ( * )()) listArgumentTypes, c);
  ndx = methodNdxOf( m, m->object );
  fprintf (c->file, ")) _ILU_C_FindMethod (_handle, &_%s__ILUClassRecord, %d);\n",
	   c_type_name( m->object ), ndx);
  fprintf (c->file, "#ifdef CORBA11\n  _status->_major = StExcep_NO_EXCEPTION;\n#endif /* def CORBA11 */\n");
  fprintf (c->file, "#ifdef CORBA12\n  _status->_major = CORBA_NO_EXCEPTION;\n#endif /* def CORBA12 */\n");
  if (t != void_Type)
    fprintf (c->file, "  return (");
  else
    fprintf (c->file, "  ");
  fprintf (c->file, "(*_f)(_handle, _status" );
  list_enumerate (m->arguments, (void (*)()) listArgumentNames, c);
  if (t != void_Type)
    fprintf (c->file, "));\n");
  else
    fprintf (c->file, ");\n");
  fprintf (c->file, "}\n\n");
}

static void generateUserDataAccessors (Type type, Context context)
{
  char *tn;
 
  tn = c_type_name(type);
  fprintf (context->file, "void %s__SetUserData (%s self, void *userData)\n", tn, tn);
  fprintf (context->file, "{\n  ((ILU_C_Object *) self)->data = userData;\n}\n\n");
  fprintf (context->file, "void *%s__GetUserData (%s self)\n", tn, tn);
  fprintf (context->file, "{\n  return(((ILU_C_Object *) self)->data);\n}\n\n");
}

void generateCommonCode (Interface  interface,
			 FILE  *file)
{
  struct context_s context;

  context.file = file;
  context.interface = interface;
  fprintf (file, 
	   "#include \"%s.h\"\n\n",
	   c_interface_name (interface));
  list_enumerate (interface->classes, 
		  (void (*)(void *,void *)) generateMethodsTable, 
		  &context);
  list_enumerate (interface->classes, 
		  (void (*)(void *,void *)) generateClassTable, 
		  &context);
  list_enumerate (interface->classes, 
		  (void (*)(void *,void *)) generateDispatch, 
		  &context);
  list_enumerate (interface->classes, 
		  (void (*)(void *,void *)) generateUserDataAccessors,
		  &context);
  generateGlobalCode (interface, &context);
  list_enumerate (interface->types, 
		  (void (*)(void *,void *)) generateSequenceCode, 
		  &context);
  list_enumerate (interface->classes, 
		  (void (*)(void *,void *)) generateExceptionLinking, 
		  &context);
  generateClientRegistrationCode (interface, &context);
}
