MODULE Bench EXPORTS Main;
IMPORT IO, Params, Text, OO7, OO7Rep, CRandom, AtomicPart, ODMG,
       Module, Scan, Lex, Fmt, FloatMode, SetParams, Globals, Transaction,
       Database, Utime, Uresource, Support, Stdio, Process, VarParams,
       BaseAssemblyList, BenchParams, CompositePart, BaseAssembly,
       CompositePartList;
FROM IO IMPORT Put, PutInt;
FROM Support IMPORT ComputeUserTime, ComputeSystemTime, ComputeWallClockTime;

VAR     
  startUsage, endUsage: Uresource.struct_rusage;
  startWallTime, endWallTime, startWarmTime: Utime.struct_timeval;
  tz: Utime.struct_timezone;

(*
  ///////////////////////////////////////////////////////////////////////////
  // ParseCommandLine parses the original shell call to "bench", determining 
  // which operation to run, how many times to run it, and whether the
  // individual runs are distinct transactions or are lumped together
  // into one large transaction.
  //////////////////////////////////////////////////////////////////////////
*)
PROCEDURE ParseCommandLine (VAR opIndex, repeatCount: INTEGER;
                            VAR whichOp: OO7.BenchmarkOp;
                            VAR manyXACTS: BOOLEAN) =
  CONST
    usage1 =
      " <configFileName> <count> <op> <xacts> [-d]\n" &
      "<count> = number of times to repeat <op>.\n";
    usage2 =
      "Currently supported <op>s: t1, t1ww, t2a, t2b, t2c, t3a, t3b, t3c\n" &
      "\t t4, t5do, t5undo, t6, t7, t8, t9, t10\n" &
      "\t q1, q2, q3, q4, q5, q6, q7, q8\n" &
      "\t i, d, r1, r2, wu\n";
    usage3 =
      "<xacts> = [single|one|many]\n" &
      "\t(all repetitions done in one or many distinct transactions)\n";
  BEGIN 
    Put("Call was: ");
    FOR foo := 0 TO Params.Count-1 DO           
      Put(Params.Get(foo)); Put("  ");
    END;
    Put("\n");
    opIndex := 3;

    IF Text.Equal(Params.Get(Params.Count-1), "-d") THEN
      Globals.debugMode := TRUE;
    END;
    IF Params.Count < 5 THEN
      Put("Usage: ", Stdio.stderr);
      Put(Params.Get(0), Stdio.stderr);
      Put(usage1, Stdio.stderr);
      Put(usage2, Stdio.stderr);
      Put(usage3, Stdio.stderr);
      Process.Exit(1);
    END;

    TRY
      repeatCount := Scan.Int(Params.Get(2));
    EXCEPT
    | FloatMode.Trap, Lex.Error =>
      Put("Usage: ", Stdio.stderr);
      Put(Params.Get(0), Stdio.stderr);
      Put(usage1, Stdio.stderr);
      Process.Exit(1);
    END;

    IF repeatCount <= 0 THEN
      Put("Usage: ", Stdio.stderr);
      Put(Params.Get(0), Stdio.stderr);
      Put(usage1, Stdio.stderr);
      Put(usage2, Stdio.stderr);
      Put(usage3, Stdio.stderr);
      Process.Exit(1);
    END;

    IF Text.Equal(Params.Get(4), "many") THEN 
      manyXACTS := TRUE;
    END;
    
    WITH op = Params.Get(opIndex) DO
      IF Text.Equal(op, "t1") THEN
        whichOp := OO7.BenchmarkOp.Trav1;
      ELSIF Text.Equal(op, "t1ww") THEN
        whichOp := OO7.BenchmarkOp.Trav1WW;
        TRY
          IF Globals.debugMode THEN
            AtomicPart.WorkAmount := Scan.Int(Params.Get(Params.Count-2));
          ELSE
            AtomicPart.WorkAmount := Scan.Int(Params.Get(Params.Count-1));
          END;
        EXCEPT
        | FloatMode.Trap, Lex.Error =>
          Put("Usage: ", Stdio.stderr);
          Put(Params.Get(0), Stdio.stderr);
          Put(usage1, Stdio.stderr);
          Process.Exit(1);
        END;
        Put("WorkAmount = "); PutInt(AtomicPart.WorkAmount); Put("\n");
      ELSIF Text.Equal(op, "t2a") THEN
        whichOp := OO7.BenchmarkOp.Trav2a;
      ELSIF Text.Equal(op, "t2b") THEN
        whichOp := OO7.BenchmarkOp.Trav2b;
      ELSIF Text.Equal(op, "t2c") THEN 
        whichOp := OO7.BenchmarkOp.Trav2c;
      ELSIF Text.Equal(op, "t3a") THEN
        whichOp := OO7.BenchmarkOp.Trav3a;
      ELSIF Text.Equal(op, "t3b") THEN
        whichOp := OO7.BenchmarkOp.Trav3b;
      ELSIF Text.Equal(op, "t3c") THEN
        whichOp := OO7.BenchmarkOp.Trav3c;
      ELSIF Text.Equal(op, "t4") THEN
        whichOp := OO7.BenchmarkOp.Trav4;
      ELSIF Text.Equal(op, "t5do") THEN
        whichOp := OO7.BenchmarkOp.Trav5do;
      ELSIF Text.Equal(op, "t5undo") THEN
        whichOp := OO7.BenchmarkOp.Trav5undo;
      ELSIF Text.Equal(op, "t6") THEN
        whichOp := OO7.BenchmarkOp.Trav6;
      ELSIF Text.Equal(op, "t7") THEN
        whichOp := OO7.BenchmarkOp.Trav7;
      ELSIF Text.Equal(op, "t8") THEN
        whichOp := OO7.BenchmarkOp.Trav8;
      ELSIF Text.Equal(op, "t9") THEN 
        whichOp := OO7.BenchmarkOp.Trav9;
      ELSIF Text.Equal(op, "t10") THEN
        whichOp := OO7.BenchmarkOp.Trav10;
      ELSIF Text.Equal(op, "q1") THEN
        whichOp := OO7.BenchmarkOp.Query1;
      ELSIF Text.Equal(op, "q2") THEN
        whichOp := OO7.BenchmarkOp.Query2;
      ELSIF Text.Equal(op, "q3") THEN
        whichOp := OO7.BenchmarkOp.Query3;
      ELSIF Text.Equal(op, "q4") THEN
        whichOp := OO7.BenchmarkOp.Query4;
      ELSIF Text.Equal(op, "q5") THEN
        whichOp := OO7.BenchmarkOp.Query5;
      ELSIF Text.Equal(op, "q6") THEN
        whichOp := OO7.BenchmarkOp.Query6;
      ELSIF Text.Equal(op, "q7") THEN
        whichOp := OO7.BenchmarkOp.Query7;
      ELSIF Text.Equal(op, "q8") THEN
        whichOp := OO7.BenchmarkOp.Query8;
      ELSIF Text.Equal(op, "i") THEN
        whichOp := OO7.BenchmarkOp.Insert;
      ELSIF Text.Equal(op, "d") THEN
        whichOp := OO7.BenchmarkOp.Delete;
      ELSIF Text.Equal(op, "r1") THEN
        whichOp := OO7.BenchmarkOp.Reorg1;
      ELSIF Text.Equal(op, "r2") THEN
        whichOp := OO7.BenchmarkOp.Reorg2;
      ELSIF Text.Equal(op, "wu") THEN
        whichOp := OO7.BenchmarkOp.WarmUpdate;
      ELSE
        Put("ERROR: Illegal OO7 operation specified.\n", Stdio.stderr);
        Put(usage2, Stdio.stderr);
        Process.Exit(1);
      END
    END;
  END ParseCommandLine;

<*FATAL Database.NotFound*>
<*FATAL Database.Opened*>
<*FATAL Database.Closed*>
<*FATAL Transaction.InProgress*>
<*FATAL Transaction.NotInProgress*>
<*FATAL Transaction.Disabled*>

PROCEDURE bench () =
  VAR
    db := ODMG.Open("OO7");
    tr := NEW(Transaction.T);
    moduleH: Module.T;
    resultText: TEXT;			 (* buffer to hold result of        *)
					 (* operation for printing outside  *)
					 (* of timing region.               *)
    ref: REFANY;
    moduleName: TEXT;
    opIndex := 2;
    repeatCount := 1;
    whichOp := OO7.BenchmarkOp.Trav1;
    manyXACTS := FALSE;
  BEGIN
    tr.begin();

    (* See if debug mode is desired, see which operation to run,
       and how many times to run it. *)
    ParseCommandLine(opIndex, repeatCount, whichOp, manyXACTS);

    (* initialize parameters for benchmark. *)
    SetParams.FromFile(Params.Get(1));
    Globals.nextAtomicId := VarParams.TotalAtomicParts + 1;
    Globals.nextCompositeId := VarParams.TotalCompParts + 1;

    Globals.InitGlobals(db);

    tr.commit();

    (* Compute structural info needed by the update operations,
       since these operations need to know which id's should
       be used next. *)

    VAR
      baseCnt := VarParams.NumAssmPerAssm;
      complexCnt := 1;
    BEGIN
      FOR i := 1 TO VarParams.NumAssmLevels-2 DO
        baseCnt := baseCnt * VarParams.NumAssmPerAssm;
        INC(complexCnt, complexCnt * VarParams.NumAssmPerAssm);
      END;
      Globals.nextBaseAssemblyId := VarParams.TotalModules * baseCnt + 1;
      Globals.nextComplexAssemblyId := VarParams.TotalModules * complexCnt + 1;
      Globals.nextAtomicId := VarParams.TotalAtomicParts + 1;
      Globals.nextCompositeId := VarParams.TotalCompParts + 1;
    END;

    (* needed for insert and delete tests *)
    Globals.shared_cp  := NEW(REF ARRAY OF BaseAssemblyList.T,
                              VarParams.TotalCompParts +
                              BenchParams.NumNewCompParts + 1);
    Globals.private_cp := NEW(REF ARRAY OF BaseAssemblyList.T,
                              VarParams.TotalCompParts +
                              BenchParams.NumNewCompParts + 1);

    (* Actually run the darn thing. *)
    FOR iter := 0 TO repeatCount-1 DO
      (*
        //////////////////////////////////////////////////////////////////
        // Run an OO7 Benchmark Operation
        //
        //////////////////////////////////////////////////////////////////
      *)

      Put("RUNNING OO7 BENCHMARK OPERATION "); Put(Params.Get(opIndex));
      Put(", iteration = "); PutInt(iter); Put(".\n");

      (* get wall clock time *)
      EVAL Utime.gettimeofday(startWallTime, tz);

      (* get starting usage values. *)
      EVAL Uresource.getrusage(Uresource.RUSAGE_SELF, startUsage);

      (* Start a new transaction if either this is the first iteration
         of a multioperation transaction or we we are running each
         operate as a separate transaction *)

      IF iter = 0 THEN
        tr.begin();
      ELSIF manyXACTS THEN
        IF Globals.chain_tx THEN
          (* do nothing *)
        ELSE
          tr.begin();
        END;
      END;

      (* set random seed so "hot" runs are truly hot *)
      EVAL CRandom.srandom(1);

      (* Use random module for the operation *)
      (* moduleId := (CRandom.random() MOD VarParams.TotalModules) + 1 *)
      FOR moduleId := 1 TO VarParams.TotalModules DO
        moduleName := Fmt.F("Module %08s", Fmt.Int(moduleId));
        Put("Traversing Module= "); Put(moduleName); Put("\n");
        IF NOT Globals.ModuleIdx.get(moduleName, ref) THEN
          Put("ERROR: Unable to access ", Stdio.stderr);
          Put(moduleName, Stdio.stderr);
          Put(".\n", Stdio.stderr);
          tr.abort();
          Process.Exit(1);
        END;
        moduleH := ref;

        (* Perform the requested operation on the chosen module *)
        VAR
          count := 0;
        BEGIN
          CASE (whichOp) OF
          | OO7.BenchmarkOp.Trav1 =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 1 DFS visited %s atomic parts.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav1WW =>
            AtomicPart.RealWork := TRUE;
            whichOp := OO7.BenchmarkOp.Trav1; (* so traverse methods don't complain *)
            count := moduleH.traverse(whichOp);
            whichOp := OO7.BenchmarkOp.Trav1WW;  (* for next (hot) traversal *)
            resultText :=
                Fmt.F("Traversal 1WW DFS visited %s atomic parts.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav2a =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 2A swapped %s pairs of (X,Y) coordinates.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav2b =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 2B swapped %s pairs of (X,Y) coordinates.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav2c =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 2C swapped %s pairs of (X,Y) coordinates.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav3a =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 3A toggled %s dates.\n", Fmt.Int(count));
          | OO7.BenchmarkOp.Trav3b =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 3B toggled %s dates.\n", Fmt.Int(count));
          | OO7.BenchmarkOp.Trav3c =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 3C toggled %s dates.\n", Fmt.Int(count));
          | OO7.BenchmarkOp.Trav4 =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 4: %s instances of the character found\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav5do =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 5(DO): %s string replacements performed\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav5undo=>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 5(UNDO): %s string replacements performed\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav6 =>
            count := moduleH.traverse(whichOp);
            resultText :=
                Fmt.F("Traversal 6: visited %s atomic part roots.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav7 =>
            count := traverse7();
            resultText :=
                Fmt.F("Traversal 7: found %s assemblies using random atomic part.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav8 =>
            count := moduleH.scanManual();
            resultText :=
                Fmt.F("Traversal 8: found %s occurrences of character in manual.\n",
                      Fmt.Int(count));
          | OO7.BenchmarkOp.Trav9 =>
            count := moduleH.firstLast();
            resultText :=
                Fmt.F("Traversal 9: match was %s.\n", Fmt.Int(count));
          | OO7.BenchmarkOp.Trav10 =>
            (* run traversal #1 on every module. *)
            count := 0;
            whichOp := OO7.BenchmarkOp.Trav1;  (* so object methods don't complain *)
            FOR moduleId := 1 TO VarParams.TotalModules DO
              moduleName := Fmt.F("Module %08s", Fmt.Int(moduleId));
              IF NOT Globals.ModuleIdx.get(moduleName, ref) THEN
                Put("ERROR: t10 Unable to access ", Stdio.stderr);
                Put(moduleName, Stdio.stderr);
                Put(".\n", Stdio.stderr);
                tr.abort();
                Process.Exit(1);
              END;
              moduleH := ref;
              INC(count, moduleH.traverse(whichOp));
            END;
            resultText :=
                Fmt.F("Traversal 10 visited %s atomic parts in %s modules.\n",
                      Fmt.Int(count),
                      Fmt.Int(VarParams.TotalModules));
            whichOp := OO7.BenchmarkOp.Trav10;  (* for next time around *)
          | OO7.BenchmarkOp.Insert =>
            insert1();
            resultText :=
                Fmt.F("Inserted %s composite parts (a total of %s atomic parts.)\n",
                      Fmt.Int(BenchParams.NumNewCompParts),
                      Fmt.Int(BenchParams.NumNewCompParts *
                              VarParams.NumAtomicPerComp));
          | OO7.BenchmarkOp.WarmUpdate =>
            (* first do the t1 traversal to warm the cache *)
            count := moduleH.traverse(OO7.BenchmarkOp.Trav1);
            (* then call T2 to do the update *)
            count := moduleH.traverse(OO7.BenchmarkOp.Trav2a);
            resultText :=
                Fmt.F("Warm update swapped %s pairs of (X,Y) coordinates.\n",
                      Fmt.Int(count));
          ELSE
            Put("Sorry, that operation isn't available yet.\n", Stdio.stderr);
            tr.abort();
            Process.Exit(1);
          END;
          Put("Visited="); PutInt(count); Put("\n");
        END;

        (* Commit the current transaction if
           we are running the last iteration
           or running a multitransaction test and not chaining
           Chain the tx if we are chaining and not on
           the last iteration *)
        IF iter = repeatCount-1 THEN
          tr.commit();
        ELSIF manyXACTS THEN
          IF Globals.chain_tx THEN
            tr.chain();
          ELSE
            tr.commit();
          END
        END
      END;

      (* compute and report wall clock time *)
      EVAL Utime.gettimeofday(endWallTime, tz);
      Put("PM3, operation= "); Put(Params.Get(opIndex));
      Put(", iteration= "); PutInt(iter);
      Put(", elapsedTime= ");
      Put(Fmt.LongReal(ComputeWallClockTime(startWallTime, endWallTime)));
      Put(" seconds\n");
      IF (iter = 1) THEN
        startWarmTime := startWallTime;
      END;

      (* Compute and report CPU time. *)
      EVAL Uresource.getrusage(Uresource.RUSAGE_SELF, endUsage);
      Put(resultText);
      Put("CPU time: ");
      Put(Fmt.LongReal(
              ComputeUserTime(startUsage, endUsage) +
              ComputeSystemTime(startUsage, endUsage)));
      Put(".\n(");
      Put(Fmt.LongReal(ComputeUserTime(startUsage, endUsage)));
      Put(" seconds user, ");
      Put(Fmt.LongReal(ComputeSystemTime(startUsage, endUsage)));
      Put(" seconds system.)\n");
        
      Put("maximum resident set size: "); PutInt(endUsage.ru_maxrss);
      Put("\nintegral resident set size: "); PutInt(endUsage.ru_idrss);
      Put("\npage faults not requiring physical I/O: ");
      PutInt(endUsage.ru_minflt - startUsage.ru_minflt);
      Put("\npage faults requiring physical I/O: ");
      PutInt(endUsage.ru_majflt - startUsage.ru_majflt);
      Put("\nswaps: ");
      PutInt(endUsage.ru_nswap - startUsage.ru_nswap);
      Put("\nblock input operations: ");
      PutInt(endUsage.ru_inblock - startUsage.ru_inblock);
      Put("\nblock output operations: ");
      PutInt(endUsage.ru_oublock - startUsage.ru_oublock);
      Put("\nmessages sent: ");
      PutInt(endUsage.ru_msgsnd - startUsage.ru_msgsnd);
      Put("\nmessages received: ");
      PutInt(endUsage.ru_msgrcv - startUsage.ru_msgrcv);
      Put("\nsignals received: ");
      PutInt(endUsage.ru_nsignals - startUsage.ru_nsignals);
      Put("\nvoluntary context switches: ");
      PutInt(endUsage.ru_nvcsw - startUsage.ru_nvcsw);
      Put("\ninvoluntary context switches: ");
      PutInt(endUsage.ru_nivcsw - startUsage.ru_nivcsw);
      Put("\n\n");

      IF (repeatCount > 2) AND (iter = repeatCount-2) THEN
        (* compute average hot time for 2nd through n-1 th iterations *)
        Put("PM3, operation="); Put(Params.Get(opIndex));
        Put(", average hot elapsedTime=");
        Put(Fmt.LongReal(
                ComputeWallClockTime(startWarmTime, endWallTime) /
                FLOAT(repeatCount-2, LONGREAL)));
        Put(" seconds\n");
      END;
    END;

    (*
      //////////////////////////////////////////////////////////////////
      //
      // Shutdown 
      //
      //////////////////////////////////////////////////////////////////
    *)
    db.close();
  END bench;

(*
  //////////////////////////////////////////////////////////////////
  //
  // Insert - Create NumNewCompParts new composite parts (which
  // means also creating a set of AtomicParts for each.)  Add
  // a reference to each new composite part from a randomly chosen
  // base assembly (a different random choice of base assembly for
  // each new composite part.)
  //
  //////////////////////////////////////////////////////////////////
*)

PROCEDURE insert1 ()=
  (* now create the desired number of new composite parts, adding each
     one as a (private) composite part that a randomly selected base
     assembly now wishes to use *)
  VAR
    compId, assmId: INTEGER;
    compH : CompositePart.T;
    assmH : BaseAssembly.T;
    ref: REFANY;
  BEGIN
    IF Globals.debugMode THEN
      Put("In Insert, nextCompositeId = ");
      PutInt(Globals.nextCompositeId);
      Put("\nIn Insert, nextAtomicId = ");
      PutInt(Globals.nextAtomicId);
      Put("\nIn Insert, nextBaseAssemblyId = ");
      PutInt(Globals.nextBaseAssemblyId);
      Put("\n");
    END;

    FOR i := 1 TO BenchParams.NumNewCompParts DO
      (*  add a new composite part to the design library *)
      IF Globals.debugMode THEN
        Put("In Insert, making composite part ");
        PutInt(Globals.nextCompositeId);
        Put("\n");
      END;
      compId := Globals.nextCompositeId;
      INC(Globals.nextCompositeId);
      compH := NEW (CompositePart.T).init(compId);

      (* add composite part to its index *)
      EVAL Globals.CompPartIdx.put(compId, compH);
      
      (* randomly select a base assembly that should use it and figure
         out which module it resides in *)
      assmId := (CRandom.random() MOD (Globals.nextBaseAssemblyId-1)) + 1;

      (* now locate the appropriate base assembly object within its module *)
      IF NOT Globals.BaseAssemblyIdx.get(assmId, ref) THEN
        Put("ERROR: Can't find base assembly ", Stdio.stderr);
        PutInt(assmId, Stdio.stderr);
        Put("\n", Stdio.stderr);
        Process.Exit(1);
      END;
      assmH := ref;

      (* finally, add the newly created composite part as a privately used
         member of the randomly selected base assembly *)

      (* first add this assembly to the list of assemblies in which
         this composite part is used as a private member  *)
      compH.usedInPriv := BaseAssemblyList.Cons(assmH, compH.usedInPriv);
      (* then add the composite part compH to the list of private parts used
         in this assembly *)
      assmH.componentsPriv :=
          CompositePartList.Cons(compH, assmH.componentsPriv);

      IF Globals.debugMode THEN
        Put("[just made it be used by base assembly ");
        PutInt(assmId);
        Put("]\n");
      END
    END
  END insert1;

(*
  /////////////////////////////////////////////////////////////////
  //
  // Traversal #7.  Pick a random atomic part; traverse from the 
  // part to its containing composite part P.  From P, traverse
  // backwards up to all base assemblies referencing P; for each
  // such base assembly, traverse upward through the assembly
  // hierarchy until the root is reached.  This traversal coupled
  // with traversal #1 ensures
  // that all pointers are bi-directional.
  //
  /////////////////////////////////////////////////////////////////
*)

PROCEDURE traverse7 (): INTEGER =
  VAR
    partId: INTEGER;
    atomH: AtomicPart.T;
    ref: REFANY;
  BEGIN
    (* get access to the design library containing atomic parts *)
       
    (* prepare to select a random atomic part via the id index
       now randomly select a part and traverse upwards from there *)
       
    (* set random seed so "hot" runs are truly hot *)
    EVAL CRandom.srandom(1);

    partId := (CRandom.random() MOD VarParams.TotalAtomicParts) + 1;
    IF NOT Globals.AtomicPartIdx.get(partId, ref) THEN
      Put("ERROR: Unable to find atomic part ", Stdio.stderr);
      PutInt(partId, Stdio.stderr);
      Put("\n", Stdio.stderr);
      Process.Exit(1);
    END;
    atomH := ref;

    IF Globals.debugMode THEN
      Put("  In Traverse7, starting at atomic part ");
      PutInt(partId);
      Put("\n");
    END;
    RETURN atomH.partOf.traverse7();
  END traverse7;

BEGIN
  bench();
END Bench. 
