UNSAFE MODULE Database EXPORTS Database, Resource;

IMPORT RTIO, RTDB, Fingerprint, Pathname, Access, PageFile, TextF,
       VirtualResourceSystem, VirtualResource,
       VirtualPage, InternalVirtualPage,
       VirtualFile, VirtualRemoteFile, Txn,
       TextTextBPlusTree, FPRefFPBPlusTree,
       IntIntTransientTbl AS PageTbl, TextRefTransientTbl AS TextRefTbl,
       Atom, AtomList, RTProcess, RTParams, Lex, Scan, Env, Config, FloatMode,
       PageCache, Transaction;

FROM Text IMPORT Length;
FROM RTHeapRep IMPORT Page, Nil;
FROM Transaction IMPORT LockMode;

REVEAL Private = RTDB.T BRANDED "Database.Private" OBJECT END;
REVEAL T = Public BRANDED "Database.T" OBJECT
  file: VirtualFile.T;
  pageMap: PageTbl.T;
OVERRIDES
  map := Map;
  unmap := Unmap;
  pin := Pin;
  unpin := Unpin;
  createRoot := CreateRoot;
  lock := Lock;
  newPages := NewPages;
  getFP := GetFP;
  text := Text;
  getRoot := GetRoot;
  setRoot := SetRoot;
END;

TYPE Root = BRANDED "Database.Root" REF RECORD
  pages: Page := 0;
  ref: REFANY := NIL;
  texts : TextTextBPlusTree.T;
  fingerprints : FPRefFPBPlusTree.T;
END;

EXCEPTION FatalError; <*FATAL FatalError*>
PROCEDURE errorAtoms (t: AtomList.T) =
  BEGIN
    WHILE t # NIL DO
      RTIO.PutText(Atom.ToText(t.head)); RTIO.PutChar('\n');
      t := t.tail;
    END;
    RTIO.Flush();
  END errorAtoms;
 
PROCEDURE errorText (t: TEXT) =
  BEGIN
    RTIO.PutText(t); RTIO.PutChar('\n'); RTIO.Flush();
  END errorText;

PROCEDURE Create(name: Pathname.T)
  RAISES { Exists, Transaction.InProgress, Transaction.Disabled } = 
  VAR
    db: T := NEW(T, pageMap := NEW(PageTbl.Default).init());
  BEGIN
    LOCK m DO
      TRY
        IF resource.getTransactionLevel() # Txn.EnvelopeLevel THEN
          RAISE Transaction.InProgress;
        END;
        IF resource.existsFile (name, local := FALSE) THEN
          RAISE Exists;
        END;
        db.file :=
            NEW(VirtualRemoteFile.T).open (resource, name,
                                           Access.Mode.ReadWriteExclusive,
                                           Access.Kind.Data,
                                           new := TRUE);
        resource.beginTransaction();
        RTDB.Invalidate();
        IF NOT RTDB.Flush(db) THEN
          resource.abortTransaction(); <*NOWARN*>
          resource.deleteFile(name, local := FALSE);
          RAISE Transaction.Disabled;
        END;
        db.file.close();
        resource.commitTransaction(); <*NOWARN*>
        RTDB.Release();
      EXCEPT
      | Access.Denied(t) => errorText(t); RAISE FatalError;
      | PageFile.NoAccess(t) => errorText(t); RAISE FatalError;
      | VirtualResource.FatalError(t) => errorAtoms(t); RAISE FatalError;
      END;
    END;
  END Create;

VAR open := NEW(TextRefTbl.Default).init();

PROCEDURE Open(name: Pathname.T): T
  RAISES { NotFound, Opened, Transaction.InProgress } =
  VAR
    self: T := NEW(T, pageMap := NEW(PageTbl.Default).init());
    file: <*TRANSIENT*> REFANY;
  BEGIN
    LOCK m DO
      TRY
        IF open.get(name, file) THEN
          RAISE Opened;
        END;
        IF resource.getTransactionLevel() # Txn.EnvelopeLevel THEN
          RAISE Transaction.InProgress;
        END;
        IF NOT resource.existsFile (name, local := FALSE) THEN
          RAISE NotFound;
        END;
        self.file :=
            NEW(VirtualRemoteFile.T).open (resource, name,
                                           Access.Mode.ReadWriteShared,
                                           Access.Kind.Data, new := FALSE);
        EVAL open.put(name, self.file);
      EXCEPT
      | Access.Denied(t) => errorText(t); RAISE FatalError;
      | PageFile.NoAccess(t) => errorText(t); RAISE FatalError;
      | VirtualResource.FatalError(t) => errorAtoms(t); RAISE FatalError;
      END;
    END;
    RETURN self;
  END Open;

PROCEDURE Lock(self: T; READONLY p: Page; mode: LockMode) =
  VAR page := self.file.getPage(p-1);
  BEGIN
    TRY
      PageCache.BeginAccess();
      TRY
        CASE mode OF
        | LockMode.READ    => EVAL page.readAccess();
        | LockMode.UPGRADE => EVAL page.writeAccess(load := FALSE);
        | LockMode.WRITE   => EVAL page.writeAccess(load := FALSE);
        END;
      EXCEPT
      | Access.Locked => RAISE FatalError;
      | VirtualPage.FatalError(t) => errorAtoms(t); RAISE FatalError;
      END
    FINALLY
      PageCache.EndAccess();
    END;
  END Lock;

PROCEDURE Map(self: T; READONLY p: Page; n: CARDINAL) : Page =
  VAR pp: INTEGER;
  BEGIN
    IF NOT self.pageMap.get(p, pp) THEN
      pp := RTDB.Map(self, p, n);
      FOR i := 0 TO n-1 DO
        IF self.pageMap.put(p + i, pp + i) THEN
          <* ASSERT FALSE *>
        END
      END
    END;
    RETURN pp;
  END Map;

PROCEDURE Unmap(self: T; READONLY p: Page) =
  VAR pp: INTEGER;
  BEGIN
    IF NOT self.pageMap.delete(p, pp) THEN
      <* ASSERT FALSE *>
    END;
  END Unmap;

PROCEDURE SetRoot(self: T; object: REFANY)
  RAISES { Transaction.NotInProgress } =
  VAR root: Root;
  BEGIN
    LOCK m DO
      IF resource.getTransactionLevel() = Txn.EnvelopeLevel THEN
        RAISE Transaction.NotInProgress;
      END;
      root := self.root;
      IF root = NIL THEN
        root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
        self.root := root;
      END;
      root.ref := object;
    END;
  END SetRoot;

PROCEDURE GetRoot(self: T): REFANY
  RAISES { Transaction.NotInProgress } =
  VAR root: Root;
  BEGIN
    LOCK m DO
      IF resource.getTransactionLevel() = Txn.EnvelopeLevel THEN
        RAISE Transaction.NotInProgress;
      END;
      root := self.root;
      IF root = NIL THEN
        root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
        self.root := root;
      END;
    END;
    RETURN root.ref;
  END GetRoot;

PROCEDURE CreateRoot(self: T) =
  BEGIN
    self.root := NEW(Root,
                     fingerprints := NEW(FPRefFPBPlusTree.T).init(),
                     texts := NEW(TextTextBPlusTree.T).init());
  END CreateRoot;

PROCEDURE NewPages(self: T; pp: Page; n: CARDINAL): Page =
  VAR
    root: Root := self.root;
    p: Page;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    p := root.pages + 1;
    INC(root.pages, n);
    FOR i := 0 TO n-1 DO
      IF self.pageMap.put(p + i, pp + i) THEN
        <* ASSERT FALSE *>
      END
    END;
    RETURN p;
  END NewPages;

VAR
  pinned: Page := Nil;

PROCEDURE Pin (self: T; p: Page; load: BOOLEAN): ADDRESS =
  VAR page := self.file.getPage(p-1);
  BEGIN
    <* ASSERT pinned = Nil *>
    pinned := p;
    TRY
      RETURN ADR(page.pin(load).data);
    EXCEPT
    | Access.Locked => RAISE FatalError;
    | VirtualPage.FatalError(t) => errorAtoms(t); RAISE FatalError;
    END;
  END Pin;

PROCEDURE Unpin (self: T; p: Page) =
  VAR page := self.file.getPage(p-1);
  BEGIN
    <* ASSERT pinned = p *>
    page.unpin();
    pinned := Nil;
  END Unpin;

PROCEDURE GetFP(self: T; fp: UNTRACED REF Fingerprint.T): REF Fingerprint.T =
  VAR
    root: Root := self.root;
    ref: REF Fingerprint.T;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    IF NOT root.fingerprints.get(fp^, ref) THEN
      ref := NEW(REF Fingerprint.T);
      ref^ := fp^;
      IF root.fingerprints.put(fp^, ref) THEN
        <* ASSERT FALSE *>
      END;
    END;
    RETURN ref;
  END GetFP;

PROCEDURE Text(self: T; t: TEXT): TEXT =
  VAR
    root: Root := self.root;
    res: TEXT;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    IF NOT root.texts.get(t, res) THEN
      res := TextF.New(Length(t));
      res^ := t^;
      IF root.texts.put(res, res) THEN
        <* ASSERT FALSE *>
      END;
    END;
    RETURN res;
  END Text;

PROCEDURE Close () =
  BEGIN
    LOCK m DO
      WHILE resource.getTransactionLevel() # Txn.EnvelopeLevel DO
        TRY
          resource.abortTransaction();
        EXCEPT
        | VirtualResource.FatalError(t) => errorText(Atom.ToText(t.head))
        | VirtualResource.NotInTransaction =>
        END
      END;
      VAR it := open.iterate();
          k: TEXT;
          v: <*TRANSIENT*> REFANY;
      BEGIN
        WHILE it.next(k, v) DO
          TRY
            VAR file: VirtualFile.T := v; BEGIN file.close(); END;
          EXCEPT
          | VirtualResource.FatalError(t) => errorText(Atom.ToText(t.head));
          END
        END
      END;
      TRY
        resource.close();
      EXCEPT
      | VirtualResource.FatalError(t) => errorText(Atom.ToText(t.head));
      END
    END;
  END Close;

BEGIN
  VAR
    root := Env.Get("PM3ROOT");
    cacheSize := Config.DefaultCacheSize;
    agent := Config.DefaultNameServer;
    newres: BOOLEAN;
  BEGIN
    IF RTParams.IsPresent("root") THEN
      root := RTParams.Value("root");
    END;
    IF root = NIL OR Length(root) = 0 THEN
      root := Pathname.Current;
    END;
    IF RTParams.IsPresent("cachesize") THEN
      TRY
        cacheSize := Scan.Int(RTParams.Value("cachesize"));
      EXCEPT
      | FloatMode.Trap, Lex.Error =>
        RTIO.PutText("@M3cachesize requires an integer argument");
        RTIO.PutChar('\n');
        RTIO.Flush();
      END;
    END;
    IF RTParams.IsPresent("agent") THEN
      agent := RTParams.Value("agent");
    END;
    VirtualResourceSystem.Login(root, cacheSize, nameserver := agent);
    TRY
      newres := NOT VirtualResourceSystem.ExistsResource("PM3");
    EXCEPT
    | VirtualResourceSystem.FatalError(t) => errorAtoms(t); RAISE FatalError;
    END;
    TRY
      resource := NEW(VirtualResource.T).open("PM3",
                                              Access.Mode.ReadWriteShared,
                                              new := newres);
    EXCEPT
    | Access.Denied(t) => errorText(t); RAISE FatalError;
    | PageFile.NoAccess(t) => errorText(t); RAISE FatalError;
    END;
  END;
  RTProcess.RegisterExitor (Close);
END Database.
