UNSAFE MODULE Database;

IMPORT Shore, ShoreError, SHerror, Uerror, AtomList, RTLinker;
IMPORT RTDB, M3toC, IO, Fingerprint, TextF;
FROM Text IMPORT Length;
IMPORT TextTextBPlusTree, FPRefFPBPlusTree;
IMPORT PageTbl, PageTableEntry;
IMPORT Transaction;

FROM ShoreErrorPosix IMPORT ErrnoAtom;

FROM RTHeapRep IMPORT Page, BytesPerPage;

FROM Transaction IMPORT LockMode;

<* FATAL ShoreError.E *>

VAR m := NEW(MUTEX);

TYPE bytes = <*TRANSIENT*> REF ARRAY OF CHAR;

REVEAL Private = RTDB.T BRANDED "Database.Private" OBJECT END;
REVEAL T = Public BRANDED "Database.T" OBJECT
  shoreDB: bytes;
  pageMap: PageTbl.T;
OVERRIDES
  map := Map;
  unmap := Unmap;
  createRoot := CreateRoot;
  lock := Lock;
  newPages := NewPages;
  pin := Pin;
  unpin := Unpin;
  getFP := GetFP;
  text := Text;
  getRoot := GetRoot;
  setRoot := SetRoot;
END;

TYPE Root = BRANDED "Database.Root" REF RECORD
  pages: Page := 0;
  ref: REFANY := NIL;
  texts : TextTextBPlusTree.T;
  fingerprints : FPRefFPBPlusTree.T;
END;

PROCEDURE Create(database_name: TEXT)
  RAISES { Exists, Transaction.InProgress, Transaction.Disabled } = 
  VAR
    db: T := NEW(T, pageMap := NEW(PageTbl.Default).init());
  BEGIN
    LOCK m DO
      TRY
        Shore.begin();
        RTDB.Invalidate();
        db.shoreDB := NEW(bytes, Shore.dbTSize);
        Shore.create(M3toC.TtoS(database_name), 8_644, ADR(db.shoreDB[0]));
        IF NOT RTDB.Flush(db) THEN
          Shore.abort();
          RAISE Transaction.Disabled;
        END;
        Shore.commit();
        RTDB.Release();
      EXCEPT ShoreError.E(code) =>
        IF AtomList.Member(code, ErrnoAtom(Uerror.EEXIST)) THEN
          RAISE Exists;
        ELSIF AtomList.Member(code, ErrnoAtom(SHerror.SH_Already)) THEN
          RAISE Transaction.InProgress;
        ELSIF AtomList.Member(code, ErrnoAtom(SHerror.SH_TxNotAllowed)) THEN
          RAISE Transaction.InProgress;
        ELSE
          RAISE ShoreError.E(code);
        END;
      END;
    END;
  END Create;

PROCEDURE Open(name: TEXT): T
  RAISES { NotFound, Opened, Transaction.InProgress } =
  VAR
    self: T := NEW(T, pageMap := NEW(PageTbl.Default).init());
  BEGIN
    LOCK m DO
      TRY
        self.shoreDB := NEW(bytes, Shore.dbTSize);
        Shore.open(M3toC.TtoS(name), ADR(self.shoreDB[0]));
      EXCEPT ShoreError.E(code) =>
        IF AtomList.Member(code, ErrnoAtom(SHerror.SH_NotFound)) THEN
          RAISE NotFound;
        ELSIF AtomList.Member(code, ErrnoAtom(Uerror.EEXIST)) THEN
          RAISE Opened;
        ELSIF AtomList.Member(code, ErrnoAtom(SHerror.SH_TxNotAllowed)) THEN
          RAISE Transaction.InProgress;
        ELSE
          RAISE ShoreError.E(code);
        END;
      END;
    END;
    RETURN self;
  END Open;

PROCEDURE Map(self: T; READONLY p: Page; n: CARDINAL): Page =
  VAR
    pp: Page;
    entry: PageTableEntry.T;
  BEGIN
    IF self.pageMap.get(p, entry) THEN
      pp := entry.page;
    ELSE
      pp := RTDB.Map(self, p, n);
      FOR i := 0 TO n-1 DO
        entry.page := pp + i;
        entry.ref := NIL;
        IF self.pageMap.put(p + i, entry) THEN
          <* ASSERT FALSE *>
        END
      END
    END;
    RETURN pp;
  END Map;

PROCEDURE Unmap(self: T; READONLY p: Page) =
  VAR entry: PageTableEntry.T;
  BEGIN
    IF NOT self.pageMap.delete(p, entry) THEN
      <* ASSERT FALSE *>
    END;
    WITH ref = entry.ref DO
      IF ref # NIL THEN
        Shore.Flush(ADR(ref[0]));
      END
    END
  END Unmap;

PROCEDURE SetRoot(self: T; object: REFANY) =
  VAR root: Root;
  BEGIN
    LOCK m DO
      root := self.root;
      IF root = NIL THEN
        root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
        self.root := root;
      END;
      root.ref := object;
    END;
  END SetRoot;

PROCEDURE GetRoot(self: T): REFANY =
  VAR root: Root;
  BEGIN
    LOCK m DO
      root := self.root;
      IF root = NIL THEN
        root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
        self.root := root;
      END;
    END;
    RETURN root.ref;
  END GetRoot;

PROCEDURE CreateRoot(self: T) =
  BEGIN
    self.root := NEW(Root,
                     fingerprints := NEW(FPRefFPBPlusTree.T).init(),
                     texts := NEW(TextTextBPlusTree.T).init());
  END CreateRoot;

PROCEDURE NewPages(self: T; pp: Page; n: CARDINAL): Page =
  VAR
    entry: PageTableEntry.T;
    root: Root := self.root;
    p: Page;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    p := root.pages + 1;
    INC(root.pages, n);
    FOR i := 0 TO n-1 DO
      entry.page := pp + i;
      entry.ref := NIL;
      IF self.pageMap.put(p + i, entry) THEN
        <* ASSERT FALSE *>
      END
    END;
    RETURN p;
  END NewPages;

PROCEDURE Pin (self: T; p: Page; load: BOOLEAN): ADDRESS =
  VAR entry: PageTableEntry.T;
  BEGIN
    IF NOT self.pageMap.get(p, entry) THEN
      <* ASSERT FALSE *>
    END;
    WITH ref = entry.ref DO
      IF ref = NIL THEN
        ref := NEW(bytes, Shore.pageTSize);
        Shore.GetPage(ADR(self.shoreDB[0]), p, BytesPerPage, ADR(ref[0]));
      END;
      IF load THEN
        RETURN Shore.Fetch(ADR(ref[0]));
      ELSE
        RETURN Shore.Update(ADR(ref[0]));
      END
    END
  END Pin;

PROCEDURE Unpin (self: T; p: Page) =
  VAR entry: PageTableEntry.T;
  BEGIN
    IF NOT self.pageMap.get(p, entry) THEN
      <* ASSERT FALSE *>
    END;
  END Unpin;

PROCEDURE Lock(self: T; READONLY p: Page; mode: LockMode) =
  VAR entry: PageTableEntry.T;
  BEGIN
    IF NOT self.pageMap.get(p, entry) THEN
      <* ASSERT FALSE *>
    END;
    WITH ref = entry.ref DO
      IF ref = NIL THEN
        ref := NEW(bytes, Shore.pageTSize);
        Shore.GetPage(ADR(self.shoreDB[0]), p, BytesPerPage, ADR(ref[0]));
      END;
      CASE mode OF
      | LockMode.READ    => Shore.Valid(ADR(ref[0]), Shore.LockMode.SH);
      | LockMode.UPGRADE => Shore.Valid(ADR(ref[0]), Shore.LockMode.SIX);
      | LockMode.WRITE   => Shore.Valid(ADR(ref[0]), Shore.LockMode.EX);
      END
    END
  END Lock;

PROCEDURE GetFP(self: T; fp: UNTRACED REF Fingerprint.T): REF Fingerprint.T =
  VAR
    root: Root := self.root;
    ref: REF Fingerprint.T;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    IF NOT root.fingerprints.get(fp^, ref) THEN
      ref := NEW(REF Fingerprint.T);
      ref^ := fp^;
      IF root.fingerprints.put(fp^, ref) THEN
        <* ASSERT FALSE *>
      END;
    END;
    RETURN ref;
  END GetFP;

PROCEDURE Text(self: T; t: TEXT): TEXT =
  VAR
    root: Root := self.root;
    res: TEXT;
  BEGIN
    IF root = NIL THEN
      root := LOOPHOLE(RTDB.SwizzleRoot(self), Root);
      self.root := root;
    END;
    IF NOT root.texts.get(t, res) THEN
      res := TextF.New(Length(t));
      res^ := t^;
      IF root.texts.put(res, res) THEN
        <* ASSERT FALSE *>
      END;
    END;
    RETURN res;
  END Text;

BEGIN
  LOCK m DO
    TRY
      Shore.init(RTLinker.info.argc, RTLinker.info.argv);
    EXCEPT ShoreError.E(code) =>
      IF AtomList.Member(code, ErrnoAtom(SHerror.SH_ROFailure)) THEN
        IO.Put("\nServer not started\n");
      END;
      RAISE ShoreError.E(code);
    END
  END
END Database.
