(* Copyright (C) 1993, Digital Equipment Corporation         *)
(* All rights reserved.                                      *)
(* See the file COPYRIGHT for a full description.            *)
(*                                                           *)
(* portions Copyright 1996, Critical Mass, Inc.              *)
(* portions Copyright 1998, Purdue Research Foundation       *)
(*                                                           *)
(*| Last modified on Fri Apr 26 10:29:11 PDT 1996 by heydon  *)
(*|      modified on Sat Nov 19 09:37:57 PST 1994 by kalsow  *)
(*|      modified on Fri Aug  5 14:04:35 PDT 1994 by jdd     *)
(*|      modified on Wed Jun  2 15:00:17 PDT 1993 by muller  *)
(*|      modified on Wed Apr 21 13:14:37 PDT 1993 by mcjones *)
(*|      modified on Wed Mar 10 11:01:47 PST 1993 by mjordan *)

UNSAFE MODULE RTCollector EXPORTS RTCollector, RTCollectorSRC,
                                  RTHeapRep, RTWeakRef, RTDB;

IMPORT RT0, RT0u, RTIO, RTHeapEvent, RTHeapDep, RTHeapMap, RTMachine;
IMPORT RTMisc, RTOS, RTParams, RTPerfTool, RTProcess, RTType;
IMPORT Word, Cstdlib, Thread, ThreadF;
IMPORT RTDB, RTTypeMap, RTTypeFP, Fingerprint;

FROM RT0 IMPORT Typecode, TypeDefn;
FROM RTIO IMPORT PutText, PutInt, PutChars;
IMPORT Cstring;

(* The allocator/garbage collector for the traced heap is an adaptation of
   the algorithm presented in the WRL Research Report 88/2, ``Compacting
   Garbage Collection with Ambiguous Roots'', by Joel F.  Bartlett; see
   this report for a detailed presentation.  John DeTreville modified it to
   be incremental, generational, and VM-synchronized.

   The allocator/collector for the untraced heap is simply malloc/free. *)

(* Much of the code below incorrectly assumes no difference between ADRSIZE
   and BYTESIZE. *)

(* In the following procedures, "RTType.Get(tc)" will fail if "tc" is not
   proper. *)

(*** RTCollector ***)

PROCEDURE Disable () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishVM();
      INC(disableCount);
      partialCollectionNext := FALSE;
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END Disable;

PROCEDURE Enable () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END Enable;

PROCEDURE DisableMotion () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      INC(disableMotionCount);
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END DisableMotion;

PROCEDURE EnableMotion () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableMotionCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END EnableMotion;

PROCEDURE Collect () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishGC();
      StartGC();
      FinishGC();
    END;
    RTOS.UnlockHeap();
  END Collect;

(*** RTCollectorSRC ***)

(* StartCollection starts a total collection, if none is in progress and if
   collection and motion are enabled. *)

PROCEDURE StartCollection () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      CollectorOn();
      IF collectorState = CollectorState.Zero
           AND disableCount + disableMotionCount = 0 THEN
        partialCollectionNext := FALSE;
        REPEAT CollectSome(); UNTIL collectorState # CollectorState.Zero;
        IF NOT (incremental AND RTHeapDep.VM AND disableVMCount = 0) THEN
          REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
        END;
      END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END StartCollection;

(* FinishCollection finishes the current collection, if one is on
   progress. *)

PROCEDURE FinishCollection () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      CollectorOn();
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END FinishCollection;

(* DisableVM disables the use of VM protection.  While VM protection is
   disabled, no objects on the heap will be protected.*)

PROCEDURE DisableVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishVM();
      INC(disableVMCount);
    END;
    RTOS.UnlockHeap();
  END DisableVM;

(* EnableVM reenables the use of VM protection if EnableVM has been called
   as many times as DisableVM.  It is a checked runtime error to call
   EnableVM more times than DisableVM. *)

PROCEDURE EnableVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableVMCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
  END EnableVM;

(* FinishVM is equivalent to DisableVM{}; EnableVM().  FinishVM unprotects
   all heap pages, and is intended for use from the debugger. *)

PROCEDURE FinishVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishGC();
      CollectorOn();
      (* no gray pages now; only protected pages are in older generation *)
      FOR p := p0 TO p1 - 1 DO
        IF desc[p - p0].protected
          AND desc[p - p0].resident
          AND desc[p - p0].modified THEN
          Unprotect(p);
        END;
      END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END FinishVM;

(* StartBackgroundCollection starts the background thread, if not already
   started *)

VAR startedBackground := FALSE;

PROCEDURE StartBackgroundCollection () =
  VAR start := FALSE;
  BEGIN
    RTOS.LockHeap();
    BEGIN
      IF NOT startedBackground THEN
        start := TRUE;
        startedBackground := TRUE;
      END;
    END;
    RTOS.UnlockHeap();
    IF start THEN
      EVAL Thread.Fork(NEW(Thread.Closure, apply := BackgroundThread));
    END;
  END StartBackgroundCollection;

(* ------------------------------- low-level allocation and collection *)

(* We assume that references (values of the types ADDRESS and REFANY) are
   the addresses of addressable locations and that locations with
   successive addresses are contiguous (that is, if a points to a
   n-locations referent then these n locations are at addresses a, a+1,
   ..., a+n-1).

   The memory is viewed as a collection of pages.  Each page has a number
   that identifies it, based on the addresses that are part of this page:
   page p contains the addresses p * BytesPerPage to (p+1) * BytesPerPage -
   1.

   The page size must be a multiple of the header size (see below).  Given
   our conventions about page boundaries, this implies that the first
   location of a page is properly aligned for a Header. *)

(* The array desc and the global variables p0, and p1 describe the pages
   that are part of the traced heap.  Either p0 and p1 are equal to Nil and
   no pages are allocated; or both are valid pages and page p is allocated
   iff

|          p0 <= p < p1
|      AND desc[p - p0] != Unallocated

   NUMBER (desc) must be equal to p1 - p0 if there are allocated pages.
   Index i in desc correspond to page i + p0; that is p0 is the number of
   the first page available in desc, and it must be in [p0 ..  p1) if there
   are allocated pages. *)

(* We keep the number of allocated pages in a global variable; it should
   satify the invariant:

|     allocatedPages = sigma (i = p0, p1-1,
|                              space [i - p0] # Unallocated)
|                                  if there are allocated pages,
|                      = 0 otherwise.

   We also keep the number of active pages in a global; it satisfies:

|     activePages = sigma (i = p0, p1-1,
|                           space [i - p0] = nextSpace)
|                                if there are allocated pages,
|                 = 0 otherwise. *)

(* Each referent is immediately preceded by a header that describes the
   type of the referent.  In the user world, this header is not visible;
   that is, a REFANY is the address of the referent, not the address of the
   header.

   Each referent is immediately followed by padding space so the combined
   size referent size + padding is a multiple of the header size.
   Actually, the low level routines are given a data size which is the sum
   of the referent size and padding size and assume this data size is a
   multiple of the header size.

   With this padding, addresses of headers and referent will always be
   multiple of ADRSIZE (Header).

   The combination of header/referent/padding space is called a "heap
   object".  The size of a heap object is the size of the header, plus the
   size of the referent, plus the size of the padding.  The alignment of a
   heap object is the greatest of the alignment of header and the alignment
   of the referent.

   We make the following assumptions:

   - alignment of headers is such what the addressable location following
   any properly aligned header is properly aligned for the type ADDRESS;
   and, for every referent: referent adrSize + padding adrSize >= ADRSIZE
   (ADDRESS)

   [During the garbage collection, we move heap objects.  But we need to
   keep the forwarding information somewhere.  This condition ensures that
   we can store the new address of the referent in the first word of the
   old referent.]

   - the pages are aligned more strictly than the headers (this means that
   the page size is a multiple of the header alignment).

   [We can put a header at the beginning of a page] *)

TYPE
  RefReferent = ADDRESS;

PROCEDURE HeaderOf (r: RefReferent): RefHeader =
  BEGIN
    RETURN LOOPHOLE(r - ADRSIZE(Header), RefHeader);
  END HeaderOf;

(* If a page is allocated, it can be normal or continued.  In the first
   case, there is a heap object just at the beginning of the page and
   others following.  The second case occurs when a heap object was too
   large to fit on a page: it starts at the beginning of a normal page and
   overflows on contiguous continued pages.  Whatever space is left on the
   last continued page is never used for another object or filler.  In
   other words, all the headers are on normal pages.

   Heap objects do not need to be adjacent.  Indeed, alignment constraints
   would make it difficult to ensure that property.  Filler objects may
   appear before objects to align them, or after the last object on a
   normal page to fill the page. *)

(* We need to be able to determine the size of an referent during
   collection; here is a functions to do just that.  It must be called with
   a non-nil pointer to the Header of a heap object that is there (has not
   been moved). *)

PROCEDURE OpenArraySize(h: RefHeader; def: TypeDefn): CARDINAL =
    (* the referent is an open array; it has the following layout:
|         pointer to the elements (ADDRESS)
|         size 1
|         ....
|         size n
|         optional padding
|         elements
|         ....
       where n is the number of open dimensions (given by the definition)
       and each size is the number of elements along the dimension *)
  VAR
    res: INTEGER;
    sizes: UNTRACED REF INTEGER := h + ADRSIZE(Header) + ADRSIZE(ADDRESS);
                                                           (* ^ elt pointer*)
  BEGIN
    res := 1;
    FOR i := 0 TO def.nDimensions - 1 DO
      res := res * sizes^;
      INC(sizes, ADRSIZE(sizes^));
    END;
    res := res * def.elementSize;
    res := RTMisc.Upper(res + def.dataSize, BYTESIZE(Header));
    RETURN res;
  END OpenArraySize;

PROCEDURE ReferentSize (h: RefHeader): CARDINAL =
  VAR
    res: INTEGER;
    tc: Typecode := h.typecode;
    def: TypeDefn;
  BEGIN
    IF tc = Fill_1_type THEN RETURN 0; END;
    IF tc = Fill_N_type THEN
      res := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
      RETURN res - BYTESIZE(Header);
    END;
    def := RTType.Get (tc);
    IF def.nDimensions = 0 THEN
      (* the typecell datasize tells the truth *)
      RETURN def.dataSize;
    END;
    (* ELSE, the referent is an open array *)
    RETURN OpenArraySize(h, def);
  END ReferentSize;

(* The convention about page numbering allows for a simple conversion from
   an address to the number of the page in which it is, as well as from a
   page number to the first address is contains: *)

PROCEDURE ReferentToPage (r: RefReferent): Page =
  (* VAR p: INTEGER := LOOPHOLE(r, INTEGER) DIV BytesPerPage; *)
  VAR p: INTEGER := Word.RightShift (LOOPHOLE(r, INTEGER), LogBytesPerPage);
  BEGIN
    IF p < p0 OR p >= p1 OR desc[p - p0].space = Space.Unallocated
      THEN RETURN Nil;
      ELSE RETURN p;
    END;
  END ReferentToPage;

PROCEDURE HeaderToPage (r: RefHeader): Page =
  (* VAR p: INTEGER := LOOPHOLE(r, INTEGER) DIV BytesPerPage; *)
  VAR p: INTEGER := Word.RightShift (LOOPHOLE(r, INTEGER), LogBytesPerPage);
  BEGIN
    IF p < p0 OR p >= p1 OR desc[p - p0].space = Space.Unallocated
      THEN RETURN Nil;
      ELSE RETURN p;
    END;
  END HeaderToPage;

PROCEDURE PageToHeader (p: Page): RefHeader =
  BEGIN
    RETURN LOOPHOLE(p * BytesPerPage, RefHeader);
  END PageToHeader;

PROCEDURE PageToAddress (p: Page): ADDRESS =
  BEGIN
    RETURN LOOPHOLE(p * BytesPerPage, ADDRESS);
  END PageToAddress;

PROCEDURE RefPageMap (object: REFANY;
                      VAR mapDB: RTDB.T;
                      VAR mapId: Page): BOOLEAN =
  VAR
    p := ReferentToPage(LOOPHOLE(object, RefReferent));
  BEGIN
    IF p = Nil OR mapDBs[p - p0] = NIL THEN
      RETURN FALSE;
    ELSE
      mapDB := mapDBs[p - p0];
      mapId := mapIds[p - p0];
      RETURN TRUE;
    END;
  END RefPageMap;

(* We remember where we should look for free space with the following
   globals: *)

(* The inner-loop collector action is to pick a gray page and completely
   clean it (i.e., make its referents at least gray, so that the page
   becomes black).  The current gray page, "impureCopy.page" is
   distinguished; it's the page that newly gray objects are copied to.

   To improve locality of referene in the new space, we keep the set of
   gray pages as a stack.  This helps approximate a depth-first copy to
   newspace.  The current page is not a member of the stack, but will
   become one when it becomes full.  The current page is always the page
   that contains new.ptr.

   To reduce page faults, we separate the "pure" copy pages (those whose
   objects contain no REFs) from the "impure" ones (those with REFs).  Only
   impure pages become gray, since pure pages can have no REFs into the old
   space (since they have no REFs at all). *)

(* By analogy, we also maintain "pureCopy.page" and "pureCopy.stack".  These
   are not used, but maintaining them simplifies the code. *)

(* By analogy, we also maintain "new.page" and "new.stack".  As with
   pureCopy.page and pureCopy.stack, these are not used, but maintaining them
   simplifies the code. *)

TYPE
  AllocRec = RECORD
    ptr, boundary: RefHeader;
    page:  Page := Nil; (* the current page *)
    stack: Page := Nil; (* threaded through the "link" field; ends at Nil *)
    db: RTDB.T;
  END;

VAR
  new: AllocRec;
  (* memory in [new.ptr, new.boundary) is available to AllocForNew *)

  newTransient: AllocRec;
  (* memory in [newTransient.ptr, newTransient.boundary) is available to
     AllocForNewTransient *)

  pureCopy: AllocRec;
  (* memory in [pureCopy.ptr, pureCopy.boundary) is available to AllocForCopy
     for pure objects (objects with no REFs) *)

  impureCopy: AllocRec;
  (* memory in [impureCopy.ptr, impureCopy.boundary) is available to
     AllocForCopy for impure objects (objects with REFs) *)

  pureTransientCopy: AllocRec;
  (* memory in [pureTransientCopy.ptr, pureTransientCopy.boundary) is available
     to AllocForCopy for pure objects (objects with no REFs) *)

  impureTransientCopy: AllocRec;
  (* memory in [impureTransientCopy.ptr, impureTransientCopy.boundary) is
     available to AllocForCopy for impure objects (objects with
     REFs) *)

(* To move a heap object to the new space, modifying the original
   reference to it *)

TYPE Mover = RTHeapMap.Visitor OBJECT OVERRIDES apply := Move END;

PROCEDURE Move (<*UNUSED*> self: Mover;  cp: ADDRESS) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil THEN RETURN; END;
      VAR
        pi        := p - p0;
        oldHeader := HeaderOf(ref);
      BEGIN
        IF desc[pi].space # Space.Previous THEN
          RETURN;                (* nothing to do *)
        END;
        IF desc[pi].persistent THEN
          (* if this is a persistent object, just promote the pages *)
          IF desc[pi].pure THEN
            PromotePage(
                p, Desc{space := Space.Current, generation := copyGeneration,
                        pure := TRUE, note := Note.Persistent,
                        gray := FALSE, protected := desc[pi].protected,
                        continued := FALSE, persistent := TRUE,
                        resident := desc[pi].resident,
                        modified := desc[pi].modified});
          ELSE
            PromotePage(
                p, Desc{space := Space.Current, generation := copyGeneration,
                        pure := FALSE, note := Note.Persistent,
                        gray := TRUE, protected := desc[pi].protected,
                        continued := FALSE, persistent := TRUE,
                        resident := TRUE, modified := desc[pi].modified});
            desc[pi].link := impureCopy.stack;
            impureCopy.stack := p;
          END;
        ELSIF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if this is a large object, just promote the pages *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := TRUE, note := Note.Large, gray := FALSE,
                          protected := FALSE, continued := FALSE,
                          persistent := FALSE, resident := TRUE,
                          modified := TRUE});
            ELSE
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := FALSE, note := Note.Large, gray := TRUE,
                          protected := FALSE, continued := FALSE,
                          persistent := FALSE, resident := TRUE,
                          modified := TRUE});
              IF def.traced = 1 THEN
                desc[pi].link := impureCopy.stack;
                impureCopy.stack := p;
              ELSE
                <*ASSERT def.traced = 3*>
                desc[pi].link := impureTransientCopy.stack;
                impureTransientCopy.stack := p;
              END;
            END;
          END;
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF (def.gc_map # NIL) OR (def.parent # NIL) THEN
              IF def.traced = 1 THEN
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   impureCopy,
                                   pure := FALSE);
              ELSE
                <* ASSERT def.traced = 3 *>
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   impureTransientCopy,
                                   pure := FALSE);
              END
            ELSE
              IF def.traced = 1 THEN
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   pureCopy,
                                   pure := TRUE);
              ELSE
                <* ASSERT def.traced = 3 *>
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   pureTransientCopy,
                                   pure := TRUE);
              END
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END;
        END;
      END;
    END;
  END Move;

TYPE TMover = RTHeapMap.Visitor OBJECT OVERRIDES apply := TMove END;

PROCEDURE TMove (<*UNUSED*> self: TMover; cp: ADDRESS) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil THEN RETURN; END;
      VAR
        pi        := p - p0;
        oldHeader := HeaderOf(ref);
      BEGIN
        IF desc[pi].space # Space.Previous THEN
          RETURN;                (* nothing to do *)
        END;
        IF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if this is a large object, just promote the pages *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF def.traced # 3 THEN
              RETURN;
            END;
            <* ASSERT NOT desc[pi].persistent *>
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := TRUE, note := Note.Large, gray := FALSE,
                          protected := FALSE, continued := FALSE,
                          persistent := FALSE, resident := TRUE,
                          modified := TRUE});
            ELSE
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := FALSE, note := Note.Large, gray := TRUE,
                          protected := FALSE, continued := FALSE,
                          persistent := FALSE, resident := TRUE,
                          modified := TRUE});
              desc[pi].link := impureTransientCopy.stack;
              impureTransientCopy.stack := p;
            END;
          END;
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF def.traced # 3 THEN
              RETURN;
            END;
            <* ASSERT NOT desc[pi].persistent *>
            IF (def.gc_map # NIL) OR (def.parent # NIL) THEN
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 impureTransientCopy,
                                 pure := FALSE);
            ELSE
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 pureTransientCopy,
                                 pure := TRUE);
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END;
        END;
      END;
    END;
  END TMove; 

TYPE PMover = RTTypeMap.Visitor OBJECT
  db: RTDB.T
OVERRIDES
  apply := PMove
END;

PROCEDURE PMove (self: PMover; cp: ADDRESS; k: RTTypeMap.Kind) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    <* ASSERT k = RTTypeMap.Kind.Ref *>
    VAR p := ReferentToPage(ref);
        oldHeader := HeaderOf(ref);
    BEGIN
      IF p = Nil THEN (* must be statically allocated text constant *)
        <* ASSERT oldHeader.typecode = RT0.TextTypecode *>
        refref^ := LOOPHOLE(self.db.text(LOOPHOLE(ref, TEXT)), RefReferent);
        RETURN;
      END;
      VAR
        pi := p - p0;
      BEGIN
        IF desc[pi].space = Space.Current THEN
          IF desc[pi].persistent THEN
            (* check for cross file reference *)
            <* ASSERT self.db = mapDBs[pi] *>
          ELSE
            (* page must be pinned by ambiguous root, make it persistent *)
            <* ASSERT desc[pi].note = Note.AmbiguousRoot *>
            VAR def := RTType.Get (oldHeader.typecode);
            BEGIN
              IF def.traced # 1 THEN RETURN; END;
            END;
            VAR
              n := PageCount(p);
              pp := self.db.newPages(p, n);
            BEGIN
              FOR i := 0 TO n-1 DO
                desc[pi + i].gray := TRUE;
                desc[pi + i].persistent := TRUE;
                mapIds[pi + i] := pp + i;
                mapDBs[pi + i] := self.db;
              END;
              IF desc[pi].pure THEN
                desc[pi].link := pureCopy.stack;
                pureCopy.stack := p;
              ELSE
                desc[pi].link := impureCopy.stack;
                impureCopy.stack := p;
              END;
              IF perfOn THEN PerfChange(p, n); END;
              RETURN;
            END;
          END;
          RETURN;
        END;
        <* ASSERT desc[pi].space = Space.Previous
              AND NOT desc[pi].persistent *>
        IF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if large, make persistent and promote to current space *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF def.traced # 1 THEN RETURN; END;
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              PromotePage(
                  p, Desc{space := Space.Current,
                          generation := copyGeneration, pure := TRUE,
                          note := Note.Large, gray := TRUE,
                          protected := FALSE, continued := FALSE,
                          persistent := TRUE, resident := TRUE,
                          modified := TRUE});
              desc[pi].link := pureCopy.stack;
              pureCopy.stack := p;
            ELSE
              PromotePage(
                p, Desc{space := Space.Current,
                        generation := copyGeneration, pure := FALSE,
                        note := Note.Large, gray := TRUE,
                        protected := FALSE, continued := FALSE,
                        persistent := TRUE, resident := TRUE,
                        modified := TRUE});
              desc[pi].link := impureCopy.stack;
              impureCopy.stack := p;
            END;
            VAR
              n := PageCount(p);
              pp := self.db.newPages(p, n);
            BEGIN
              FOR i := 0 TO n-1 DO
                mapIds[pi + i] := pp + i;
                mapDBs[pi + i] := self.db;
              END;
            END;
          END;
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF def.traced # 1 THEN RETURN; END;
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 pureCopy, pure := TRUE);
            ELSE
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 impureCopy, pure := FALSE);
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END
        END
      END
    END
  END PMove;

(* Determines whether a REF has yet been moved into the new space.  Follows
   the logic in "Move".*)

PROCEDURE Moved (ref: RefReferent): BOOLEAN =
  BEGIN
    IF ref = NIL THEN RETURN TRUE; END;
    (* check the space *)
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil OR desc[p - p0].space # Space.Previous THEN
        RETURN TRUE;
      END;
    END;
    (* check the forwarded bit *)
    IF HeaderOf(LOOPHOLE(ref, ADDRESS)).forwarded THEN RETURN TRUE; END;
    (* not moved *)
    RETURN FALSE;
  END Moved;

(* When an allocated page is referenced by the stack, we have to move it to
   the next space and insert it in the list of promoted pages.  In the case
   where the page is actually part of a group of pages for a big referent,
   we have to promote all these pages to the new space, but only the first
   one needs to be inserted in the queue, as it is the only one containing
   referent headers.

   This routine is passed to the Threads implementation.  It is called for
   each stack, where start and stop are the addresses of the first and last
   word of the stack under consideration. *)

PROCEDURE NoteStackLocations (start, stop: ADDRESS) =
  VAR
    fp                                := start;
    firstAllocatedAddress             := PageToAddress(p0);
    firstNonAllocatedAddress          := PageToAddress(p1);
    p                       : ADDRESS;
    pp                      : Page;
  BEGIN
    WHILE fp <= stop DO
      p := LOOPHOLE(fp, UNTRACED REF ADDRESS)^;
      IF firstAllocatedAddress <= p AND p < firstNonAllocatedAddress THEN
        pp := LOOPHOLE(p, INTEGER) DIV BytesPerPage;
        IF desc[pp - p0].space = Space.Previous THEN
          VAR fp := FirstPage(pp);
          BEGIN
            IF desc[fp - p0].pure THEN
              PromotePage(fp, Desc{space := Space.Current, pure := TRUE,
                                   note := Note.AmbiguousRoot, gray := FALSE,
                                   generation := copyGeneration,
                                   protected := desc[fp-p0].protected,
                                   continued := FALSE,
                                   persistent := desc[fp-p0].persistent,
                                   resident := desc[fp-p0].resident,
                                   modified := desc[fp-p0].modified});
            ELSE
              PromotePage(fp, Desc{space := Space.Current, pure := FALSE,
                                   note := Note.AmbiguousRoot, gray := TRUE,
                                   generation := copyGeneration,
                                   protected := desc[fp-p0].protected,
                                   continued := FALSE,
                                   persistent := desc[fp-p0].persistent,
                                   resident := TRUE,
                                   modified := desc[fp-p0].modified});
              desc[fp - p0].link := impureTransientCopy.stack;
              impureTransientCopy.stack := fp;
            END;
          END;
        END;
      END;
      INC(fp, RTMachine.PointerAlignment);
    END;
  END NoteStackLocations;

PROCEDURE PromotePage (p: Page;  READONLY d: Desc) =
  BEGIN
    <* ASSERT desc[p - p0].space = Space.Previous *>
    <* ASSERT NOT desc[p - p0].continued*>
    VAR n := PageCount(p);
    BEGIN
      desc[p - p0] := d;
      IF n > 1 THEN
        VAR dd := d;
        BEGIN
          dd.continued := TRUE;
          FOR pp := p + 1 TO p + n - 1 DO desc[pp - p0] := dd; END;
        END;
      END;
      IF perfOn THEN PerfChange(p, n); END;
      IF d.space = Space.Current THEN
        IF n = 1 THEN
          INC(smallPromotionPages, 1);
        ELSE
          INC(largePromotionPages, n);
        END;
      END;
    END;
  END PromotePage;

PROCEDURE InsertFiller (start: RefHeader; n: INTEGER) =
  BEGIN
    IF n = 0 THEN
      (* nothing to do *)
    ELSIF n = ADRSIZE(Header) THEN
      start^ := FillHeader1;
    ELSIF n >= ADRSIZE(Header) + ADRSIZE(INTEGER) THEN
      start^ := FillHeaderN;
      LOOPHOLE(start + ADRSIZE(Header), UNTRACED REF INTEGER)^ := n;
    ELSE
      <* ASSERT FALSE *>
    END;
  END InsertFiller;

PROCEDURE Fill (VAR (*INOUT*) current: AllocRec) =
  BEGIN
    InsertFiller(current.ptr, current.boundary - current.ptr);
    current.page := Nil;
    current.stack := Nil;
    current.ptr := NIL;
    current.boundary := NIL;
    <* ASSERT current.db = NIL *>
  END Fill;

PROCEDURE Close (VAR (*INOUT*) current: AllocRec; gray := FALSE) =
  BEGIN
    InsertFiller(current.ptr, current.boundary - current.ptr);
    IF current.page # Nil THEN
      desc[current.page - p0].gray := gray;
      IF perfOn THEN PerfChange(current.page, 1); END;
      IF NOT desc[current.page - p0].pure
        AND desc[current.page - p0].generation = Generation.Older
       THEN
        <* ASSERT desc[current.page - p0].modified *>
        <* ASSERT desc[current.page - p0].space = Space.Current *>
        Protect(current.page, readable := TRUE, writable := FALSE);
      END;
      current.page := Nil;
    END;
    <* ASSERT current.stack = Nil *>
    current.ptr := NIL;
    current.boundary := NIL;
  END Close;

PROCEDURE Push (VAR (*INOUT*) current: AllocRec; db: RTDB.T) =
  BEGIN
    IF current.db # db THEN
      IF current.page # Nil THEN
        InsertFiller(current.ptr, current.boundary - current.ptr);
        desc[current.page - p0].link := current.stack;
        current.stack := current.page;
        current.page := Nil;
        current.ptr := NIL;
        current.boundary := NIL;
      END;
      current.db := db;
    END;
  END Push;    

PROCEDURE SetDB (db: RTDB.T) =
  BEGIN
    pmover.db := db;
    Push(new, db);
    Push(pureCopy, db);
    Push(impureCopy, db);
  END SetDB;

TYPE CollectorState = {Zero, One, Two, Three, Four, Five};

VAR collectorState := CollectorState.Zero;

VAR
  threshold := ARRAY [0 .. 1] OF
                 REAL{FLOAT(InitialBytes DIV 4 DIV BytesPerPage - 1), 1.0};
  (* start a collection as soon as current space reaches threshold[0] /
     threshold[1] pages; the initial value is 64KB *)

  partialCollection: BOOLEAN;
  (* whether the collection in progress is partial, involving only the newer
     generation *)

  partialCollectionNext: BOOLEAN := FALSE;
  (* whether the next collection should be partial *)

  collectorOn: BOOLEAN := FALSE;

  copyGeneration: Generation := Generation.Younger;

  signalBackground := FALSE;
  (* should signal background collector thread? *)

  signalWeak := FALSE;
  (* should signal weak cleaner thread? *)

TYPE Unswizzler = RTTypeMap.Visitor OBJECT
  mapDB: RTDB.T;
  p: Page;
  data: ADDRESS;
OVERRIDES
  apply := Unswizzle
END;

PROCEDURE Unswizzle (self: Unswizzler; cp: ADDRESS;
                     <*UNUSED*> k: RTTypeMap.Kind) =
  VAR refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
      ref := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    VAR
      p := Word.RightShift(LOOPHOLE(refref, INTEGER), LogBytesPerPage);
      pos := Word.And(LOOPHOLE(refref, INTEGER), BytesPerPage - 1);
    BEGIN
      IF p # self.p THEN
        <* ASSERT self.p # Nil *>
        self.mapDB.unpin(mapIds[self.p - p0]);
        self.p := p;
        self.data := self.mapDB.pin(mapIds[p - p0], load := FALSE);
        RTMisc.Copy(PageToAddress(p), self.data, BytesPerPage);
      END;
      LOOPHOLE(self.data + pos, UNTRACED REF RefReferent)^ :=
          UnswizzleRef(self.mapDB, ref);
    END;
  END Unswizzle;

PROCEDURE UnswizzleRef (db: RTDB.T; ref: RefReferent): RefReferent =
  VAR p := ReferentToPage(ref);
  BEGIN
    IF p = Nil OR NOT desc[p - p0].persistent THEN RETURN NIL; END;
    VAR offset := ref - PageToAddress(p);
        mapDB := mapDBs[p - p0];
        nbPages := PageCount(p);
    BEGIN
      <* ASSERT Word.And(offset, 3) = 0 *>
      <* ASSERT db = mapDB *>
      IF nbPages > 1 THEN
        CASE offset OF
        |   4 => offset := 2_0001;
        |   8 => offset := 2_0011;
        |  16 => offset := 2_0101;
        |  32 => offset := 2_0111;
        |  64 => offset := 2_1001;
        | 128 => offset := 2_1011;
        | 256 => offset := 2_1101;
        | 512 => offset := 2_1111;
        ELSE
          <* ASSERT FALSE *>
        END;
        IF nbPages >= Word.RightShift(BytesPerPage, 4) THEN
          nbPages := 0;
        ELSE
          nbPages := Word.LeftShift(nbPages, 4);
        END;
        INC(offset, nbPages);
      END;
      p := mapIds[p - p0];
      RETURN PageToAddress(p) + offset;
    END
  END UnswizzleRef;

PROCEDURE UnswizzleBetween (h, he: RefHeader; new: BOOLEAN) =
  VAR
    tc: Typecode;
    def: TypeDefn;
    fp: RefReferent;
    referentSize: CARDINAL;
    p: Page;
    pos: INTEGER;
    mapDB: RTDB.T;
  BEGIN
    IF h < he THEN
      p := Word.RightShift(LOOPHOLE(h, INTEGER), LogBytesPerPage);
      mapDB := mapDBs[p - p0];
      unswizzler.mapDB := mapDB;
      unswizzler.p := Nil;
      REPEAT
        <* ASSERT NOT h.forwarded *>

        p := Word.RightShift(LOOPHOLE(h, INTEGER), LogBytesPerPage);
        pos := Word.And(LOOPHOLE(h, INTEGER), BytesPerPage - 1);
        IF p # unswizzler.p THEN
          IF unswizzler.p # Nil THEN
            mapDB.unpin(mapIds[unswizzler.p - p0]);
          END;
          unswizzler.p := p;
          unswizzler.data := mapDB.pin(mapIds[p - p0], load := FALSE);
          RTMisc.Copy(PageToAddress(p), unswizzler.data, BytesPerPage);
        END;

        tc := h.typecode;
        IF tc = Fill_1_type THEN
          LOOPHOLE(unswizzler.data + pos, UNTRACED REF ADDRESS)^ := 
              LOOPHOLE(0, ADDRESS);
          referentSize := 0;
        ELSIF tc = Fill_N_type THEN
          LOOPHOLE(unswizzler.data + pos, UNTRACED REF ADDRESS)^ :=
              LOOPHOLE(1, ADDRESS);
          referentSize := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
          DEC(referentSize, ADRSIZE(Header));
        ELSE
          def := RTType.Get(tc);
          <* ASSERT def.traced = 1 *>
          fp := LOOPHOLE(mapDB.getFP(ADR(def.fp)), RefReferent);
          LOOPHOLE(unswizzler.data + pos, UNTRACED REF ADDRESS)^ :=
              UnswizzleRef(mapDB, fp);
          IF def.nDimensions = 0 THEN
            referentSize := def.dataSize;
          ELSE
            referentSize := OpenArraySize(h, def);
          END;
          IF new AND stats.print THEN
            WITH z = stats.objects[tc] DO z := Word.Plus(z, 1); END;
            WITH z = stats.bytes[tc] DO
              z := Word.Plus(z, BYTESIZE(Header) + referentSize)
            END;
          END;
          IF def.gc_map # NIL OR def.parent # NIL THEN
            <*FATAL ANY*>
            BEGIN
              RTTypeMap.DoWalkRef(def, h + ADRSIZE(Header),
                                  RTTypeMap.Mask{RTTypeMap.Kind.Ref},
                                  unswizzler);
            END
          END
        END;
        INC(h, ADRSIZE(Header) + referentSize);
      UNTIL NOT h < he;
      <* ASSERT unswizzler.p # Nil *>
      mapDB.unpin(mapIds[unswizzler.p - p0]);
    END
  END UnswizzleBetween;

PROCEDURE Stabilize (bootstrapDB: RTDB.T) =
  BEGIN
    (* compute some costs relative to previous collection *)
    INC(cycleNews, smallNewPages + largeNewPages);
    VAR prefixAvgCost := cycleCost / FLOAT(cycleNews);
    BEGIN
      IF prefixAvgCost < minPrefixAvgCost THEN
        minPrefixAvgCost := prefixAvgCost;
        minCycleL := cycleL;
      END;
    END;

    (* we want to do full collection *)
    copyGeneration := Generation.Younger;
    partialCollection := FALSE;
    partialCollectionNext := TRUE;
    
    (* not partial collection *)
    cycleL := 1;
    cycleCost := 0.0;
    cycleNews := 0;
    minPrefixAvgCost := LAST(REAL);
    minCycleL := 0;

    InvokeMonitors (before := TRUE);

    IF perfOn THEN PerfBegin(); END;

    (* fill the rest of the current page *)
    Fill(new);
    Fill(newTransient);

    INC(collections);

    (* flip spaces; newspace becomes oldspace *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].space = Space.Current THEN
        desc[p - p0].space := Space.Previous;
        IF perfOn THEN PerfChange(p, 1); END;
      END;
    END;

    IF perfOn THEN PerfFlip(); END;

    (* The 'new' nextSpace is empty *)
    smallNewPages := 0;
    largeNewPages := 0;
    smallCopyPages := 0;
    largeCopyPages := 0;
    smallPromotionPages := 0;
    largePromotionPages := 0;

    FOR p := p0 TO p1 - 1 DO
      VAR d := desc[p - p0];
      BEGIN
        IF d.space = Space.Previous AND NOT d.continued THEN
          IF d.persistent THEN
            (* Promote ALL persistent pages: stabilize may trigger faults
               when accessing meta-data, which may swizzle references to
               previously unreachable pages, so we need to make sure these
               pages are retained.  Also, updates to persistent pages during
               closure copying need to be captured when unswizzling. *)
            IF d.resident AND d.modified THEN
              IF d.pure THEN
                <* ASSERT NOT d.protected *>
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration,
                            pure := TRUE, note := Note.Persistent,
                            gray := TRUE, protected := FALSE,
                            continued := FALSE, persistent := TRUE});
                desc[p - p0].link := pureCopy.stack;
                pureCopy.stack := p;
              ELSE
                IF d.protected THEN Unprotect(p); END;
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration,
                            pure := FALSE, note := Note.Persistent,
                            gray := TRUE, protected := FALSE,
                            continued := FALSE, persistent := TRUE});
                desc[p - p0].link := impureCopy.stack;
                impureCopy.stack := p;
              END
            ELSE
              <* ASSERT d.protected *>
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := d.pure, note := Note.Persistent,
                          gray := FALSE, protected := TRUE, continued := FALSE,
                          persistent := TRUE, resident := d.resident,
                          modified := d.modified});
            END;
          ELSIF d.generation = Generation.Older THEN
            <* ASSERT d.modified *>
            IF d.protected THEN Unprotect(p); END;
          ELSE
            <* ASSERT NOT d.protected *>
          END
        END
      END
    END;

    (* get these in the new space *)
    mover := NEW (Mover);
    swizzler := NEW (Swizzler);
    unswizzler := NEW (Unswizzler);
    tmover := NEW (TMover);
    pmover := NEW (PMover);

    (* mark from roots *)
    ThreadF.SuspendOthers();
    BEGIN
      (* Examine the stacks for possible pointers *)
      ThreadF.ProcessStacks(NoteStackLocations);

      <* ASSERT impureTransientCopy.page = Nil *>
      <* ASSERT impureTransientCopy.ptr = NIL *>
      <* ASSERT impureTransientCopy.boundary = NIL *>

      <* ASSERT pureTransientCopy.page = Nil *>
      <* ASSERT pureTransientCopy.stack = Nil *>
      <* ASSERT pureTransientCopy.ptr = NIL *>
      <* ASSERT pureTransientCopy.boundary = NIL *>

      <* ASSERT impureCopy.page = Nil *>
      <* ASSERT impureCopy.ptr = NIL *>
      <* ASSERT impureCopy.boundary = NIL *>

      <* ASSERT pureCopy.page = Nil *>
      <* ASSERT pureCopy.ptr = NIL *>
      <* ASSERT pureCopy.boundary = NIL *>

      (* fill current page in preparation for persistent allocations *)
      Fill(new);

      (* Create root if this is a new database *)
      IF bootstrapDB # NIL THEN
        SetDB(bootstrapDB);
        bootstrapDB.createRoot();
        SetDB(NIL);
      END;

      (* Copy transient closure from globals and user-level database handles,
         since we may need them to access the database when unswizzling *)
      RTHeapMap.WalkGlobals(tmover);
      FOR p := p0 TO p1 - 1 DO
        IF mapDBs[p - p0] # NIL THEN
          TMove (NIL, ADR(mapDBs[p - p0]));
        END;
      END;
      WHILE CleanSome(impureTransientCopy, CleanTransient, gray:= TRUE) DO END;
      Close(impureTransientCopy, gray := TRUE);
      tmover := NIL;

      (* Copy persistence closure *)
      WHILE CleanSome(impureCopy, CleanPersistent, gray := TRUE) DO END;
      Close(impureCopy, gray := TRUE);

      (* now allocate metadata for pure pages *)
      WHILE CleanSome(pureCopy, CleanPersistent) DO END;
      Close(pureCopy);

      (* now the metadata itself *)
      WHILE CleanSome(new, CleanPersistent) DO END;
      Close(new);

      <* ASSERT impureTransientCopy.stack = Nil *>
      <* ASSERT impureTransientCopy.page = Nil *>
      <* ASSERT impureTransientCopy.ptr = NIL *>
      <* ASSERT impureTransientCopy.boundary = NIL *>

      <* ASSERT impureCopy.stack = Nil *>
      <* ASSERT impureCopy.page = Nil *>
      <* ASSERT impureCopy.ptr = NIL *>
      <* ASSERT impureCopy.boundary = NIL *>

      <* ASSERT pureCopy.page = Nil *>
      <* ASSERT pureCopy.stack = Nil *>
      <* ASSERT pureCopy.ptr = NIL *>
      <* ASSERT pureCopy.boundary = NIL *>

      SetDB(NIL);
      pmover := NIL;

      (* All modified persistent pages are in Space.Current; unswizzle *)
      FOR  i := 0 TO p1 - p0 - 1 DO
        VAR d := desc[i];
        BEGIN
          IF d.space = Space.Current AND NOT d.continued THEN
            IF d.persistent THEN
              IF d.resident THEN
                IF d.modified THEN
                  VAR
                    p := p0 + i;
                    n := PageCount(p);
                  BEGIN
                    IF d.note = Note.AmbiguousRoot AND stats.print THEN
                      INC(stats.ambiguousPages, n);
                    ELSIF d.note # Note.Persistent AND stats.print THEN
                      INC(stats.accuratePages, n);
                    END;
                    UnswizzleBetween(PageToHeader(p), PageToHeader(p + 1),
                                     new := d.note # Note.Persistent);
                    d.modified := FALSE;
                    desc[i] := d;
                    VAR dd := d;
                    BEGIN
                      dd.link := Nil;
                      dd.continued := TRUE;
                      FOR ii := 1 TO n-1 DO
                        desc[i + ii] := dd;
                      END
                    END;
                    <*ASSERT NOT d.protected*>
                    Protect(p, readable := TRUE, writable := FALSE);
                  END
                ELSE (* NOT d.modified *)
                  <* ASSERT d.protected *>
                END
              ELSE (* NOT d.resident *)
                <* ASSERT d.pure AND d.protected *>
              END
            END;
            IF d.gray THEN
              <* ASSERT d.resident AND NOT d.pure *>
              d.link := impureCopy.stack;
              desc[i] := d;
              impureCopy.stack := p0 + i;
            END
          END
        END
      END;
      unswizzler := NIL;

      IF stats.print THEN
        PutText("\nAccurate pages:  "); PutInt(stats.accuratePages);
        PutText("\nAmbiguous pages: "); PutInt(stats.ambiguousPages);
        PutText("\n");
        stats.accuratePages := 0; stats.ambiguousPages := 0;
        FOR i := 0 TO RT0u.nTypes - 1 DO
          IF stats.objects[i] # 0 THEN
            VAR t := RTType.Get(i);
                str: ADDRESS := t.name;
            BEGIN
              IF str # NIL THEN
                PutChars(str, Cstring.strlen(str));
              ELSE
                PutText("<tc="); PutInt(i); PutText(">");
              END;
            END;
            PutText(": ");
            PutInt(stats.objects[i]); PutText(" objects ");
            PutInt(stats.bytes[i]); PutText(" bytes\n");
            stats.objects[i] := 0;
            stats.bytes[i] := 0;
          END;
        END;
        RTIO.Flush();
      END;

      (* Examine the global variables for possible pointers *)
      RTHeapMap.WalkGlobals (mover);
    END;
    ThreadF.ResumeOthers();

    IF perfOn THEN PerfPromotedRoots(); END;

    collectorState := CollectorState.One;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END Stabilize;

PROCEDURE Invalidate() =
  BEGIN
    ThreadF.SuspendOthers();
    BEGIN
      FOR i := 0 TO p1 - p0 - 1 DO
        VAR d := desc[i];
        BEGIN
          IF d.space = Space.Current OR d.space = Space.Previous THEN
            IF d.persistent THEN
              IF d.resident THEN
                d.modified := FALSE;
                d.resident := FALSE;
                d.gray := FALSE;
                d.pure := TRUE;
                desc[i] := d;
                IF NOT d.continued THEN
                  Protect(p0 + i, readable := FALSE, writable := FALSE);
                ELSE
                  IF perfOn THEN PerfChange(p0 + i, 1); END;
                END
              ELSIF d.modified THEN
                <* ASSERT d.pure AND d.protected AND NOT d.gray *>
                d.modified := FALSE;
                desc[i] := d;
                IF perfOn THEN PerfChange(p0 + i, 1); END;
              ELSE
                <* ASSERT d.pure AND d.protected AND NOT d.gray *>
              END
            END
          END
        END
      END
    END;
    ThreadF.ResumeOthers();
  END Invalidate;

PROCEDURE Release() =
  BEGIN
    ThreadF.SuspendOthers();
    BEGIN
      FOR i := 0 TO p1 - p0 - 1 DO
        VAR d := desc[i];
        BEGIN
          IF d.space = Space.Current OR d.space = Space.Previous THEN
            IF d.persistent THEN
              IF d.resident THEN
                IF NOT d.modified THEN
                  d.modified := TRUE;
                  desc[i] := d;
                  IF d.continued THEN
                    IF perfOn THEN PerfChange(p0 + i, 1); END;
                  ELSIF d.gray THEN
                    <*ASSERT d.protected*>
                    IF perfOn THEN PerfChange(p0 + i, 1); END;
                  ELSIF d.generation = Generation.Older THEN
                    <*ASSERT d.protected AND d.space = Space.Current*>
                    IF perfOn THEN PerfChange(p0 + i, 1); END;
                  ELSE
                    Unprotect(p0 + i);
                  END
                END
              ELSE
                <*ASSERT d.pure AND d.protected AND NOT d.gray*>
              END
            END
          END
        END
      END
    END;
    ThreadF.ResumeOthers();
  END Release;

PROCEDURE Flush (db: RTDB.T): BOOLEAN =
  BEGIN
    RTOS.LockHeap();
    IF disableCount + disableMotionCount > 0 THEN
      RTOS.UnlockHeap();
      RETURN FALSE;
    END;
    BEGIN
      CollectorOn();
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      Stabilize(db);
      IF NOT (incremental AND RTHeapDep.VM AND disableVMCount = 0) THEN
        REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
      END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
    RETURN TRUE;
  END Flush;

PROCEDURE CollectEnough () =
  BEGIN
    IF collectorOn THEN RETURN; END;
    IF Behind() THEN
      CollectorOn();
      IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
        REPEAT CollectSome(); UNTIL NOT Behind();
      ELSE
        WHILE collectorState = CollectorState.Zero DO CollectSome(); END;
        REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
      END;
      CollectorOff();
    END;
  END CollectEnough;

PROCEDURE Behind (): BOOLEAN =
  BEGIN
    IF disableCount + disableMotionCount > 0
         AND collectorState = CollectorState.Zero THEN
      RETURN FALSE;
    END;
    IF collectorState = CollectorState.Zero THEN
      RETURN FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                     + largePromotionPages + smallNewPages + largeNewPages)
               * threshold[1] >= threshold[0];
    ELSE
      RETURN FLOAT(smallNewPages + largeNewPages) * gcRatio
               >= FLOAT(smallCopyPages + largeCopyPages);
    END;
  END Behind;

VAR timeUsedOnEntry: REAL;       (* time used when entered collector *)

PROCEDURE CollectorOn () =
  BEGIN
    (* ASSERT locked *)
    <* ASSERT NOT collectorOn *>
    collectorOn := TRUE;

    IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
      (* The VM-synchronized collector doesn't worry about running threads. *)
    ELSE
      ThreadF.SuspendOthers ();
    END;

    IF RTHeapDep.VM THEN timeUsedOnEntry := RTHeapDep.TimeUsed(); END;
    IF impureCopy.page # Nil THEN
      <* ASSERT desc[impureCopy.page - p0].gray *>
      <* ASSERT desc[impureCopy.page - p0].protected *>
      Unprotect(impureCopy.page);
    END;
    IF impureTransientCopy.page # Nil THEN
      <* ASSERT desc[impureTransientCopy.page - p0].gray *>
      <* ASSERT desc[impureTransientCopy.page - p0].protected *>
      Unprotect(impureTransientCopy.page);
    END;
  END CollectorOn;

PROCEDURE CollectorOff () =
  BEGIN
    (* ASSERT locked *)
    <* ASSERT collectorOn *>
    IF impureCopy.page # Nil THEN
      <* ASSERT desc[impureCopy.page - p0].gray *>
      <* ASSERT NOT desc[impureCopy.page - p0].protected *>
      Protect(impureCopy.page, readable := FALSE, writable := FALSE);
    END;
    VAR p := impureCopy.stack;
    BEGIN
      WHILE p # Nil DO
        IF desc[p - p0].gray
          AND NOT (desc[p - p0].modified AND desc[p - p0].protected) THEN
          Protect(p, readable := FALSE, writable := FALSE);
        END;
        p := desc[p - p0].link;
      END;
    END;
    IF impureTransientCopy.page # Nil THEN
      <* ASSERT desc[impureTransientCopy.page - p0].gray *>
      <* ASSERT NOT desc[impureTransientCopy.page - p0].protected *>
      Protect(impureTransientCopy.page, readable := FALSE, writable := FALSE);
    END;
    VAR p := impureTransientCopy.stack;
    BEGIN
      WHILE p # Nil DO
        IF desc[p - p0].gray
          AND NOT (desc[p - p0].modified AND desc[p - p0].protected) THEN
          Protect(p, readable := FALSE, writable := FALSE);
        END;
        p := desc[p - p0].link;
      END;
    END;

    IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
      (* The VM-synchronized collector doesn't worry about running threads. *)
    ELSE
      ThreadF.ResumeOthers ();
    END;

    collectorOn := FALSE;
    IF signalBackground OR signalWeak THEN
      signalBackground := FALSE;
      signalWeak := FALSE;
      Broadcast();
    END;
    IF RTHeapDep.VM THEN
      cycleCost := cycleCost + (RTHeapDep.TimeUsed() - timeUsedOnEntry);
    END;
  END CollectorOff;

PROCEDURE CollectSome () =
  BEGIN
    <* ASSERT disableCount = 0 *>
    CASE collectorState OF
    | CollectorState.Zero => CollectSomeInStateZero();
    | CollectorState.One => CollectSomeInStateOne();
    | CollectorState.Two => CollectSomeInStateTwo();
    | CollectorState.Three => CollectSomeInStateThree();
    | CollectorState.Four => CollectSomeInStateFour();
    | CollectorState.Five => CollectSomeInStateFive();
    END;
  END CollectSome;

(* Start a collection *)

VAR
  mover      : Mover    := NIL;
  tmover     : TMover   := NIL;
  pmover     : PMover   := NIL;
  swizzler   := NEW(Swizzler);
  unswizzler : Unswizzler := NIL;
  cycleCost  : REAL     := 0.0;  (* running cost of current cycle *)
  cycleLength: CARDINAL := 1;    (* current planned cycle length *)
  cycleL     : CARDINAL := 0;    (* length of current cycle, so far *)
  cycleNews  : CARDINAL;         (* the number of new pages this cycle *)
  minPrefixAvgCost: REAL;        (* minimum average cost for a prefix of
                                    this cycle *)
  minCycleL: CARDINAL;           (* the length of that prefix *)

PROCEDURE CollectSomeInStateZero () =
  BEGIN
    <* ASSERT disableCount + disableMotionCount = 0 *>
    (* compute some costs relative to previous collection *)
    INC(cycleNews, smallNewPages + largeNewPages);
    VAR prefixAvgCost := cycleCost / FLOAT(cycleNews);
    BEGIN
      IF prefixAvgCost < minPrefixAvgCost THEN
        minPrefixAvgCost := prefixAvgCost;
        minCycleL := cycleL;
      END;
    END;

    (* make generational decisions *)
    IF generational AND RTHeapDep.VM AND disableVMCount = 0 THEN
      copyGeneration := Generation.Older;
      partialCollection := partialCollectionNext AND cycleL < cycleLength;
      IF NOT partialCollection THEN
        IF minCycleL = cycleLength THEN
          cycleLength := cycleLength + 1;
        ELSE
          cycleLength := MAX(cycleLength - 1, 1);
        END;
      END;
    ELSE
      copyGeneration := Generation.Younger;
      partialCollection := FALSE;
    END;
    partialCollectionNext := TRUE;

    IF partialCollection THEN
      INC(cycleL);
    ELSE
      cycleL := 1;
      cycleCost := 0.0;
      cycleNews := 0;
      minPrefixAvgCost := LAST(REAL);
      minCycleL := 0;
    END;

    InvokeMonitors (before := TRUE);

    IF perfOn THEN PerfBegin(); END;

    (* fill the rest of the current page *)
    Fill(new);
    Fill(newTransient);

    INC(collections);

    (* flip spaces; newspace becomes oldspace *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].space = Space.Current THEN
        desc[p - p0].space := Space.Previous;
        IF perfOn THEN PerfChange(p, 1); END;
      END;
    END;

    IF perfOn THEN PerfFlip(); END;

    (* The 'new' nextSpace is empty *)
    smallNewPages := 0;
    largeNewPages := 0;
    smallCopyPages := 0;
    largeCopyPages := 0;
    smallPromotionPages := 0;
    largePromotionPages := 0;

    FOR p := p0 TO p1 - 1 DO
      VAR d := desc[p - p0];
      BEGIN
        IF d.space = Space.Previous AND NOT d.continued THEN
          IF d.persistent AND d.resident AND d.modified THEN
            (* promote modified persistent pages to avoid deleting them *)
            IF d.pure OR partialCollection AND d.protected THEN
              PromotePage(
                  p, Desc{space := Space.Current,
                          generation := copyGeneration, pure := d.pure,
                          note := Note.Persistent, gray := FALSE,
                          protected := d.protected, continued := FALSE,
                          persistent := TRUE});
            ELSE (* impure, neither partial collection nor protected *)
              IF d.protected THEN Unprotect(p); END;
              PromotePage(
                  p, Desc{space := Space.Current,
                          generation := copyGeneration, pure := FALSE,
                          note := Note.Persistent, gray := TRUE,
                          protected := FALSE, continued := FALSE,
                          persistent := TRUE});
              desc[p - p0].link := impureCopy.stack;
              impureCopy.stack := p;
            END
          ELSIF d.generation = Generation.Older THEN
            IF partialCollection THEN
              <* ASSERT copyGeneration = Generation.Older *>
              IF d.persistent THEN
                <* ASSERT d.protected *>
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := d.pure,
                            note := Note.OlderGeneration, gray := FALSE,
                            protected := TRUE, continued := FALSE,
                            persistent := TRUE, resident := d.resident,
                            modified := d.modified});
              ELSIF d.protected THEN
                <* ASSERT NOT d.pure *>
                PromotePage(p, Desc{space := Space.Current,
                                    generation := copyGeneration,
                                    pure := FALSE,
                                    note := Note.OlderGeneration,
                                    gray := FALSE, protected := TRUE,
                                    continued := FALSE});
              ELSIF d.pure THEN
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := TRUE,
                            note := Note.OlderGeneration, gray := FALSE,
                            protected := FALSE, continued := FALSE});
              ELSE
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := FALSE,
                            note := Note.OlderGeneration, gray := TRUE,
                            protected := FALSE, continued := FALSE});
                desc[p - p0].link := impureCopy.stack;
                impureCopy.stack := p;
              END;
            ELSIF d.protected AND NOT d.persistent THEN
              Unprotect(p);
            END
          ELSE
            <* ASSERT NOT (d.protected AND d.resident AND d.modified) *>
          END
        END
      END
    END;
    (* now nothing in the previous space is protected or in the older
       generation *)

    (* get these in the new space *)
    mover := NEW (Mover);
    swizzler := NEW (Swizzler);

    (* mark from roots *)
    ThreadF.SuspendOthers();
    BEGIN
      (* Examine the stacks for possible pointers *)
      ThreadF.ProcessStacks(NoteStackLocations);

      (* Examine the global variables for possible pointers *)
      RTHeapMap.WalkGlobals (mover);

      (* Maintain open databases, but drop roots *)
      FOR p := p0 TO p1 - 1 DO
        IF mapDBs[p - p0] # NIL THEN
          mapDBs[p - p0].root := NIL;
          Move (NIL, ADR(mapDBs[p - p0]));
        END;
      END;
    END;
    ThreadF.ResumeOthers();

    IF perfOn THEN PerfPromotedRoots(); END;

    collectorState := CollectorState.One;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateZero;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateOne () =
  BEGIN
    IF NOT CopySome() THEN collectorState := CollectorState.Two; END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateOne;

(* Walk weakly-referenced nodes to determine order in which to do cleanup,
   then cleanup gray nodes.  This should be broken down into parts, since
   it may be a lengthy operation. *)

PROCEDURE CollectSomeInStateTwo () =
  BEGIN
    PreHandleWeakRefs();
    collectorState := CollectorState.Three;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateTwo;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateThree () =
  BEGIN
    (* recursively copy all objects reachable from promoted objects.  marks
       "marka" and "markb" are cleared when objects move to the new
       space. *)
    IF NOT CopySome() THEN
      (* That's the last chance for unmodified persistent pages since
         they may contain WRNNC objects that are about to get finalized.
         Thus, we now unmap them so swizzling can't resurrect them. *)
      ThreadF.SuspendOthers();
      BEGIN
        FOR i := 0 TO p1 - p0 - 1 DO
          VAR d := desc[i];
          BEGIN
            IF d.space = Space.Previous AND d.persistent THEN
              IF NOT d.continued THEN
                <* ASSERT d.protected AND NOT (d.resident AND d.modified) *>
                Unprotect(p0 + i);
              END;
              desc[i].persistent := FALSE;
              (* unmap pages *)
              mapDBs[i].unmap(mapIds[i]);
              mapDBs[i] := NIL;
              mapIds[i] := Nil;
              IF perfOn THEN PerfChange(p0 + i, 1); END;
            END
          END
        END
      END;
      ThreadF.ResumeOthers();

      PostHandleWeakRefs();      (* must be called with no gray objects *)
      signalWeak := TRUE;
      collectorState := CollectorState.Four;
    END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateThree;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateFour () =
  BEGIN
    IF NOT CopySome() THEN collectorState := CollectorState.Five; END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateFour;

PROCEDURE CollectSomeInStateFive () =
  BEGIN
    (* free all oldspace pages; oldspace becomes freespace *)
    FOR i := 0 TO p1 - p0 - 1 DO
      IF desc[i].space = Space.Previous THEN
        desc[i].space := Space.Free;
        desc[i].continued := FALSE;
        <* ASSERT NOT desc[i].protected *>
        IF perfOn THEN PerfChange(p0 + i, 1); END;
      END;
    END;

    RebuildFreelist();

    (* fill the rest of the current copy pages *)
    Close(impureCopy);
    Close(impureTransientCopy);
    Fill(pureCopy);
    Fill(pureTransientCopy);

    IF perfOn THEN PerfEnd(); END;

    InvokeMonitors(before := FALSE);

    IF partialCollection THEN
      IF FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                 + largePromotionPages) * threshold[1] >= threshold[0] THEN
        partialCollectionNext := FALSE;
      ELSE
        partialCollectionNext := TRUE;
      END;
    ELSE
      threshold[0] :=
        FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                + largePromotionPages) * (gcRatio + 1.0);
      threshold[1] := gcRatio;
      partialCollectionNext := TRUE;
    END;

    collectorState := CollectorState.Zero;
  END CollectSomeInStateFive;

(* CopySome attempts to make progress toward cleaning the new space.  It
   returns FALSE iff there was no more work to do.

   It operates by cleaning the current copy page.  It may also clean some
   number of pages on the stack.  When it returns, there is a new copy
   page. *)

PROCEDURE CopySome (): BOOLEAN =
  VAR
    originalPage := impureCopy.page;
    originalBoundary := impureCopy.boundary;
    cleanTo := PageToHeader(impureCopy.page);
    originalTransientPage := impureTransientCopy.page;
    originalTransientBoundary := impureTransientCopy.boundary;
    cleanTransientTo := PageToHeader(impureTransientCopy.page);
  BEGIN
    LOOP
      IF cleanTo < impureCopy.ptr THEN
        VAR ptr := impureCopy.ptr;
        BEGIN
          CleanBetween(cleanTo, ptr);
          cleanTo := ptr;
        END;
      ELSIF cleanTransientTo < impureTransientCopy.ptr THEN
        VAR ptr := impureTransientCopy.ptr;
        BEGIN
          CleanBetween(cleanTransientTo, ptr);
          cleanTransientTo := ptr;
        END;
      ELSE
        IF impureCopy.stack = Nil AND impureTransientCopy.stack = Nil THEN
          RETURN FALSE;
        END;
        IF impureCopy.stack # Nil THEN
          VAR p := impureCopy.stack;
          BEGIN
            impureCopy.stack := desc[p - p0].link;
            IF desc[p - p0].gray THEN
              <* ASSERT desc[p - p0].resident *>
              IF desc[p - p0].protected THEN Unprotect(p); END;
              CleanBetween(PageToHeader(p), PageToHeader(p + 1));
              FOR i := 0 TO PageCount(p) - 1 DO
                desc[p + i - p0].gray := FALSE;
              END;
              IF perfOn THEN PerfChange(p, PageCount(p)); END;
              IF desc[p - p0].generation = Generation.Older
                OR NOT desc[p - p0].modified THEN
                <* ASSERT desc[p - p0].space = Space.Current *>
                Protect(p, readable := TRUE, writable := FALSE);
              END;
            END;
          END;
        END;
        IF impureTransientCopy.stack # Nil THEN
          VAR p := impureTransientCopy.stack;
          BEGIN
            impureTransientCopy.stack := desc[p - p0].link;
            IF desc[p - p0].gray THEN
              <* ASSERT desc[p - p0].resident *>
              IF desc[p - p0].protected THEN Unprotect(p); END;
              CleanBetween(PageToHeader(p), PageToHeader(p + 1));
              FOR i := 0 TO PageCount(p) - 1 DO
                desc[p + i - p0].gray := FALSE;
              END;
              IF perfOn THEN PerfChange(p, PageCount(p)); END;
              IF desc[p - p0].generation = Generation.Older
                OR NOT desc[p - p0].modified THEN
                <* ASSERT desc[p - p0].space = Space.Current *>
                Protect(p, readable := TRUE, writable := FALSE);
              END
            END
          END
        END
      END;
      IF impureCopy.page # originalPage
        OR impureTransientCopy.page # originalTransientPage THEN EXIT; END;
    END;
    IF originalPage # Nil
      AND originalPage # impureCopy.page
      AND desc[originalPage - p0].gray THEN
      (* originalPage is now in the stack; mark it not gray *)
      CleanBetween(cleanTo, originalBoundary);
      desc[originalPage - p0].gray := FALSE;
      IF perfOn THEN PerfChange(originalPage, 1); END;
      IF desc[originalPage - p0].generation = Generation.Older THEN
        <* ASSERT desc[originalPage - p0].modified *>
        <* ASSERT desc[originalPage - p0].space = Space.Current *>
        Protect(originalPage, readable := TRUE, writable := FALSE);
      END;
      IF originalTransientPage # Nil
        AND originalTransientPage # impureTransientCopy.page
        AND desc[originalTransientPage - p0].gray THEN
        (* originalTransientPage is now in the stack; mark it not gray *)
        CleanBetween(cleanTransientTo, originalTransientBoundary);
        desc[originalTransientPage - p0].gray := FALSE;
        IF perfOn THEN PerfChange(originalTransientPage, 1); END;
        IF desc[originalTransientPage - p0].generation = Generation.Older THEN
          <* ASSERT desc[originalTransientPage - p0].modified *>
          <* ASSERT desc[originalTransientPage - p0].space = Space.Current *>
          Protect(originalTransientPage, readable := TRUE, writable := FALSE);
        END
      END
    ELSIF originalTransientPage # Nil
      AND originalTransientPage # impureTransientCopy.page
      AND desc[originalTransientPage - p0].gray THEN
      (* originalTransientPage is now in the stack; mark it not gray *)
      CleanBetween(cleanTransientTo, originalTransientBoundary);
      desc[originalTransientPage - p0].gray := FALSE;
      IF perfOn THEN PerfChange(originalTransientPage, 1); END;
      IF desc[originalTransientPage - p0].generation = Generation.Older THEN
        <* ASSERT desc[originalTransientPage - p0].modified *>
        <* ASSERT desc[originalTransientPage - p0].space = Space.Current *>
        Protect(originalTransientPage, readable := TRUE, writable := FALSE);
      END;
      IF originalPage # Nil
        AND originalPage # impureCopy.page
        AND desc[originalPage - p0].gray THEN
        (* originalPage is now in the stack; mark it not gray *)
        CleanBetween(cleanTo, originalBoundary);
        desc[originalPage - p0].gray := FALSE;
        IF perfOn THEN PerfChange(originalPage, 1); END;
        IF desc[originalPage - p0].generation = Generation.Older THEN
          <* ASSERT desc[originalPage - p0].modified *>
          <* ASSERT desc[originalPage - p0].space = Space.Current *>
          Protect(originalPage, readable := TRUE, writable := FALSE);
        END
      END
    END;
    RETURN TRUE;
  END CopySome;

PROCEDURE CleanSome (VAR current: AllocRec; cleaner := CleanBetween;
                     gray := FALSE): BOOLEAN =
  VAR
    originalPage := current.page;
    originalBoundary := current.boundary;
    cleanTo := PageToHeader(current.page);
  BEGIN
    LOOP
      IF cleanTo < current.ptr THEN
        VAR ptr := current.ptr;
        BEGIN
          cleaner(cleanTo, ptr);
          cleanTo := ptr;
        END;
      ELSE
        IF current.stack = Nil THEN RETURN FALSE; END;
        VAR p := current.stack;
        BEGIN
          current.stack := desc[p - p0].link;
          IF desc[p - p0].gray THEN
            <* ASSERT desc[p - p0].resident *>
            IF desc[p - p0].protected THEN Unprotect(p); END;
            cleaner(PageToHeader(p), PageToHeader(p + 1));
            FOR i := 0 TO PageCount(p) - 1 DO
              desc[p + i - p0].gray := gray;
            END;
            IF perfOn THEN PerfChange(p, PageCount(p)); END;
            IF desc[p - p0].generation = Generation.Older
              OR NOT desc[p - p0].modified THEN
              <* ASSERT desc[p - p0].space = Space.Current *>
              Protect(p, readable := TRUE, writable := FALSE);
            END;
          END;
        END;
      END;
      IF current.page # originalPage THEN EXIT; END;
    END;
    cleaner(cleanTo, originalBoundary);
    (* originalPage is now in the stack; mark it not gray *)
    IF originalPage # Nil THEN
      desc[originalPage - p0].gray := gray;
      IF perfOn THEN PerfChange(originalPage, 1); END;
      IF desc[originalPage - p0].generation = Generation.Older THEN
        <* ASSERT desc[originalPage - p0].modified *>
        <* ASSERT desc[originalPage - p0].space = Space.Current *>
        Protect(originalPage, readable := TRUE, writable := FALSE);
      END;
    END;
    RETURN TRUE;
  END CleanSome;

PROCEDURE CleanBetween (h, he: RefHeader) =
  BEGIN
    WHILE h < he DO
      <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
      <* ASSERT NOT h.forwarded *>
      h.marka := FALSE;
      h.markb := FALSE;
      RTHeapMap.WalkRef (h, mover);
      INC(h, ADRSIZE(Header) + ReferentSize(h));
    END;
  END CleanBetween;

PROCEDURE CleanTransient (h, he: RefHeader) =
  BEGIN
    WHILE h < he DO
      <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
      <* ASSERT NOT h.forwarded *>
      h.marka := FALSE;
      h.markb := FALSE;
      RTHeapMap.WalkRef (h, tmover);
      INC(h, ADRSIZE(Header) + ReferentSize(h));
    END;
  END CleanTransient; 

PROCEDURE CleanPersistent (h, he: RefHeader) =
  VAR
    p: Page;
    pi: INTEGER;
    mapDB: RTDB.T;
    tc: Typecode;
    def: TypeDefn;
    pure: BOOLEAN;
  BEGIN
    IF h < he THEN
      p := HeaderToPage(h);
      <* ASSERT p # Nil *>
      pi := p - p0;
      <* ASSERT desc[pi].persistent *>
      pure := desc[pi].pure;
      mapDB := mapDBs[pi];
      <* ASSERT mapDB # NIL *>
      SetDB(mapDB);
      IF mapIds[pi] = Nil THEN
        VAR
          n := PageCount(p);
          pp := mapDB.newPages(p, n);
        BEGIN
          FOR i := 0 TO n-1 DO
            mapIds[pi + i] := pp + i;
          END
        END
      END;

      REPEAT
        <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
        <* ASSERT NOT h.forwarded *>
        h.marka := FALSE;
        h.markb := FALSE;
        tc := h.typecode;
        IF tc = Fill_1_type THEN
          INC(h, ADRSIZE(Header));
        ELSIF tc = Fill_N_type THEN
          INC(h, LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^);
        ELSE
          def := RTType.Get(tc);
          <* ASSERT def.traced = 1 *>
          EVAL mapDB.getFP(ADR(def.fp))^;
          IF NOT pure THEN
            <*FATAL ANY*>
            BEGIN
              RTTypeMap.DoWalkRef(def, h + ADRSIZE(Header),
                                  RTTypeMap.Mask{RTTypeMap.Kind.Ref}, pmover);
            END;
          END;
          IF def.nDimensions = 0 THEN
            INC(h, ADRSIZE(Header) + def.dataSize);
          ELSE
            INC(h, ADRSIZE(Header) + OpenArraySize(h, def));
          END;
        END;
      UNTIL NOT h < he;
    END;
  END CleanPersistent;

(* We maintain a list in weakTable, starting at weakLive0, of weak refs and
   the objects they reference.  This table is not considered a root.  When
   HandleWeakRefs is entered, any object mentioned in that list is a
   candidate for cleanup.

   First, we determine which weakly-referenced objects with non-NIL
   cleanups ("WRNNC objects") are reachable from other WRNNC objects, by
   walking the old space.  All such WRNNC objects are copied to new space,
   and all the objects they reference.

   All the weakly-referenced objects left in the old space can then be
   scheduled for cleanup; we move them from the list starting at weakLive0
   to the list starting at weakDead0 in weakTable.  A separate thread runs
   WeakCleaner, which does the calls to the procedures.

   Note that the refs in weakTable must be updated to point to new
   space. *)

(* PreHandleWeakRefs walks the weakly-references structures in old-space,
   deciding on a cleanup order. *)

PROCEDURE PreHandleWeakRefs () =
  VAR s: Stacker;
  BEGIN
    (* get ready to allocate on a new page (take this out!) *)
    InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
    InsertFiller(pureCopy.ptr, pureCopy.boundary - pureCopy.ptr);
    InsertFiller(impureTransientCopy.ptr,
                 impureTransientCopy.boundary - impureTransientCopy.ptr);
    InsertFiller(pureTransientCopy.ptr,
                 pureTransientCopy.boundary - pureTransientCopy.ptr);
    (* allocate a stack on the side for walking the old space *)
    s := InitStack();
    (* iterate over the weak refs to walk the old space *)
    VAR i := weakLive0;
    BEGIN
      WHILE i # -1 DO
        (* here, all old-space WRNNC objects that have already been scanned
           have marka set, as do all old-space objects reachable from them;
           all old-space WRNNC objects that were reachable from other
           already-scanned WRNNC objects have been promoted to the new
           space. *)
        WITH entry = weakTable[i] DO
          IF entry.p # NIL AND NOT Moved(entry.r) THEN
            (* we haven't seen this WRNNC object before *)
            VAR header := HeaderOf(LOOPHOLE(entry.r, ADDRESS));
            BEGIN
              IF NOT header.marka THEN
                <* ASSERT NOT header.markb *>
                (* visit all old-space objects reachable from it; promote
                   all other old-space WRNNC objects reachable from it;
                   promote all old-space objects reachable from it that
                   have "marka" set.  mark all visited nodes with
                   "markb". *)
                WeakWalk1(s, entry.r);
                <* ASSERT NOT header.marka *>
                <* ASSERT header.markb *>
                (* then change all "markb" to "marka" *)
                WeakWalk2(s, entry.r);
                <* ASSERT header.marka *>
                <* ASSERT NOT header.markb *>
              END;
            END;
          END;
          i := entry.next;
        END;
      END;
    END;
  END PreHandleWeakRefs;

(* WeakWalk1 starts at a WRNNC object and visits all objects in old space
   reachable from it, using "markb" to keep from visiting them more than
   once.  All other WRNNC objects visited are promoted, as are all objects
   already visited from other WRNNC objects. *)

PROCEDURE WeakWalk1 (s: Stacker; ref: RefReferent) =
  VAR ref0 := ref;
  BEGIN
    <* ASSERT s.empty() *>
    LOOP
      IF NOT Moved(ref) THEN
        VAR header := HeaderOf(ref);
        BEGIN
          IF header.marka THEN
            <* ASSERT NOT header.markb *>
            Move(NIL, ADR(ref));
          ELSIF NOT header.markb THEN
            IF header.weak AND ref # ref0 THEN
              Move(NIL, ADR(ref));
            ELSE
              header.markb := TRUE;
              RTHeapMap.WalkRef (header, s);
            END;
          END;
        END;
      END;
      IF s.empty() THEN EXIT; END;
      ref := s.pop();
    END;
  END WeakWalk1;

(* WeakWalk2 starts at a WRNNC objects and visits all objects in the old
   space that are reachable from it, changing "markb" to "marka" *)

PROCEDURE WeakWalk2 (s: Stacker;  ref: RefReferent) =
  BEGIN
    <* ASSERT s.empty() *>
    LOOP
      IF NOT Moved(ref) THEN
        VAR header := HeaderOf(ref);
        BEGIN
          IF header.markb THEN
            header.markb := FALSE;
            header.marka := TRUE;
            RTHeapMap.WalkRef (header, s);
          END;
        END;
      END;
      IF s.empty() THEN EXIT; END;
      ref := s.pop();
    END;
  END WeakWalk2;

PROCEDURE PostHandleWeakRefs () =
  BEGIN
    (* move to a new page (take this out!) *)
    InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
    InsertFiller(pureCopy.ptr, pureCopy.boundary - pureCopy.ptr);
    InsertFiller(impureTransientCopy.ptr,
                 impureTransientCopy.boundary - impureTransientCopy.ptr);
    InsertFiller(pureTransientCopy.ptr,
                 pureTransientCopy.boundary - pureTransientCopy.ptr);
    (* iterate over all weak refs.  if the object hasn't been promoted,
       schedule a cleanup *)
    VAR
      i        := weakLive0;
      previous := -1;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          IF Moved(entry.r) THEN
            (* no cleanup this time; note new address *)
            Move(NIL, ADR(entry.r));
            previous := i;
            i := entry.next;
          ELSE
            (* the weak ref is dead; there are no cleanups *)
            VAR header := HeaderOf(LOOPHOLE(entry.r, ADDRESS));
            BEGIN
              header.weak := FALSE;
            END;
            (* move the entry from the weakLive0 list into the weakDead0 or
               weakFree0 list *)
            VAR next := entry.next;
            BEGIN
              IF previous = -1 THEN
                weakLive0 := next;
              ELSE
                weakTable[previous].next := next;
              END;
              entry.t.a := -1;   (* keep ToRef from succeeding *)
              IF entry.p # NIL THEN
                entry.next := weakDead0;
                weakDead0 := i;
              ELSE
                entry.next := weakFree0;
                weakFree0 := i;
              END;
              i := next;
            END;
          END;
        END;
      END;
    END;
    (* for all entries on the weakDead0 list, including those just placed
       there, note the new address *)
    VAR i := weakDead0;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          <* ASSERT entry.t.a = -1 *>
          Move(NIL, ADR(entry.r));
          i := entry.next;
        END;
      END;
    END;
    (* finally, check for objects with final cleanup enabled *)
    VAR
      i        := weakFinal0;
      previous := -1;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          IF Moved(entry.r) THEN
            (* no cleanup this time; note new address *)
            Move(NIL, ADR(entry.r));
            previous := i;
            i := entry.next;
          ELSE
            (* call the cleanup procedure *)
            LOOPHOLE(entry.p, PROCEDURE (p: REFANY))(
              LOOPHOLE(entry.r, REFANY));
            (* take the entry off the weakFinal0 list and put it on the
               weakFree0 list; on to the next entry *)
            VAR next := entry.next;
            BEGIN
              IF previous = -1 THEN
                weakFinal0 := next;
              ELSE
                weakTable[previous].next := next;
              END;
              entry.next := weakFree0;
              weakFree0 := i;
              i := next;
            END;
          END;
        END;
      END;
    END;
  END PostHandleWeakRefs;

(* The stack for walking the old space is maintained on the heap in the new
   space. *)

TYPE
  Stacker = RTHeapMap.Visitor OBJECT
    data : <*TRANSIENT*> REF ARRAY OF RefReferent;
    x0   : UNTRACED REF RefReferent;
    x1   : UNTRACED REF RefReferent;
    xA   : UNTRACED REF RefReferent;
    xN   : CARDINAL;
  METHODS
    empty (): BOOLEAN     := StackEmpty;
    pop   (): RefReferent := PopStack;
  OVERRIDES
    apply := PushStack;
  END;

(* InitStack allocates an initial stack of 100 elements. *)

PROCEDURE InitStack (): Stacker =
  VAR s := NEW (Stacker);
  BEGIN
    s.data := NEW(<*TRANSIENT*> REF ARRAY OF RefReferent, 100);
    s.xN   := NUMBER (s.data^);
    s.x0   := ADR(s.data[0]);
    s.x1   := s.x0 + s.xN * ADRSIZE(RefReferent);
    s.xA   := s.x0;
    RETURN s;
  END InitStack;

(* PushStack pushes an object onto the stack, growing it if necessary. *)

PROCEDURE PushStack (s: Stacker;  cp: ADDRESS) =
  VAR ref: RefReferent := LOOPHOLE(cp, UNTRACED REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      IF s.xA = s.x1 THEN ExpandStack (s); END;
      s.xA^ := ref;
      INC(s.xA, ADRSIZE(RefReferent));
    END;
  END PushStack;

PROCEDURE ExpandStack (s: Stacker) =
  VAR
    newStackN := 2 * s.xN;
    newStack := NEW(<*TRANSIENT*> REF ARRAY OF RefReferent, newStackN);
  BEGIN
    SUBARRAY(newStack^, 0, s.xN) := SUBARRAY(s.data^, 0, s.xN);
    s.x0   := ADR(newStack^[0]);
    s.xA   := s.x0 + s.xN * ADRSIZE(RefReferent);
    s.x1   := s.x0 + newStackN * ADRSIZE(RefReferent);
    s.data := newStack;
    s.xN   := newStackN;
  END ExpandStack;

(* PopStack pops an object off the stack. *)

PROCEDURE PopStack (s: Stacker): RefReferent =
  BEGIN
    DEC(s.xA, ADRSIZE(RefReferent));
    RETURN s.xA^;
  END PopStack;

(* StackEmpty tells if the stack is empty. *)

PROCEDURE StackEmpty (s: Stacker): BOOLEAN =
  BEGIN
    RETURN s.xA = s.x0;
  END StackEmpty;

(* Malloc returns the address of "size" bytes of untraced, zeroed storage *)

PROCEDURE Malloc (size: INTEGER): ADDRESS =
  VAR res: ADDRESS;
  BEGIN
    RTOS.LockHeap();
    BEGIN
      res := Cstdlib.malloc(size);
      IF res = NIL THEN
        RTMisc.FatalError(NIL, 0, "malloc failed, unable to get more memory");
      END;
    END;
    RTOS.UnlockHeap();
    RTMisc.Zero(res, size);
    RETURN res;
  END Malloc;

(* AllocForNew allocates space for a NEW. *)

PROCEDURE AllocForNew (dataSize, dataAlignment: CARDINAL): RefReferent =
  VAR
    alignment : INTEGER;
    nextPtr   : RefHeader;
  BEGIN
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       new.ptr? *)
    VAR referentTry := new.ptr + ADRSIZE(Header);
    BEGIN
      (* ---------------- see CheckTypes ---------------------------------
|      WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|        alignment := a - referentTry;
|        nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
         ------------------------------------------------------------------ *)
      alignment :=
        align[Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask),
              dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;
    (* If this is not ok, take the long route *)
    IF nextPtr > new.boundary THEN
      nextPtr := NIL;         (* clear in case of GC *)
      VAR
        res := LongAlloc(dataSize, dataAlignment, new,
                         allocMode := AllocMode.New, pure := FALSE);
      BEGIN
        RTOS.UnlockHeap();
        RETURN res;
      END;
    END;
    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(new.ptr, alignment);
      new.ptr := new.ptr + alignment;
    END;
    VAR res := LOOPHOLE(new.ptr + ADRSIZE(Header), RefReferent);
    BEGIN
      new.ptr := nextPtr;
      RTOS.UnlockHeap();
      RETURN res;
    END;
  END AllocForNew;

(* AllocForNewTransient allocates space for a NEW. *)

PROCEDURE AllocForNewTransient (dataSize, dataAlignment: CARDINAL)
  : RefReferent =
  VAR
    alignment : INTEGER;
    nextPtr   : RefHeader;
  BEGIN
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       newTransient.ptr? *)
    VAR referentTry := newTransient.ptr + ADRSIZE(Header);
    BEGIN
      (* ---------------- see CheckTypes ---------------------------------
|      WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|        alignment := a - referentTry;
|        nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
         ------------------------------------------------------------------ *)
      alignment :=
        align[Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask),
              dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;
    (* If this is not ok, take the long route *)
    IF nextPtr > newTransient.boundary THEN
      nextPtr := NIL;         (* clear in case of GC *)
      VAR
        res := LongAlloc(dataSize, dataAlignment, newTransient,
                         allocMode := AllocMode.New, pure := FALSE);
      BEGIN
        RTOS.UnlockHeap();
        RETURN res;
      END;
    END;
    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(newTransient.ptr, alignment);
      newTransient.ptr := newTransient.ptr + alignment;
    END;
    VAR res := LOOPHOLE(newTransient.ptr + ADRSIZE(Header), RefReferent);
    BEGIN
      newTransient.ptr := nextPtr;
      RTOS.UnlockHeap();
      RETURN res;
    END;
  END AllocForNewTransient;

(* AllocForCopy allocates space to copy an object from oldspace; it has the
   same logic as AllocForNew. *)

PROCEDURE AllocForCopy (dataSize, dataAlignment: CARDINAL;
                        VAR current: AllocRec;
                        pure: BOOLEAN) : RefReferent =
  VAR
    alignment       : INTEGER;
    nextPtr         : RefHeader;
    res, referentTry: RefReferent;
  BEGIN
    <* ASSERT collectorOn *>
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       current.ptr? *)
    referentTry := current.ptr + ADRSIZE(Header);

    (* ---------------- see CheckTypes ---------------------------------
|    WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|      alignment := a - referentTry;
|      nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
       ------------------------------------------------------------------ *)
    VAR tmp := Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask);
    BEGIN
      alignment := align[tmp, dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;

    (* If this is not ok, take the long route *)
    IF nextPtr > current.boundary THEN
      res := LongAlloc(dataSize, dataAlignment, current,
                       allocMode := AllocMode.Copy, pure := pure);
      RTOS.UnlockHeap();
      RETURN res;
    END;

    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(current.ptr, alignment);
      current.ptr := current.ptr + alignment;
    END;

    res := LOOPHOLE(current.ptr + ADRSIZE(Header), RefReferent);
    current.ptr := nextPtr;
    RTOS.UnlockHeap();
    RETURN res;
  END AllocForCopy;

TYPE AllocMode = {New, Copy};

PROCEDURE LongAlloc (              dataSize, dataAlignment : CARDINAL;
                     VAR (*INOUT*) current                 : AllocRec;
                                   allocMode               : AllocMode;
                                   pure                    : BOOLEAN)
  : RefReferent =
  VAR
    nbBytes := RTMisc.Upper(ADRSIZE(Header), dataAlignment) + dataSize;
    nbPages := (nbBytes + AdrPerPage - 1) DIV AdrPerPage;
    res     : RefReferent;
    notAfter: SET OF Note;
  BEGIN
    IF allocMode = AllocMode.New THEN
      CollectEnough();
      notAfter := SET OF Note{Note.Copied};
    ELSE
      notAfter := SET OF Note{Note.Allocated};
    END;

    VAR
      thisPage := FindFreePages(nbPages, notAfter := notAfter);
      (* thisPage points to a block of nbPages contiguous, free pages; just
         what we need! *)
      thisPtr                  := PageToHeader(thisPage);
      thisBoundary             := PageToHeader(thisPage + 1);
      persistent := current.db # NIL;
      gray := persistent OR allocMode = AllocMode.Copy AND NOT pure;
      generation  : Generation;
      note        : Note;
    BEGIN
      (* maybe we have to put a filler to align this thing *)
      res := RTMisc.Align(thisPtr + ADRSIZE(Header), dataAlignment);
      InsertFiller(thisPtr, res - ADRSIZE(Header) - thisPtr);

      (* allocate the object *)
      thisPtr := LOOPHOLE(res + dataSize, RefHeader);

      IF allocMode = AllocMode.New THEN
        generation := Generation.Younger;
        note := Note.Allocated;
      ELSE
        generation := copyGeneration;
        IF generation = Generation.Older THEN <* ASSERT gray OR pure *> END;
        note := Note.Copied;
      END;
      desc[thisPage - p0] :=
          Desc{space := Space.Current, generation := generation,
               pure := pure, note := note, gray := gray, protected := FALSE,
               continued := FALSE, persistent := persistent,
               resident := TRUE, modified := TRUE};
      mapDBs[thisPage - p0] := current.db;
      mapIds[thisPage - p0] := Nil;
      IF nbPages = 1 THEN
        CASE allocMode OF
        | AllocMode.New => INC(smallNewPages);
        | AllocMode.Copy => INC(smallCopyPages);
        END;
      ELSE
        CASE allocMode OF
        | AllocMode.New => INC(largeNewPages, nbPages);
        | AllocMode.Copy => INC(largeCopyPages, nbPages);
        END;
        FOR i := 1 TO nbPages - 1 DO
          desc[thisPage + i - p0] :=
              Desc{space := Space.Current, generation := generation,
                   pure := pure, note := note, gray := gray,
                   protected := FALSE, continued := TRUE,
                   persistent := persistent, resident := TRUE,
                   modified := TRUE};
          mapDBs[thisPage + i - p0] := current.db;
          mapIds[thisPage + i - p0] := Nil;
        END
      END;
      IF perfOn THEN PerfChange(thisPage, nbPages); END;

      IF nbPages = 1 THEN
        IF thisBoundary - thisPtr > current.boundary - current.ptr THEN
          (* more allocation space available on this page; fill and file
             the current page *)
          InsertFiller(current.ptr, current.boundary - current.ptr);
          IF current.page # Nil THEN
            <* ASSERT desc[current.page - p0].space = Space.Current *>
            desc[current.page - p0].link := current.stack;
            current.stack := current.page;
            IF allocMode = AllocMode.Copy THEN
              <* ASSERT desc[current.page - p0].gray
                   OR desc[current.page - p0].pure *>
            END;
          END;
          current.ptr := thisPtr;
          current.boundary := thisBoundary;
          current.page := thisPage;
        ELSE
          (* more allocation space available on current page; fill and file
             this page *)
          InsertFiller(thisPtr, thisBoundary - thisPtr);
          desc[thisPage - p0].link := current.stack;
          current.stack := thisPage;
        END;
      ELSE
        (* file this page *)
        desc[thisPage - p0].link := current.stack;
        current.stack := thisPage;
      END;
    END;
    RETURN res;
  END LongAlloc;

(*--------------------------------------------------*)

VAR
  backgroundWaiting   := FALSE;

(* The background thread may be present or not.  If it is present, it
   speeds collection asynchronously.  Because it makes progress slowly, it
   should impose only a small overhead when the mutator is running, but
   quickly complete a collection if the collector pauses. *)

PROCEDURE BackgroundThread (<* UNUSED *> closure: Thread.Closure): REFANY =
  BEGIN
    LOOP
      backgroundWaiting := TRUE; (* no locks, unfortunately *)
      WHILE collectorState = CollectorState.Zero DO Wait(); END;
      backgroundWaiting := FALSE;
      WHILE collectorState # CollectorState.Zero DO
        RTOS.LockHeap();
        BEGIN
          IF collectorState # CollectorState.Zero THEN
            CollectorOn();
            CollectSome();
            CollectorOff();
          END;
        END;
        RTOS.UnlockHeap();
        Thread.Pause(1.0d0);       (* one second *)
      END;
    END;
  END BackgroundThread;


(* --------------------------------------------------------- collector *)

PROCEDURE StartGC () =
  BEGIN
    StartCollection();
  END StartGC;

PROCEDURE FinishGC () =
  BEGIN
    FinishCollection();
  END FinishGC;

PROCEDURE Crash (): BOOLEAN =
  VAR result: BOOLEAN;
  BEGIN
    RTOS.LockHeap();        (* left incremented *)
    IF collectorState = CollectorState.Zero THEN
      (* no collection in progress *)
      collectorOn := TRUE;       (* left on *)
      result := TRUE;
    ELSIF NOT collectorOn THEN
      CollectorOn();             (* left on *)
      (* finish collection *)
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      result := TRUE;
    ELSE
      collectorOn := TRUE;       (* left on *)
      result := FALSE;
    END;
    (* unprotect all pages *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].protected THEN Unprotect(p); END;
    END;
    RETURN result;
  END Crash;

(* --------------------------------------------------------- debugging *)

VAR
  protectedCheck, refCheck: RTHeapMap.Visitor;

PROCEDURE InstallSanityCheck () =
  BEGIN
    RegisterMonitor(
      NEW(MonitorClosure, before := SanityCheck, after := SanityCheck));
    IF (refCheck = NIL) THEN
      protectedCheck := NEW (RTHeapMap.Visitor,
                             apply := ProtectedOlderRefSanityCheck);
      refCheck := NEW (RTHeapMap.Visitor, apply := RefSanityCheck);
    END;
  END InstallSanityCheck;

(* SanityCheck checks the heap for correctness when no collection is in
   progress. *)

PROCEDURE SanityCheck (<*UNUSED*> self: MonitorClosure) =
  VAR p := p0;
  BEGIN
    WHILE p < p1 DO
      CASE desc[p - p0].space OF
      | Space.Unallocated => INC(p);
      | Space.Previous =>        <* ASSERT FALSE *>
      | Space.Current =>
          <* ASSERT NOT desc[p - p0].gray *>
          <* ASSERT NOT desc[p - p0].continued *>
          IF desc[p - p0].persistent THEN
            <* ASSERT mapIds[p - p0] # Nil *>
            <* ASSERT mapDBs[p - p0] # NIL *>
            IF desc[p - p0].resident THEN
              IF desc[p - p0].modified THEN
                IF desc[p - p0].protected THEN
                  <* ASSERT desc[p - p0].generation = Generation.Older *>
                END;
              ELSE
                <* ASSERT desc[p - p0].protected *>
              END;
            ELSE
              <* ASSERT desc[p - p0].protected *>
              <* ASSERT desc[p - p0].pure *>
            END;
          ELSE
            <* ASSERT desc[p - p0].resident *>
            <* ASSERT desc[p - p0].modified *>
            IF desc[p - p0].protected THEN
              <* ASSERT desc[p - p0].generation = Generation.Older *>
            END;
          END;
          IF desc[p - p0].resident THEN
            (* visit the objects on the page *)
            VAR
              h  := PageToHeader(p);
              he := PageToHeader(p + 1);
            BEGIN
              IF    p = new.page          THEN he := new.ptr;
              ELSIF p = newTransient.page THEN he := newTransient.ptr; END;
              WHILE h < he DO
                (* check the references in the object *)
                IF desc[p - p0].modified AND desc[p - p0].protected THEN
                  RTHeapMap.WalkRef (h, protectedCheck);
                ELSE
                  RTHeapMap.WalkRef (h, refCheck);
                END;
                INC(h, ADRSIZE(Header) + ReferentSize(h));
              END;
              IF h > he THEN
                <* ASSERT HeaderToPage(h - 1) = p + PageCount(p) - 1 *>
              ELSE
                <* ASSERT PageCount(p) = 1 *>
              END;
            END;
          END;
          VAR
            n := PageCount(p);
            d := desc[p - p0];
          BEGIN
            d.continued := TRUE;
            d.link := Nil;
            LOOP
              INC(p);
              DEC(n);
              IF n = 0 THEN EXIT; END;
              VAR dd := desc[p - p0];
              BEGIN
                dd.link := Nil;
                <* ASSERT dd = d *>
                IF dd.persistent THEN
                  <* ASSERT mapIds[p - p0] # Nil *>
                  <* ASSERT mapDBs[p - p0] # NIL *>
                ELSE
                  <* ASSERT mapIds[p - p0] = Nil *>
                  <* ASSERT mapDBs[p - p0] = NIL *>
                END;
              END;
            END;
          END;
      | Space.Free =>
          <* ASSERT NOT desc[p - p0].continued *>
          INC(p);
      END;
    END;
    <* ASSERT p = p1 *>
  END SanityCheck;

PROCEDURE RefSanityCheck (<*UNUSED*>v: RTHeapMap.Visitor;  cp  : ADDRESS) =
  VAR ref := LOOPHOLE(cp, REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      VAR
        p := ReferentToPage(ref);
        h := HeaderOf(ref);
      BEGIN
        IF p0 <= p AND p < p1 THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          <* ASSERT NOT desc[p - p0].continued *>
          IF desc[p - p0].resident THEN
            VAR tc := h.typecode;
            BEGIN
              <* ASSERT (0 < tc AND tc < RT0u.nTypes)
                        OR tc = Fill_1_type
                        OR tc = Fill_N_type *>
            END;
          END;
        ELSE
          (* the compiler generates Text.T that are not in the traced heap *)
          <* ASSERT h.typecode = RT0.TextTypecode *>
        END;
      END;
    END;
  END RefSanityCheck;

PROCEDURE ProtectedOlderRefSanityCheck (<*UNUSED*> v  : RTHeapMap.Visitor;
                                                   cp : ADDRESS) =
  VAR ref := LOOPHOLE(cp, REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      VAR
        p := ReferentToPage(ref);
        h := HeaderOf(ref);
      BEGIN
        IF p0 <= p AND p < p1 THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          <* ASSERT desc[p - p0].generation = Generation.Older *>
          <* ASSERT NOT desc[p - p0].continued *>
          IF desc[p - p0].resident THEN
            VAR tc := h.typecode;
            BEGIN
              <* ASSERT (0 < tc AND tc < RT0u.nTypes)
                        OR tc = Fill_1_type
                        OR tc = Fill_N_type *>
            END;
          END;
        ELSE
          (* the compiler generates Text.T that are not in the traced heap *)
          <* ASSERT h.typecode = RT0.TextTypecode *>
        END;
      END;
    END;
  END ProtectedOlderRefSanityCheck;

(* ----------------------------------------------------------------------- *)

PROCEDURE VisitAllRefs (v: RefVisitor) =
  VAR tc: Typecode;
  BEGIN
    TRY
      Disable();
      FOR p := p0 TO p1 - 1 DO
        IF desc[p - p0].space = Space.Current
             AND NOT desc[p - p0].continued THEN
          VAR
            h             := PageToHeader(p);
            he            := PageToHeader(p + 1);
            size: INTEGER;
          BEGIN
            IF    p = new.page          THEN he := new.ptr;
            ELSIF p = newTransient.page THEN he := newTransient.ptr; END;
            WHILE h < he DO
              size := ReferentSize(h);
              tc := h.typecode;
              IF tc # Fill_1_type AND tc # Fill_N_type THEN
                IF NOT v.visit(
                         tc, LOOPHOLE(h + ADRSIZE(Header), REFANY), size) THEN
                  RETURN;
                END;
              END;
              INC(h, ADRSIZE(Header) + size);
            END;
          END;
        END;
      END;
    FINALLY
      Enable();
    END;
  END VisitAllRefs;

TYPE
  CountClosure = MonitorClosure OBJECT
                   tcs    : <*TRANSIENT*> REF ARRAY OF Typecode;
                   counts : <*TRANSIENT*> REF ARRAY OF CARDINAL;
                   visitor: RefVisitor;
                 OVERRIDES
                   after := CountRefsForTypecodes;
                 END;

TYPE
  CountAllClosure = MonitorClosure OBJECT
                      counts : <*TRANSIENT*> REF ARRAY OF CARDINAL;
                      visitor: RefVisitor;
                    OVERRIDES
                      after := CountRefsForAllTypecodes;
                    END;

TYPE
  CountVisitor =
    RefVisitor OBJECT cl: CountClosure OVERRIDES visit := One; END;

  CountAllVisitor =
    RefVisitor OBJECT cl: CountAllClosure OVERRIDES visit := All; END;

PROCEDURE One (           self: CountVisitor;
                          tc  : Typecode;
               <*UNUSED*> r   : REFANY;
               <*UNUSED*> size: CARDINAL      ): BOOLEAN =
  BEGIN
    FOR i := FIRST(self.cl.tcs^) TO LAST(self.cl.tcs^) DO
      IF self.cl.tcs[i] = tc THEN INC(self.cl.counts[i]); RETURN TRUE; END;
    END;
    RETURN TRUE;
  END One;

PROCEDURE All (           self: CountAllVisitor;
                          tc  : Typecode;
               <*UNUSED*> r   : REFANY;
               <*UNUSED*> size: CARDINAL         ): BOOLEAN =
  BEGIN
    INC(self.cl.counts[tc]);
    RETURN TRUE;
  END All;

PROCEDURE CountRefsForTypecodes (cl: CountClosure) =
  BEGIN
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      cl.counts[i] := 0;
    END;
    VisitAllRefs(cl.visitor);
    FOR i := FIRST(cl.tcs^) TO LAST(cl.tcs^) DO
      PutText("count[");
      PutInt(cl.tcs[i]);
      PutText("] = ");
      PutInt(cl.counts[i]);
      IF i # LAST(cl.tcs^) THEN PutText(",  "); END;
    END;
    PutText("\n");
  END CountRefsForTypecodes;

PROCEDURE CountRefsForAllTypecodes (cl: CountAllClosure) =
  BEGIN
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      cl.counts[i] := 0;
    END;
    VisitAllRefs(cl.visitor);
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      IF cl.counts[i] > 1 THEN
        PutInt(i);
        PutText(": ");
        PutInt(cl.counts[i]);
        IF i # LAST(cl.counts^) THEN PutText(", "); END;
      END;
    END;
    PutText("\n");
  END CountRefsForAllTypecodes;

(* ---------------------------------------------------- showheap hooks *)

VAR
  perfW  : RTPerfTool.Handle;
  perfOn : BOOLEAN := FALSE;

CONST
  EventSize = (BITSIZE(RTHeapEvent.T) + BITSIZE(CHAR) - 1) DIV BITSIZE(CHAR);

PROCEDURE PerfStart () =
  VAR i, j: Page;
  BEGIN
    IF RTPerfTool.Start("showheap", perfW) THEN
      perfOn := TRUE;
      RTProcess.RegisterExitor(PerfStop);
      PerfGrow(p0, p1 - p0);

      i := p0;
      WHILE i # Nil AND i < p1 DO
        j := i + 1;
        WHILE j < p1 AND desc[j - p0].continued DO INC(j); END;
        IF desc[i - p0].space # Space.Free THEN PerfChange(i, j - i); END;
        i := j;
      END;
    END;
  END PerfStart;

PROCEDURE PerfFlip () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Flip};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfFlip;

PROCEDURE PerfPromotedRoots () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Roots};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfPromotedRoots;

PROCEDURE PerfStop () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Bye};
  BEGIN
    (* UNSAFE, but needed to prevent deadlock if we're crashing! *)
    EVAL RTPerfTool.Send (perfW, ADR(e), EventSize);
    RTPerfTool.Close (perfW);
  END PerfStop;

PROCEDURE PerfAllow (<*UNUSED*> n: INTEGER := 0) =
  VAR
    e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Off, nb :=
                       disableCount + disableMotionCount};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfAllow;

PROCEDURE PerfBegin () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Begin};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfBegin;

PROCEDURE PerfEnd () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.End};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfEnd;

PROCEDURE PerfChange (first: Page; nb: CARDINAL) =
  VAR
    e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Change, first := first,
                       nb := nb, desc := desc[first - p0]};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfChange;

PROCEDURE PerfGrow (firstNew: Page; nb: CARDINAL) =
  VAR
    e := RTHeapEvent.T{
           kind := RTHeapEvent.Kind.Grow, first := firstNew, nb := nb};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfGrow;

(* ----------------------------------------------------------------------- *)

(* RTWeakRef *)

(* weakTable contains four singly-linked lists: for entries in use (rooted
   at index weakLive0), entries with final cleanup (at weakFinal0), dead
   entries awaiting cleanup (at weakDead0), and free entries (at
   weakFree0).

   Entries in use contain the weak ref, the REF, and the procedure.  The
   "a" field of the weak ref is the index in the table; this speeds lookup.
   The "b" field is a unique value, taken from a 32-bit counter.

   Dead entries contain the same dields, but the "a" field of the weak ref
   is set to -1 to keep lookups from succeeding.  When the cleanup
   procedure is to be called, the original weak ref can still be
   reconstructed, since the "a" field was the index. *)

VAR
  weakTable: UNTRACED REF ARRAY OF WeakEntry; (* allocated in "Init" *)
             (* := NEW(UNTRACED REF ARRAY OF WeakEntry, 0); *)
  weakLive0  := -1;              (* the root of the in-use list *)
  weakFinal0 := -1;              (* the root of the thread-cleanup list *)
  weakDead0  := -1;              (* the root of the dead list *)
  weakFree0  := -1;              (* the root of the free list *)

TYPE
  Int32 = BITS 32 FOR [-16_7fffffff-1 .. 16_7fffffff];
  WeakRefAB = RECORD
                a: Int32;
                b: Int32;
              END;
  WeakEntry = RECORD
                t: WeakRefAB;    (* the weak ref, if well-formed *)
                r: RefReferent;  (* the traced reference *)
                p: ADDRESS;      (* a WeakRefCleanUpProc or a PROCEDURE(r:
                                    REFANY) *)
                next: INTEGER;   (* the next entry on the list *)
              END;

(* This is WeakRef.FromRef, which returns a new weak ref for an object. *)

VAR startedWeakCleaner := FALSE;

PROCEDURE WeakRefFromRef (r: REFANY; p: WeakRefCleanUpProc := NIL): WeakRef =
  VAR
    start           := FALSE;
    result: WeakRef;
  BEGIN
    <* ASSERT r # NIL *>
    RTOS.LockHeap();
    BEGIN
      (* create a WeakCleaner thread the first time through *)
      IF p # NIL AND NOT startedWeakCleaner THEN
        start := TRUE;
        startedWeakCleaner := TRUE;
      END;
      (* if necessary, expand weakTable *)
      IF weakFree0 = -1 THEN ExpandWeakTable(); END;
      IF p # NIL THEN
        (* mark the object as having a weak ref with non-nil cleanup *)
        VAR header := HeaderOf(LOOPHOLE(r, ADDRESS));
        BEGIN
          <* ASSERT NOT header^.weak *>
          header^.weak := TRUE;
        END;
      END;
      (* allocate a new entry *)
      VAR i := weakFree0;
      BEGIN
        weakFree0 := weakTable[i].next;
        (* generate a new weak ref *)
        VAR t := WeakRefAB{a := i, b := Word.Plus(weakTable[i].t.b, 1)};
        BEGIN
          <* ASSERT t.b # 0 *>
          (* set up the entry *)
          weakTable[i] :=
            WeakEntry{t := t, r := LOOPHOLE(r, RefReferent), p :=
                      LOOPHOLE(p, ADDRESS), next := weakLive0};
          weakLive0 := i;
          result := LOOPHOLE(t, WeakRef);
        END;
      END;
    END;
    RTOS.UnlockHeap();
    IF start THEN
      EVAL Thread.Fork(NEW(Thread.Closure, apply := WeakCleaner));
    END;
    RETURN result;
  END WeakRefFromRef;

PROCEDURE ExpandWeakTable () =
  VAR
    newTable := NEW(UNTRACED REF ARRAY OF WeakEntry,
                    2 * NUMBER(weakTable^) + 1);
  BEGIN
    SUBARRAY(newTable^, 0, NUMBER(weakTable^)) := weakTable^;
    FOR i := NUMBER(weakTable^) TO NUMBER(newTable^) - 1 DO
      WITH entry = newTable[i] DO
        entry.t.b := 0;
        entry.next := weakFree0;
        weakFree0 := i;
      END;
    END;
    weakTable := newTable;
  END ExpandWeakTable;

(* This is WeakRef.ToRef, which inverts FromRef *)

PROCEDURE WeakRefToRef (READONLY t: WeakRef): REFANY =
  VAR ab: WeakRefAB;  r: REFANY := NIL;
  BEGIN
    LOOPHOLE (ab, WeakRef) := t;
    RTOS.LockHeap();
    (* if the weak ref is not dead, we know the index *)
    IF ab.a < NUMBER(weakTable^) THEN
      WITH entry = weakTable[ab.a] DO
        (* check the weak ref there *)
        IF entry.t = ab THEN
          <* ASSERT entry.r # NIL *>
          IF collectorState # CollectorState.Zero THEN
            VAR p := ReferentToPage(entry.r);
            BEGIN
              <* ASSERT p # Nil *>
              IF desc[p - p0].space = Space.Previous THEN
                CollectorOn();
                Move(NIL, ADR(entry.r));
                CollectorOff();
              END;
            END;
          END;
          r := LOOPHOLE(entry.r, REFANY);
        END;
      END;
    END;
    RTOS.UnlockHeap();
    RETURN r;
  END WeakRefToRef;

(* This is RTHeapRef.RegisterFinalCleanup, which registers final cleanup
   for a heap object. *)

PROCEDURE RegisterFinalCleanup (r: REFANY; p: PROCEDURE (r: REFANY)) =
  BEGIN
    <* ASSERT r # NIL *>
    <* ASSERT p # NIL *>
    RTOS.LockHeap();
    BEGIN
      (* if necessary, expand weakTable *)
      IF weakFree0 = -1 THEN ExpandWeakTable(); END;
      (* allocate a new entry *)
      VAR i := weakFree0;
      BEGIN
        weakFree0 := weakTable[i].next;
        (* set up the entry, without a weak ref *)
        weakTable[i].r := LOOPHOLE(r, RefReferent);
        weakTable[i].p := LOOPHOLE(p, ADDRESS);
        weakTable[i].next := weakFinal0;
        weakFinal0 := i;
      END;
    END;
    RTOS.UnlockHeap();
  END RegisterFinalCleanup;

(* WeakCleaner waits for entries to be placed on the dead list, then cleans
   them up and puts them on the free list. *)

PROCEDURE WeakCleaner (<*UNUSED*> closure: Thread.Closure): REFANY =
  VAR
    i   : INTEGER;
    copy: WeakEntry;
  BEGIN
    LOOP
      (* get an entry to handle.  copy its contents, then put it on the
         free list. *)
      WHILE weakDead0 = -1 DO Wait(); END;
      RTOS.LockHeap();
      IF weakDead0 = -1 THEN
        RTOS.UnlockHeap();
      ELSE
        i := weakDead0;
        WITH entry = weakTable[i] DO
          <* ASSERT entry.t.a = -1 *>
          CollectorOn();
          Move(NIL, ADR(entry.r));
          CollectorOff();
          copy := entry;
          weakDead0 := entry.next;
          entry.next := weakFree0;
          weakFree0 := i;
        END;
        RTOS.UnlockHeap();
        (* call the registered procedure.  note that collections are
           allowed; the copy is kept on the stack so the object won't be
           freed during the call. *)
        IF copy.p # NIL THEN
          LOOPHOLE(copy.p, WeakRefCleanUpProc)(
            LOOPHOLE(WeakRefAB{a := i, b := copy.t.b}, WeakRef),
            LOOPHOLE(copy.r, REFANY));
        END;
        copy.r := NIL;           (* to help conservative collector *)
      END;
    END;
  END WeakCleaner;

(* ----------------------------------------------------------------------- *)

PROCEDURE FirstPage (p: Page): Page =
  BEGIN
    WHILE desc[p - p0].continued DO DEC(p); END;
    RETURN p;
  END FirstPage;

PROCEDURE PageCount (p: Page): CARDINAL =
  VAR n := 0;
  BEGIN
    <* ASSERT NOT desc[p - p0].continued *>
    REPEAT INC(p); INC(n); UNTIL p >= p1 OR NOT desc[p - p0].continued;
    RETURN n;
  END PageCount;

(* ----------------------------------------------------------------------- *)

PROCEDURE Protect (p: Page; readable, writable: BOOLEAN) =
  VAR n := PageCount(p);
  BEGIN
    <* ASSERT collectorOn OR (readable AND writable) OR desc[p-p0].persistent*>
    <* ASSERT RTHeapDep.VM *>
    <* NOWARN *><* ASSERT TRUE *>
    RTHeapDep.Protect(p, n, readable, writable);
    VAR protected := NOT (readable AND writable);
    BEGIN
      FOR i := 0 TO n - 1 DO desc[p + i - p0].protected := protected; END;
    END;
    IF perfOn THEN PerfChange(p, n); END;
  END Protect;

PROCEDURE Unprotect (p: Page) =
  BEGIN
    Protect(p, readable := TRUE, writable := TRUE);
  END Unprotect;

TYPE Swizzler = RTTypeMap.Visitor OBJECT
  mapDB: RTDB.T;
  stack: Page;
OVERRIDES
  apply := Swizzle
END;

PROCEDURE Swizzle (self: Swizzler; cp: ADDRESS; k: RTTypeMap.Kind) =
  VAR refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
  BEGIN
    CASE k OF
    | RTTypeMap.Kind.UntracedRef, RTTypeMap.Kind.TransientRef =>
      refref^ := NIL;
    | RTTypeMap.Kind.Ref =>
      VAR ref := refref^;
      BEGIN
        IF ref # NIL THEN
          refref^ := SwizzleRef(self.mapDB, ref);
        END
      END
    ELSE
    END;
  END Swizzle;

PROCEDURE SwizzleRef (db: RTDB.T; ref: RefReferent): RefReferent =
  VAR
    p: Page := Word.RightShift(LOOPHOLE(ref, INTEGER), LogBytesPerPage);
    offset := Word.And(LOOPHOLE(ref, INTEGER), BytesPerPage - 1);
    nbPages := 1;
  BEGIN
    IF Word.And(offset, 1) # 0 THEN
      (* large object *)
      nbPages := Word.RightShift(offset, 4);
      CASE Word.And(offset, 2_1111) OF
      | 2_0001 => offset :=   4;
      | 2_0011 => offset :=   8;
      | 2_0101 => offset :=  16;
      | 2_0111 => offset :=  32;
      | 2_1001 => offset :=  64;
      | 2_1011 => offset := 128;
      | 2_1101 => offset := 256;
      | 2_1111 => offset := 512;
      ELSE
        <* ASSERT FALSE *>
      END;
    END;
    p := db.map(p, nbPages);
    VAR pi := p - p0;
    BEGIN
      IF desc[pi].space = Space.Previous THEN
        IF desc[pi].pure THEN
          PromotePage(
              p, Desc{space := Space.Current, generation := copyGeneration,
                      pure := TRUE, note := Note.Persistent,
                      gray := FALSE, protected := desc[pi].protected,
                      continued := FALSE, persistent := TRUE,
                      resident := desc[pi].resident,
                      modified := desc[pi].modified});
        ELSE
          PromotePage(
              p, Desc{space := Space.Current, generation := copyGeneration,
                      pure := FALSE, note := Note.Persistent,
                      gray := TRUE, protected := desc[pi].protected,
                      continued := FALSE, persistent := TRUE,
                      resident := TRUE, modified := desc[pi].modified});
          desc[pi].link := impureCopy.stack;
          impureCopy.stack := p;
          IF NOT collectorOn THEN
            Protect(p, readable := FALSE, writable := FALSE);
          END;
        END
      END
    END;
    RETURN PageToAddress(p) + offset;
  END SwizzleRef;

PROCEDURE Map (mapDB: RTDB.T; mapId: Page; nbPages: CARDINAL): Page =
  (* page has not yet been mapped *)
  BEGIN IF nbPages = 0 THEN nbPages := MapPageCount(mapDB, mapId); END;
  VAR
    notAfter: SET OF Note := SET OF Note{Note.Allocated};
    p := FindFreePages(nbPages, notAfter := notAfter);
    d := Desc{space := Space.Current, generation := copyGeneration,
              pure := TRUE, note := Note.Persistent, gray := FALSE,
              protected := FALSE, continued := FALSE, persistent := TRUE,
              resident := FALSE, modified := FALSE};
  BEGIN
    desc[p - p0] := d;
    mapIds[p - p0] := mapId;
    mapDBs[p - p0] := mapDB;
    IF nbPages > 1 THEN
      d.continued := TRUE;
      FOR i := 1 TO nbPages - 1 DO
        desc[p + i - p0] := d;
        INC(mapId);
        mapIds[p + i - p0] := mapId;
        mapDBs[p + i - p0] := mapDB;
      END;
    END;
    Protect(p, readable := FALSE, writable := FALSE);
    RETURN p;
  END;
  END Map;

PROCEDURE MapPageCount (mapDB: RTDB.T; mapId: Page): CARDINAL =
  (* large object of unknown size: get size from first page *)
  VAR
    data := mapDB.pin(mapId);
    h := LOOPHOLE(data, RefHeader);
    type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
    def: TypeDefn;
    dataSize, nbBytes: CARDINAL;
  BEGIN
    IF type = LOOPHOLE(0, ADDRESS) THEN
      INC(h, ADRSIZE(Header));
      type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
    ELSIF type = LOOPHOLE(1, ADDRESS) THEN
      INC(h, LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^);
      type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
    END;
    def := SwizzleType(mapDB, type);
    IF def.nDimensions = 0 THEN
      dataSize := def.dataSize;
    ELSE
      dataSize := OpenArraySize(h, def);
    END;
    mapDB.unpin(mapId);
    nbBytes := RTMisc.Upper(ADRSIZE(Header), def.dataAlignment) + dataSize;
    RETURN (nbBytes + AdrPerPage - 1) DIV AdrPerPage;
  END MapPageCount;

PROCEDURE SwizzleType (db: RTDB.T; ref: RefReferent): TypeDefn =
  VAR
    fp := LOOPHOLE(SwizzleRef(db, ref), UNTRACED REF Fingerprint.T);
    p: Page := Word.RightShift(LOOPHOLE(fp, INTEGER), LogBytesPerPage);
    tc: Typecode;
  BEGIN
    IF desc[p - p0].protected THEN
      IF desc[p - p0].pure THEN
        (* don't fault page, just get fingerprint *)
        <* ASSERT desc[p - p0].persistent *>
        IF desc[p - p0].resident THEN
          tc := RTTypeFP.FromFingerprint(fp^);
          <* ASSERT desc[p - p0].protected *>
        ELSIF desc[p - p0].modified THEN
          Unprotect(p);
          tc := RTTypeFP.FromFingerprint(fp^);
          Protect(p, readable := FALSE, writable := FALSE);
        ELSE
          VAR
            mapId := mapIds[p - p0];
            data := db.pin(mapId);
          BEGIN
            Unprotect(p);
            RTMisc.Copy(data, PageToAddress(p), BytesPerPage);
            tc := RTTypeFP.FromFingerprint(fp^);
            Protect(p, readable := FALSE, writable := FALSE);
            db.unpin(mapId);
          END;
          desc[p - p0].modified := TRUE;
          desc[p - p0].link := swizzler.stack;
          swizzler.stack := p;
        END
      ELSIF desc[p - p0].gray THEN
        (* don't clean page, just get fingerprint *)
        Unprotect(p);
        tc := RTTypeFP.FromFingerprint(fp^);
        Protect(p, readable := FALSE, writable := FALSE);
      ELSE
        tc := RTTypeFP.FromFingerprint(fp^);
        <* ASSERT desc[p - p0].protected *>
      END;
    ELSE
      tc := RTTypeFP.FromFingerprint(fp^);
    END;
    RETURN RTType.Get(tc);
  END SwizzleType;

PROCEDURE SwizzleBetween (h, he: RefHeader): BOOLEAN =
  VAR
    pure := TRUE;
    type: RefReferent;
    def: TypeDefn;
    referentSize: CARDINAL;
  BEGIN
    WHILE h < he DO
      type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
      IF type = LOOPHOLE(0, ADDRESS) THEN
        h^ := Header{typecode := Fill_1_type};
        referentSize := 0;
      ELSIF type = LOOPHOLE(1, ADDRESS) THEN
        h^ := Header{typecode := Fill_N_type};
        referentSize := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
        DEC(referentSize, ADRSIZE(Header));
      ELSE
        def := SwizzleType(swizzler.mapDB, type);
        <* ASSERT def.traced = 1 *>
        h^ := Header{typecode := def.typecode};
        IF def.nDimensions = 0 THEN
          referentSize := def.dataSize;
          IF def.defaultMethods # NIL THEN
            (* object: set up methods pointer *)
            LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF RT0.MethodSuite)^ :=
                def.defaultMethods;
          END;
        ELSE
          (* open array: set up the internal pointer *)
          LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF ADDRESS)^ :=
              h + ADRSIZE(Header) + def.dataSize;
          referentSize := OpenArraySize(h, def);
        END;
        IF def.gc_map # NIL OR def.parent # NIL THEN
          pure := FALSE;
          <*FATAL ANY*>
          BEGIN
            RTTypeMap.DoWalkRef(def, h + ADRSIZE(Header),
                                RTTypeMap.Mask{RTTypeMap.Kind.Ref,
                                               RTTypeMap.Kind.UntracedRef,
                                               RTTypeMap.Kind.TransientRef},
                                swizzler);
          END;
        END;
      END;
      INC(h, ADRSIZE(Header) + referentSize);
    END;
    RETURN pure;
  END SwizzleBetween;

PROCEDURE SwizzleRoot (db: RTDB.T): RefReferent =
  (* root should be first object on page 1 *)  
  VAR rootId := LOOPHOLE(BytesPerPage, ADDRESS) + ADRSIZE(Header);
  BEGIN
    RETURN SwizzleRef(db, rootId);
  END SwizzleRoot;

PROCEDURE Fault (addr: ADDRESS): BOOLEAN =
  VAR p := LOOPHOLE(addr, INTEGER) DIV BytesPerPage;
  BEGIN
    ThreadF.SuspendOthers();
    <* ASSERT RTHeapDep.VM *>
    <* NOWARN *><* ASSERT TRUE *>
    cycleCost :=
      cycleCost + RTHeapDep.VMFaultTime(); (* otherwise unaccounted for *)
    IF NOT (p0 <= p AND p < p1) OR desc[p - p0].space = Space.Unallocated THEN
      ThreadF.ResumeOthers();
      RETURN FALSE;              (* not in heap *)
    END;
    IF NOT desc[p - p0].protected THEN
      ThreadF.ResumeOthers();
      RETURN TRUE;               (* was protected, but not any more *)
    END;

    IF desc[p - p0].pure THEN
      <* ASSERT desc[p - p0].persistent *> (* pure + protected = persistent *)
      p := FirstPage(p);
      IF desc[p - p0].resident THEN
        (* pure resident persistent page being modified *)
        IF NOT desc[p - p0].modified THEN
          mapDBs[p - p0].lock(mapIds[p - p0], RTDB.LockMode.WRITE);
          desc[p - p0].modified := TRUE;
        END;
        Unprotect(p);
      ELSE (* non-resident, so fault the page(s) *)
        VAR
          mapDB := mapDBs[p - p0];
        BEGIN
          FOR pp := p TO p + PageCount(p) - 1 DO
            IF NOT desc[pp - p0].modified THEN
              VAR
                mapId := mapIds[pp - p0];
                data := mapDB.pin(mapId);
              BEGIN
                Unprotect(p);
                RTMisc.Copy(data, PageToAddress(pp), BytesPerPage);
                Protect(p, readable := FALSE, writable := FALSE);
                mapDB.unpin(mapId);
              END;
              desc[pp - p0].modified := TRUE;
            END
          END
        END;

        <*ASSERT swizzler.stack = Nil*>
        desc[p - p0].link := Nil;
        swizzler.stack := p;

        IF NOT collectorOn THEN Disable(); END;
        WHILE swizzler.stack # Nil DO
          p := swizzler.stack;
          swizzler.stack := desc[p - p0].link;
          swizzler.mapDB := mapDBs[p - p0];
          <*ASSERT desc[p - p0].protected
               AND NOT desc[p - p0].resident
               AND desc[p - p0].modified*>
          Unprotect(p);
          desc[p - p0].pure :=
              SwizzleBetween(PageToHeader(p), PageToHeader(p + 1));
          desc[p - p0].resident := TRUE;
          desc[p - p0].modified := FALSE;
          VAR d := desc[p - p0];
          BEGIN
            d.continued := TRUE;
            FOR pp := p+1 TO p + PageCount(p) - 1 DO
              desc[pp - p0] := d;
            END;
          END;
          Protect(p, readable := TRUE, writable := FALSE);
        END;          
        IF NOT collectorOn THEN Enable(); END;
      END;
      ThreadF.ResumeOthers();
      RETURN TRUE;
    END;

    IF desc[p - p0].gray THEN
      CollectorOn();
      IF p = impureCopy.page THEN
        EVAL CopySome();
        IF desc[p - p0].gray THEN
          <* ASSERT p = impureCopy.page *>
          InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
          impureCopy.page := Nil;
          impureCopy.ptr := NIL;
          impureCopy.boundary := NIL;
          CleanBetween(PageToHeader(p), PageToHeader(p + 1));
          desc[p - p0].gray := FALSE;
          IF perfOn THEN PerfChange(p, 1); END;
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].modified *>
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, readable := TRUE, writable := FALSE);
          END;
        END;
      ELSIF p = impureTransientCopy.page THEN
        EVAL CopySome();
        IF desc[p - p0].gray THEN
          <* ASSERT p = impureTransientCopy.page *>
          InsertFiller(impureTransientCopy.ptr,
                       impureTransientCopy.boundary - impureTransientCopy.ptr);
          impureTransientCopy.page := Nil;
          impureTransientCopy.ptr := NIL;
          impureTransientCopy.boundary := NIL;
          CleanBetween(PageToHeader(p), PageToHeader(p + 1));
          desc[p - p0].gray := FALSE;
          IF perfOn THEN PerfChange(p, 1); END;
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].modified *>
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, readable := TRUE, writable := FALSE);
          END;
        END;
      ELSE
        p := FirstPage(p);
        Unprotect(p);
        CleanBetween(PageToHeader(p), PageToHeader(p + 1));
        FOR i := 0 TO PageCount(p) - 1 DO
          desc[p + i - p0].gray := FALSE;
        END;
        IF perfOn THEN PerfChange(p, PageCount(p)); END;
        IF desc[p - p0].generation = Generation.Older 
          OR NOT desc[p - p0].modified THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          Protect(p, readable := TRUE, writable := FALSE);
        END;
      END;
      CollectorOff();
      ThreadF.ResumeOthers();
      RETURN TRUE;
    END;

    p := FirstPage(p);
    IF desc[p - p0].modified THEN
      <* ASSERT desc[p - p0].generation = Generation.Older *>
    ELSE
      <*ASSERT desc[p - p0].space = Space.Current*>
      <*ASSERT desc[p - p0].persistent *>
      mapDBs[p - p0].lock(mapIds[p - p0], RTDB.LockMode.WRITE);
      desc[p - p0].modified := TRUE;
    END;
    Unprotect(p);
    ThreadF.ResumeOthers();
    RETURN TRUE;                 (* was protected, protection cleared *)
  END Fault;

(* ----------------------------------------------------------------------- *)

(****** Page-level allocator ******)

(* The freelist is sorted by blocksize, linked through the first page in
   each block, using the "link" field in the "desc" array.  Page allocation
   is best-fit.  For elements of the same blocksize, they are sorted by
   page number, to make the showheap display more easily readable, and to
   slightly reduce fragmentation. *)

(* FindFreePages allocates a run of "n" free pages, which we would prefer
   not be near pages in the current space with notes in notAfter.  The
   allocator can thus be used to separate pages with different notes, since
   they will have different lifetimes.  This is a concern only when
   incremental and generational collection are combined. *)

PROCEDURE FindFreePages (n: INTEGER; notAfter: SET OF Note): Page =
  VAR p: Page;
  BEGIN
    IF collectorState = CollectorState.Zero THEN
      p := AllocateFreePagesFromBlock(n, SET OF Note{}, TRUE);
      IF p # Nil THEN RETURN p; END;
    ELSE
      p := AllocateFreePagesFromBlock(n, notAfter, TRUE);
      IF p # Nil THEN RETURN p; END;
      p := AllocateFreePagesFromBlock(n, SET OF Note{}, FALSE);
      IF p # Nil THEN RETURN p; END;
    END;
    GrowHeap(n);
    p := AllocateFreePagesFromBlock(n, SET OF Note{}, TRUE);
    <* ASSERT p # Nil *>
    RETURN p;
  END FindFreePages;

VAR free: Page;                  (* the head of the freelist *)

(* AllocateFreePagesFromBlock finds the first block large enough to satisfy
   the request.  "notAfter" is the set of page notes in the current space
   that the block allocated from must not immediately follow; this is used
   to separate Note.Allocated pages from Note.Copied pages.  If "front" is
   TRUE, the pages will be allocated from the beginning of the block, else
   from the end; this is also used to separate Note.Allocated Pages from
   Note.Copied pages.  If the block is bigger than the request, the
   remainder is left at the right point in the freelist.  If no block
   exists, Nil is returned. *)

PROCEDURE AllocateFreePagesFromBlock (n       : INTEGER;
                                      notAfter: SET OF Note;
                                      front   : BOOLEAN      ): Page =
  VAR
    p                   := free;
    prevP               := Nil;
    prevLength          := 0;
    length    : INTEGER;
  BEGIN
    LOOP
      IF p = Nil THEN RETURN Nil; END;
      length := FreeLength(p);
      IF length >= n
           AND NOT (p > p0 AND desc[(p - 1) - p0].space = Space.Current
                      AND desc[(p - 1) - p0].note IN notAfter) THEN
        EXIT;
      END;
      prevP := p;
      prevLength := length;
      p := desc[p - p0].link;
    END;
    IF length = n THEN
      IF prevP = Nil THEN
        free := desc[p - p0].link;
      ELSE
        desc[prevP - p0].link := desc[p - p0].link;
      END;
      RETURN p;
    ELSE
      VAR
        newP, fragP: Page;
        fragLength : CARDINAL;
      BEGIN
        IF front THEN
          newP := p;
          fragP := p + n;
        ELSE
          newP := p + length - n;
          fragP := p;
        END;
        fragLength := length - n;
        IF fragLength > prevLength THEN
          IF prevP = Nil THEN
            free := fragP;
          ELSE
            desc[prevP - p0].link := fragP;
          END;
          desc[fragP - p0].link := desc[p - p0].link;
        ELSE
          IF prevP = Nil THEN
            free := desc[p - p0].link;
          ELSE
            desc[prevP - p0].link := desc[p - p0].link;
          END;
          VAR
            pp     := free;
            prevPP := Nil;
          BEGIN
            LOOP
              IF pp = Nil THEN EXIT; END;
              VAR length := FreeLength(pp);
              BEGIN
                IF length > fragLength
                     OR (length = fragLength AND pp > fragP) THEN
                  EXIT;
                END;
              END;
              prevPP := pp;
              pp := desc[pp - p0].link;
            END;
            desc[fragP - p0].link := pp;
            IF prevPP = Nil THEN
              free := fragP;
            ELSE
              desc[prevPP - p0].link := fragP;
            END;
          END;
        END;
        RETURN newP;
      END;
    END;
  END AllocateFreePagesFromBlock;

(* RebuildFreelist rebuilds the free list, from the "desc" array.  It first
   links all free blocks into the free list, then it sorts the free list.
   The sort used is insertion sort, which is quadratic in the number of
   different block sizes, but only linear in the number of pages. *)

PROCEDURE RebuildFreelist () =
  BEGIN
    VAR
      prevP     := Nil;
      prevSpace := Space.Unallocated;
    BEGIN
      (* link together the first pages of all free blocks *)
      FOR p := p0 TO p1 - 1 DO
        VAR space := desc[p - p0].space;
        BEGIN
          IF space = Space.Free AND prevSpace # Space.Free THEN
            IF prevP = Nil THEN
              free := p;
            ELSE
              desc[prevP - p0].link := p;
            END;
            prevP := p;
          END;
          prevSpace := space;
        END;
      END;
      IF prevP = Nil THEN
        free := Nil;
      ELSE
        desc[prevP - p0].link := Nil;
      END;
    END;
    (* sort them, using insertion sort *)
    VAR
      n     := 1;                (* smallest block size *)
      p     := free;             (* start of sublist we're examining *)
      prevP := Nil;              (* element before sublist *)
    BEGIN
      LOOP
        VAR
          excess     := Nil;
          prevExcess := Nil;
        BEGIN
          (* separate off blocks over "n" long into excess list *)
          WHILE p # Nil DO
            VAR length := FreeLength(p);
            BEGIN
              <* ASSERT length >= n *>
              IF length > n THEN
                IF prevExcess = Nil THEN
                  excess := p;
                ELSE
                  desc[prevExcess - p0].link := p;
                END;
                IF prevP = Nil THEN
                  free := desc[p - p0].link;
                ELSE
                  desc[prevP - p0].link := desc[p - p0].link;
                END;
                prevExcess := p;
              ELSE
                prevP := p;
              END;
            END;
            p := desc[p - p0].link;
          END;
          (* maybe done *)
          IF excess = Nil THEN EXIT; END;
          <* ASSERT prevExcess # Nil *>
          (* link longer blocks onto end *)
          IF prevP = Nil THEN
            free := excess;
          ELSE
            desc[prevP - p0].link := excess;
          END;
          desc[prevExcess - p0].link := Nil;
          p := excess;
        END;
        (* find smallest element size of remaining bocks *)
        n := LAST(CARDINAL);
        VAR pp := p;
        BEGIN
          REPEAT
            VAR length := FreeLength(pp);
            BEGIN
              IF length < n THEN n := length; END;
            END;
            pp := desc[pp - p0].link;
          UNTIL pp = Nil;
        END;
      END;
    END;
  END RebuildFreelist;

(* FreeLength returns the number of free pages starting at page p. *)

PROCEDURE FreeLength (p: Page): CARDINAL =
  BEGIN
    <* ASSERT desc[p - p0].space = Space.Free *>
    VAR pp := p + 1;
    BEGIN
      LOOP
        IF pp >= p1 THEN EXIT; END;
        IF desc[pp - p0].space # Space.Free THEN EXIT; END;
        INC(pp);
      END;
      RETURN pp - p;
    END;
  END FreeLength;

(* GrowHeap adds a block of at least "minNewPages" free pages to the heap,
   and links it into the free list. *)

VAR fragment0, fragment1: ADDRESS := NIL;

CONST
  InitialBytes = 262144;         (* initial heap size is 256K *)
  MinNewBytes  = 262144;         (* grow the heap by at least 256K *)
  MinNewFactor = 0.2;            (* grow the heap by at least 20% *)

VAR
  mapIds: UNTRACED REF ARRAY OF Page;
  mapDBs: UNTRACED REF ARRAY OF RTDB.T;

PROCEDURE GrowHeap (pp: INTEGER) =
  VAR
    newChunk    : ADDRESS;
    newSideSpan : INTEGER;
    firstNewPage: Page;
    lastNewPage : Page;
    newP0       : Page;
    newP1       : Page;
  BEGIN
    IF allocatedPages = 0 THEN
      pp := MAX(pp, (InitialBytes + BytesPerPage - 1) DIV BytesPerPage);
    ELSE
      pp := MAX(pp, (MinNewBytes + BytesPerPage - 1) DIV BytesPerPage);
      pp := MAX(pp, CEILING(FLOAT(allocatedPages) * MinNewFactor));
    END;
    VAR bytes := (pp + 1) * BytesPerPage;
    BEGIN
      newChunk := RTOS.GetMemory(bytes);
      IF newChunk = NIL OR newChunk = LOOPHOLE(-1, ADDRESS) THEN
        RTMisc.FatalError(NIL, 0, "gc: Could not extend the traced heap");
      END;
      IF fragment1 = newChunk THEN
        newChunk := fragment0;
        bytes := bytes + (fragment1 - fragment0);
      END;
      VAR excess := (-LOOPHOLE(newChunk, INTEGER)) MOD BytesPerPage;
      BEGIN
        INC(newChunk, excess);
        DEC(bytes, excess);
      END;
      VAR pages := bytes DIV BytesPerPage;
      BEGIN
        firstNewPage := LOOPHOLE(newChunk, INTEGER) DIV BytesPerPage;
        lastNewPage := firstNewPage + pages - 1;
        fragment0 :=
          LOOPHOLE((firstNewPage + pages) * BytesPerPage, ADDRESS);
        fragment1 := newChunk + bytes;
      END;
    END;
    (* determine the new boundaries of the allocated pages *)
    IF p0 = Nil THEN
      newP0 := firstNewPage;
      newP1 := lastNewPage + 1;
    ELSIF firstNewPage < p0 THEN
      newP0 := firstNewPage;
      newP1 := p1;
    ELSIF p1 <= lastNewPage THEN
      newP0 := p0;
      newP1 := lastNewPage + 1;
    ELSE
      newP0 := p0;
      newP1 := p1;
    END;
    (* extend the side arrays if necessary *)
    newSideSpan := newP1 - newP0;
    IF desc = NIL OR newSideSpan # NUMBER(desc^) THEN
      WITH newDesc  = NEW(UNTRACED REF ARRAY OF Desc,   newSideSpan) DO
      WITH newMapId = NEW(UNTRACED REF ARRAY OF Page,   newSideSpan) DO
      WITH newMapDB = NEW(UNTRACED REF ARRAY OF RTDB.T, newSideSpan) DO
        IF desc # NIL THEN
          <* ASSERT mapIds # NIL AND mapDBs # NIL *>
          FOR i := FIRST(desc^) TO LAST(desc^) DO
            newDesc [i + p0 - newP0] := desc[i];
            newMapId[i + p0 - newP0] := mapIds[i];
            newMapDB[i + p0 - newP0] := mapDBs[i];
          END;
          FOR i := p1 TO firstNewPage - 1 DO
            newDesc [i - newP0].space := Space.Unallocated;
            newMapId[i - newP0]       := Nil;
            newMapDB[i - newP0]       := NIL;
          END;
          FOR i := lastNewPage + 1 TO p0 - 1 DO
            newDesc [i - newP0].space := Space.Unallocated;
            newMapId[i - newP0]       := Nil;
            newMapDB[i - newP0]       := NIL;
          END;
          DISPOSE(desc);
          DISPOSE(mapIds);
          DISPOSE(mapDBs);
        END;
        desc   := newDesc;
        mapIds := newMapId;
        mapDBs := newMapDB;
      END;
      END;
      END;
    END;
    p0 := newP0;
    p1 := newP1;
    FOR i := firstNewPage TO lastNewPage DO
      desc[i - p0].space := Space.Free;
    END;
    IF perfOn THEN
      PerfGrow(firstNewPage, lastNewPage - firstNewPage + 1);
    END;
    INC(allocatedPages, lastNewPage - firstNewPage + 1);
    RebuildFreelist();
  END GrowHeap;

(*** SYNCHRONIZATION ***)

(* Wait() waits on a condition, implemented compatibly with RTOS.LockHeap().
   Wait() will be called only from background threads.  The caller
   will be in a critical section.  *)

VAR
  mutex     := NEW(MUTEX);
  condition := NEW(Thread.Condition);

PROCEDURE Wait () =
  BEGIN
    (* This procedure may not be called from inside the collector. *)
    LOCK mutex DO Thread.Wait(mutex, condition); END;
  END Wait;

(* Broadcast() broadcasts a condition, implemented compatibly with
   RTOS.LockHeap().  Broadcast will not be called from inside the collector, so
   that references to the heap will succeed. *)

PROCEDURE Broadcast () =
  BEGIN
    Thread.Broadcast(condition);
  END Broadcast;

(*** INITIALIZATION ***)

CONST MaxAlignment  = 8;
CONST MaxAlignMask  = 2_0111;     (* bit mask to capture MaxAlignment *)
TYPE  MaxAlignRange = [0 .. MaxAlignment - 1];

VAR align: ARRAY MaxAlignRange, [1 .. MaxAlignment] OF CARDINAL;
(* align[i,j] == RTMisc.Align (i, j) - i *)

PROCEDURE Init () =
  BEGIN
    weakTable := NEW(UNTRACED REF ARRAY OF WeakEntry, 0);

    (* initialize the alignment array *)
    FOR i := FIRST(align) TO LAST(align) DO
      FOR j := FIRST(align[0]) TO LAST(align[0]) DO
        align[i, j] := RTMisc.Upper(i, j) - i;
      END;
    END;
  END Init;

VAR
  stats: RECORD
    print := FALSE;
    accuratePages, ambiguousPages := 0;
    objects, bytes: UNTRACED REF ARRAY OF INTEGER;
  END;

BEGIN
  IF RTParams.IsPresent("nogc") THEN disableCount := 1; END;
  IF RTParams.IsPresent("novm") THEN disableVMCount := 1; END;
  IF RTParams.IsPresent("noincremental") THEN incremental := FALSE; END;
  IF RTParams.IsPresent("nogenerational") THEN generational := FALSE; END;
  IF RTParams.IsPresent("paranoidgc") THEN InstallSanityCheck(); END;
  IF RTParams.IsPresent("stats") THEN
    stats.print := TRUE;
    stats.objects := NEW(UNTRACED REF ARRAY OF INTEGER, RT0u.nTypes);
    stats.bytes   := NEW(UNTRACED REF ARRAY OF INTEGER, RT0u.nTypes);
  END;
  PerfStart();
END RTCollector.
