INTERFACE RGRASGraph;

(***************************************************************************)
(** Created by:  Roland Baumann						   *)

(** $Author: hosking $
    $Revision: 1.1.1.1 $
    $Date: 1998/12/03 18:54:57 $
    $Log: RGRASGraph.i3,v $
    Revision 1.1.1.1  1998/12/03 18:54:57  hosking
    Import of GRAS3-1.1

    Revision 1.9  1998/07/29 09:30:47  roland
    New parameter profile for AGLogin.

    Revision 1.8  1998/07/27 13:31:01  roland
    Procedure AGDeleteScheme added.

    Revision 1.7  1998/05/15 12:21:58  renehuel
    Adapted the RGRASGraph interface to handle events.

    Revision 1.6  1998/02/24 15:46:23  renehuel
    AGDeleteGraphPool now has a new parameter force, which forces it to
    delete the pool, disregarding the presence of graphs and schemes.
    AGShowAllNodesWithIndex and AGShowNodeWithKez now have a paramter
    attLength, determining the maximum length of the given attribute
    value.
    New procedure AGFileSize, returning the size of a given file of a pool
    in pages.

    Revision 1.5  1998/01/14 17:52:26  roland
    AGAppendNodeType only for schemes.
    Bugfix in AGPutAttributeSubstr and AGDeclareDependency.

    Revision 1.4  1997/12/23 12:41:23  renehuel
    The parameters poolName, graphName and schemeName now dont use the
    type TEXT in the interface any more, but the type PoolName, GraphName
    and SchemeName.

    Revision 1.3  1997/10/24 14:39:05  renehuel
    These files implement the new RGRASGraph interface.

    Revision 1.2  1997/05/13 09:30:02  roland
    Interface cleaned up a little.
    Implementation uses current modul names.

    Revision 1.1  1996/09/13 15:44:33  walkmark
    Interface to use old GRAS Implementations with new GRAS (basic
    procedures) (hope it will work)

*)
(***************************************************************************)

(** An interface to enable applications of the old RGRAS-System to use
    GRAS_3 with as few changes as possible *)

FROM RGGlobal IMPORT GraphPoolMode, TStatus, ExternNumber,
                     GraphNumber, SimpleSet, SimpleElement, GraphType,
                     RelSet, GroupMode, GroupName, GraphMode, GroupType,
                     NodeNumber, TypeNumber, AttributeNumber, SchemeNumber,
                     Cardinality, AttributeKind, IndexProperties,
                     EvalFunction, ValueTypeNumber, DependencyKind,
                     PoolName, GraphName, SchemeName, GraphEvent,
                     ActionPriority, ActionProc;
IMPORT Word;

(**                      Error description                       *)
(**                                                              *)
(** All error messages which can occur during the RGRAS          *)
(** operations are explained in the following lines.             *)
(**                                                              *)
(**                                                              *)
(** Access.Denied:                                               *)
(**   Conflicting access modes.                                  *)
(** Access.Locked:                                               *)
(**   Currently not available.                                   *)
(** ChgMgmtGraph.InternalError:                                  *)
(**   An internal error occurred.                                *)
(** ChgMgmtGraph.LogError:                                       *)
(**   Operations on the log failed.                              *)
(** ChgMgmtGraph.NoLog:                                          *)
(**   Graph is in LogMode.None. User-Recovery is not allowed.    *)
(** ChgMgmtGraphPool.Existent:                                   *)
(**   Operation would overwrite an existing graph.               *)
(** ChgMgmtGraphPool.InternalError:                              *)
(**   An internal error occurred.                                *)
(** ChgMgmtGraphPool.InUse:                                      *)
(**   The operation is only allowed on graphs which are not used *)
(**  (open) by any  clients.                                     *)
(** ChgMgmtGraphPool.NotExistent:                                *)
(**   Operation tries to access a not existing graph             *)
(** DaemonList.EntryNotInList:                                   *)
(**   The known triggers are kept in a list as                   *)
(**   graphNumber/graphEvent/action tuples. Referring to a tuple *)
(**   not in the list will raise this exception.                 *)
(** Graph.AlreadyBound:                                          *)
(**   An evaluation function different from the one given is     *)
(**   bound to name.                                             *)
(** EventType.Mismatch:                                          *)
(**                                                              *)
(** EventType.Unknown:                                           *)
(**                                                              *)
(** EventTypes.Unknown:                                          *)
(**                                                              *)
(** Graph.CardinalityError:                                      *)
(**   Edge or attribute value cardinalities are violated.        *)
(** Graph.CylicEvaluation:                                       *)
(**   An attribute occurred twice in an attribute evaluation     *)
(**   sequence.                                                  *)
(** Graph.InternalError:                                         *)
(**   An internal error occurred.                                *)
(** Graph.InUse:                                                 *)
(**   The graph is still in use.                                 *)
(** Graph.LogError:                                              *)
(**   Operations on the log failed.                              *)
(** Graph.NodeNotFound:                                          *)
(**   Raised when error checks are active and the node does      *)
(**   not exist.                                                 *)
(** Graph.NoIndex:                                               *)
(**   This operation is only allowed for index attributes.       *)
(** Graph.NoScheme:                                              *)
(**   Either no scheme found/specified or the name given is no   *)
(**   scheme.                                                    *)
(** Graph.NotExistent:                                           *)
(**   The graph does not exist.                                  *)
(** Graph.NotOwner:                                              *)
(**   Operation must be called on a different graph.             *)
(** Graph.Unknown:                                               *)
(**   The given node/edge class/type or attribute number is not  *)
(**   declared.                                                  *)
(** Graph.WrongType:                                             *)
(**   This operation is not allowed for the given node/edge      *)
(**   class/type.                                                *)
(** GraphList.EntryNotInList:                                    *)
(**   The open graphs are internally kept in a list              *)
(**   an can be referred to by an internal number. If a number   *)
(**   is used, which can not be found in the list, that means    *)
(**   that there is no corresponding open graph, this error is   *)
(**   displayed.                                                 *)
(** GraphPool.CardinalityError:                                  *)
(**   If a graph detects a cardinality error during transaction  *)
(**   commit, it reports it with this exception. The number is   *)
(**   the pool number of the faulting graph.                     *)
(** GraphPool.Existent:                                          *)
(**   Operation would overwrite an existing graph.               *)
(** GraphPool.InternalError:                                     *)
(**   An internal error occurred.                                *)
(** GraphPool.InTransaction:                                     *)
(**   Copying files between two pools is only possible outside   *)
(**   of transactionstions.                                      *)
(** GraphPool.InUse:                                             *)
(**   The operation is only allowed on graphs which are not used *)
(**   (open) by any clients.                                     *)
(** GraphPool.NoGraph:                                           *)
(**   The graph with the given name does not exist in the pool.  *)
(** GraphPool.NoScheme:                                          *)
(**   The scheme with the given name does not exist in the pool. *)
(** GraphPool.NotExistent:                                       *)
(**   Operation tries to access a not existing graph.            *)
(** GraphPool.NotInTransaction:                                  *)
(**   Commit/AbortTransaction called outside a transaction.      *)
(** GraphSystem.InternalError:                                   *)
(**   An internal error occurred.                                *)
(** More than one source node:                                   *)
(**   The operation can only return a reasonable result if       *)
(**   exactly one source node was found, but there were more.    *)
(** More than one target node:                                   *)
(**   The operation can only return a reasonable result if       *)
(**   exactly one target node was found, but there were more.    *)
(** Names.Undeclared:                                            *)
(**   An operation was called with an undeclared collection or   *)
(**   relation.                                                  *)
(** Names.Unknown:                                               *)
(**   A name was referenced that is not in the collection.       *)
(** PageFile.NoAccess:                                           *)
(**   The graph/file cannot be accessed because of access rights.*)
(** Pending transactions:                                        *)
(**   The desired action cannot be executed due to pending       *)
(**   transactions in the pool                                   *)
(** PersistentGraph.IndexUnused:                                 *)
(**   'deleteIndex' was called with an index number that was not *)
(**   used for this node.                                        *)
(** PersistentGraph.IndexUsed:                                   *)
(**   Raised when put index is called for an already set index   *)
(**   no.                                                        *)
(** PersistentGraph.InternalError:                               *)
(**   An internal error occurred.                                *)
(** PersistentGraph.NodeNotFound:                                *)
(**   Raised when error checks are active and the node does      *)
(**   not exist.                                                 *)
(** PersistentGraph.NotOwner:                                    *)
(**   Node or edge does not belong to this graph.                *)
(** RessourceTriggerStorage.TriggerDelayed:                      *)
(**                                                              *)
(** RGRASNames.InternalError:                                    *)
(**   An internal error occurred.                                *)
(** Scheme.AlreadyDeclared:                                      *)
(**   An item with this name already exists.                     *)
(** Scheme.AttributeNameClash:                                   *)
(**   A class inherits two attributes with the same name from    *)
(**   different super-classes.                                   *)
(** Scheme.AttributePlacementClash                               *)
(**   A class has two attributes with overlapping placements     *)
(** Scheme.Cyclic:                                               *)
(**   The inheritance relation would form a cycle.               *)
(** Scheme.Existent:                                             *)
(**   The part that should be created already exists in          *)
(**   the scheme.                                                *)
(** Scheme.InternalError:                                        *)
(**   An internal error occurred.                                *)
(** Scheme.InUse:                                                *)
(**   The scheme cannot be changed, cause a graph using this     *)
(**   scheme does exist.                                         *)
(** Scheme.NotDeclared:                                          *)
(**   A given type/class/attribute is not declared.              *)
(** Scheme.NotExistent:                                          *)
(**   The referred part of the scheme does not exist.            *)
(** Scheme.NoValidScheme:                                        *)
(**   The object is not valid scheme                             *)
(** The graph is not open:                                       *)
(**   A graph is referred to as open, but it is not open.        *)
(** The pool could not be found:                                 *)
(**   The pool with the given name could not be found in the     *)
(**   graph system because it does not exist.                    *)
(** The pool is not open:                                        *)
(**   A pool is referred to as open, but it is not open.         *)
(** The scheme is already open:                                  *)
(**   The scheme which is to be opened is already open.          *)
(** The scheme is not open:                                      *)
(**   A scheme is referred to as open, but it is not open.       *)
(** The set has not been created:                                *)
(**   In the most set operations the set has to be created       *)
(**   before it can be used with the operation. If it was        *)
(**   not created, this error will appear                        *)
(** The sourceset has not been created:                          *)
(**   see above                                                  *)
(** The target or sourceset has not been created:                *)
(**   see above                                                  *)
(** TypedNames.InternalError:                                    *)
(**   An internal error occurred.                                *)
(** TypedNames.Unknown:                                          *)
(**  One of the used identifiers are not known.                  *)
(** VirtualResource.FatalError                                   *)
(**  A fatal error occurred.                                     *)
(**                                                              *)



PROCEDURE AGLogin (ClientRoot: TEXT         := NIL;
                   CacheSize : CARDINAL     := 0;
                   ServerId  : TEXT         := NIL;
                   Agent     : TEXT         := NIL);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Initialization of the GRAS system.                 *)
  (**                                                      *)
  (**   ClientRoot is the directory in which private       *)
  (**   client data is stored. IF not given, RGRAS tries   *)
  (**   to read the environment variable GRAS3 instead.    *)
  (**                                                      *)
  (**   CacheSize determines the size of the GRAS client   *)
  (**   cache in 8 kByte pages (default is 100).           *)
  (**                                                      *)
  (**   ServerId is the name of the GRAS page server to    *)
  (**   which this client should try to connect. Default   *)
  (**   is GRAS-3.<uid>, with <uid> being the numerical    *)
  (**   user id of the user running the program.           *)
  (**                                                      *)
  (**   Agent is a host name on which the netobjd process  *)
  (**   is running. Default is given in Config.i3          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGLogout ();
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Logging out the GRAS system.                       *)
  (**   All opened schemes, graphs and pools are closed.   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   RessourceTriggerStorage.TriggerDelayed             *)
  (**   GraphPool.InternalError                            *)
  (**   Graph.InternalError                                *)
  (**   Scheme.InternalError                               *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                Operations for managing graph pools                   *)
(**                                                                      *)
(*************************************************************************)

PROCEDURE AGOpenGraphPool (    poolName: PoolName;
                               mode    : GraphPoolMode;
                           VAR status  : TStatus        );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creation of a graph pool which can hold several    *)
  (**   graphs.                                            *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   AlreadyExistent: pool already exists               *)
  (**   AlreadyOpen : pool already open                    *)
  (**   NotExistent : pool does not exist                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   RGRASNames.InternalError                           *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   GraphSystem.InternalError                          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCloseGraphPool (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Closes the graph pool.                             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   GraphPool.InternalError                            *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteGraphPool (    poolName: PoolName;
                             VAR status  : TStatus;
                                 force   : BOOLEAN    := FALSE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes a pool if there are no graphs  or schemes  *)
  (**   left in it, or force = TRUE.                       *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   NotExistent: pool does not exist                   *)
  (**   NotEmpty   : there is at least one graph or        *)
  (**                scheme left in the pool.              *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphSystem.InternalError                          *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   GraphPool.InternalError                            *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

(*PROCEDURE AGReorganizeGraphPool (Pool: PoolName; VAR Status: TStatus);*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Reorganizes the graph pool by removing all unused  *)
(**   information from the management graph. Besides,    *)
(**   all unnecessary GRAS files are deleted. Deltas are *)
(**   packed and combined if possible. Furthermore all   *)
(**   existing locks on graphs are released (which are   *)
(**   due to application or system crashes) and recovery *)
(**   is initiated for all previously locked graphs      *)
(**   based on information in TMPGRAS (see also AGCheck- *)
(**   Graph).                                            *)
(**                                                      *)
(** Warning:                                             *)
(**                                                      *)
(**   This PROCEDURE may only be called by privileged    *)
(**   application modules, because any access to the     *)
(**   graph pool is locked until AGReorganizeGraphPool   *)
(**   is terminated.                                     *)
(**                                                      *)
(** Note:                                                *)
(**                                                      *)
(**   AGReorganizeGraphPool may take a rather long time  *)
(**   to do its job!                                     *)
(**                                                      *)
(** Errors Returned:                                     *)
(**                                                      *)
(**   AlreadyExistent: pool already exists               *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**                                                      *)
(*********************************************************)

(*PROCEDURE AGRecoverGraphPool (Pool: PoolName; VAR Status: TStatus);*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   are due to application or system crashes) and      *)
(**   recovery is initiated for all previously locked    *)
(**   graphs based on information in TMPGRAS.            *)
(**                                                      *)
(** Warning:                                             *)
(**                                                      *)
(** Note:                                                *)
(**                                                      *)
(** Errors Returned:                                     *)
(**                                                      *)
(**   AlreadyExistent: pool already exists               *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**                                                      *)
(*********************************************************)

(*PROCEDURE AGResetGraphPool (Pool: PoolName; VAR Status: TStatus);*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   All existing locks on graphs are released (which   *)
(**   are due to application or system crashes).         *)
(** Warning:                                             *)
(**                                                      *)
(** Note:                                                *)
(**                                                      *)
(** Errors Returned:                                     *)
(**                                                      *)
(**   AlreadyExistent: pool already exists               *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**                                                      *)
(*********************************************************)

PROCEDURE AGGetGraphsInGraphPool (poolName: PoolName; VAR existent: BOOLEAN);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Tests, whether the pool exists. If so, a list of   *)
  (**   all external graph numbers of the graphs in this   *)
  (**   pool is created. It can be referred to with the    *)
  (**   PROCEDURE "AGGiveNextGraphInGraphPool".            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGiveNextGraphInGraphPool (VAR extNumber: ExternNumber;
                                      VAR existent : BOOLEAN       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   The result of this function is the next external   *)
  (**   graph number in the list created by the PROCEDURE  *)
  (**   "AGGetGraphsInGraphPool".                          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCopyGraph (    sourceGraph   : GraphNumber;
                           targetPoolName: PoolName;
                           targetGrName  : GraphName;
                       VAR status        : TStatus      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creates a copy of an open graph and places it in   *)
  (**   pool 'targetPoolName'. 'targetGraph' can be used   *)
  (**   directly and doesn't need to be reconstructed with *)
  (**   a delta.                                           *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   NotExistent    : TargetPool doesn't exist          *)
  (**   AlreadyExistent: graph already exists              *)
  (**   StillPendingTransactions : The targetpool is still *)
  (**   in transaction                                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Access.Denied                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   Graph.InternalError                                *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.Existent                                 *)
  (**   GraphPool.NotExistent                              *)
  (**   GraphPool.NotInTransaction                         *)
  (**   GraphPool.InTransaction                            *)
  (**   GraphPool.InUse                                    *)
  (**   GraphSystem.InternalError                          *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   TypedNames.InternalError                           *)
  (**   TypedNames.Unknown                                 *)
  (**   GraphPool.NoGraph                                  *)
  (**   GraphPool.NoScheme                                 *)
  (**   GraphPool.NotExistent                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeltaCopyGraph (    poolName    : PoolName;
                                sourceGrName: GraphName;
                                targetGrName: GraphName;
                                forward     : BOOLEAN;
                            VAR Status      : TStatus    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Copies the closed graph "sourceGrName" from the    *)
  (**   pool "poolName" logically to "targetGrName" in the *)
  (**   same pool. If "Forward" is set to TRUE, the copy is*)
  (**   performed with a forward delta, else with a back-  *)
  (**   ward delta.                                        *)
  (**                                                      *)
  (** Note:                                                *)
  (**   Take care of the fact, that AGDeltaCopyGraph needs *)
  (**   a closed source graph whereas AGCopyGraph needs an *)
  (**   open one!                                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraphPool.Existent                          *)
  (**   ChgMgmtGraphPool.InUse                             *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**                                                      *)
  (*********************************************************)

(*
  PROCEDURE AGGetFirstGroupInfo (Pool : PoolName;  GrName :
   GraphName; VAR OneGroup: BOOLEAN; VAR Existent: BOOLEAN; VAR Group :
   GroupName; VAR LockMode: GroupType );
*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Tests wether the graph is opened AND an access     *)
(**   group exists. Then the parameter OneGroup is       *)
(**   TRUE, if the graph might be accessed only          *)
(**   by members in exactly this group (i.e. no other    *)
(**   groups can exist at the same time). The identi-    *)
(**   fication of the group is returned in Group.        *)
(**   The lock duration to the graph for all group       *)
(**   members is returned in LockMode.                   *)
(**                                                      *)
(** Note:                                                *)
(**                                                      *)
(**   If the parameter Existent is FALSE just as it was  *)
(**   in the call to AGGetWriterGroupInfo then the graph *)
(**   is not opened.                                     *)
(**   If the parameter OneGroup is TRUE, a subsequent    *)
(**   call to AGGetNextReaderGroupInfo always yields     *)
(**   Existent = FALSE.                                  *)
(**                                                      *)
(*********************************************************)

(*PROCEDURE AGGetNextGroupInfo (VAR Existent: BOOLEAN; VAR Group :
   GroupName; VAR LockMode: GroupType );*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Returns information on the next group if existent. *)
(**                                                      *)
(*********************************************************)

(*PROCEDURE AGGetInfoByGrNo ( Graph : GraphNumber; VAR IsOpen : BOOLEAN;
   VAR Pool : PoolName; VAR GrName : GraphName; VAR ExtNumber :
   ExternNumber; VAR GrType : GraphType; VAR Group : GroupName; VAR
   LockMode : GroupType; VAR SingleWriter: BOOLEAN );*)
(*********************************************************)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**                                                      *)
(*********************************************************)

PROCEDURE AGGetInfoByExtNo (    poolName : PoolName;
                                extNumber: ExternNumber;
                            VAR existent : BOOLEAN;
                            VAR grName   : GraphName;
                            VAR grType   : GraphType;
                            VAR isOpen   : BOOLEAN;
                            VAR graph    : GraphNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Checks, whether a graph with the external number   *)
  (**   "extNumber" exists in the pool. If so, the         *)
  (**   following information is returned:                 *)
  (**   - its name                                         *)
  (**   - whether the graph is opened by the current       *)
  (**     application and if so,                           *)
  (**     - its temporary graph number                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetInfoByName (    poolName   : PoolName;
                               grName     : GraphName;
                           VAR existent   : BOOLEAN;
                           VAR grType     : GraphType;
                           VAR extNumber  : ExternNumber;
                           VAR isOpen     : BOOLEAN;
                           VAR graphNumber: GraphNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Checks, whether a graph with the specified name    *)
  (**   "grName" exists in the pool. If so, the following  *)
  (**   information is returned:                           *)
  (**   - its external number                              *)
  (**   - whether the graph is opened by the current       *)
  (**     application and if so,                           *)
  (**     - its temporary graph number                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGOpenGraph (    poolName  : PoolName;
                           grName    : GraphName;
                       VAR grType    : GraphType;
                           grMode    : GraphMode;
                           gpName    : GroupName;
                           gpType    : GroupType;
                           gpMode    : GroupMode;
                           oneGroup  : BOOLEAN;
                           noWait    : BOOLEAN;
                       VAR status    : TStatus;
                       VAR extNumber : ExternNumber;
                       VAR grNumber  : GraphNumber;
                           schemeName: SchemeName     := NIL);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**   Creates or opens a graph. If already open, then    *)
  (**   closes the graph and uses the appropriate          *)
  (**   parameters for opening the graph again.            *)
  (**                                                      *)
  (** Functionality:                                       *)
  (**                                                      *)
  (**   Opens a graph referred to by poolName and grName   *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   NotExistent          : the graph or the pool does  *)
  (**                          not exist                   *)
  (**   NoScheme             : the scheme does not exist   *)
  (**   AlreadyExistent      : the graph does already      *)
  (**                          exist                       *)
  (**   StillPendingTransaction : The pool is still in     *)
  (**                             transaction              *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   In addition to the error codes mentioned above     *)
  (**   errors are returned which occur due to the use     *)
  (**   of the standard open function AGOpenGraph.         *)
  (**                                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Graph.InUse                                        *)
  (**   Graph.InternalError                                *)
  (**   Graph.NoScheme                                     *)
  (**   Graph.NotExistent                                  *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   RGRASNames.InternalError                           *)
  (**   GraphPool.NoGraph                                  *)
  (**   GraphPool.NoScheme                                 *)
  (**   GraphPool.NotExistent                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCloseGraph (graphNumber: GraphNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Closes a graph and resets the corresponding access *)
  (**   locks. This PROCEDURE may only be called at a time *)
  (**   when all transactions concerning the graph are     *)
  (**   committed.                                         *)
  (**                                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**   Pending transactions                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteGraph (    poolName: PoolName;
                             grName  : GraphName;
                             force   : BOOLEAN;
                         VAR status  : TStatus    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   The closed graph "grName" in the GraphPool         *)
  (**   "poolName" will be deleeted.                       *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   NotExistent: graph or pool does not exist          *)
  (**   StillOpen  : the graph is still opened by          *)
  (**                an application program                *)
  (**   StillPendingTransactions : The pool is still in    *)
  (**                              transaction             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Locked                                      *)
  (**   Access.Denied                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.InUse                                    *)
  (**   GraphPool.NoGraph                                  *)
  (**   GraphPool.NotExistent                              *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteScheme (    poolName: PoolName;
                              schema  : SchemeName;
                          VAR status  : TStatus    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   The closed schema "schema" in the GraphPool        *)
  (**   "poolName" will be deleted.                        *)
  (**                                                      *)
  (** Errors Returned:                                     *)
  (**                                                      *)
  (**   NotExistent: schema or pool does not exist         *)
  (**   StillOpen  : the schema is still opened by         *)
  (**                an application program                *)
  (**   StillPendingTransactions : The pool is still in    *)
  (**                              transaction             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphSystem.InternalError                          *)
  (**   Access.Locked                                      *)
  (**   Access.Denied                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.InUse                                    *)
  (**   GraphPool.NoGraph                                  *)
  (**   GraphPool.NotExistent                              *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**            Operations for Primitive Recovery Mechanisms              *)
(**                                                                      *)
(*************************************************************************)


PROCEDURE AGStartTransaction (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Starts a new transaction. This transaction can     *)
  (**   either be committed normally (then it has no       *)
  (**   effects on the state of the graph) or be aborted   *)
  (**   (that means, that the graph is reset to the state  *)
  (**   when AGStartTransaction was called).               *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   All restrictions and effects of this PROCEDURE are *)
  (**   mentioned in the module comments.                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphPool.InternalError                            *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCommitToplevelTransaction (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Commits all transactions that were started on the  *)
  (**   specified graph.                                   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCommitTransaction (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Commits a transaction that was started by calling  *)
  (**   AGStartTransaction in the normal way. That means,  *)
  (**   that all graph changing operations performed       *)
  (**   during the transaction take place in the graph.    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGAbortToplevelTransaction (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Resets the graph to the state right before         *)
  (**   initiating the first transaction.                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGAbortTransaction (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Aborts a transaction that was started by calling   *)
  (**   AGStartTransaction. The effect is, that the graph  *)
  (**   is reset into the state at the time the trans-     *)
  (**   action was started.                                *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphPool.InternalError                            *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**      Operations for Requests and Modifications on Single Graphs      *)
(**                                                                      *)
(** Note:                                                                *)
(**                                                                      *)
(**   The effects of these operations can be reset by aborting trans-    *)
(**   actions (but this does not change the state of sets!)              *)
(**                                                                      *)
(*************************************************************************)

PROCEDURE AGCreateNode (    graphNumber: GraphNumber;
                            nType      : TypeNumber;
                            environment: NodeNumber;
                        VAR nodeNr     : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creates a new node in graph "graphNumber".         *)
  (**   This node gets the type "nType".                   *)
  (**   The parameter "environment" may                    *)
  (**   specify another node, which represents a logical   *)
  (**   neighbor to the current one. "nodeNr" passes the   *)
  (**   number of the new node to the application. It is   *)
  (**   set to "environment", if that number is not yet    *)
  (**   used.                                              *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   The parameter "environment" is internally used to  *)
  (**   place the nodes onto pages in order to minimize    *)
  (**   disk accesses. Therefore logically neighbored      *)
  (**   nodes should always be stored as physically neigh- *)
  (**   bored ones.                                        *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.LogError                                     *)
  (**   Graph.Unknown                                      *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCreateEdgeAndNode (    graphNumber: GraphNumber;
                                   sourceNode : NodeNumber;
                                   eType      : TypeNumber;
                                   nType      : TypeNumber;
                               VAR nodeNr     : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Inserts a new edge from node "sourceNode" to a     *)
  (**   node of type "nType". The edge gets the type       *)
  (**   "eType". "nodeNr" returns the number of the new    *)
  (**   node. The new node is placed in physical neigh-    *)
  (**   borhood to the existing one.                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.LogError                                     *)
  (**   Graph.Unknown                                      *)
  (**   Graph.CardinalityError                             *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.WrongType                                    *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteNodeAndEdges (graphNumber: GraphNumber; node: NodeNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes the specified node with all its incoming   *)
  (**   and outgoing edges from the graph.                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NotOwner                                     *)
  (**   graph.LogError                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetUniqueNodeId (    graphNumber: GraphNumber;
                             VAR uniqueID   : CARDINAL     );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   This PROCEDURE computes a number which is unique   *)
  (**   for the whole graph pool.                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   RGRASNames.InternalError                           *)
  (**   Access.Locked                                      *)
  (**   Names.Undeclared                                   *)
  (**   Names.Unknown                                      *)
  (**   GraphPool.InernalError                             *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool could not be found                        *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowAllNodesWithIndex (    graphNumber: GraphNumber;
                                       attNo      : TypeNumber;
                                       attLength  : Word.T;
                                       value      : TEXT;
                                   VAR setOfNodes : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns a set of nodes the given attribute value   *)
  (**   is associated with.                                *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   This PROCEDURE may not be used within evaluation-  *)
  (**   functions of attributes.                           *)
  (**                                                      *)
  (**   Furthermore, in the case of a (boolean) constraint *)
  (**   attribute this query is only supported for "value" *)
  (**   equal to "Violated" = FALSE (with "Length" = 1) in *)
  (**   order to avoid creation of a large useless index   *)
  (**   for all nodes which respect a certain constraint.  *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.CyclicEvaluation                             *)
  (**   Graph.LogError                                     *)
  (**   Graph.NoIndex                                      *)
  (**   Graph.Unknown                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowNodeWithKey (    graphNumber: GraphNumber;
                                 attNo      : TypeNumber;
                                 attLength  : Word.T;
                                 value      : TEXT;
                             VAR exist      : BOOLEAN;
                             VAR node       : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the number of the node, the given attribute*)
  (**   value is associated with.                          *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   This PROCEDURE may not be used within evaluation-  *)
  (**   functions of attributes.                           *)
  (**                                                      *)
  (**   Furthermore, in the case of a (boolean) constraint *)
  (**   attribute this query is only supported for "value" *)
  (**   equal to "Violated" = FALSE (with "Length" = 1) in *)
  (**   order to avoid creation of a large useless index   *)
  (**   for all nodes which respect a certain constraint.  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.CyclicEvaluation                             *)
  (**   Graph.LogError                                     *)
  (**   Graph.NoIndex                                      *)
  (**   Graph.Unknown                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowTypeOfNode (    graphNumber: GraphNumber;
                                node       : NodeNumber;
                            VAR nType      : TypeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the type of a given node. If the node does *)
  (**   not exist, nType.entity is set to                  *)
  (**   UndefinedTypeNumber=0.                             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCreateEdge (graphNumber: GraphNumber;
                        sourceNode : NodeNumber;
                        targetNode : NodeNumber;
                        eType      : TypeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Inserts a new directed edge from node "sourceNode" *)
  (**   to node "targetNode". This edge gets the type      *)
  (**   "eType".                                           *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Edges of the composite category can only be        *)
  (**   created as incoming ones, when the node is created.*)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.CardinalityError                             *)
  (**   Graph.LogError                                     *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   Graph.WrongType                                    *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteEdge (graphNumber: GraphNumber;
                        sourceNode : NodeNumber;
                        targetNode : NodeNumber;
                        eType      : TypeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes the edge of type "eType" between the       *)
  (**   nodes "sourceNode" and "targetNode".               *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Edges of the composite category can only be        *)
  (**   deleted together with their corresponding nodes.   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.CardinalityError                             *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.WrongType                                    *)
  (**   Graph.LogError                                     *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGEdgeExists (graphNumber: GraphNumber;
                        sourceNode : NodeNumber;
                        targetNode : NodeNumber;
                        eType      : TypeNumber   ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Tests, whether there is a directed edge between    *)
  (**   the nodes.                                         *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Graph.InternalError                                *)
  (**   Access.Locked                                      *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowSourceNode (    graphNumber: GraphNumber;
                                targetNode : NodeNumber;
                                eType      : TypeNumber;
                            VAR sourceNode : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the source node to a given edge and target *)
  (**   node.                                              *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   There must be one incoming edge of type "eType"    *)
  (**   at the node "targetNode".                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**   More than one source node                          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGTestAndShowSourceNode (    graphNumber: GraphNumber;
                                       targetNode : NodeNumber;
                                       eType      : TypeNumber;
                                   VAR sourceNrs  : CARDINAL;
                                   VAR sourceNode : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   "sourceNrs" specifies, how many edges of type      *)
  (**   "eType" come into "targetNode". If this count      *)
  (**   is exactly 1, the number of the source node is     *)
  (**   returned.                                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowAllSourceNodes (    graphNumber  : GraphNumber;
                                    targetNode   : NodeNumber;
                                    eType        : TypeNumber;
                                VAR sourceNodeSet: SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns all numbers of source nodes to a given     *)
  (**   pair (edge,target node).                           *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowTargetNode (    graphNumber: GraphNumber;
                                sourceNode : NodeNumber;
                                eType      : TypeNumber;
                            VAR targetNode : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the target node to a given edge and source *)
  (**   node.                                              *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   There must be one outgoing edge of type "eType"    *)
  (**   at the node "sourceNode".                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**   More than one target node                          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowAllTargetNodes (    graphNumber  : GraphNumber;
                                    sourceNode   : NodeNumber;
                                    eType        : TypeNumber;
                                VAR targetNodeSet: SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns all numbers of target nodes to a given     *)
  (**   pair (source node,edge).                           *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGTestAndShowTargetNode (    graphNumber: GraphNumber;
                                       sourceNode : NodeNumber;
                                       eType      : TypeNumber;
                                   VAR targetNrs  : CARDINAL;
                                   VAR targetNode : NodeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   "targetNrs" specifies, how many edges of type      *)
  (**   "eType" go out of "sourceNode". If this count      *)
  (**   is exactly 1, the number of the target node is     *)
  (**   returned.                                          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGTestIncomingEdge (graphNumber: GraphNumber;
                              targetNode : NodeNumber;
                              eType      : TypeNumber   ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Tests, whether there is at least one edge of type  *)
  (**   "eType" coming into "targetNode".                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGTestOutgoingEdge (graphNumber: GraphNumber;
                              sourceNode : NodeNumber;
                              eType      : TypeNumber   ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Tests, whether there is at least one edge of type  *)
  (**   "eType" going out of "sourceNode".                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetInContextOfNode (    graphNumber: GraphNumber;
                                    targetNode : NodeNumber;
                                VAR tupleSet   : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   This PROCEDURE calculates a set of tuples of the   *)
  (**   form (eType,sourceNode). Each element of this      *)
  (**   set fulfills the condition, that the edge of type  *)
  (**   edgeType runs from sourceNode to the specified     *)
  (**   targetNode.                                        *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Use "AGRelRemoveAnyTuple" to access all tuples     *)
  (**   stored within the temporarily existing "RelSet"    *)
  (**   and use "AGRelKill" afterwards to release all      *)
  (**   storage occupied by this relation.                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetOutContextOfNode (    graphNumber: GraphNumber;
                                     sourceNode : NodeNumber;
                                 VAR tupleSet   : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   This PROCEDURE calculates a set of tuples of the   *)
  (**   form (edgeType,targetNode). Each element of this   *)
  (**   set fulfills the condition, that the edge of type  *)
  (**   edgeType runs from the specified sourceNode to     *)
  (**   targetNode.                                        *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   Use "AGRelRemoveAnyTuple" to access all tuples     *)
  (**   stored within the temporarily existing "RelSet"    *)
  (**   and use "AGRelKill" afterwards to release all      *)
  (**    storage occupied by this relation.                *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGPutAttributeSubstr (graphNumber: GraphNumber;
                                node       : NodeNumber;
                                attNo      : TypeNumber;
                                attBegin   : Word.T;
                                attLength  : Word.T;
                                attValue   : TEXT;
                                truncate   : BOOLEAN      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Replaces a substring of the attNo'th (intrin-      *)
  (**   sic) attribute of a node by a new value. The sub-  *)
  (**   string begins at position "attBegin" and has a     *)
  (**   length of "attLength" characters. The truncate     *)
  (**   argument specifies whether the attribute length    *)
  (**   will be set to attBegin+attLength-1.               *)
  (**                                                      *)
  (** Remarks:                                             *)
  (**                                                      *)
  (** - A typical call of AGPutAttributeSubstr might look  *)
  (**   like "AGPutAttributeSubstr(g,n,0,0,SIZE(I),I,      *)
  (**   FALSE)", making use of SYSTEM's PROCEDURE SIZE.    *)
  (** - Writing a range starting at a position > 0 has the *)
  (**   effect that reading below this position will not   *)
  (**   result in error messages or strings of length 0!   *)
  (**   The contents however may be undefined!             *)
  (** - The call "AGPutAttributeSubstr(g,n,X,0,0,I,TRUE)"  *)
  (**   will release the internal storage of the X'th      *)
  (**   attribute by setting its physical length to zero.  *)
  (**   It's easier to use AGDeleteAttribute(g,n,X).       *)
  (** - index and key attributes must have a length up to  *)
  (**   250 bytes, normal attributes must be shorter       *)
  (**   than 16449286 bytes (see constants MaxNormalAttri- *)
  (**   buteLength and MaxIndexAttributeLength in module   *)
  (**   AGGlobal).                                         *)
  (** - The first about 250 bytes of the 0th normal attri- *)
  (**   bute should be preferred by any application because*)
  (**   GRAS' internal storage mechanisms handle them in a *)
  (**   more efficient way.                                *)
  (**   If efficient storage of attribute values is essen- *)
  (**   tial then please store all short attribute values  *)
  (**   in UNDECLARED 0 attributes (the first attribute    *)
  (**   value at position 0, the second at position 0 +    *)
  (**   size of first attribute value etc.). But not that  *)
  (**   maintaining indexes or incremental attribute eva-  *)
  (**   luation doesn't work for these effficiently stored *)
  (**   attributes.                                        *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.LogError                                     *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   Graph.WrongType                                    *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteAttribute (graphNumber: GraphNumber;
                             node       : NodeNumber;
                             attNo      : AttributeNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes the attribute attNo from the node.         *)
  (**                                                      *)
  (** Remarks:                                             *)
  (**                                                      *)
  (**   A call to AGDeleteAttribute is basically a call    *)
  (**   AGPutAttributeSubstr(graphNumber,node,attNo,0,0,i, *)
  (**   TRUE)                                              *)
  (**   (with a dummy parameter i - see AGPutAttribute-    *)
  (**   Substr).                                           *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.WrongType                                    *)
  (**   Graph.Unknown                                      *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.LogError                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetAttributeSubstr (    graphNumber   : GraphNumber;
                                    node          : NodeNumber;
                                    attNo         : TypeNumber;
                                    attBegin      : Word.T;
                                    attLength     : Word.T;
                                VAR attValue      : TEXT;
                                VAR returnedLength: Word.T       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   This PROCEDURE performs the inverse action of      *)
  (**   AGPutAttributeSubstr. The value of the attribute   *)
  (**   substring is returned in the "generic" parameter   *)
  (**   "attValue". ReturnedLength returns the number of   *)
  (**   bytes that could actually be read. If the value of *)
  (**   the attribute is undefined (due to the fact that   *)
  (**   the attribute didn't receive an initial value or   *)
  (**   previous assignments - to a derived attribute -    *)
  (**   became invalid due to certain graph changes), then *)
  (**   ReturnedLength will be 0!                          *)
  (**   In the case of an invalid derived attribute, the   *)
  (**   application may reevaluate this attribute and      *)
  (**   assign a new value (using 'AGPutAttributeS.')      *)
  (**   It is, however, possible to let GRAS do this       *)
  (**   automatically. In order to do this, information    *)
  (**   about the function name must be stored in the graph*)
  (**   scheme using "AGSetEvaluationFunction".            *)
  (**   Additionally, the application must specify the     *)
  (**   current address of the function code at runtime    *)
  (**   using "AGBindEvaluationFunction".                  *)
  (**   In this case any attempt to read an invalid attri- *)
  (**   bute will cause GRAS to reevaluate it and yield    *)
  (**   the new value.                                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.LogError                                     *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   Graph.WrongType                                    *)
  (**   Graph.CyclicEvaluation                             *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                    Operations for Event Handling                     *)
(**                                                                      *)
(*************************************************************************)

PROCEDURE AGActivateDaemon (Graph   : GraphNumber;
                            Event   : GraphEvent;
                            Priority: ActionPriority;
                            Action  : ActionProc      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**  Activates a daemon. If "Event" occurs in the graph  *)
  (**  for which "Action" was defined, "Action" will be	   *)
  (**  called with parameters as listed above. For every   *)
  (**  tuple (Action,Event) there will be max. one daemon  *)
  (**  started.					           *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphList.EntryNotInList                           *)
  (**   EventType.Mismatch                                 *)
  (**   EventType.Unknown                                  *)
  (**   EventTypes.Unknown                                 *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGKillDaemon (Graph : GraphNumber;
                        Event : GraphEvent;
                        Action: ActionProc   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deactivates a daemon that has be activated by a    *)
  (**   call of AGActivateDaemon. If a graph is closed, all*)
  (**   its active daemons will be killed, so the next time*)
  (**   it is opened, there will be no initially active    *)
  (**   daemons.                                           *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   The execution of activated actions may not be      *)
  (**   locked for the specified graph!                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   DaemonList.EntryNotInList                          *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRaiseEvent (Graph: GraphNumber; p1, p2, p3: CARDINAL);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   All daemons, that wait for UserDefined events	   *)
  (**   (with matching parameters) will invoke their 	   *)
  (**   actions. This can be used to communicate with	   *)
  (**   other clients. But be careful: the meaning of	   *)
  (**   p1, p2 and p3 is totally undefined from GRAS and   *)
  (**   can be different from client to client.            *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   RaiseEvent can also be called from a client        *)
  (**   different from the owner of the graph who may have *)
  (**   defined the EventPattern and activated the daemon, *)
  (**   provided that he has the right to work on the      *)
  (**   graph under consideration.                         *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   GraphList.EntryNotInList                           *)
  (**   EventTypes.Unknown                                 *)
  (**   EventType.Unknown                                  *)
  (**   EventType.Mismatch                                 *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDelayActionExecution (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Locks execution of all actions that were activated *)
  (**   for the pool. Locking may be nested, which means   *)
  (**   that a certain number of AGDelayActionExecution    *)
  (**   calls must be followed be the same number of calls *)
  (**   to AGReleaseActionExecution to release the lock.   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGReleaseActionExecution (poolName: PoolName);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Unlocks execution of activated actions for a pool. *)
  (**   Locked active actions are executed. Locking may be *)
  (**   nested, which means that a certain number of       *)
  (**   AGDelayActionExecution calls must be followed be   *)
  (**   the same number of calls to AGReleaseAction-       *)
  (**   Execution to release the lock.                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The pool is not open                               *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                      Operations on Local Sets                        *)
(**                                                                      *)
(** Note:                                                                *)
(**                                                                      *)
(**   Aborting transactions does not effect local sets! They are not     *)
(**   reset into a previous state.                                       *)
(**                                                                      *)
(*************************************************************************)

PROCEDURE AGSetCreate (graphNumber: GraphNumber; VAR emptySet: SimpleSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creates an empty temporary set.                    *)
  (**   A set is a structure with                          *)
  (**   the properties of mathematical sets. It can store  *)
  (**   elements which are present at most once. Several   *)
  (**   operations are presented for these sets. SimpleSet *)
  (**   structures contain NodeNumbers      .              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetClear (graphNumber: GraphNumber; VAR clearSet: SimpleSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Clears an temporary set.                           *)
  (**   A set is a structure with the properties           *)
  (**   of mathematical sets. It can store elements which  *)
  (**   are present at most once. Several operations are   *)
  (**   presented for these sets. SimpleSet structures con-*)
  (**   tain NodeNumbers.                                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetKill (graphNumber: GraphNumber; VAR set: SimpleSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Releases the amount of memory used by "set". It    *)
  (**   may then be reused by AGSetCreate.                 *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetCopy (    graphNumber: GraphNumber;
                         sourceSet  : SimpleSet;
                     VAR targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Copies "sourceSet" to "targetSet".                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The sourceset has not been created                 *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetInsertElement (graphNumber: GraphNumber;
                              element    : SimpleElement;
                              set        : SimpleSet      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Inserts an element into a set.                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetDeleteElement (graphNumber: GraphNumber;
                              element    : SimpleElement;
                              set        : SimpleSet      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes an element from a set.                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGSetRemoveAnyElement (    graphNumber: GraphNumber;
                                 VAR found      : BOOLEAN;
                                 VAR element    : SimpleElement;
                                     set        : SimpleSet      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes an arbitrary element from the set and      *)
  (**   returns it to the calling module. If the set is    *)
  (**   empty, the result of "found" is FALSE.             *)
  (**   "Arbitrary" means that the resulting sequence of   *)
  (**   returned elements is the most efficient one with   *)
  (**   respect to the system's internal storage layout.   *)
  (**   Use "AGSetRemoveRndElement" if real random results *)
  (**   are required.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGSetRemoveRndElement (    graphNumber: GraphNumber;
                                 VAR found      : BOOLEAN;
                                 VAR element    : SimpleElement;
                                     set        : SimpleSet      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes a real random element from the set and     *)
  (**   returns it to the calling module. If the set is    *)
  (**   empty, the result of "found" is FALSE.             *)
  (**   This PROCEDURE is considerably less efficient than *)
  (**   "AGSetRemoveAnyElement" and should only be used in *)
  (**   cases where "random" access is really important.   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetIsElement (graphNumber: GraphNumber;
                          element    : SimpleElement;
                          set        : SimpleSet      ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns whether "element" is in "set".             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetSize (graphNumber: GraphNumber; set: SimpleSet): Word.T;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes the cardinal number of "set".             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetUnion (graphNumber: GraphNumber;
                      sourceSet  : SimpleSet;
                      targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes the union of targetSet and sourceSet and  *)
  (**   places the result in targetSet.                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The target or sourceset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetIntersection (graphNumber: GraphNumber;
                             sourceSet  : SimpleSet;
                             targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes targetSet := targetSet intersected with   *)
  (**                         sourceSet                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The target or sourceset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetDifference (graphNumber: GraphNumber;
                           sourceSet  : SimpleSet;
                           targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes targetSet := targetSet without sourceSet. *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The target or sourceset has not been created       *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                   Operations on Local Relations                      *)
(**                                                                      *)
(** Note:                                                                *)
(**                                                                      *)
(**   Aborting transactions does not effect local relations! They are    *)
(**   not reset into a previous state.                                   *)
(**                                                                      *)
(*************************************************************************)


PROCEDURE AGRelCreate (graphNumber: GraphNumber; VAR emptySet: RelSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creates an empty temporary tuple set               *)
  (**   A relation set is a                                *)
  (**   structure with the properties of mathematical sets.*)
  (**   It can store elements which are present at most    *)
  (**   once. Several operations are presented for these   *)
  (**   sets. RelSet structures contain elements which are *)
  (**   each a pair of NodeNumbers.                        *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelClear (graphNumber: GraphNumber; VAR clearSet: RelSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Clears an temporary tuple set.                     *)
  (**   A relation set is a                                *)
  (**   structure with the properties of mathematical sets.*)
  (**   It can store elements which are present at most    *)
  (**   once. Several operations are presented for these   *)
  (**   sets. RelSet structures contain elements which are *)
  (**   each a pair of NodeNumbers.     .                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelKill (graphNumber: GraphNumber; VAR set: RelSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Releases the amount of memory used by "set". It    *)
  (**   may then be re-used be AGRelCreate.                *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelCopy (    graphNumber: GraphNumber;
                         sourceSet  : RelSet;
                     VAR targetSet  : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Copies "sourceSet" to "targetSet".                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source set has not been created                *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelInsertTuple (graphNumber : GraphNumber;
                            surr1, surr2: TypeNumber;
                            set         : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Inserts the Tuple ("surr1","surr2") into a set.    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelDeleteTuple (graphNumber : GraphNumber;
                            surr1, surr2: TypeNumber;
                            set         : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes the tuple ("surr1","surr2") from the set.  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelRemoveAnyTuple (    graphNumber : GraphNumber;
                               VAR found       : BOOLEAN;
                               VAR surr1, surr2: TypeNumber;
                                   set         : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes an arbitrary tuple from the set and        *)
  (**   returns its components "surr1" and "surr2" to the  *)
  (**   calling module. If the set is empty, the result of *)
  (**   "found" is FALSE.                                  *)
  (**   "Arbitrary" means that the resulting sequence of   *)
  (**   returned elements is the most efficient one with   *)
  (**   respect to the system's internal storage layout.   *)
  (**   Use "AGRelRemoveRndElement" if real random results *)
  (**   are required.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGRelRemoveRndTuple (    graphNumber : GraphNumber;
                               VAR found       : BOOLEAN;
                               VAR surr1, surr2: TypeNumber;
                                   set         : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Deletes an arbitrary tuple from the set and        *)
  (**   returns its components "surr1" and "surr2" to the  *)
  (**   calling module. If the set is empty, the result of *)
  (**   "found" is FALSE.                                  *)
  (**   This PROCEDURE is considerably less efficient than *)
  (**   "AGRelRemoveAnyElement" and should only be used in *)
  (**   cases where "random" access is really important.   *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelIsElement (graphNumber : GraphNumber;
                          surr1, surr2: TypeNumber;
                          set         : RelSet       ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   The result of this function is true, if the pair   *)
  (**   ("surr1","surr2") is in "set".                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelSize (graphNumber: GraphNumber; set: RelSet): Word.T;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes the cardinal number of "set".             *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelUnion (graphNumber: GraphNumber;
                      sourceSet  : RelSet;
                      targetSet  : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes the union of targetSet and sourceSet and  *)
  (**   places the result in targetSet.                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source or targetset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelIntersection (graphNumber: GraphNumber;
                             sourceSet  : RelSet;
                             targetSet  : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes targetSet := targetSet intersected with   *)
  (**                         sourceSet                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source or targetset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRelDifference (graphNumber: GraphNumber;
                           sourceSet  : RelSet;
                           targetSet  : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes targetSet := targetSet without sourceSet. *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source or targetset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGCartesian (    graphNumber: GraphNumber;
                           sourceSet1 : SimpleSet;
                           sourceSet2 : SimpleSet;
                       VAR targetSet  : RelSet       );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Computes the cartesian product of the two source   *)
  (**   sets and places the result in the tuple set        *)
  (**   "targetSet".                                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source or targetset has not been created       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGProjectionFirst (    graphNumber: GraphNumber;
                                 sourceSet  : RelSet;
                             VAR targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Projection of the first component of "sourceSet"   *)
  (**   into "targetSet".                                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source set has not been created                *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGProjectionSecond (    graphNumber: GraphNumber;
                                  sourceSet  : RelSet;
                              VAR targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Projection of the second component of "sourceSet"  *)
  (**   into "targetSet", which must have been created     *)
  (**   before.                                            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source set has not been created                *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGQueryProjection (    graphNumber: GraphNumber;
                                 sourceSet  : RelSet;
                                 surr       : TypeNumber;
                             VAR targetSet  : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Projection of the second component of "sourceSet"  *)
  (**   into "targetSet", if the first component matches   *)
  (**   "surr". "targetSet" must have been created before. *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The source set has not been created                *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGSingleQueryProjection (    graphNumber: GraphNumber;
                                       set        : RelSet;
                                       surr1      : TypeNumber;
                                   VAR surr2      : TypeNumber;
                                   VAR found      : BOOLEAN      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   AGSingleQueryProjection calculates *one* second    *)
  (**   component surr2 for a given first component surr1. *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGChange (graphNumber: GraphNumber; VAR set: RelSet);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Exchanges the components of a local tuple set.     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The set has not been created                       *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                    PROCEDUREs to work with Logs                      *)
(**                                                                      *)
(*************************************************************************)

PROCEDURE AGSetCheckpoint (graphNumber: GraphNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Sets a checkpoint.                                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   ChgMgmtGraph.NoLog                                 *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGUndo (graphNumber: GraphNumber; VAR n: CARDINAL);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   AGUndo performs an UNDO back for n check points.   *)
  (**   If there are less than n check points set, the     *)
  (**   number of successful UNDOs is returned in n. This  *)
  (**   PROCEDURE should only be called, if there were no  *)
  (**   graph changing operations since the last setting   *)
  (**   of a check point. Otherwise, a new checkpoint is   *)
  (**   created internally before executing AGUndo itself. *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   ChgMgmtGraph.NoLog                                 *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGRedo (graphNumber: GraphNumber; VAR n: CARDINAL);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   AGRedo performs a REDO of all graph changing       *)
  (**   operations that were performed since n check       *)
  (**   points before. This PROCEDURE may only be called,  *)
  (**   if there were no graph changing operations since   *)
  (**   the last setting of a check point. The number of   *)
  (**   successful REDOs is returned as parameter n.       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   ChgMgmtGraph.NoLog                                 *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

(*************************************************************************)
(**                                                                      *)
(**                           Scheme Management                          *)
(**                                                                      *)
(**  N O T E :  The following PROCEDUREs are not necessary for the mani- *)
(**             pulation of attributed graphs (if the error check mode   *)
(**             is deactivated and incremental attribute evaluation as   *)
(**             well as index or key attributes are not used).           *)
(**                                                                      *)
(*************************************************************************)

(*************************************************************************)
(**                                                                      *)
(**         Operations for managing schemes                              *)
(**                                                                      *)
(** Every component of the graph scheme has two identifiers. One is a    *)
(** (human-readable) name, the other one an arbitrary internally used    *)
(** number which must be unique across all types of declared graph       *)
(** compontents.                                                         *)
(**                                                                      *)
(** ATTENTION: Even components belonging to different categories of data *)
(**            - like node classes and edge types - must have different  *)
(**            names and internally used type numbers.                   *)
(**                                                                      *)
(** This part of the GRAS interface is not yet complete. Explanations of *)
(** error messages must be added and a number of PROCEDUREs for querying *)
(** an already existing graph are missing.                               *)
(**                                                                      *)
(*************************************************************************)


PROCEDURE AGDeclareScheme (    schemeName  : SchemeName;
                           VAR schemeNumber: SchemeNumber;
                               poolName    : PoolName      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Creation of a scheme with the name "Name" that     *)
  (**   will get the number "Scheme". This number can be   *)
  (**   used to reference it. It is possible to have more  *)
  (**   than one (sub-)scheme in one graph.                *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   AGDeclareScheme may be used to extend already      *)
  (**   existing graph schemes and to redefine the         *)
  (**   evaluation functions and attribute dependencies    *)
  (**   of previously declared attributes, i.e. AGDeclare- *)
  (**   Scheme for an already existing graph scheme starts *)
  (**   the extension of this graph scheme.                *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   Scheme.Existent                                    *)
  (**   Scheme.InUse                                       *)
  (**   Scheme.InternalError                               *)
  (**   GraphPool.InternalError                            *)
  (**   Scheme.NoValidScheme                               *)
  (**   Scheme.NotExistent                                 *)
  (**   GraphPool.CardinalityError                         *)
  (**   GraphPool.NotInTransaction                         *)
  (**   The pool is not open                               *)
  (**   The scheme is already open                         *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGCommitScheme (schemeNumber: SchemeNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Commit all declarations that were made for this    *)
  (**   scheme. This PROCEDURE is used to tell the GRAS    *)
  (**   system that there are no more declarations. There- *)
  (**   fore it must be called AFTER all other scheme      *)
  (**   PROCEDUREs. All declarations are then carried out  *)
  (**   in the proper order. That means, that you can call *)
  (**   all declarations in arbitrary order and then simply*)
  (**   commit the scheme.                                 *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   AGCommitScheme may be used to extend already       *)
  (**   existing graph schemes and to redefine the         *)
  (**   evaluation functions and attribute dependencies    *)
  (**   of previously declared attributes.                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGExistsScheme (poolName: PoolName; schemeName: SchemeName):
  BOOLEAN;

(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Look for the scheme "schemeName".                  *)
(**   If it is existent, TRUE is returned .              *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**   GraphPool.InternalError                            *)
(**   Access.Locked                                      *)
(**   The pool is not open                               *)
(**                                                      *)
(*********************************************************)

PROCEDURE AGShowSchemeName (    schemeNumber: SchemeNumber;
                            VAR name        : SchemeName    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the scheme "schemeNumber"          *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

(* unimplemented *)
(*
PROCEDURE AGSetSchemeVersion (Scheme: SchemeNumber;  Version: TEXT);
  *)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Set the version of the scheme "Scheme".            *)
(**                                                      *)
(*********************************************************)
(*
PROCEDURE AGShowSchemeVersion (Scheme: SchemeNumber; VAR Version: TEXT);
  *)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Get the version of the scheme "Scheme".            *)
(**                                                      *)
(*********************************************************)

(******************************************************************************)
(**                                                                           *)
(**  PROCEDUREs for managing node classes                                     *)
(**                                                                           *)
(******************************************************************************)

PROCEDURE AGDeclareNodeClass (    schemeNumber: SchemeNumber;
                                  name        : TEXT;
                              VAR nodeClass   : TypeNumber    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Declare the node class "name" for the scheme       *)
  (**   "schemeNumber". This node class will get the number*)
  (**   "nodeClass". The name and the number of the node   *)
  (**   class must be unique for this graph.               *)
  (**   Node class no. 0 is reserved.                      *)
  (**                                                      *)
  (** Note:                                                *)
  (**   This action will not take place before the scheme  *)
  (**   is committed.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.AlreadyDeclared                             *)
  (**   Scheme.InternalError                               *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGTestAndShowNodeClassNumber (graphOrSchemeNumber: GraphNumber;
                                        className          : TEXT;
                                        VAR existent : BOOLEAN;
                                        VAR nodeClass: TypeNumber;
                                        isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Look for the node class "className".               *)
  (**   If it is existent, its number is returned in       *)
  (**   "nodeClass".                                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowNodeClassName (    graphOrSchemeNumber: GraphNumber;
                                   nodeClass          : TypeNumber;
                               VAR className          : TEXT;
                               VAR existent           : BOOLEAN;
                               isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the node class "nodeClass". If the *)
  (**   node class is declared, the name is returned in    *)
  (**   "className".                                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGShowAllNodeTypes (    graphOrSchemeNumber: GraphNumber;
                                  nodeClass          : TypeNumber;
                              VAR setOfTypes         : SimpleSet;
                              isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**  Get all node types of a given class and all of its  *)
  (**  subclasses in "setOfTypes"                          *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all types    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**    storage occupied by this set.                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGAppendNodeClass (schemeNumber  : SchemeNumber;
                             nodeClassName : TEXT;
                             superClassName: TEXT          );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Add a specialization/generalization relation to    *)
  (**   the scheme. The node class "nodeClassName" will    *)
  (**   then be a subclass of "superClassName" and inherits*)
  (* all attributes and attribute evaluation rules. *)
  (**                                                      *)
  (** Note:                                                *)
  (**   This action will not take place before the scheme  *)
  (**   is committed.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.NotDeclared                                 *)
  (**   Scheme.AttributeNameClash                          *)
  (**   Scheme.AttributePlacementClash                     *)
  (**   Scheme.Cyclic                                      *)
  (**   Scheme.InternalError                               *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

(******************************************************************************)
(**                                                                           *)
(** PROCEDUREs for managing node types                                        *)
(**                                                                           *)
(******************************************************************************)

PROCEDURE AGDeclareNodeType (    schemeNumber: SchemeNumber;
                                 name        : TEXT;
                             VAR nType       : TypeNumber    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Declare node type "name" for scheme "schemeNumber".*)
  (**   This node type will get  number "nType".           *)
  (**   The name and the number of the node type must be   *)
  (**   unique for this graph. Node type no. 0 is reserved.*)
  (**   If the "UseMembership" flag is set, GRAS will be   *)
  (**   able to find all nodes of that type using the      *)
  (**   PROCEDURE "AGGetAllNodesOfNodeType".               *)
  (**   The "UseMembership" flag may not be used if more   *)
  (**   than 5000 nodes of this type will be created.      *)
  (**                                                      *)
  (** Notes:                                               *)
  (**   1) This action will not take place before the      *)
  (**      scheme is committed.                            *)
  (**   2) Creating and deleting nodes of this type        *)
  (**      will be more efficient if the                   *)
  (**      "UseMembership" option is NOT set.              *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.AlreadyDeclared                             *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGTestAndShowNodeTypeNumber (    graphOrSchemeNumber: CARDINAL;
                                           name               : TEXT;
                                       VAR existent           : BOOLEAN;
                                       VAR nType: TypeNumber;
                                       isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Look for the node type "name". If it is existent,  *)
  (**   its number is returned in "nType".                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowNodeTypeName (    graphOrSchemeNumber: CARDINAL;
                                  nType              : TypeNumber;
                              VAR name               : TEXT;
                              VAR existent           : BOOLEAN;
                                  isGraphNumber      : BOOLEAN      := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the node type "nType". If the      *)
  (**   node type was declared, the name is returned in    *)
  (**   "name".                                            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGAppendNodeType (graphOrSchemeNumber: CARDINAL;
                            nType              : TEXT;
                            nodeClass          : TEXT      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Add a specialization/generalization relation to    *)
  (**   the scheme. The node type "nType" will then        *)
  (**   belong to the class "nodeClass" and inherits all   *)
  (**   attributes and attribute evaluation rules.         *)
  (**                                                      *)
  (** Note:                                                *)
  (**   This action will not take place before the scheme  *)
  (**   is committed.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   Scheme.AlreadyDeclared                             *)
  (**   Scheme.AttributePlacementClash                     *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

(******************************************************************************)
(**                                                                           *)
(** PROCEDUREs for managing edge types                                        *)
(**                                                                           *)
(******************************************************************************)


PROCEDURE AGDeclareEdgeType (    schemeNumber         : SchemeNumber;
                                 name                 : TEXT;
                                 sourceNodeClassOrType: TEXT;
                                 sourceCardinality    : Cardinality;
                                 targetNodeClassOrType: TEXT;
                                 targetCardinality    : Cardinality;
                             VAR edge                 : TypeNumber    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Declare the edge type "name" for                   *)
  (**   the scheme "schemeNumber". The edge type will get  *)
  (**   the number "edge". Edge type no. 0 is reserved.    *)
  (**   All nodes that belong to "sourceNodeClassOrType"   *)
  (**   are allowed as source nodes for edges of this type.*)
  (**   All nodes that belong to "targetNodeClassOrType"   *)
  (**   are allowed as target nodes for edges of this type.*)
  (**   "sourceCardinality" and "targetCardinality" are    *)
  (**   used for runtime checks. "targetCardinality" =     *)
  (**   "ObligateSet" means e.g. that any node of class    *)
  (**   "sourceNodeClassOrType" is expected to have at     *)
  (**   least one emanating edge of this type.             *)
  (**                                                      *)
  (** Note:                                                *)
  (**   This action will not take place before the scheme  *)
  (**   is committed.                                      *)
  (**                                                      *)
  (** ATTENTION:                                           *)
  (**   Up to now, cardinality checks in "AGCreateEdge"    *)
  (**   etc. are not yet implemented.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   Scheme.AlreadyDeclared                             *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGTestAndShowEdgeTypeNumber (    graphOrSchemeNumber: CARDINAL;
                                           name               : TEXT;
                                       VAR existent           : BOOLEAN;
                                       VAR edge: TypeNumber;
                                       isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Look for the edge type "name". If it is existent,  *)
  (**   its number is returned in "edge".                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowEdgeTypeName (    graphOrSchemeNumber: CARDINAL;
                                  edge               : TypeNumber;
                              VAR name               : TEXT;
                              VAR existent           : BOOLEAN;
                                  isGraphNumber      : BOOLEAN      := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the edge type "edge". If the       *)
  (**   edge type was declared, the name is returned in    *)
  (**   "name".                                            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGShowEdgeTypeProps (    graphOrSchemeNumber: CARDINAL;
                                   edge               : TypeNumber;
                               VAR sourceCT           : TypeNumber;
                               VAR sourceCardinality  : Cardinality;
                               VAR targetCT           : TypeNumber;
                               VAR targetCardinality  : Cardinality;
                               VAR existent           : BOOLEAN;
                               isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the properties of the edge type "edge". If the *)
  (**   edge type was declared, its cardinalities and its  *)
  (**   source and target types/classes are returned.      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

(******************************************************************************)
(**                                                                           *)
(** PROCEDUREs for managing attributes                                        *)
(**                                                                           *)
(******************************************************************************)


PROCEDURE AGDeclareAttribute (    schemeNumber    : SchemeNumber;
                                  classOrType     : TEXT;
                                  name            : TEXT;
                                  kind            : AttributeKind;
                                  props           : IndexProperties;
                                  valueType       : ValueTypeNumber;
                                  valueCardinality: Cardinality;
                                  constantLength  : BOOLEAN;
                                  length          : Word.T;
                              VAR attNo           : AttributeNumber  );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Declare the Attribute "name" for the scheme        *)
  (**   "schemeNumber". All nodes that belong to the       *)
  (**   class/type "classOrType" will carry that attribute.*)
  (**   The attribute will get the number "attNo". The     *)
  (**   name and the number of the attribute must be       *)
  (**   unique.                                            *)
  (**                                                      *)
  (**   "kind" specifies whether the attribute is derived  *)
  (**   (with its subcase dummy), i.e. depends on other    *)
  (**   attributes, or whether it is an intrinsic attribute*)
  (**   (with its subcase meta attribute which has the same*)
  (**   value for all nodes of a certain type, i.e. is an  *)
  (**   attribute of the node type itself).                *)
  (**                                                      *)
  (**   If constant length flag is set, "length" specifies *)
  (**   the size of the values that will be stored in      *)
  (**   that attribute (in Bytes).                         *)
  (**                                                      *)
  (**   "props" contains information about the properties  *)
  (**   of the attribute, i.e. whether it is a normal, an  *)
  (**   index or a key attribute.                          *)
  (**                                                      *)
  (**   "valueType" is an almost uninterpreted number which*)
  (**   gives applications the chance to store information *)
  (**   about the type of an attribute's value (which is   *)
  (**   normally considered to be an uninterpreted byte    *)
  (**   stream. The numbers for standard types like integer*)
  (**   boolean etc. are fixed! All remaining numbers may  *)
  (**   be used as "pointers" to application defined types.*)
  (**                                                      *)
  (**   "valueCardinality" specifies whether a value for   *)
  (**   this attribute is optional ( -> "Opt..." ) or      *)
  (**   obligate ( -> "Obl..." ) and whether the attribute *)
  (**   contains a set of values ( -> "...Set" ) or only   *)
  (**   a single element ( -> "...Unique" )                *)
  (**   Note that "ValueCardinality" is an attribute prop. *)
  (**   which has no meaning for GRAS (like "ValueType")   *)
  (**   because (up to now) all (sets of) attribute values *)
  (**   are uninterpreted arrays of byte for GRAS.         *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   1) The attribute number 0 should be used for the   *)
  (**      most frequently used NON-INDEX (and NON-KEY)    *)
  (**      attribute. If time and space consumption are    *)
  (**      very important, then you might even store many  *)
  (**      short attribute values within the 0 attribute   *)
  (**      (or within another attribute) without declaring *)
  (**  	   this attribute. This trick deactivates a number *)
  (**      of checks (for keeping derived attributes and   *)
  (**      attribute indexes consistent).                  *)
  (**   2) Meta and dummy index or key attributes are not  *)
  (**      supported.                                      *)
  (**   3) Dummy attributes may be used for propagating    *)
  (**      graph changes along arbitrarily long paths in   *)
  (**      a graph (by translating an n-context dependency *)
  (**      between two attributes into n 1-context depen-  *)
  (**      dencies between dummy attributes).              *)
  (**   4) Dummy attributes only are allowed to have a     *)
  (**      constant length of size 0.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.AlreadyDeclared                             *)
  (**   Scheme.AttributeNameClash                          *)
  (**   Scheme.AttributePlacementClash                     *)
  (**   Scheme.NotDeclared                                 *)
  (**   Scheme.InternalError                               *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGTestAndShowAttributeNumber (graphOrSchemeNumber: GraphNumber;
                                        name               : TEXT;
                                        classOrType        : TypeNumber;
                                        VAR existent: BOOLEAN;
                                        VAR attNo   : AttributeNumber;
                                        isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Look for the attribute "name". If it is existent,  *)
  (**   its number is returned in "attNo".                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGShowAttributeName (    graphOrSchemeNumber: GraphNumber;
                                   attNo              : AttributeNumber;
                               VAR name               : TEXT;
                               VAR existent           : BOOLEAN;
                               isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the attribute "attNo". If the      *)
  (**   attribute was declared, the name is returned in    *)
  (**   "name".                                            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGShowAttributeProps (    graphOrSchemeNumber: GraphNumber;
                                    attNo              : AttributeNumber;
                                VAR kind               : AttributeKind;
                                VAR props              : IndexProperties;
                                VAR valueType          : ValueTypeNumber;
                                VAR attCard            : Cardinality;
                                VAR constLength        : BOOLEAN;
                                VAR length             : CARDINAL;
                                VAR existent           : BOOLEAN;
                                isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the properties of the attribute "attNo".       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGPutMetaAttribute (schemeNumber: GraphNumber;
                              nType       : TypeNumber;
                              attNo       : AttributeNumber;
                              attBegin    : Word.T;
                              attLength   : Word.T;
                              attValue    : TEXT;
                              truncate    : BOOLEAN          );

(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Basically works like 'AGPutAttributeSubstr' but is *)
(**   used to define the "meta" attribute value of       *)
(**   'attNo' at a node type object (and not at a node). *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**   Scheme.InternalError                               *)
(**   Scheme.NotDeclared                                 *)
(**   The scheme is not open                             *)
(**                                                      *)
(*********************************************************)

PROCEDURE AGGetMetaAttribute (    graphOrSchemeNumber: GraphNumber;
                                  nType              : TypeNumber;
                                  attNo              : AttributeNumber;
                                  attBegin           : Word.T;
                                  attLength          : Word.T;
                              VAR attValue           : TEXT;
                              VAR returnedLength     : Word.T;
                              isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Basically works like 'AGGetAttributeSubstr' but is *)
  (**   used to access the "meta" attribute value of       *)
  (**   'attNo' at a node type object (and not at a node)  *)
  (**   and is therefore the inverse function to "AGPut-   *)
  (**   MetaAttribute".                                    *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGSetEvaluationFunction (schemeNumber   : SchemeNumber;
                                   nodeClassOrType: TEXT;
                                   attribute      : TEXT;
                                   name           : TEXT          );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Set the name of the evaluation function for the    *)
  (**   attribute "attribute". This name will be           *)
  (**   valid for all nodes that belong to the class/type  *)
  (**   "nodeClassOrType" with the exception of those      *)
  (**   attributes which belong to a subclass (type) with  *)
  (**   a redefinition of the evaluation function.         *)
  (**   To get the valid evaluation function for a specific*)
  (**   class or type use "AGShowEvaluationFunction".      *)
  (**   The purpose of an evaluation function depends on   *)
  (**   the attribute's kind:                              *)
  (**      derived:   Recompute an attribute's value.      *)
  (**      intrinsic: Initialize new attribute instances   *)
  (**                 (not for intrinsic KEY attribute).   *)
  (**      meta:      Compute constant value for all nodes *)
  (**                 of given node type (i.e. the 'Name'  *)
  (**                 function has a node type instead of  *)
  (**                 a node as one of its parameters).    *)
  (**      dummy:     Not useful!                          *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   1) This action will not take place before the      *)
  (**      scheme is committed.                            *)
  (**   2) The name of the evaluation function may be      *)
  (**      redefined in subclasses/subtypes.               *)
  (**   3) You have to use "AGBindEvaluationFunction" to   *)
  (**      provide GRAS at runtime with the code addresses *)
  (**      of these functions.                             *)
  (**                                                      *)
  (** Feature:                                             *)
  (**                                                      *)
  (**   The main purpose of an evaluation function is to   *)
  (**   compute a new value for an invalid attribute and   *)
  (**   to store this value within the proper attribute    *)
  (**   itself by using 'AGPutAttributeSubstr'.            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGShowEvaluationFunction (graphOrSchemeNumber: GraphNumber;
                                    attNo              : AttributeNumber;
                                    nodeClassOrType    : TypeNumber;
                                    VAR name         : TEXT;
                                    VAR existent     : BOOLEAN;
                                        isGraphNumber: BOOLEAN   := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get the name of the evaluation function for the    *)
  (**   attribute "attNo" and the node class/type          *)
  (**   "nodeClassOrType". If the evaluation               *)
  (**   function was set, the name is returned in          *)
  (**   "name".                                            *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGBindEvaluationFunction (graphNumber: GraphNumber;
                                    name       : TEXT;
                                    function   : EvalFunction );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Specify the current address of the evaluation      *)
  (**   function "name" at runtime.                        *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   This function must be used after each "AGLogin" to *)
  (**   provide GRAS at runtime with the code addresses    *)
  (**   of the evaluation functions.                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Graph.AlreadyBound                                 *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGDeclareDependency (schemeNumber    : SchemeNumber;
                               nodeClassOrType : TEXT;
                               dependent       : TEXT;
                               dependsOn       : TEXT;
                               nameOfEdge      : TEXT;
                               kindOfDependency: DependencyKind);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Statically declare a dependency of the attribute   *)
  (**   "dependent" from the attribute "dependsOn". This   *)
  (**   dependency will be valid for all nodes that belong *)
  (**   to the class/type "nodeClassOrType".               *)
  (**   "kindOfDependency" specifies the kind of dependency*)
  (**   of the attribute (see AGGlobal for details).       *)
  (**   GRAS will delete all attributes that are affected  *)
  (**   by attribute writing or structural changes of the  *)
  (**   graph. This values must be reevaluated later on.   *)
  (**                                                      *)
  (** Note:                                                *)
  (**   This action will not take place before the scheme  *)
  (**   is committed.                                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGDeleteDependencies (graphOrSchemeNumber: GraphNumber;
                                nodeClassOrType    : TEXT;
                                dependent          : TEXT;
                                restoreInheritance : BOOLEAN;
                                isGraphNumber      : BOOLEAN       := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Delete all dependencies of the attribute           *)
  (**   "dependent" for the node class/type                *)
  (**   "nodeClassOrType". This will also delete           *)
  (**   any evaluation functions that were declared for    *)
  (**   this attribute/class combination.                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGShowAllAttributesOfNodeType (graphOrSchemeNumber: GraphNumber;
                                         nType              : TypeNumber;
                                         VAR attributes: SimpleSet;
                                         isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all attributes that were declared for the node *)
  (**   type "nType".                                      *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all attnos.  *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetAllNodesOfGraph (    graphNumber: GraphNumber;
                                VAR setOfNodes : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all nodes in the graph. All node types must use*)
  (**   the membership relation.                           *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and "AGSetKill" afterwards to release all storage  *)
  (**   occupied by this set.                              *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetAllNodeclasses (    graphOrSchemeNumber: GraphNumber;
                               VAR setOfClasses       : SimpleSet;
                               isGraphNumber: BOOLEAN := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all declared classes in the graph.             *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all classes  *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGGetAllEdgetypes (    graphOrSchemeNumber: GraphNumber;
                             VAR setOfEdgetypes     : SimpleSet;
                                 isGraphNumber      : BOOLEAN       := TRUE);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all declared edge types in the graph.          *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all types    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   The graph is not open                              *)
  (**   The scheme is not open                             *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGGetAllNodesOfNodeClass (    graphOrSchemeNumber: GraphNumber;
                                        nodeClass          : TypeNumber;
                                    VAR setOfNodes         : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all nodes of the class "nodeClass". The set    *)
  (**   must have been created. All node types must use    *)
  (**   the membership relation.                           *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGGetAllNodesOfNodeType (    graphNumber: GraphNumber;
                                       nType      : TypeNumber;
                                   VAR setOfNodes : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all nodes of type "nType". All subtypes must   *)
  (**   use the membership relation.                       *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetKill" afterwards to release all      *)
  (**   storage occupied by this set.                      *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetComplementOfNodeSet (    graphNumber: GraphNumber;
                                    VAR setOfNodes : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Get all nodes of the graph that do not belong to   *)
  (**   "setOfNodes". All node types must use the          *)
  (**   membership relation.                               *)
  (**                                                      *)
  (**   Note:                                              *)
  (**                                                      *)
  (**   Use "AGSetRemoveAnyElement" to access all nodes    *)
  (**   stored within the temporarily existing "SimpleSet" *)
  (**   and use "AGSetCreate" to initialize the set before *)
  (**   calling this PROCEDURE and use "AGSetKill" after-  *)
  (**   wards to release all storage occupied by this set. *)
  (**                                                      *)
  (*********************************************************)


PROCEDURE AGTestIsNodeOfNodeClass (graphNumber: GraphNumber;
                                   nodeClass  : TypeNumber;
                                   node       : TypeNumber   ): BOOLEAN;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Self explanatory.                                  *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.NotOwner                                     *)
  (**   Graph.Unknown                                      *)
  (**   Scheme.InternalError                               *)
  (**   Scheme.NotDeclared                                 *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

(******************************************************************************)
(**                                                                           *)
(** PROCEDUREs for setting the error check mode                               *)
(**                                                                           *)
(******************************************************************************)

PROCEDURE AGSetErrorCheckMode (graphNumber : GraphNumber;
                               errorcheckOn: BOOLEAN      );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Switch on/off the runtime legality tests. The      *)
  (**   default setting is OFF.                            *)
  (**   This affects only expensive checks against graph   *)
  (**   schemes, i.e. even with error check "OFF"  many    *)
  (**   consistency checks are active in order to avoid    *)
  (**   "serious" corruptions of graphs. Set this switch   *)
  (**   to "ON" (TRUE) for debugging purposes and to "OFF" *)
  (**   (FALSE) in your production version.                *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Graph.InternalError                                *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGFileSize (poolName: PoolName; fileName: TEXT): CARDINAL;
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the filesize of the file 'fileName'        *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   The pool is not open                               *)
  (**   PageFile.NoAccess                                  *)
  (**   VirtualResource.FatalError                         *)
  (**                                                      *)
  (*********************************************************)

(***********************************************************************)
(**                                                                    *)
(** Here starts the 'hack section'! The PROCEDUREs below are to be     *)
(** handled with care because they may lead to inconsistent states     *)
(** (if not carefully used) which may result in loss of data!          *)
(**                                                                    *)
(** The reasons for still offering these PROCEDUREs are twofold:       *)
(**    (1) Compatibility with older GRAS versions                      *)
(**    (2) Efficient usage of GRAS resources in the absence of a graph *)
(**        scheme                                                      *)
(**                                                                    *)
(**                               *****                                *)
(**                               *   *                                *)
(**                               *   *                                *)
(**                               *   *                                *)
(**                               *   *                                *)
(**                            ****   ****                             *)
(**                             *       *                              *)
(**                              *     *                               *)
(**                               *   *                                *)
(**                                * *                                 *)
(**                                 *                                  *)
(**                                                                    *)
(***********************************************************************)


PROCEDURE AGPutIndexAttribute (graphNumber: GraphNumber;
                               node       : NodeNumber;
                               attNo      : AttributeNumber;
                               attLength  : Word.T;
                               attValue   : TEXT             );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Without scheme                                     *)
  (**   --------------                                     *)
  (**   Assigns a new value to an index attribute and over-*)
  (**   rides the old value, if existent.                  *)
  (**                                                      *)
  (**   With scheme                                        *)
  (**   -----------                                        *)
  (**   Creates an index attribute for "node". If the      *)
  (**   scheme is used, we recommend to use AGPutAttribute-*)
  (**   Substr rather than this PROCEDURE!                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   PersistentGraph.IndexUsed                          *)
  (**   PersistentGraph.NotOwner                           *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetIndexAttribute (    graphNumber: GraphNumber;
                                   node       : NodeNumber;
                                   attNo      : AttributeNumber;
                               VAR attValue   : TEXT;
                               VAR attLength  : Word.T           );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Without scheme                                     *)
  (**   --------------                                     *)
  (**   Returns the value of the attribute "AttNo".        *)
  (**                                                      *)
  (**   With scheme                                        *)
  (**   -----------                                        *)
  (**   Returns the value of an index attribute for Node.  *)
  (**   If the scheme is used, we recommend to use AGGet-  *)
  (**   AttributeSubstr rather than this PROCEDURE!        *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   PersistentGraph.InternalError                      *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   PersistentGraph.NotOwner                           *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGDeleteIndexAttribute (graphNumber: GraphNumber;
                                  node       : NodeNumber;
                                  attNo      : AttributeNumber);
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Without scheme                                     *)
  (**   --------------                                     *)
  (**   Deletes an index attribute from a node.            *)
  (**                                                      *)
  (**   With scheme                                        *)
  (**   -----------                                        *)
  (**   Deletes an index attribute from a node. If the     *)
  (**   scheme is used, we recommend to use AGPutAttribute-*)
  (**   Substr rather than this PROCEDURE!                 *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   PersistentGraph.IndexUnused                        *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   PersistentGraph.NotOwner                           *)
  (**   Graph.CyclicEvaluation                             *)
  (**   Graph.InternalError                                *)
  (**   Graph.NodeNotFound                                 *)
  (**   Graph.LogError                                     *)
  (**   Graph.Unknown                                      *)
  (**   Graph.WrongType                                    *)
  (**   Graph.NotOwner                                     *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

(* the following procedure is left unimplemented unless needed. *)
(*
PROCEDURE AGGetAllNodesWithUndefinedIndex (Graph: GraphNumber;
                                           AttNo: AttributeNumber;
                                           VAR SetOfNodes: SimpleSet);
*)
(*********************************************************)
(**                                                      *)
(** Purpose:                                             *)
(**                                                      *)
(**   Returns all nodes which have currently invalid     *)
(**   index/key attribute "AttNo". This PROCEDURE is for *)
(**   applications which do not use the automatic evalu- *)
(**   facility of GRAS but have their own method to re-  *)
(**   compute invalid derived attributes. These appli-   *)
(**   cations need knowledge about invalid index attri-  *)
(**   butes in order to be able to compute all these     *)
(**   attributes before starting a corresponding index   *)
(**   query.                                             *)
(**                                                      *)
(** Error Messages:                                      *)
(**                                                      *)
(**                                                      *)
(*********************************************************)

PROCEDURE AGGetAllIndexAttributes (    graphNumber: GraphNumber;
                                       node       : NodeNumber;
                                   VAR indexSet   : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Without scheme                                     *)
  (**   --------------                                     *)
  (**   Returns all attribute numbers of index attributes  *)
  (**   corresponding to node whose value is currently     *)
  (**   defined.                                           *)
  (**                                                      *)
  (**   With scheme                                        *)
  (**   -----------                                        *)
  (**   Returns all index attribute numbers corresponding  *)
  (**   to Node which are NOT declared in the graph scheme.*)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   PersistentGraph.InternalError                      *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   PersistentGraph,NotOwner                           *)
  (**   The graph is not open                              *)
  (**   Access.Denied                                      *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGGetAllDefinedAttributes (    graphNumber: GraphNumber;
                                         node       : NodeNumber;
                                     VAR attSet     : SimpleSet    );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Returns the numbers of all attributes of a node    *)
  (**   which have currently a defined value. The result of*)
  (**   this PROCEDURE is unpredictable if there are some  *)
  (**   derived attributes whose value is not up to date.  *)
  (**   (the evaluation of these attributes will not be    *)
  (**   triggered automatically).                          *)
  (**   For index attributes, AGGetAllDefinedAttributes    *)
  (**   returns all those, which are declared in the scheme*)
  (**   and whose value is currently defined. You can use  *)
  (**   AGGetAllIndexAttributes to get the attributes      *)
  (**   which have not been defined.                       *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Denied                                      *)
  (**   Access.Locked                                      *)
  (**   PersistentGraph.InternalError                      *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   PersistentGraph.NotOwner                           *)
  (**   The graph is not open                              *)
  (**   PageFile.NoAccess                                  *)
  (**   GraphPool.InternalError                            *)
  (**   ChgMgmtGraphPool.InternalError                     *)
  (**   ChgMgmtGraphPool.NotExistent                       *)
  (**   ResourceTriggerStorage.TriggerDelayed              *)
  (**                                                      *)
  (*********************************************************)

PROCEDURE AGChangeNodeType (graphNumber: GraphNumber;
                            node       : NodeNumber;
                            newType    : TypeNumber   );
  (*********************************************************)
  (**                                                      *)
  (** Purpose:                                             *)
  (**                                                      *)
  (**   Replaces the type of the given node with "newType".*)
  (**   Attributes, names and edges of the node are not    *)
  (**   affected.                                          *)
  (**                                                      *)
  (**   DON'T USE this PROCEDURE within new applications!  *)
  (**                                                      *)
  (** Note:                                                *)
  (**                                                      *)
  (**   This operation is only admissible if the new and   *)
  (**   the old node type have the same attributes and     *)
  (**   evaluation functions (this property won't be       *)
  (**   assured or even checked)! Furthermore, instances   *)
  (**   of the new type must be allowed to have the same   *)
  (**   edge context as instances of the old type.         *)
  (**   Last but not least this function does not work for *)
  (**   node types which are declared within the graph     *)
  (**   scheme with "UseMembership"=YES (for efficiency    *)
  (**   reasons only).                                     *)
  (**                                                      *)
  (** Error Messages:                                      *)
  (**                                                      *)
  (**   Access.Locked                                      *)
  (**   ChgMgmtGraph.InternalError                         *)
  (**   ChgMgmtGraph.LogError                              *)
  (**   PersistentGraph.NodeNotFound                       *)
  (**   PersistentGraph.NotOwner                           *)
  (**   PersistentGraph.IndexUnused                        *)
  (**   PersistentGraph.InternalError                      *)
  (**   PersistentGraph.IndexUsed                          *)
  (**   The graph is not open                              *)
  (**                                                      *)
  (*********************************************************)

END RGRASGraph.
