UNSAFE MODULE PStore;

IMPORT RTDB;
IMPORT Transaction;
IMPORT Database;
IMPORT Thread;
FROM RTHeapDep IMPORT PageData;
IMPORT IntRefTransientTbl AS RequestTable; 
IMPORT IntRefTransientTbl AS DatabaseCachedPagesTable;
IMPORT IntRefTransientTbl AS CachedPagesTable;
IMPORT RefSeq AS RequestList;
IMPORT Shore;
FROM Shore IMPORT RequestType;
FROM Shore IMPORT ResponseType;

(* type definitions need to be here to avoid
   circular dependencies *)
TYPE
  (* structure for passing request/response info *)
  ReqInfo = RECORD
    id: RTDB.Id;    (* database or transaction id *)
    p_id: RTDB.Id;  (* page id                    *)
    cond: Thread.Condition;
    type: RequestType;
    status: ResponseType;
  END;

  (* describes status of a cache entry *)
  CacheStatusType = {VALID, INVALID};

  (* structure defining cache entry *)
  CacheEntry = RECORD
    status: CacheStatusType;
    data: PageData;
  END;


VAR
  req_table: RequestTable.T := NEW(RequestTable.Default).init();
  req_list: RequestList.T := NEW(RequestList.T).init();
  database_cached_pages := NEW(DatabaseCachedPagesTable.Default).init();
  m := NEW(MUTEX);
  dispatcher_thread: Thread.T;

PROCEDURE create(xct_id: RTDB.Id; database_name: TEXT): RTDB.Id
  RAISES { Database.Exists } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.CREATE_R;
      Shore.createReq(xct_id, database_name);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.EXISTS THEN
        RAISE Database.Exists;
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
    END;
    RETURN req_info.id;
  END create;

PROCEDURE open(xct_id: RTDB.Id; database_name: TEXT): RTDB.Id
  RAISES { Database.NotFound, Database.Opened} =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.OPEN_R;
      Shore.openReq(xct_id, database_name);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.NOT_FOUND THEN
        RAISE Database.NotFound;
      ELSIF req_info^.status = ResponseType.OPENED THEN
        RAISE Database.Opened;
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
    END;
    RETURN req_info.id;
  END open;


PROCEDURE getPage(xct_id: RTDB.Id; 
                   db_id: RTDB.Id; 
                   p_id: RTDB.Id;
                   swizzler: Swizzler;
                   type: RequestType)
  RAISES { Thread.Aborted } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    cached_pages:  CachedPagesTable.T;
    cache_entry: <* TRANSIENT *> REF CacheEntry;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := database_cached_pages.get(db_id, ref);
      IF result = FALSE THEN
        cached_pages := NEW(CachedPagesTable.Default).init();
        EVAL database_cached_pages.put(db_id, cached_pages);
      ELSE
        cached_pages := NARROW(ref, CachedPagesTable.T);
      END;
      
      result := cached_pages.get(p_id, ref);
      IF result = FALSE THEN
        cache_entry := NEW(<* TRANSIENT *> REF CacheEntry);
        EVAL cached_pages.put(p_id, cache_entry);
      ELSE
        cache_entry := NARROW(ref, <* TRANSIENT *> REF CacheEntry);
      END;

      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := type;
      req_info^.id := db_id;
      req_info^.p_id := p_id;
      Shore.getPageReq(xct_id, db_id, p_id, NUMBER(cache_entry^.data), type);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.DEADLOCK THEN
        RAISE Thread.Aborted;
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
      swizzler(cache_entry^.data);
    END;
  END getPage; 


PROCEDURE read(xct_id: RTDB.Id; 
               db_id: RTDB.Id;
               p_id: RTDB.Id; 
               swizzler: Swizzler)
  RAISES { Thread.Aborted } =
  BEGIN
    getPage(xct_id, db_id, p_id, swizzler, RequestType.READ_PAGE_R);
  END read; 


PROCEDURE peek(xct_id: RTDB.Id; 
               db_id: RTDB.Id;
               p_id: RTDB.Id; 
               swizzler: Swizzler)
  RAISES { Thread.Aborted } =
  BEGIN
    getPage(xct_id, db_id, p_id, swizzler, RequestType.PEEK_PAGE_R);
  END peek;

PROCEDURE write(xct_id: RTDB.Id; 
                db_id: RTDB.Id; 
                p_id: RTDB.Id;
                unswizzler: Unswizzler)
  RAISES { Thread.Aborted } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    cached_pages:  CachedPagesTable.T;
    cache_entry: <* TRANSIENT *> REF CacheEntry;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := database_cached_pages.get(db_id, ref);
      IF result = FALSE THEN
        cached_pages := NEW(CachedPagesTable.Default).init();
        EVAL database_cached_pages.put(db_id, cached_pages);
      ELSE
        cached_pages := NARROW(ref, CachedPagesTable.T);
      END;
    
      result := cached_pages.get(p_id, ref);
      IF result = FALSE THEN
        cache_entry := NEW(<* TRANSIENT *> REF CacheEntry);
        EVAL cached_pages.put(p_id, cache_entry);
      ELSE
        cache_entry := NARROW(ref, <* TRANSIENT *> REF CacheEntry);
      END;
      
      unswizzler(cache_entry^.data);

      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.WRITE_PAGE_R;
      Shore.putPageReq(xct_id, db_id, p_id, cache_entry^.data);  <* NOWARN *>
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.DEADLOCK THEN
        RAISE Thread.Aborted;
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END
    END
  END write;

PROCEDURE lockPage(xct_id: RTDB.Id; db_id: RTDB.Id; p_id: RTDB.Id;
                   mode: Transaction.LockMode) 
  RAISES {Thread.Aborted} =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.LOCK_PAGE_R;
      Shore.lockPageReq(xct_id, db_id, p_id, mode);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.DEADLOCK THEN
        RAISE Thread.Aborted;
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END
    END
  END lockPage;

PROCEDURE begin(): RTDB.Id
  RAISES { Transaction.InProgress } =
  VAR
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN; 
  BEGIN
    req_info := NEW(<* TRANSIENT *> REF ReqInfo); 
    req_info^ := ReqInfo{id := 0,
                         p_id := 0,
                         cond := NEW(Thread.Condition),
                         type:=RequestType.BEGIN_R, 
                         status := ResponseType.OK};
    
    LOCK m DO
      req_list.addhi(req_info);
      Shore.beginReq();
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.IN_PROG THEN
        RAISE Transaction.InProgress
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
      result :=  req_table.put(req_info^.id, req_info);
      <* ASSERT result = FALSE *>
    END;
    RETURN req_info^.id;
  END begin;

PROCEDURE commit(xct_id: RTDB.Id)
  RAISES { Transaction.NotInProgress } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN; 
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.COMMIT_R;
      Shore.commitReq(xct_id);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.N_IN_PROG THEN
        RAISE Transaction.NotInProgress
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
      EVAL req_table.delete(xct_id, ref);
    END; 
  END commit;

PROCEDURE chain(xct_id: RTDB.Id)
  RAISES { Transaction.NotInProgress } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN;
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.CHAIN_R;
      Shore.chainReq(xct_id);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.N_IN_PROG THEN
        RAISE Transaction.NotInProgress
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
    END;
  END chain;

PROCEDURE abort(xct_id: RTDB.Id)
  RAISES { Transaction.NotInProgress } =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo;  
    result: BOOLEAN; 
  BEGIN
    LOCK m DO
      result := req_table.get(xct_id, ref);
      <* ASSERT result = TRUE *>
      req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
      req_info^.type := RequestType.ABORT_R;
      Shore.abortReq(xct_id);
      Thread.Wait(m, req_info^.cond);
      IF req_info^.status = ResponseType.N_IN_PROG THEN
        RAISE Transaction.NotInProgress
      ELSIF req_info^.status # ResponseType.OK THEN
        RAISE FatalError;   <* NOWARN *>
      END;
      EVAL req_table.delete(xct_id, ref);
    END;
  END abort;


(* Dispatcher thread taking server operation requests from transactional 
   threads and handing server responses back to them                    *)

PROCEDURE dispatch(<* UNUSED *> cl: Thread.Closure): REFANY =
  VAR
    ref: <* TRANSIENT *> REFANY;
    req_info: <* TRANSIENT *> REF ReqInfo; 
    xct_id: RTDB.Id;
    result: BOOLEAN;
    cached_pages:  CachedPagesTable.T;
    cache_entry: <* TRANSIENT *> REF CacheEntry;
  BEGIN
      WHILE TRUE DO        
        (* blocks until getting a response for
           some transaction  *)
        xct_id := Shore.getResponseXct();
        LOCK m DO           
          result := req_table.get(xct_id, ref);
          IF result = TRUE THEN
            req_info := NARROW(ref, <* TRANSIENT *> REF ReqInfo);
            IF req_info^.type = RequestType.OPEN_R THEN
              req_info^.status := Shore.openResp(req_info^.id); 
            ELSIF req_info^.type = RequestType.CREATE_R THEN
              req_info^.status := Shore.createResp(req_info^.id); 
            ELSIF req_info^.type = RequestType.READ_PAGE_R OR
              req_info^.type = RequestType.PEEK_PAGE_R THEN
              result := database_cached_pages.get(req_info^.id, ref);
              <* ASSERT result = TRUE *>
              cached_pages := NARROW(ref, CachedPagesTable.T);
              
              result := cached_pages.get(req_info^.p_id, ref);
              <* ASSERT result = TRUE *>      
              cache_entry := NARROW(ref, <* TRANSIENT *> REF CacheEntry);

              req_info^.status := Shore.getPageResp(cache_entry^.data); 
            ELSIF req_info^.type = RequestType.WRITE_PAGE_R THEN
              req_info^.status := Shore.putPageResp(); 
            ELSIF req_info^.type = RequestType.LOCK_PAGE_R THEN
              req_info^.status := Shore.lockPageResp(); 
            ELSIF req_info^.type = RequestType.COMMIT_R THEN
              req_info^.status := Shore.commitResp(); 
            ELSIF req_info^.type = RequestType.CHAIN_R THEN
              req_info^.status := Shore.chainResp(); 
            ELSIF req_info^.type = RequestType.ABORT_R THEN
              req_info^.status := Shore.abortResp(); 
            ELSE
              <* ASSERT FALSE *>
            END;
          ELSE
            req_info := req_list.remlo();
            req_info^.status := Shore.beginResp(); 
            req_info^.id := xct_id;
          END;
          Thread.Signal(req_info^.cond);
        END
      END;
    RETURN NIL;
  END dispatch;


BEGIN  
  dispatcher_thread := Thread.Fork(NEW(Thread.Closure, apply := dispatch) );
END PStore.
















