MODULE Transaction EXPORTS Transaction, InternalTransaction;

IMPORT RTHeapDB;
IMPORT PStore;
IMPORT RTDB;
IMPORT ThreadF;
IMPORT BaseDatabase, InternalDatabase;
IMPORT BaseTransaction;

REVEAL T = Internal BRANDED "Transaction.T" OBJECT
  open := FALSE;
OVERRIDES
  (* user-level API *)
  init := Init;
  begin := Begin;
  commit := Commit;
  chain := Chain;
  abort := Abort;
  checkpoint := Checkpoint;
  isOpen := IsOpen;
  lock := Lock;

END;

PROCEDURE Init(self: T): BaseTransaction.T =
  BEGIN
    EVAL BaseTransaction.T.init(self);
    RETURN self;
  END Init;

PROCEDURE Begin(self: T) RAISES { InProgress } =
  BEGIN
    LOCK self DO
      EVAL self.init();
      IF self.open THEN RAISE InProgress; END;
      self.id := PStore.begin();
      ThreadF.TxnBegin(self);
      self.open := TRUE;
    END
  END Begin;

PROCEDURE Commit(self: T)
  RAISES { NotInProgress, Disabled } =
  BEGIN
    LOCK self DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      TRY
        RTHeapDB.Flush();
      EXCEPT
        RTHeapDB.Disabled => RAISE Disabled;
      END;
      PStore.commit(self.id);
      ThreadF.TxnCommit();
      self.open := FALSE;
    END;
  END Commit;

PROCEDURE Chain(self: T)
  RAISES { NotInProgress, Disabled } =
  BEGIN
    LOCK self DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      TRY
        RTHeapDB.Flush();
      EXCEPT
        RTHeapDB.Disabled => RAISE Disabled;
      END;
      PStore.chain(self.id);
    END;
   END Chain;

PROCEDURE Abort(self: T) RAISES { NotInProgress } =
  BEGIN
    LOCK self DO
      IF NOT self.open THEN RAISE NotInProgress END;
      PStore.abort(self.id);
      ThreadF.TxnAbort();
      self.open := FALSE;
    END;
  END Abort;

PROCEDURE Checkpoint(self: T)
  RAISES { NotInProgress, Disabled } =
  BEGIN
    LOCK self DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      TRY
        RTHeapDB.Flush();
      EXCEPT
        RTHeapDB.Disabled => RAISE Disabled;
      END;
    END;
  END Checkpoint;

PROCEDURE IsOpen(self: T): BOOLEAN =
  BEGIN
    LOCK self DO
      RETURN self.open;
    END;
  END IsOpen;

PROCEDURE Lock(self: T; object: REFANY; mode: LockMode)
  RAISES { NotInProgress } =
  VAR p: RTDB.Page;
  BEGIN
    LOCK self DO
      IF NOT self.open THEN RAISE NotInProgress; END;
      p := RTHeapDB.RefPageMap(object);
      IF p # NIL THEN
        PStore.lockPage(self.id,                        <* NOWARN *>
                        NARROW(p.db, InternalDatabase.Internal).id, 
                        p.id, mode);  
      END;
    END;
  END Lock;


BEGIN
END Transaction.
