(* Copyright (C) 1993, Digital Equipment Corporation         *)
(* All rights reserved.                                      *)
(* See the file COPYRIGHT for a full description.            *)
(*                                                           *)
(* portions Copyright 1996, Critical Mass, Inc.              *)
(* portions Copyright 1998, Purdue Research Foundation       *)
(*                                                           *)
(*| Last modified on Fri Apr 26 10:29:11 PDT 1996 by heydon  *)
(*|      modified on Sat Nov 19 09:37:57 PST 1994 by kalsow  *)
(*|      modified on Fri Aug  5 14:04:35 PDT 1994 by jdd     *)
(*|      modified on Wed Jun  2 15:00:17 PDT 1993 by muller  *)
(*|      modified on Wed Apr 21 13:14:37 PDT 1993 by mcjones *)
(*|      modified on Wed Mar 10 11:01:47 PST 1993 by mjordan *)

UNSAFE MODULE RTCollector EXPORTS RTCollector, RTCollectorSRC,
                                  RTHeapRep, RTWeakRef, RTHeapDB;

IMPORT RT0, RT0u, RTIO, RTHeapEvent, RTHeapDep, RTHeapMap, RTMachine;
IMPORT RTMisc, RTOS, RTParams, RTPerfTool, RTProcess, RTType;
IMPORT Word, Cstdlib, Thread, ThreadF;
IMPORT RTDB, RTTxn, RTTypeMap, RTTypeFP, Fingerprint;
IMPORT Text, TextF;

FROM RT0 IMPORT Typecode, TypeDefn;
FROM RTIO IMPORT PutText, PutInt, PutChars;
IMPORT Cstring;

(* The allocator/garbage collector for the traced heap is an adaptation of
   the algorithm presented in the WRL Research Report 88/2, ``Compacting
   Garbage Collection with Ambiguous Roots'', by Joel F.  Bartlett; see
   this report for a detailed presentation.  John DeTreville modified it to
   be incremental, generational, and VM-synchronized.

   The allocator/collector for the untraced heap is simply malloc/free. *)

(* Much of the code below incorrectly assumes no difference between ADRSIZE
   and BYTESIZE. *)

(* In the following procedures, "RTType.Get(tc)" will fail if "tc" is not
   proper. *)

(*** RTCollector ***)

PROCEDURE Disable () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishVM();
      INC(disableCount);
      partialCollectionNext := FALSE;
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END Disable;

PROCEDURE Enable () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END Enable;

PROCEDURE DisableMotion () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      INC(disableMotionCount);
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END DisableMotion;

PROCEDURE EnableMotion () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableMotionCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
    IF perfOn THEN PerfAllow(); END;
  END EnableMotion;

PROCEDURE Collect () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishGC();
      StartGC();
      FinishGC();
    END;
    RTOS.UnlockHeap();
  END Collect;

(*** RTCollectorSRC ***)

(* StartCollection starts a total collection, if none is in progress and if
   collection and motion are enabled. *)

PROCEDURE StartCollection () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      CollectorOn();
      IF collectorState = CollectorState.Zero
           AND disableCount + disableMotionCount = 0 THEN
        partialCollectionNext := FALSE;
        REPEAT CollectSome(); UNTIL collectorState # CollectorState.Zero;
        IF NOT (incremental AND RTHeapDep.VM AND disableVMCount = 0) THEN
          REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
        END;
      END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END StartCollection;

(* FinishCollection finishes the current collection, if one is on
   progress. *)

PROCEDURE FinishCollection () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      CollectorOn();
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END FinishCollection;

(* DisableVM disables the use of VM protection.  While VM protection is
   disabled, no objects on the heap will be protected.*)

PROCEDURE DisableVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishVM();
      INC(disableVMCount);
    END;
    RTOS.UnlockHeap();
  END DisableVM;

(* EnableVM reenables the use of VM protection if EnableVM has been called
   as many times as DisableVM.  It is a checked runtime error to call
   EnableVM more times than DisableVM. *)

PROCEDURE EnableVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      DEC(disableVMCount);
      CollectEnough();
    END;
    RTOS.UnlockHeap();
  END EnableVM;

(* FinishVM is equivalent to DisableVM{}; EnableVM().  FinishVM unprotects
   all heap pages, and is intended for use from the debugger. *)

PROCEDURE FinishVM () =
  BEGIN
    RTOS.LockHeap();
    BEGIN
      FinishGC();
      CollectorOn();
      (* no gray pages now; only protected pages are in older generation *)
      ThreadF.SuspendOthers();
      FOR p := p0 TO p1 - 1 DO
        IF ThreadF.myTxn = NIL
          OR map[p - p0] = NIL
          OR map[p - p0].writer = ThreadF.myTxn
         THEN
          IF desc[p - p0].access # Access.ReadWrite THEN
            desc[p - p0].dirty := TRUE;
            Unprotect(p);
          END
        ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
          IF desc[p - p0].access = Access.None THEN
            Protect(p, Access.ReadOnly);
          END
        END
      END;
      ThreadF.ResumeOthers();
      CollectorOff();
    END;
    RTOS.UnlockHeap();
  END FinishVM;

(* StartBackgroundCollection starts the background thread, if not already
   started *)

VAR startedBackground := FALSE;

PROCEDURE StartBackgroundCollection () =
  VAR start := FALSE;
  BEGIN
    RTOS.LockHeap();
    BEGIN
      IF NOT startedBackground THEN
        start := TRUE;
        startedBackground := TRUE;
      END;
    END;
    RTOS.UnlockHeap();
    IF start THEN
      EVAL Thread.Fork(NEW(Thread.Closure, apply := BackgroundThread));
    END;
  END StartBackgroundCollection;

(* ------------------------------- low-level allocation and collection *)

(* We assume that references (values of the types ADDRESS and REFANY) are
   the addresses of addressable locations and that locations with
   successive addresses are contiguous (that is, if a points to a
   n-locations referent then these n locations are at addresses a, a+1,
   ..., a+n-1).

   The memory is viewed as a collection of pages.  Each page has a number
   that identifies it, based on the addresses that are part of this page:
   page p contains the addresses p * BytesPerPage to (p+1) * BytesPerPage -
   1.

   The page size must be a multiple of the header size (see below).  Given
   our conventions about page boundaries, this implies that the first
   location of a page is properly aligned for a Header. *)

(* The array desc and the global variables p0, and p1 describe the pages
   that are part of the traced heap.  Either p0 and p1 are equal to Nil and
   no pages are allocated; or both are valid pages and page p is allocated
   iff

|          p0 <= p < p1
|      AND desc[p - p0] != Unallocated

   NUMBER (desc) must be equal to p1 - p0 if there are allocated pages.
   Index i in desc correspond to page i + p0; that is p0 is the number of
   the first page available in desc, and it must be in [p0 ..  p1) if there
   are allocated pages. *)

(* We keep the number of allocated pages in a global variable; it should
   satify the invariant:

|     allocatedPages = sigma (i = p0, p1-1,
|                              space [i - p0] # Unallocated)
|                                  if there are allocated pages,
|                      = 0 otherwise.

   We also keep the number of active pages in a global; it satisfies:

|     activePages = sigma (i = p0, p1-1,
|                           space [i - p0] = nextSpace)
|                                if there are allocated pages,
|                 = 0 otherwise. *)

(* Each referent is immediately preceded by a header that describes the
   type of the referent.  In the user world, this header is not visible;
   that is, a REFANY is the address of the referent, not the address of the
   header.

   Each referent is immediately followed by padding space so the combined
   size referent size + padding is a multiple of the header size.
   Actually, the low level routines are given a data size which is the sum
   of the referent size and padding size and assume this data size is a
   multiple of the header size.

   With this padding, addresses of headers and referent will always be
   multiple of ADRSIZE (Header).

   The combination of header/referent/padding space is called a "heap
   object".  The size of a heap object is the size of the header, plus the
   size of the referent, plus the size of the padding.  The alignment of a
   heap object is the greatest of the alignment of header and the alignment
   of the referent.

   We make the following assumptions:

   - alignment of headers is such what the addressable location following
   any properly aligned header is properly aligned for the type ADDRESS;
   and, for every referent: referent adrSize + padding adrSize >= ADRSIZE
   (ADDRESS)

   [During the garbage collection, we move heap objects.  But we need to
   keep the forwarding information somewhere.  This condition ensures that
   we can store the new address of the referent in the first word of the
   old referent.]

   - the pages are aligned more strictly than the headers (this means that
   the page size is a multiple of the header alignment).

   [We can put a header at the beginning of a page] *)

TYPE
  RefReferent = ADDRESS;

PROCEDURE HeaderOf (r: RefReferent): RefHeader =
  BEGIN
    RETURN LOOPHOLE(r - ADRSIZE(Header), RefHeader);
  END HeaderOf;

(* If a page is allocated, it can be normal or continued.  In the first
   case, there is a heap object just at the beginning of the page and
   others following.  The second case occurs when a heap object was too
   large to fit on a page: it starts at the beginning of a normal page and
   overflows on contiguous continued pages.  Whatever space is left on the
   last continued page is never used for another object or filler.  In
   other words, all the headers are on normal pages.

   Heap objects do not need to be adjacent.  Indeed, alignment constraints
   would make it difficult to ensure that property.  Filler objects may
   appear before objects to align them, or after the last object on a
   normal page to fill the page. *)

(* We need to be able to determine the size of an referent during
   collection; here is a functions to do just that.  It must be called with
   a non-nil pointer to the Header of a heap object that is there (has not
   been moved). *)

PROCEDURE OpenArraySize(h: RefHeader; def: TypeDefn): CARDINAL =
    (* the referent is an open array; it has the following layout:
|         pointer to the elements (ADDRESS)
|         size 1
|         ....
|         size n
|         optional padding
|         elements
|         ....
       where n is the number of open dimensions (given by the definition)
       and each size is the number of elements along the dimension *)
  VAR
    res: INTEGER;
    sizes: UNTRACED REF INTEGER := h + ADRSIZE(Header) + ADRSIZE(ADDRESS);
                                                           (* ^ elt pointer*)
  BEGIN
    res := 1;
    FOR i := 0 TO def.nDimensions - 1 DO
      res := res * sizes^;
      INC(sizes, ADRSIZE(sizes^));
    END;
    res := res * def.elementSize;
    res := RTMisc.Upper(res + def.dataSize, BYTESIZE(Header));
    RETURN res;
  END OpenArraySize;

PROCEDURE ReferentSize (h: RefHeader): CARDINAL =
  VAR
    res: INTEGER;
    tc: Typecode := h.typecode;
    def: TypeDefn;
  BEGIN
    IF tc = Fill_1_type THEN RETURN 0; END;
    IF tc = Fill_N_type THEN
      res := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
      RETURN res - BYTESIZE(Header);
    END;
    def := RTType.Get (tc);
    IF def.nDimensions = 0 THEN
      (* the typecell datasize tells the truth *)
      RETURN def.dataSize;
    END;
    (* ELSE, the referent is an open array *)
    RETURN OpenArraySize(h, def);
  END ReferentSize;

(* The convention about page numbering allows for a simple conversion from
   an address to the number of the page in which it is, as well as from a
   page number to the first address is contains: *)

PROCEDURE ReferentToPage (r: RefReferent): Page =
  (* VAR p: INTEGER := LOOPHOLE(r, INTEGER) DIV BytesPerPage; *)
  VAR p: INTEGER := Word.RightShift (LOOPHOLE(r, INTEGER), LogBytesPerPage);
  BEGIN
    IF p < p0 OR p >= p1 OR desc[p - p0].space = Space.Unallocated
      THEN RETURN Nil;
      ELSE RETURN p;
    END;
  END ReferentToPage;

PROCEDURE HeaderToPage (r: RefHeader): Page =
  (* VAR p: INTEGER := LOOPHOLE(r, INTEGER) DIV BytesPerPage; *)
  VAR p: INTEGER := Word.RightShift (LOOPHOLE(r, INTEGER), LogBytesPerPage);
  BEGIN
    IF p < p0 OR p >= p1 OR desc[p - p0].space = Space.Unallocated
      THEN RETURN Nil;
      ELSE RETURN p;
    END;
  END HeaderToPage;

PROCEDURE PageToHeader (p: Page): RefHeader =
  BEGIN
    RETURN LOOPHOLE(p * BytesPerPage, RefHeader);
  END PageToHeader;

PROCEDURE PageToAddress (p: Page): ADDRESS =
  BEGIN
    RETURN LOOPHOLE(p * BytesPerPage, ADDRESS);
  END PageToAddress;

PROCEDURE PageToData (p: Page): UNTRACED REF RTHeapDep.PageData =
  BEGIN
    RETURN LOOPHOLE(p * BytesPerPage, UNTRACED REF RTHeapDep.PageData);
  END PageToData;

PROCEDURE RefPageMap (object: REFANY): RTDB.Page =
  VAR p: Page;
  BEGIN
    TRY
      RTOS.LockHeap();
      p := ReferentToPage(LOOPHOLE(object, RefReferent));
      IF p = Nil THEN
        RETURN NIL;
      ELSE
        RETURN map[p - p0];
      END
    FINALLY
      RTOS.UnlockHeap();
    END
  END RefPageMap;

(* We remember where we should look for free space with the following
   globals: *)

(* The inner-loop collector action is to pick a gray page and completely
   clean it (i.e., make its referents at least gray, so that the page
   becomes black).  The current gray page, "impureCopy.page" is
   distinguished; it's the page that newly gray objects are copied to.

   To improve locality of referene in the new space, we keep the set of
   gray pages as a stack.  This helps approximate a depth-first copy to
   newspace.  The current page is not a member of the stack, but will
   become one when it becomes full.  The current page is always the page
   that contains new.ptr.

   To reduce page faults, we separate the "pure" copy pages (those whose
   objects contain no REFs) from the "impure" ones (those with REFs).  Only
   impure pages become gray, since pure pages can have no REFs into the old
   space (since they have no REFs at all). *)

(* By analogy, we also maintain "pureCopy.page" and "pureCopy.stack".  These
   are not used, but maintaining them simplifies the code. *)

(* By analogy, we also maintain "new.page" and "new.stack".  As with
   pureCopy.page and pureCopy.stack, these are not used, but maintaining them
   simplifies the code. *)

TYPE
  AllocRec = RECORD
    ptr, boundary: RefHeader;
    page:  Page := Nil; (* the current page *)
    stack: Page := Nil; (* threaded through the "link" field; ends at Nil *)
    db: RTDB.T;
  END;

VAR
  new: AllocRec;
  (* memory in [new.ptr, new.boundary) is available to AllocForNew *)

  newTransient: AllocRec;
  (* memory in [newTransient.ptr, newTransient.boundary) is available to
     AllocForNewTransient *)

  pureCopy: AllocRec;
  (* memory in [pureCopy.ptr, pureCopy.boundary) is available to AllocForCopy
     for pure objects (objects with no REFs) *)

  impureCopy: AllocRec;
  (* memory in [impureCopy.ptr, impureCopy.boundary) is available to
     AllocForCopy for impure objects (objects with REFs) *)

  pureTransientCopy: AllocRec;
  (* memory in [pureTransientCopy.ptr, pureTransientCopy.boundary) is available
     to AllocForCopy for pure objects (objects with no REFs) *)

  impureTransientCopy: AllocRec;
  (* memory in [impureTransientCopy.ptr, impureTransientCopy.boundary) is
     available to AllocForCopy for impure objects (objects with
     REFs) *)

(* To move a heap object to the new space, modifying the original
   reference to it *)

TYPE Mover = RTHeapMap.Visitor OBJECT OVERRIDES apply := Move END;

PROCEDURE Move (<*UNUSED*> self: Mover;  cp: ADDRESS) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil THEN RETURN; END;
      VAR
        pi        := p - p0;
        oldHeader := HeaderOf(ref);
      BEGIN
        IF desc[pi].space # Space.Previous THEN
          RETURN;                (* nothing to do *)
        END;
        IF map[pi] # NIL THEN
          (* if this is a persistent object, just promote the pages *)
          IF desc[pi].pure THEN
            PromotePage(
                p, Desc{space := Space.Current, generation := copyGeneration,
                        pure := TRUE, note := Note.Persistent,
                        gray := FALSE, access := desc[pi].access,
                        continued := FALSE, resident := desc[pi].resident,
                        dirty := FALSE});
          ELSE
            IF desc[pi].access # Access.ReadWrite THEN Unprotect(p) END;
            PromotePage(
                p, Desc{space := Space.Current, generation := copyGeneration,
                        pure := FALSE, note := Note.Persistent,
                        gray := TRUE, access := Access.ReadWrite,
                        continued := FALSE, resident := TRUE,
                        dirty := TRUE});
            desc[pi].link := impureCopy.stack;
            impureCopy.stack := p;
          END;
        ELSIF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if this is a large object, just promote the pages *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := TRUE, note := Note.Large, gray := FALSE,
                          access := Access.ReadWrite, continued := FALSE,
                          resident := TRUE, dirty := FALSE});
            ELSE
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := FALSE, note := Note.Large, gray := TRUE,
                          access := Access.ReadWrite, continued := FALSE,
                          resident := TRUE, dirty := TRUE});
              IF def.traced = 1 THEN
                desc[pi].link := impureCopy.stack;
                impureCopy.stack := p;
              ELSE
                <*ASSERT def.traced = 3*>
                desc[pi].link := impureTransientCopy.stack;
                impureTransientCopy.stack := p;
              END;
            END;
          END;
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF (def.gc_map # NIL) OR (def.parent # NIL) THEN
              IF def.traced = 1 THEN
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   impureCopy,
                                   pure := FALSE);
              ELSE
                <* ASSERT def.traced = 3 *>
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   impureTransientCopy,
                                   pure := FALSE);
              END
            ELSE
              IF def.traced = 1 THEN
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   pureCopy,
                                   pure := TRUE);
              ELSE
                <* ASSERT def.traced = 3 *>
                np := AllocForCopy(dataSize, def.dataAlignment,
                                   pureTransientCopy,
                                   pure := TRUE);
              END
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END;
        END;
      END;
    END;
  END Move;

TYPE TMover = RTHeapMap.Visitor OBJECT OVERRIDES apply := TMove END;

PROCEDURE TMove (<*UNUSED*> self: TMover; cp: ADDRESS) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil THEN RETURN; END;
      VAR
        pi        := p - p0;
        oldHeader := HeaderOf(ref);
      BEGIN
        IF desc[pi].space # Space.Previous THEN
          RETURN;                (* nothing to do *)
        END;
        IF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if this is a large object, just promote the pages *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF def.traced # 3 THEN RETURN END;
            <* ASSERT map[pi] = NIL *>
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := TRUE, note := Note.Large, gray := FALSE,
                          access := Access.ReadWrite, continued := FALSE,
                          resident := TRUE, dirty := FALSE});
            ELSE
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := FALSE, note := Note.Large, gray := TRUE,
                          access := Access.ReadWrite, continued := FALSE,
                          resident := TRUE, dirty := TRUE});
              desc[pi].link := impureTransientCopy.stack;
              impureTransientCopy.stack := p;
            END;
          END;
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF def.traced # 3 THEN RETURN END;
            <* ASSERT map[pi] = NIL *>
            IF (def.gc_map # NIL) OR (def.parent # NIL) THEN
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 impureTransientCopy,
                                 pure := FALSE);
            ELSE
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 pureTransientCopy,
                                 pure := TRUE);
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END;
        END;
      END;
    END;
  END TMove; 

TYPE PMover = RTTypeMap.Visitor OBJECT
  db: RTDB.T
OVERRIDES
  apply := PMove
END;

PROCEDURE PMove (self: PMover; cp: ADDRESS; k: RTTypeMap.Kind) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref    := refref^;
    text: TEXT;
  BEGIN
    IF ref = NIL THEN RETURN; END;
    <* ASSERT k = RTTypeMap.Kind.Ref *>
    VAR
      p := ReferentToPage(ref);
      oldHeader := HeaderOf(ref);
    BEGIN
      IF p = Nil THEN
        (* Must be statically allocated text constant. *)
        (* Get a copy in the heap *)
        <* ASSERT oldHeader.typecode = RT0.TextTypecode *>
        text := TextF.New(Text.Length(LOOPHOLE(ref, TEXT)));
        text^ := LOOPHOLE(ref, TEXT)^;
(* 
        text := NIL;
        self.db.mapText(LOOPHOLE(ref, TEXT), text);
*)
        refref^ := LOOPHOLE(text, RefReferent);
        RETURN;
      END;
      VAR pi := p - p0;
      BEGIN
        IF desc[pi].space = Space.Current THEN
          IF map[pi] # NIL THEN
            (* check for cross file reference *)
            <* ASSERT self.db = map[pi].db *>
          ELSE
            (* page must be pinned by ambiguous root, make it persistent *)
            <* ASSERT desc[pi].note = Note.AmbiguousRoot *>
            VAR def := RTType.Get (oldHeader.typecode);
            BEGIN
              IF def.traced = 1 THEN
                WITH d = desc[pi] DO
                  IF d.pure THEN
                    d.link := pureCopy.stack;
                    pureCopy.stack := p;
                  ELSE
                    d.link := impureCopy.stack;
                    impureCopy.stack := p;
                  END
                END;
                FOR i := 0 TO PageCount(p) - 1 DO
                  VAR page := self.db.newPage();
                  BEGIN
                    map[pi + i] := page;
                    page.p := p + i;
                    page.lastReader := ThreadF.myTxn;
                    page.writer := ThreadF.myTxn;
                    WITH d = desc[pi + i] DO
                      d.gray := TRUE;
                    END;
                    IF perfOn THEN PerfChange(p + i, 1); END;
                  END
                END
              END
            END
          END;
          RETURN;
        END;
        <* ASSERT desc[pi].space = Space.Previous *>
        <* ASSERT map[pi] = NIL *>
        IF p + 1 < p1 AND desc[pi + 1].continued THEN
          (* if large, make persistent and promote to current space *)
          VAR def := RTType.Get (oldHeader.typecode);
          BEGIN
            IF def.traced = 1 THEN
              IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := TRUE,
                            note := Note.Large, gray := TRUE,
                            access := Access.ReadWrite, continued := FALSE,
                            resident := TRUE, dirty := FALSE});
                desc[pi].link := pureCopy.stack;
                pureCopy.stack := p;
              ELSE
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := FALSE,
                            note := Note.Large, gray := TRUE,
                            access := Access.ReadWrite, continued := FALSE,
                            resident := TRUE, dirty := TRUE});
                desc[pi].link := impureCopy.stack;
                impureCopy.stack := p;
              END;
              FOR i := 0 TO PageCount(p) - 1 DO
                VAR page := self.db.newPage();
                BEGIN
                  map[pi + i] := page;
                  page.p := p + i;
                  page.lastReader := ThreadF.myTxn;
                  page.writer := ThreadF.myTxn;
                END
              END
            END
          END
        ELSIF oldHeader.forwarded THEN
          (* if already moved, just update the reference *)
          refref^ := LOOPHOLE(ref, UNTRACED REF RefReferent)^;
        ELSE
          (* move the object *)
          VAR
            def      := RTType.Get(oldHeader.typecode);
            dataSize := ReferentSize(oldHeader);
            np       : RefReferent;
          BEGIN
            IF def.traced # 1 THEN RETURN END;
            IF (def.gc_map = NIL) AND (def.parent = NIL) THEN
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 pureCopy, pure := TRUE);
            ELSE
              np := AllocForCopy(dataSize, def.dataAlignment,
                                 impureCopy, pure := FALSE);
            END;
            RTMisc.Copy(oldHeader, HeaderOf(np), BYTESIZE(Header) + dataSize);
            IF def.nDimensions # 0 THEN
              (* open array: update the internal pointer *)
              LOOPHOLE(np, UNTRACED REF ADDRESS)^ := np + def.dataSize;
            END;
            oldHeader.forwarded := TRUE;
            LOOPHOLE(ref, UNTRACED REF RefReferent)^ := np;
            refref^ := np;
          END
        END
      END
    END
  END PMove;

(* Determines whether a REF has yet been moved into the new space.  Follows
   the logic in "Move".*)

PROCEDURE Moved (ref: RefReferent): BOOLEAN =
  BEGIN
    IF ref = NIL THEN RETURN TRUE; END;
    (* check the space *)
    VAR p := ReferentToPage(ref);
    BEGIN
      IF p = Nil OR desc[p - p0].space # Space.Previous THEN
        RETURN TRUE;
      END;
    END;
    (* check the forwarded bit *)
    IF HeaderOf(LOOPHOLE(ref, ADDRESS)).forwarded THEN RETURN TRUE; END;
    (* not moved *)
    RETURN FALSE;
  END Moved;

(* When an allocated page is referenced by the stack, we have to move it to
   the next space and insert it in the list of promoted pages.  In the case
   where the page is actually part of a group of pages for a big referent,
   we have to promote all these pages to the new space, but only the first
   one needs to be inserted in the queue, as it is the only one containing
   referent headers.

   This routine is passed to the Threads implementation.  It is called for
   each stack, where start and stop are the addresses of the first and last
   word of the stack under consideration. *)

PROCEDURE NoteStackLocations (start, stop: ADDRESS) =
  VAR
    fp                                := start;
    firstAllocatedAddress             := PageToAddress(p0);
    firstNonAllocatedAddress          := PageToAddress(p1);
    p                       : ADDRESS;
    pp                      : Page;
  BEGIN
    WHILE fp <= stop DO
      p := LOOPHOLE(fp, UNTRACED REF ADDRESS)^;
      IF firstAllocatedAddress <= p AND p < firstNonAllocatedAddress THEN
        pp := LOOPHOLE(p, INTEGER) DIV BytesPerPage;
        IF desc[pp - p0].space = Space.Previous THEN
          VAR fp := FirstPage(pp);
          BEGIN
            IF desc[fp - p0].pure THEN
              PromotePage(fp, Desc{space := Space.Current, pure := TRUE,
                                   note := Note.AmbiguousRoot, gray := FALSE,
                                   generation := copyGeneration,
                                   access := desc[fp - p0].access,
                                   continued := FALSE,
                                   resident := desc[fp-p0].resident,
                                   dirty := FALSE});
            ELSE
              IF desc[fp - p0].access # Access.ReadWrite THEN
                Unprotect(fp);
              END;
              PromotePage(fp, Desc{space := Space.Current, pure := FALSE,
                                   note := Note.AmbiguousRoot, gray := TRUE,
                                   generation := copyGeneration,
                                   access := Access.ReadWrite,
                                   continued := FALSE,
                                   resident := TRUE,
                                   dirty := TRUE});
              desc[fp - p0].link := impureTransientCopy.stack;
              impureTransientCopy.stack := fp;
            END;
          END;
        END;
      END;
      INC(fp, RTMachine.PointerAlignment);
    END;
  END NoteStackLocations;

PROCEDURE PromotePage (p: Page;  READONLY d: Desc) =
  BEGIN
    <* ASSERT desc[p - p0].space = Space.Previous *>
    <* ASSERT NOT desc[p - p0].continued*>
    VAR n := PageCount(p);
    BEGIN
      desc[p - p0] := d;
      IF n > 1 THEN
        VAR dd := d;
        BEGIN
          dd.continued := TRUE;
          FOR pp := p + 1 TO p + n - 1 DO desc[pp - p0] := dd; END;
        END;
      END;
      IF perfOn THEN PerfChange(p, n); END;
      IF d.space = Space.Current THEN
        IF n = 1 THEN
          INC(smallPromotionPages, 1);
        ELSE
          INC(largePromotionPages, n);
        END;
      END;
    END;
  END PromotePage;

PROCEDURE InsertFiller (start: RefHeader; n: INTEGER) =
  BEGIN
    IF n = 0 THEN
      (* nothing to do *)
    ELSIF n = ADRSIZE(Header) THEN
      start^ := FillHeader1;
    ELSIF n >= ADRSIZE(Header) + ADRSIZE(INTEGER) THEN
      start^ := FillHeaderN;
      LOOPHOLE(start + ADRSIZE(Header), UNTRACED REF INTEGER)^ := n;
    ELSE
      <* ASSERT FALSE *>
    END;
  END InsertFiller;

PROCEDURE Fill (VAR (*INOUT*) current: AllocRec) =
  BEGIN
    InsertFiller(current.ptr, current.boundary - current.ptr);
    current.page := Nil;
    current.stack := Nil;
    current.ptr := NIL;
    current.boundary := NIL;
    <* ASSERT current.db = NIL *>
  END Fill;

PROCEDURE Close (VAR (*INOUT*) current: AllocRec; gray := FALSE) =
  VAR p := current.page;
  BEGIN
    InsertFiller(current.ptr, current.boundary - current.ptr);
    IF p # Nil THEN
      desc[p - p0].gray := gray;
      IF perfOn THEN PerfChange(p, 1); END;
      IF desc[p - p0].generation = Generation.Older AND NOT desc[p - p0].pure
       THEN
        <* ASSERT desc[p - p0].dirty *>
        <* ASSERT desc[p - p0].space = Space.Current *>
        <* ASSERT map[p - p0] = NIL OR map[p - p0].writer = ThreadF.myTxn *>
        desc[p - p0].dirty := FALSE;
        Protect(p, Access.ReadOnly);
      END;
      current.page := Nil;
    END;
    <* ASSERT current.stack = Nil *>
    current.ptr := NIL;
    current.boundary := NIL;
  END Close;

PROCEDURE Push (VAR (*INOUT*) current: AllocRec; db: RTDB.T) =
  BEGIN
    IF current.db # db THEN
      IF current.page # Nil THEN
        InsertFiller(current.ptr, current.boundary - current.ptr);
        desc[current.page - p0].link := current.stack;
        current.stack := current.page;
        current.page := Nil;
        current.ptr := NIL;
        current.boundary := NIL;
      END;
      current.db := db;
    END;
  END Push;    

PROCEDURE SetDB (db: RTDB.T) =
  BEGIN
    pmover.db := db;
    Push(new, db);
    Push(pureCopy, db);
    Push(impureCopy, db);
  END SetDB;

TYPE CollectorState = {Zero, One, Two, Three, Four, Five};

VAR collectorState := CollectorState.Zero;

VAR
  threshold := ARRAY [0 .. 1] OF
                 REAL{FLOAT(InitialBytes DIV 4 DIV BytesPerPage - 1), 1.0};
  (* start a collection as soon as current space reaches threshold[0] /
     threshold[1] pages; the initial value is 64KB *)

  partialCollection: BOOLEAN;
  (* whether the collection in progress is partial, involving only the newer
     generation *)

  partialCollectionNext: BOOLEAN := FALSE;
  (* whether the next collection should be partial *)

  collectorOn: BOOLEAN := FALSE;

  copyGeneration: Generation := Generation.Younger;

  signalBackground := FALSE;
  (* should signal background collector thread? *)

  signalWeak := FALSE;
  (* should signal weak cleaner thread? *)

TYPE Unswizzler = RTHeapMap.Visitor OBJECT
  page: RTDB.Page;
  data: ADDRESS;
OVERRIDES
  apply := Unswizzle
END;

PROCEDURE Unswizzle (self: Unswizzler; cp: ADDRESS) =
  VAR
    refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
    ref := refref^;
  BEGIN
    IF ref = NIL THEN RETURN END;
    VAR
      p := Word.RightShift(LOOPHOLE(refref, INTEGER), LogBytesPerPage);
      pos := Word.And(LOOPHOLE(refref, INTEGER), BytesPerPage - 1);
    BEGIN
      IF p = self.page.p THEN
        LOOPHOLE(self.data + pos, UNTRACED REF RefReferent)^ :=
            UnswizzleRef(self.page.db, ref);
      END
    END;
  END Unswizzle;

PROCEDURE UnswizzleRef (db: RTDB.T; ref: RefReferent): RefReferent =
  VAR p := ReferentToPage(ref);
  BEGIN
    IF p = Nil OR map[p - p0] = NIL THEN RETURN NIL END;
    VAR
      offset := ref - PageToAddress(p);
      targetDB := map[p - p0].db;
      nbPages := PageCount(p);
      page := map[p - p0];
    BEGIN
      <* ASSERT Word.And(offset, 3) = 0 *>
      <* ASSERT db = targetDB *>
      IF nbPages > 1 THEN
        CASE offset OF
        |   4 => offset := 2_0001;
        |   8 => offset := 2_0011;
        |  16 => offset := 2_0101;
        |  32 => offset := 2_0111;
        |  64 => offset := 2_1001;
        | 128 => offset := 2_1011;
        | 256 => offset := 2_1101;
        | 512 => offset := 2_1111;
        ELSE
          <* ASSERT FALSE *>
        END;
        IF nbPages >= Word.RightShift(BytesPerPage, 4) THEN
          nbPages := 0;
        ELSE
          nbPages := Word.LeftShift(nbPages, 4);
        END;
        INC(offset, nbPages);
      END;
      RETURN PageToAddress(page.id) + offset;
    END
  END UnswizzleRef;

PROCEDURE UnswizzlePage(page: RTDB.Page; VAR data: RTHeapDep.PageData) =
  BEGIN
    ThreadF.SuspendOthers();
  VAR
    p := FirstPage(page.p);
    n := PageCount(p);
    d := ADR(data[0]);
    db := page.db;
    h := PageToHeader(p);
    he := PageToHeader(p+1);
    tc: Typecode;
    pos: INTEGER;
    referentSize: CARDINAL;
    def: TypeDefn;
    fpRef: REF Fingerprint.T;
    fpAdr: ADDRESS;
  BEGIN
    <* ASSERT desc[p - p0].space = Space.Current *>
    IF desc[p - p0].access = Access.None THEN
      <* ASSERT desc[p - p0].gray *>
      Unprotect(p);
    END;
    data := PageToData(page.p)^;

    unswizzler.page := page;
    unswizzler.data := d;
    IF n = 1 THEN
      WHILE h < he DO
        <* ASSERT NOT h.forwarded *>
        p := Word.RightShift(LOOPHOLE(h, INTEGER), LogBytesPerPage);
        pos := Word.And(LOOPHOLE(h, INTEGER), BytesPerPage - 1);
        tc := h.typecode;
        IF tc = Fill_1_type THEN
          LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := LOOPHOLE(0, ADDRESS);
          referentSize := 0;
        ELSIF tc = Fill_N_type THEN
          LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := LOOPHOLE(1, ADDRESS);
          referentSize := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
          DEC(referentSize, ADRSIZE(Header));
        ELSE
          def := RTType.Get(tc);
          <* ASSERT def.traced = 1 *>
          fpRef := NIL;
          fpAdr := NIL;
          db.mapFP(LOOPHOLE(def.fp, Fingerprint.T), fpRef, fpAdr);
          IF fpAdr = NIL THEN
            <* ASSERT fpRef # NIL *>
            fpAdr := UnswizzleRef(db, LOOPHOLE(fpRef, RefReferent));
            db.mapFP(LOOPHOLE(def.fp, Fingerprint.T), fpRef, fpAdr);
          END;
          LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := fpAdr;
          IF def.nDimensions = 0 THEN
            referentSize := def.dataSize;
          ELSE
            referentSize := OpenArraySize(h, def);
          END;
          IF stats.print AND desc[p - p0].note # Note.Persistent THEN
            WITH z = stats.objects[tc] DO z := Word.Plus(z, 1); END;
            WITH z = stats.bytes[tc] DO
              z := Word.Plus(z, BYTESIZE(Header) + referentSize)
            END;
          END;
          IF def.gc_map # NIL OR def.parent # NIL THEN
            RTHeapMap.DoWalkRef(def, h + ADRSIZE(Header), unswizzler);
          END
        END;
        INC(h, ADRSIZE(Header) + referentSize);
      END
    ELSE
      WHILE h < he DO
        <* ASSERT NOT h.forwarded *>
        p := Word.RightShift(LOOPHOLE(h, INTEGER), LogBytesPerPage);
        pos := Word.And(LOOPHOLE(h, INTEGER), BytesPerPage - 1);
        tc := h.typecode;
        IF tc = Fill_1_type THEN
          IF p = page.p THEN
            LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := LOOPHOLE(0, ADDRESS);
          END;
          referentSize := 0;
        ELSIF tc = Fill_N_type THEN
          IF p = page.p THEN
            LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := LOOPHOLE(1, ADDRESS);
          END;
          referentSize :=
              LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
          DEC(referentSize, ADRSIZE(Header));
        ELSE
          def := RTType.Get(tc);
          <* ASSERT def.traced = 1 *>
          IF p = page.p THEN
            fpRef := NIL;
            fpAdr := NIL;
            db.mapFP(LOOPHOLE(def.fp, Fingerprint.T), fpRef, fpAdr);
            IF fpAdr = NIL THEN
              <* ASSERT fpRef # NIL *>
              fpAdr := UnswizzleRef(db, LOOPHOLE(fpRef, RefReferent));
              db.mapFP(LOOPHOLE(def.fp, Fingerprint.T), fpRef, fpAdr);
            END;
            LOOPHOLE(d + pos, UNTRACED REF ADDRESS)^ := fpAdr;
          END;
          IF def.nDimensions = 0 THEN
            referentSize := def.dataSize;
          ELSE
            referentSize := OpenArraySize(h, def);
          END;
          IF stats.print AND desc[p - p0].note # Note.Persistent THEN
            WITH z = stats.objects[tc] DO z := Word.Plus(z, 1); END;
            WITH z = stats.bytes[tc] DO
              z := Word.Plus(z, BYTESIZE(Header) + referentSize)
            END
          END;
          IF def.gc_map # NIL OR def.parent # NIL THEN
            RTHeapMap.DoWalkRef(def, h + ADRSIZE(Header), unswizzler);
          END
        END;
        INC(h, ADRSIZE(Header) + referentSize);
      END
    END;

    p := FirstPage(page.p);
    IF desc[p - p0].gray THEN
      Protect(p, Access.None);
    END
  END;
  ThreadF.ResumeOthers();
END UnswizzlePage;

PROCEDURE Stabilize (bootstrapDB: RTDB.T) =
  BEGIN
    (* compute some costs relative to previous collection *)
    INC(cycleNews, smallNewPages + largeNewPages);
    VAR prefixAvgCost := cycleCost / FLOAT(cycleNews);
    BEGIN
      IF prefixAvgCost < minPrefixAvgCost THEN
        minPrefixAvgCost := prefixAvgCost;
        minCycleL := cycleL;
      END;
    END;

    (* we want to do full collection *)
    copyGeneration := Generation.Younger;
    partialCollection := FALSE;
    partialCollectionNext := TRUE;
    
    (* not partial collection *)
    cycleL := 1;
    cycleCost := 0.0;
    cycleNews := 0;
    minPrefixAvgCost := LAST(REAL);
    minCycleL := 0;

    InvokeMonitors (before := TRUE);

    IF perfOn THEN PerfBegin(); END;

    (* fill the rest of the current page *)
    Fill(new);
    Fill(newTransient);

    INC(collections);

    (* flip spaces; newspace becomes oldspace *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].space = Space.Current THEN
        desc[p - p0].space := Space.Previous;
        IF perfOn THEN PerfChange(p, 1); END;
      END;
    END;

    IF perfOn THEN PerfFlip(); END;

    (* The 'new' nextSpace is empty *)
    smallNewPages := 0;
    largeNewPages := 0;
    smallCopyPages := 0;
    largeCopyPages := 0;
    smallPromotionPages := 0;
    largePromotionPages := 0;

    FOR p := p0 TO p1 - 1 DO
      VAR d := desc[p - p0];
      BEGIN
        IF d.space = Space.Previous AND NOT d.continued THEN
          IF map[p - p0] # NIL THEN
            (* Promote ALL persistent pages: stabilize may trigger faults
               when accessing meta-data, which may swizzle references to
               previously unreachable pages, so we need to make sure these
               pages are retained.  Also, updates to persistent pages during
               closure copying need to be captured when unswizzling. *)
            IF d.resident AND map[p - p0].writer # NIL THEN
              IF d.access # Access.ReadWrite THEN Unprotect(p) END;
              IF d.pure THEN
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration,
                            pure := TRUE, note := Note.Persistent,
                            gray := TRUE, access := Access.ReadWrite,
                            continued := FALSE, resident := TRUE,
                            dirty := FALSE});
                desc[p - p0].link := pureCopy.stack;
                pureCopy.stack := p;
              ELSE
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration,
                            pure := FALSE, note := Note.Persistent,
                            gray := TRUE, access := Access.ReadWrite,
                            continued := FALSE, resident := TRUE,
                            dirty := TRUE});
                desc[p - p0].link := impureCopy.stack;
                impureCopy.stack := p;
              END
            ELSE
              <* ASSERT d.access # Access.ReadWrite *>
              PromotePage(
                  p, Desc{space := Space.Current, generation := copyGeneration,
                          pure := d.pure, note := Note.Persistent,
                          gray := NOT d.pure, access := d.access,
                          continued := FALSE, resident := d.resident,
                          dirty := FALSE});
            END
          ELSE
            (* non-persistent page *)
            <* ASSERT d.access # Access.None *>
            IF d.generation = Generation.Older THEN
              IF d.access # Access.ReadWrite THEN Unprotect(p) END;
            ELSE
              <* ASSERT d.access = Access.ReadWrite *>
            END
          END
        END
      END
    END;

    (* get these in the new space *)
    mover := NEW (Mover);
    unswizzler := NEW (Unswizzler);
    tmover := NEW (TMover);
    pmover := NEW (PMover);

    (* mark from roots *)
    ThreadF.SuspendOthers();
    BEGIN
      (* Examine the stacks for possible pointers *)
      ThreadF.ProcessStacks(NoteStackLocations);

      <* ASSERT impureTransientCopy.page = Nil *>
      <* ASSERT impureTransientCopy.ptr = NIL *>
      <* ASSERT impureTransientCopy.boundary = NIL *>

      <* ASSERT pureTransientCopy.page = Nil *>
      <* ASSERT pureTransientCopy.stack = Nil *>
      <* ASSERT pureTransientCopy.ptr = NIL *>
      <* ASSERT pureTransientCopy.boundary = NIL *>

      <* ASSERT impureCopy.page = Nil *>
      <* ASSERT impureCopy.ptr = NIL *>
      <* ASSERT impureCopy.boundary = NIL *>

      <* ASSERT pureCopy.page = Nil *>
      <* ASSERT pureCopy.ptr = NIL *>
      <* ASSERT pureCopy.boundary = NIL *>

      (* fill current page in preparation for persistent allocations *)
      Fill(new);

      (* Create root if this is a new database *)
      IF bootstrapDB # NIL THEN
        SetDB(bootstrapDB);
        bootstrapDB.createRoot();
        WITH ref = LOOPHOLE(bootstrapDB.root, RefReferent) DO
          bootstrapDB.newId(map[ReferentToPage(ref) - p0]);
        END;
        SetDB(NIL);
      END;

      (* Copy transient closure from globals and user-level page handles,
         since we may need them to access the database when unswizzling *)
      RTHeapMap.WalkGlobals(tmover);
      FOR p := p0 TO p1 - 1 DO
        WITH m = map[p - p0] DO
          IF m # NIL THEN
            TMove (NIL, ADR(m));
          END
        END
      END;
      WHILE CleanSome(impureTransientCopy, CleanTransient, gray:= TRUE) DO END;
      Close(impureTransientCopy, gray := TRUE);
      tmover := NIL;

      (* Copy persistence closure *)
      WHILE CleanSome(impureCopy, CleanPersistent, gray := TRUE) DO END;
      Close(impureCopy, gray := TRUE);

      (* now allocate metadata for pure pages *)
      WHILE CleanSome(pureCopy, CleanPersistent) DO END;
      Close(pureCopy);

      (* now the metadata itself *)
      WHILE CleanSome(new, CleanPersistent) DO END;
      Close(new);

      <* ASSERT impureTransientCopy.stack = Nil *>
      <* ASSERT impureTransientCopy.page = Nil *>
      <* ASSERT impureTransientCopy.ptr = NIL *>
      <* ASSERT impureTransientCopy.boundary = NIL *>

      <* ASSERT impureCopy.stack = Nil *>
      <* ASSERT impureCopy.page = Nil *>
      <* ASSERT impureCopy.ptr = NIL *>
      <* ASSERT impureCopy.boundary = NIL *>

      <* ASSERT pureCopy.page = Nil *>
      <* ASSERT pureCopy.stack = Nil *>
      <* ASSERT pureCopy.ptr = NIL *>
      <* ASSERT pureCopy.boundary = NIL *>

      SetDB(NIL);
      pmover := NIL;

      (* All modified persistent pages are in Space.Current *)
      FOR  i := 0 TO p1 - p0 - 1 DO
        VAR d := desc[i];
        BEGIN
          IF d.space = Space.Current AND NOT d.continued THEN
            IF d.resident THEN
              IF map[i] # NIL THEN
                IF map[i].writer = ThreadF.myTxn THEN
                  <* ASSERT d.access = Access.ReadWrite *>
                  VAR p := p0 + i;
                  BEGIN
                    IF d.note = Note.AmbiguousRoot AND stats.print THEN
                      INC(stats.ambiguousPages, PageCount(p));
                    ELSIF d.note # Note.Persistent AND stats.print THEN
                      INC(stats.accuratePages, PageCount(p));
                    END;
                  END
                END;
                IF d.gray THEN
                  <* ASSERT d.resident AND NOT d.pure *>
                  d.link := impureCopy.stack;
                  desc[i] := d;
                  impureCopy.stack := p0 + i;
                ELSIF ThreadF.myTxn # NIL AND map[i] # NIL THEN
                  IF map[i].lastReader = ThreadF.myTxn THEN
                    Protect(p0 + i, Access.ReadOnly);
                  ELSIF map[i].writer # ThreadF.myTxn THEN
                    Protect(p0 + i, Access.None);
                  END
                END
              ELSIF d.gray THEN
                <* ASSERT d.resident AND NOT d.pure *>
                d.link := impureTransientCopy.stack;
                desc[i] := d;
                impureTransientCopy.stack := p0 + i;
              END
            ELSE
              <* ASSERT d.pure AND d.access = Access.None *>
            END
          END
        END
      END;

      IF stats.print THEN
        PutText("\nAccurate pages:  "); PutInt(stats.accuratePages);
        PutText("\nAmbiguous pages: "); PutInt(stats.ambiguousPages);
        PutText("\n");
        stats.accuratePages := 0; stats.ambiguousPages := 0;
        FOR i := 0 TO RT0u.nTypes - 1 DO
          IF stats.objects[i] # 0 THEN
            VAR t := RTType.Get(i);
                str: ADDRESS := t.name;
            BEGIN
              IF str # NIL THEN
                PutChars(str, Cstring.strlen(str));
              ELSE
                PutText("<tc="); PutInt(i); PutText(">");
              END;
            END;
            PutText(": ");
            PutInt(stats.objects[i]); PutText(" objects ");
            PutInt(stats.bytes[i]); PutText(" bytes\n");
            stats.objects[i] := 0;
            stats.bytes[i] := 0;
          END
        END;
        RTIO.Flush();
      END;

      WHILE CleanSome(impureTransientCopy) DO END;
      Close(impureTransientCopy);

      (* Examine the global variables for possible pointers *)
      RTHeapMap.WalkGlobals (mover);
    END;
    ThreadF.ResumeOthers();

    IF perfOn THEN PerfPromotedRoots(); END;

    collectorState := CollectorState.One;
    IF backgroundWaiting THEN signalBackground := TRUE END;
  END Stabilize;

PROCEDURE Transfer(from: RTTxn.T; to: RTTxn.T) =
  BEGIN
    <* ASSERT to # NIL *>
    <* ASSERT from # to *>
    FOR i := 0 TO p1 - p0 - 1 DO
      VAR d := desc[i];
      BEGIN
        IF NOT d.continued THEN
          IF d.space = Space.Current OR d.space = Space.Previous THEN
            IF d.resident THEN
              IF map[i] # NIL THEN
                IF d.gray THEN
                  <* ASSERT d.access = Access.None *>
                ELSIF map[i].writer = to THEN
                  IF d.generation = Generation.Older THEN
                    <* ASSERT d.space = Space.Current *>
                    IF (d.dirty OR d.pure) AND d.access # Access.ReadWrite
                     THEN
                      Unprotect(p0 + i);
                    ELSIF d.access = Access.None THEN
                      Protect(p0 + i, Access.ReadOnly);
                    END
                  ELSIF d.access # Access.ReadWrite THEN
                    Unprotect(p0 + i);
                  END
                ELSIF d.access = Access.None AND map[i].lastReader = to THEN
                  Protect(p0 + i, Access.ReadOnly);
                ELSIF d.access # Access.None THEN
                  Protect(p0 + i, Access.None);
                END
              END
            ELSE
              <* ASSERT d.pure AND d.access = Access.None AND NOT d.gray *>
            END
          END
        END
      END
    END
  END Transfer;

PROCEDURE Abort() =
  BEGIN
    FOR p := p0 TO p1 - 1 DO
      VAR
        pi := p - p0;
        d := desc[pi];
      BEGIN
        IF NOT d.continued THEN
          IF d.space = Space.Current OR d.space = Space.Previous THEN
            IF d.resident THEN
              IF map[pi] # NIL THEN
                IF map[pi].writer = ThreadF.myTxn THEN
                  d.dirty := FALSE;
                  d.resident := FALSE;
                  d.gray := FALSE;
                  d.pure := TRUE;
                  desc[pi] := d;
                  d.continued := TRUE;
                  FOR i := 1 TO PageCount(p) - 1 DO
                    desc[pi + i] := d;
                  END;
                  Protect(p, Access.None);
                ELSIF map[pi].lastReader = ThreadF.myTxn THEN
                  Protect(p, Access.None);
                END
              END
            ELSE
              <* ASSERT d.pure AND d.access = Access.None AND NOT d.gray *>
            END
          END
        END
      END
    END
  END Abort;

PROCEDURE Flush (db: RTDB.T)
  RAISES { Disabled, Thread.Aborted } =
  VAR page: RTDB.Page;
  BEGIN
    RTOS.LockHeap();
    IF disableCount + disableMotionCount > 0 THEN
      RTOS.UnlockHeap();
      RAISE Disabled;
    END;
    BEGIN
      CollectorOn();
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      Stabilize(db);
      IF NOT (incremental AND RTHeapDep.VM AND disableVMCount = 0) THEN
        REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
      END;
      CollectorOff();
    END;
    RTOS.UnlockHeap();
    DisableMotion();
    (* First pass to assign IDs *)
    FOR p := p0 TO p1 - 1 DO		 (* race here is OK *)
      RTOS.LockHeap();
      page := map[p - p0];
      RTOS.UnlockHeap();
      IF page # NIL AND page.id = Nil THEN page.db.newId(page); END
    END;
    (* Second pass to unswizzle *)
    FOR p := p0 TO p1 - 1 DO		 (* race here is OK *)
      RTOS.LockHeap();
      VAR
        d := desc[p - p0];
      BEGIN
        IF NOT d.continued THEN
          IF d.resident THEN
            page := map[p - p0];
            IF page # NIL AND page.writer = ThreadF.myTxn THEN
              TRY
                RTOS.UnlockHeap();
                Thread.Acquire(page);
                page.write(UnswizzlePage);
              FINALLY
                Thread.Release(page);
                RTOS.LockHeap();
              END;
              FOR i := 1 TO PageCount(p) - 1 DO
                page := map[p - p0 + i];
                TRY
                  RTOS.UnlockHeap();
                  Thread.Acquire(page);
                  page.write(UnswizzlePage);
                FINALLY
                  Thread.Release(page);
                  RTOS.LockHeap();
                END
              END;
              FOR i := 0 TO PageCount(p) - 1 DO
                page := map[p - p0 + i];
                <* ASSERT page.writer = ThreadF.myTxn *>
                page.writer := NIL;
              END;
              IF NOT desc[p - p0].gray THEN
                Protect(p, Access.ReadOnly);
              END
            END
          END
        END
      END;
      RTOS.UnlockHeap();
    END;
    EnableMotion();
  END Flush;

PROCEDURE CollectEnough () =
  BEGIN
    IF collectorOn THEN RETURN; END;
    IF Behind() THEN
      CollectorOn();
      IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
        REPEAT CollectSome(); UNTIL NOT Behind();
      ELSE
        WHILE collectorState = CollectorState.Zero DO CollectSome(); END;
        REPEAT CollectSome(); UNTIL collectorState = CollectorState.Zero;
      END;
      CollectorOff();
    END;
  END CollectEnough;

PROCEDURE Behind (): BOOLEAN =
  BEGIN
    IF disableCount + disableMotionCount > 0
         AND collectorState = CollectorState.Zero THEN
      RETURN FALSE;
    END;
    IF collectorState = CollectorState.Zero THEN
      RETURN FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                     + largePromotionPages + smallNewPages + largeNewPages)
               * threshold[1] >= threshold[0];
    ELSE
      RETURN FLOAT(smallNewPages + largeNewPages) * gcRatio
               >= FLOAT(smallCopyPages + largeCopyPages);
    END;
  END Behind;

VAR timeUsedOnEntry: REAL;       (* time used when entered collector *)

PROCEDURE CollectorOn () =
  BEGIN
    (* ASSERT locked *)
    <* ASSERT NOT collectorOn *>
    collectorOn := TRUE;

    IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
      (* The VM-synchronized collector doesn't worry about running threads. *)
    ELSE
      ThreadF.SuspendOthers ();
    END;

    IF RTHeapDep.VM THEN timeUsedOnEntry := RTHeapDep.TimeUsed(); END;
    IF impureCopy.page # Nil THEN
      <* ASSERT desc[impureCopy.page - p0].gray *>
      <* ASSERT desc[impureCopy.page - p0].access = Access.None *>
      Unprotect(impureCopy.page);
    END;
    IF impureTransientCopy.page # Nil THEN
      <* ASSERT desc[impureTransientCopy.page - p0].gray *>
      <* ASSERT desc[impureTransientCopy.page - p0].access = Access.None *>
      Unprotect(impureTransientCopy.page);
    END;
  END CollectorOn;

PROCEDURE CollectorOff () =
  BEGIN
    (* ASSERT locked *)
    <* ASSERT collectorOn *>
    IF impureCopy.page # Nil THEN
      <* ASSERT desc[impureCopy.page - p0].gray *>
      <* ASSERT desc[impureCopy.page - p0].access = Access.ReadWrite *>
      Protect(impureCopy.page, Access.None);
    END;
    VAR p := impureCopy.stack;
    BEGIN
      WHILE p # Nil DO
        IF desc[p - p0].gray AND desc[p - p0].access # Access.None THEN
          Protect(p, Access.None);
        END;
        p := desc[p - p0].link;
      END;
    END;
    IF impureTransientCopy.page # Nil THEN
      <*ASSERT desc[impureTransientCopy.page - p0].gray *>
      <*ASSERT desc[impureTransientCopy.page - p0].access = Access.ReadWrite *>
      Protect(impureTransientCopy.page, Access.None);
    END;
    VAR p := impureTransientCopy.stack;
    BEGIN
      WHILE p # Nil DO
        IF desc[p - p0].gray AND desc[p - p0].access # Access.None THEN
          Protect(p, Access.None);
        END;
        p := desc[p - p0].link;
      END;
    END;

    IF incremental AND RTHeapDep.VM AND disableVMCount = 0 THEN
      (* The VM-synchronized collector doesn't worry about running threads. *)
    ELSE
      ThreadF.ResumeOthers ();
    END;

    collectorOn := FALSE;
    IF signalBackground OR signalWeak THEN
      signalBackground := FALSE;
      signalWeak := FALSE;
      Broadcast();
    END;
    IF RTHeapDep.VM THEN
      cycleCost := cycleCost + (RTHeapDep.TimeUsed() - timeUsedOnEntry);
    END;
  END CollectorOff;

PROCEDURE CollectSome () =
  BEGIN
    <* ASSERT disableCount = 0 *>
    CASE collectorState OF
    | CollectorState.Zero => CollectSomeInStateZero();
    | CollectorState.One => CollectSomeInStateOne();
    | CollectorState.Two => CollectSomeInStateTwo();
    | CollectorState.Three => CollectSomeInStateThree();
    | CollectorState.Four => CollectSomeInStateFour();
    | CollectorState.Five => CollectSomeInStateFive();
    END;
  END CollectSome;

(* Start a collection *)

VAR
  mover      : Mover    := NIL;
  tmover     : TMover   := NIL;
  pmover     : PMover   := NIL;
  unswizzler : Unswizzler := NIL;
  cycleCost  : REAL     := 0.0;  (* running cost of current cycle *)
  cycleLength: CARDINAL := 1;    (* current planned cycle length *)
  cycleL     : CARDINAL := 0;    (* length of current cycle, so far *)
  cycleNews  : CARDINAL;         (* the number of new pages this cycle *)
  minPrefixAvgCost: REAL;        (* minimum average cost for a prefix of
                                    this cycle *)
  minCycleL: CARDINAL;           (* the length of that prefix *)

PROCEDURE CollectSomeInStateZero () =
  BEGIN
    <* ASSERT disableCount + disableMotionCount = 0 *>
    (* compute some costs relative to previous collection *)
    INC(cycleNews, smallNewPages + largeNewPages);
    VAR prefixAvgCost := cycleCost / FLOAT(cycleNews);
    BEGIN
      IF prefixAvgCost < minPrefixAvgCost THEN
        minPrefixAvgCost := prefixAvgCost;
        minCycleL := cycleL;
      END;
    END;

    (* make generational decisions *)
    IF generational AND RTHeapDep.VM AND disableVMCount = 0 THEN
      copyGeneration := Generation.Older;
      partialCollection := partialCollectionNext AND cycleL < cycleLength;
      IF NOT partialCollection THEN
        IF minCycleL = cycleLength THEN
          cycleLength := cycleLength + 1;
        ELSE
          cycleLength := MAX(cycleLength - 1, 1);
        END;
      END;
    ELSE
      copyGeneration := Generation.Younger;
      partialCollection := FALSE;
    END;
    partialCollectionNext := TRUE;

    IF partialCollection THEN
      INC(cycleL);
    ELSE
      cycleL := 1;
      cycleCost := 0.0;
      cycleNews := 0;
      minPrefixAvgCost := LAST(REAL);
      minCycleL := 0;
    END;

    InvokeMonitors (before := TRUE);

    IF perfOn THEN PerfBegin(); END;

    (* fill the rest of the current page *)
    Fill(new);
    Fill(newTransient);

    INC(collections);

    (* flip spaces; newspace becomes oldspace *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].space = Space.Current THEN
        desc[p - p0].space := Space.Previous;
        IF perfOn THEN PerfChange(p, 1); END;
      END;
    END;

    IF perfOn THEN PerfFlip(); END;

    (* The 'new' nextSpace is empty *)
    smallNewPages := 0;
    largeNewPages := 0;
    smallCopyPages := 0;
    largeCopyPages := 0;
    smallPromotionPages := 0;
    largePromotionPages := 0;

    FOR p := p0 TO p1 - 1 DO
      VAR d := desc[p - p0];
      BEGIN
        IF d.space = Space.Previous AND NOT d.continued THEN
          IF map[p - p0] # NIL THEN
            (* persistent page *)
            IF d.resident AND map[p - p0].writer # NIL THEN
              (* promote modified persistent pages to avoid deleting them *)
              IF d.pure OR partialCollection AND NOT d.dirty THEN
                <* ASSERT NOT d.dirty *>
                PromotePage(
                  p, Desc{space := Space.Current,
                          generation := copyGeneration, pure := d.pure,
                          note := Note.Persistent, gray := FALSE,
                          access := d.access, continued := FALSE,
                          resident := TRUE, dirty := FALSE});
              ELSE (* impure, neither partial collection nor clean *)
                IF d.access # Access.ReadWrite THEN Unprotect(p) END;
                PromotePage(
                    p, Desc{space := Space.Current,
                            generation := copyGeneration, pure := FALSE,
                            note := Note.Persistent, gray := TRUE,
                            access := Access.ReadWrite, continued := FALSE,
                            resident := TRUE, dirty := TRUE});
                desc[p - p0].link := impureCopy.stack;
                impureCopy.stack := p;
              END
            ELSIF d.generation = Generation.Older THEN
              IF partialCollection THEN
                <* ASSERT copyGeneration = Generation.Older *>
                IF d.pure THEN
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration, pure := TRUE,
                              note := Note.OlderGeneration, gray := FALSE,
                              access := d.access, continued := FALSE,
                              resident := d.resident, dirty := FALSE});
                ELSIF d.dirty THEN
                  <* ASSERT d.resident *>
                  IF d.access # Access.ReadWrite THEN Unprotect(p) END;
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration, pure := FALSE,
                              note := Note.OlderGeneration, gray := TRUE,
                              access := Access.ReadWrite, continued := FALSE,
                              resident := TRUE, dirty := TRUE});
                  desc[p - p0].link := impureCopy.stack;
                  impureCopy.stack := p;
                ELSE
                  <* ASSERT d.access # Access.ReadWrite *>
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration, pure := FALSE,
                              note := Note.OlderGeneration, gray := FALSE,
                              access := d.access, continued := FALSE,
                              resident := d.resident, dirty := FALSE});
                END
              END
            END
          ELSE
            (* non-persistent page *)
            IF d.generation = Generation.Older THEN
              IF partialCollection THEN
                <* ASSERT copyGeneration = Generation.Older *>
                IF d.pure THEN
                  <* ASSERT d.access = Access.ReadWrite *>
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration,
                              pure := TRUE, note := Note.OlderGeneration,
                              gray := FALSE, access := Access.ReadWrite,
                              continued := FALSE, resident := TRUE,
                              dirty := FALSE});
                ELSIF d.dirty THEN
                  <* ASSERT d.access = Access.ReadWrite *>
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration,
                              pure := FALSE, note := Note.OlderGeneration,
                              gray := TRUE, access := Access.ReadWrite,
                              continued := FALSE, resident := TRUE,
                              dirty := TRUE});
                  desc[p - p0].link := impureCopy.stack;
                  impureCopy.stack := p;
                ELSE
                  <* ASSERT d.access = Access.ReadOnly *>
                  PromotePage(
                      p, Desc{space := Space.Current,
                              generation := copyGeneration,
                              pure := FALSE, note := Note.OlderGeneration,
                              gray := FALSE, access := Access.ReadOnly,
                              continued := FALSE, resident := TRUE,
                              dirty := FALSE});
                END
              ELSE
                IF d.access # Access.ReadWrite THEN Unprotect(p) END;
              END
            ELSE
              <* ASSERT d.access = Access.ReadWrite *>
            END
          END
        END
      END
    END;
    (* now nothing in the previous space is protected or in the older
       generation *)

    (* get these in the new space *)
    mover := NEW (Mover);
    tmover := NEW(TMover);

    (* mark from roots *)
    ThreadF.SuspendOthers();
    BEGIN
      (* Examine the stacks for possible pointers *)
      ThreadF.ProcessStacks(NoteStackLocations);

      (* Copy closure from user-level page handles,
         since we need page information during GC, but drop roots *)
      FOR p := p0 TO p1 - 1 DO
        WITH m = map[p - p0] DO
          IF m # NIL THEN
            m.db.root := NIL;
            Move (NIL, ADR(m));
          END
        END
      END;
      WHILE CopySome() DO END;

      (* Examine the global variables for possible pointers *)
      RTHeapMap.WalkGlobals (mover);
    END;
    ThreadF.ResumeOthers();

    IF perfOn THEN PerfPromotedRoots(); END;

    collectorState := CollectorState.One;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateZero;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateOne () =
  BEGIN
    IF NOT CopySome() THEN collectorState := CollectorState.Two; END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateOne;

(* Walk weakly-referenced nodes to determine order in which to do cleanup,
   then cleanup gray nodes.  This should be broken down into parts, since
   it may be a lengthy operation. *)

PROCEDURE CollectSomeInStateTwo () =
  BEGIN
    PreHandleWeakRefs();
    collectorState := CollectorState.Three;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateTwo;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateThree () =
  BEGIN
    (* recursively copy all objects reachable from promoted objects.  marks
       "marka" and "markb" are cleared when objects move to the new
       space. *)
    IF NOT CopySome() THEN
      (* That's the last chance for unmodified persistent pages since
         they may contain WRNNC objects that are about to get finalized.
         Thus, we now unmap them so swizzling can't resurrect them. *)
      ThreadF.SuspendOthers();
      BEGIN
        FOR i := 0 TO p1 - p0 - 1 DO
          VAR
            d := desc[i];
            page := map[i];
          BEGIN
            IF d.space = Space.Previous AND page # NIL THEN
              IF NOT d.continued THEN
                <* ASSERT d.access # Access.ReadWrite *>
                <* ASSERT NOT d.resident OR map[i].writer = NIL *>
                Unprotect(p0 + i);
              END;
              (* unmap pages *)
              page.db.unmapPage(page.id);
              page.p := Nil;
              map[i] := NIL;
              IF perfOn THEN PerfChange(p0 + i, 1); END;
            END
          END
        END
      END;
      ThreadF.ResumeOthers();

      PostHandleWeakRefs();      (* must be called with no gray objects *)
      signalWeak := TRUE;
      collectorState := CollectorState.Four;
    END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateThree;

(* Clean gray nodes *)

PROCEDURE CollectSomeInStateFour () =
  BEGIN
    IF NOT CopySome() THEN collectorState := CollectorState.Five; END;
    IF backgroundWaiting THEN signalBackground := TRUE; END;
  END CollectSomeInStateFour;

PROCEDURE CollectSomeInStateFive () =
  BEGIN
    (* free all oldspace pages; oldspace becomes freespace *)
    FOR i := 0 TO p1 - p0 - 1 DO
      IF desc[i].space = Space.Previous THEN
        desc[i].space := Space.Free;
        desc[i].continued := FALSE;
        <* ASSERT desc[i].access = Access.ReadWrite *>
        IF perfOn THEN PerfChange(p0 + i, 1); END;
      END;
    END;

    RebuildFreelist();

    (* fill the rest of the current copy pages *)
    Close(impureCopy);
    Close(impureTransientCopy);
    Fill(pureCopy);
    Fill(pureTransientCopy);

    IF perfOn THEN PerfEnd(); END;

    InvokeMonitors(before := FALSE);

    IF partialCollection THEN
      IF FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                 + largePromotionPages) * threshold[1] >= threshold[0] THEN
        partialCollectionNext := FALSE;
      ELSE
        partialCollectionNext := TRUE;
      END;
    ELSE
      threshold[0] :=
        FLOAT(smallCopyPages + largeCopyPages + smallPromotionPages
                + largePromotionPages) * (gcRatio + 1.0);
      threshold[1] := gcRatio;
      partialCollectionNext := TRUE;
    END;

    collectorState := CollectorState.Zero;
  END CollectSomeInStateFive;

(* CopySome attempts to make progress toward cleaning the new space.  It
   returns FALSE iff there was no more work to do.

   It operates by cleaning the current copy page.  It may also clean some
   number of pages on the stack.  When it returns, there is a new copy
   page. *)

PROCEDURE CopySome (): BOOLEAN =
  VAR
    originalPage := impureCopy.page;
    originalBoundary := impureCopy.boundary;
    cleanTo := PageToHeader(impureCopy.page);
    originalTransientPage := impureTransientCopy.page;
    originalTransientBoundary := impureTransientCopy.boundary;
    cleanTransientTo := PageToHeader(impureTransientCopy.page);
    p: Page;
  BEGIN
    LOOP
      IF cleanTo < impureCopy.ptr THEN
        VAR ptr := impureCopy.ptr;
        BEGIN
          CleanBetween(cleanTo, ptr);
          cleanTo := ptr;
        END;
      ELSIF cleanTransientTo < impureTransientCopy.ptr THEN
        VAR ptr := impureTransientCopy.ptr;
        BEGIN
          CleanBetween(cleanTransientTo, ptr);
          cleanTransientTo := ptr;
        END;
      ELSE
        IF impureCopy.stack = Nil AND impureTransientCopy.stack = Nil THEN
          RETURN FALSE;
        END;

        p := impureCopy.stack;
        IF p # Nil THEN
          impureCopy.stack := desc[p - p0].link;
          IF desc[p - p0].gray THEN
            <* ASSERT desc[p - p0].resident *>
            IF desc[p - p0].access # Access.ReadWrite THEN Unprotect(p); END;
            CleanBetween(PageToHeader(p), PageToHeader(p + 1));
            FOR i := 0 TO PageCount(p) - 1 DO
              desc[p + i - p0].gray := FALSE;
              desc[p + i - p0].dirty := FALSE;
            END;
            IF perfOn THEN PerfChange(p, PageCount(p)) END;
            IF ThreadF.myTxn = NIL
              OR map[p - p0] = NIL
              OR map[p - p0].writer = ThreadF.myTxn
             THEN
              IF desc[p - p0].generation = Generation.Older THEN
                <* ASSERT desc[p - p0].space = Space.Current *>
                Protect(p, Access.ReadOnly);
              END
            ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
              Protect(p, Access.ReadOnly);
            ELSE
              Protect(p, Access.None);
            END;
          END;
        END;

        p := impureTransientCopy.stack;
        IF p # Nil THEN
          impureTransientCopy.stack := desc[p - p0].link;
          IF desc[p - p0].gray THEN
            <* ASSERT desc[p - p0].resident *>
            IF desc[p - p0].access # Access.ReadWrite THEN Unprotect(p); END;
            CleanBetween(PageToHeader(p), PageToHeader(p + 1));
            FOR i := 0 TO PageCount(p) - 1 DO
              desc[p + i - p0].gray := FALSE;
              desc[p + i - p0].dirty := FALSE;
            END;
            IF perfOn THEN PerfChange(p, PageCount(p)) END;
            IF ThreadF.myTxn = NIL
              OR map[p - p0] = NIL
              OR map[p - p0].writer = ThreadF.myTxn
             THEN
              IF desc[p - p0].generation = Generation.Older THEN
                <* ASSERT desc[p - p0].space = Space.Current *>
                Protect(p, Access.ReadOnly);
              END
            ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
              Protect(p, Access.ReadOnly);
            ELSE
              Protect(p, Access.None);
            END
          END
        END;

      END;
      IF impureCopy.page # originalPage
        OR impureTransientCopy.page # originalTransientPage THEN EXIT; END;
    END;

    p := originalPage;
    IF p # Nil AND p # impureCopy.page AND desc[p - p0].gray THEN
      (* originalPage is now in the stack; mark it not gray *)
      CleanBetween(cleanTo, originalBoundary);
      desc[p - p0].gray := FALSE;
      desc[p - p0].dirty := FALSE;
      IF perfOn THEN PerfChange(p, 1) END;
      IF ThreadF.myTxn = NIL
        OR map[p - p0] = NIL
        OR map[p - p0].writer = ThreadF.myTxn
       THEN
        IF desc[p - p0].generation = Generation.Older THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          Protect(p, Access.ReadOnly);
        END
      ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
        Protect(p, Access.ReadOnly);
      ELSE
        Protect(p, Access.None);
      END;

      p := originalTransientPage;
      IF p # Nil AND p # impureTransientCopy.page AND desc[p - p0].gray THEN
        (* originalTransientPage is now in the stack; mark it not gray *)
        CleanBetween(cleanTransientTo, originalTransientBoundary);
        desc[p - p0].gray := FALSE;
        desc[p - p0].dirty := FALSE;
        IF perfOn THEN PerfChange(p, 1) END;
        IF ThreadF.myTxn = NIL
          OR map[p - p0] = NIL
          OR map[p - p0].writer = ThreadF.myTxn
         THEN
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, Access.ReadOnly);
          END
        ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
          Protect(p, Access.ReadOnly);
        ELSE
          Protect(p, Access.None);
        END
      END;
      RETURN TRUE;
    END;

    p := originalTransientPage;
    IF p # Nil AND p # impureTransientCopy.page AND desc[p - p0].gray THEN
      (* originalTransientPage is now in the stack; mark it not gray *)
      CleanBetween(cleanTransientTo, originalTransientBoundary);
      desc[p - p0].gray := FALSE;
      desc[p - p0].dirty := FALSE;
      IF perfOn THEN PerfChange(p, 1) END;
      IF ThreadF.myTxn = NIL
        OR map[p - p0] = NIL
        OR map[p - p0].writer = ThreadF.myTxn
       THEN
        IF desc[p - p0].generation = Generation.Older THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          Protect(p, Access.ReadOnly);
        END
      ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
        Protect(p, Access.ReadOnly);
      ELSE
        Protect(p, Access.None);
      END;

      p := originalPage;
      IF p # Nil AND p # impureCopy.page AND desc[p - p0].gray THEN
        (* originalPage is now in the stack; mark it not gray *)
        CleanBetween(cleanTo, originalBoundary);
        desc[p - p0].gray := FALSE;
        desc[p - p0].dirty := FALSE;
        IF perfOn THEN PerfChange(p, 1) END;
        IF ThreadF.myTxn = NIL
          OR map[p - p0] = NIL
          OR map[p - p0].writer = ThreadF.myTxn
         THEN
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, Access.ReadOnly);
          END
        ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
          Protect(p, Access.ReadOnly);
        ELSE
          Protect(p, Access.None);
        END
      END;
      RETURN TRUE;
    END;

    RETURN TRUE;
  END CopySome;

PROCEDURE CleanSome (VAR current: AllocRec; cleaner := CleanBetween;
                     gray := FALSE): BOOLEAN =
  VAR
    originalPage := current.page;
    originalBoundary := current.boundary;
    cleanTo := PageToHeader(current.page);
    p: Page;
  BEGIN
    LOOP
      IF cleanTo < current.ptr THEN
        VAR ptr := current.ptr;
        BEGIN
          cleaner(cleanTo, ptr);
          cleanTo := ptr;
        END;
      ELSE
        p := current.stack;
        IF p = Nil THEN RETURN FALSE; END;
        current.stack := desc[p - p0].link;
        <* ASSERT desc[p - p0].gray *>
        <* ASSERT desc[p - p0].resident *>
        <* ASSERT desc[p - p0].access = Access.ReadWrite *>
        cleaner(PageToHeader(p), PageToHeader(p + 1));
        FOR i := 0 TO PageCount(p) - 1 DO
          desc[p + i - p0].gray := gray;
          desc[p + i - p0].dirty := FALSE;
        END;
        IF perfOn THEN PerfChange(p, PageCount(p)) END;
      END;
      IF current.page # originalPage THEN EXIT; END;
    END;
    cleaner(cleanTo, originalBoundary);

    (* originalPage is now in the stack; mark it not gray *)
    p := originalPage;
    IF p # Nil THEN
      desc[p - p0].gray := gray;
      desc[p - p0].dirty := FALSE;
      IF perfOn THEN PerfChange(p, 1) END;
    END;
    RETURN TRUE;
  END CleanSome;

PROCEDURE CleanBetween (h, he: RefHeader) =
  BEGIN
    WHILE h < he DO
      <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
      <* ASSERT NOT h.forwarded *>
      h.marka := FALSE;
      h.markb := FALSE;
      RTHeapMap.WalkRef (h, mover);
      INC(h, ADRSIZE(Header) + ReferentSize(h));
    END;
  END CleanBetween;

PROCEDURE CleanTransient (h, he: RefHeader) =
  BEGIN
    WHILE h < he DO
      <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
      <* ASSERT NOT h.forwarded *>
      h.marka := FALSE;
      h.markb := FALSE;
      RTHeapMap.WalkRef (h, tmover);
      INC(h, ADRSIZE(Header) + ReferentSize(h));
    END;
  END CleanTransient; 

PROCEDURE CleanPersistent (h, he: RefHeader) =
  VAR
    p: Page;
    pi: INTEGER;
    mapDB: RTDB.T;
    tc: Typecode;
    def: TypeDefn;
    pure: BOOLEAN;
    fpRef: REF Fingerprint.T;
    fpAdr: ADDRESS;
    page: RTDB.Page;
  BEGIN
    IF h < he THEN
      p := HeaderToPage(h);
      <* ASSERT p # Nil *>
      pi := p - p0;
      page := map[pi];
      <* ASSERT page # NIL *>
      IF page.writer # ThreadF.myTxn THEN RETURN END;
      pure := desc[pi].pure;
      mapDB := page.db;
      <* ASSERT mapDB # NIL *>
      SetDB(mapDB);

      REPEAT
        <* ASSERT Word.And (LOOPHOLE (h, INTEGER), 3) = 0 *>
        <* ASSERT NOT h.forwarded *>
        h.marka := FALSE;
        h.markb := FALSE;
        tc := h.typecode;
        IF tc = Fill_1_type THEN
          INC(h, ADRSIZE(Header));
        ELSIF tc = Fill_N_type THEN
          INC(h, LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^);
        ELSE
          def := RTType.Get(tc);
          <* ASSERT def.traced = 1 *>
          fpRef := NIL;
          fpAdr := NIL;
          mapDB.mapFP(LOOPHOLE(def.fp, Fingerprint.T), fpRef, fpAdr);
          IF NOT pure THEN
            <* FATAL ANY *>
            BEGIN
              RTTypeMap.DoWalkRef(def, h + ADRSIZE(Header),
                                  RTTypeMap.Mask{RTTypeMap.Kind.Ref}, pmover);
            END;
          END;
          IF def.nDimensions = 0 THEN
            INC(h, ADRSIZE(Header) + def.dataSize);
          ELSE
            INC(h, ADRSIZE(Header) + OpenArraySize(h, def));
          END;
        END;
      UNTIL NOT h < he;
    END;
  END CleanPersistent;

(* We maintain a list in weakTable, starting at weakLive0, of weak refs and
   the objects they reference.  This table is not considered a root.  When
   HandleWeakRefs is entered, any object mentioned in that list is a
   candidate for cleanup.

   First, we determine which weakly-referenced objects with non-NIL
   cleanups ("WRNNC objects") are reachable from other WRNNC objects, by
   walking the old space.  All such WRNNC objects are copied to new space,
   and all the objects they reference.

   All the weakly-referenced objects left in the old space can then be
   scheduled for cleanup; we move them from the list starting at weakLive0
   to the list starting at weakDead0 in weakTable.  A separate thread runs
   WeakCleaner, which does the calls to the procedures.

   Note that the refs in weakTable must be updated to point to new
   space. *)

(* PreHandleWeakRefs walks the weakly-references structures in old-space,
   deciding on a cleanup order. *)

PROCEDURE PreHandleWeakRefs () =
  VAR s: Stacker;
  BEGIN
    (* get ready to allocate on a new page (take this out!) *)
    InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
    InsertFiller(pureCopy.ptr, pureCopy.boundary - pureCopy.ptr);
    InsertFiller(impureTransientCopy.ptr,
                 impureTransientCopy.boundary - impureTransientCopy.ptr);
    InsertFiller(pureTransientCopy.ptr,
                 pureTransientCopy.boundary - pureTransientCopy.ptr);
    (* allocate a stack on the side for walking the old space *)
    s := InitStack();
    (* iterate over the weak refs to walk the old space *)
    VAR i := weakLive0;
    BEGIN
      WHILE i # -1 DO
        (* here, all old-space WRNNC objects that have already been scanned
           have marka set, as do all old-space objects reachable from them;
           all old-space WRNNC objects that were reachable from other
           already-scanned WRNNC objects have been promoted to the new
           space. *)
        WITH entry = weakTable[i] DO
          IF entry.p # NIL AND NOT Moved(entry.r) THEN
            (* we haven't seen this WRNNC object before *)
            VAR header := HeaderOf(LOOPHOLE(entry.r, ADDRESS));
            BEGIN
              IF NOT header.marka THEN
                <* ASSERT NOT header.markb *>
                (* visit all old-space objects reachable from it; promote
                   all other old-space WRNNC objects reachable from it;
                   promote all old-space objects reachable from it that
                   have "marka" set.  mark all visited nodes with
                   "markb". *)
                WeakWalk1(s, entry.r);
                <* ASSERT NOT header.marka *>
                <* ASSERT header.markb *>
                (* then change all "markb" to "marka" *)
                WeakWalk2(s, entry.r);
                <* ASSERT header.marka *>
                <* ASSERT NOT header.markb *>
              END;
            END;
          END;
          i := entry.next;
        END;
      END;
    END;
  END PreHandleWeakRefs;

(* WeakWalk1 starts at a WRNNC object and visits all objects in old space
   reachable from it, using "markb" to keep from visiting them more than
   once.  All other WRNNC objects visited are promoted, as are all objects
   already visited from other WRNNC objects. *)

PROCEDURE WeakWalk1 (s: Stacker; ref: RefReferent) =
  VAR ref0 := ref;
  BEGIN
    <* ASSERT s.empty() *>
    LOOP
      IF NOT Moved(ref) THEN
        VAR header := HeaderOf(ref);
        BEGIN
          IF header.marka THEN
            <* ASSERT NOT header.markb *>
            Move(NIL, ADR(ref));
          ELSIF NOT header.markb THEN
            IF header.weak AND ref # ref0 THEN
              Move(NIL, ADR(ref));
            ELSE
              header.markb := TRUE;
              RTHeapMap.WalkRef (header, s);
            END;
          END;
        END;
      END;
      IF s.empty() THEN EXIT; END;
      ref := s.pop();
    END;
  END WeakWalk1;

(* WeakWalk2 starts at a WRNNC objects and visits all objects in the old
   space that are reachable from it, changing "markb" to "marka" *)

PROCEDURE WeakWalk2 (s: Stacker;  ref: RefReferent) =
  BEGIN
    <* ASSERT s.empty() *>
    LOOP
      IF NOT Moved(ref) THEN
        VAR header := HeaderOf(ref);
        BEGIN
          IF header.markb THEN
            header.markb := FALSE;
            header.marka := TRUE;
            RTHeapMap.WalkRef (header, s);
          END;
        END;
      END;
      IF s.empty() THEN EXIT; END;
      ref := s.pop();
    END;
  END WeakWalk2;

PROCEDURE PostHandleWeakRefs () =
  BEGIN
    (* move to a new page (take this out!) *)
    InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
    InsertFiller(pureCopy.ptr, pureCopy.boundary - pureCopy.ptr);
    InsertFiller(impureTransientCopy.ptr,
                 impureTransientCopy.boundary - impureTransientCopy.ptr);
    InsertFiller(pureTransientCopy.ptr,
                 pureTransientCopy.boundary - pureTransientCopy.ptr);
    (* iterate over all weak refs.  if the object hasn't been promoted,
       schedule a cleanup *)
    VAR
      i        := weakLive0;
      previous := -1;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          IF Moved(entry.r) THEN
            (* no cleanup this time; note new address *)
            Move(NIL, ADR(entry.r));
            previous := i;
            i := entry.next;
          ELSE
            (* the weak ref is dead; there are no cleanups *)
            VAR header := HeaderOf(LOOPHOLE(entry.r, ADDRESS));
            BEGIN
              header.weak := FALSE;
            END;
            (* move the entry from the weakLive0 list into the weakDead0 or
               weakFree0 list *)
            VAR next := entry.next;
            BEGIN
              IF previous = -1 THEN
                weakLive0 := next;
              ELSE
                weakTable[previous].next := next;
              END;
              entry.t.a := -1;   (* keep ToRef from succeeding *)
              IF entry.p # NIL THEN
                entry.next := weakDead0;
                weakDead0 := i;
              ELSE
                entry.next := weakFree0;
                weakFree0 := i;
              END;
              i := next;
            END;
          END;
        END;
      END;
    END;
    (* for all entries on the weakDead0 list, including those just placed
       there, note the new address *)
    VAR i := weakDead0;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          <* ASSERT entry.t.a = -1 *>
          Move(NIL, ADR(entry.r));
          i := entry.next;
        END;
      END;
    END;
    (* finally, check for objects with final cleanup enabled *)
    VAR
      i        := weakFinal0;
      previous := -1;
    BEGIN
      WHILE i # -1 DO
        WITH entry = weakTable[i] DO
          IF Moved(entry.r) THEN
            (* no cleanup this time; note new address *)
            Move(NIL, ADR(entry.r));
            previous := i;
            i := entry.next;
          ELSE
            (* call the cleanup procedure *)
            LOOPHOLE(entry.p, PROCEDURE (p: REFANY))(
              LOOPHOLE(entry.r, REFANY));
            (* take the entry off the weakFinal0 list and put it on the
               weakFree0 list; on to the next entry *)
            VAR next := entry.next;
            BEGIN
              IF previous = -1 THEN
                weakFinal0 := next;
              ELSE
                weakTable[previous].next := next;
              END;
              entry.next := weakFree0;
              weakFree0 := i;
              i := next;
            END;
          END;
        END;
      END;
    END;
  END PostHandleWeakRefs;

(* The stack for walking the old space is maintained on the heap in the new
   space. *)

TYPE
  Stacker = RTHeapMap.Visitor OBJECT
    data : <*TRANSIENT*> REF ARRAY OF RefReferent;
    x0   : UNTRACED REF RefReferent;
    x1   : UNTRACED REF RefReferent;
    xA   : UNTRACED REF RefReferent;
    xN   : CARDINAL;
  METHODS
    empty (): BOOLEAN     := StackEmpty;
    pop   (): RefReferent := PopStack;
  OVERRIDES
    apply := PushStack;
  END;

(* InitStack allocates an initial stack of 100 elements. *)

PROCEDURE InitStack (): Stacker =
  VAR s := NEW (Stacker);
  BEGIN
    s.data := NEW(<*TRANSIENT*> REF ARRAY OF RefReferent, 100);
    s.xN   := NUMBER (s.data^);
    s.x0   := ADR(s.data[0]);
    s.x1   := s.x0 + s.xN * ADRSIZE(RefReferent);
    s.xA   := s.x0;
    RETURN s;
  END InitStack;

(* PushStack pushes an object onto the stack, growing it if necessary. *)

PROCEDURE PushStack (s: Stacker;  cp: ADDRESS) =
  VAR ref: RefReferent := LOOPHOLE(cp, UNTRACED REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      IF s.xA = s.x1 THEN ExpandStack (s); END;
      s.xA^ := ref;
      INC(s.xA, ADRSIZE(RefReferent));
    END;
  END PushStack;

PROCEDURE ExpandStack (s: Stacker) =
  VAR
    newStackN := 2 * s.xN;
    newStack := NEW(<*TRANSIENT*> REF ARRAY OF RefReferent, newStackN);
  BEGIN
    SUBARRAY(newStack^, 0, s.xN) := SUBARRAY(s.data^, 0, s.xN);
    s.x0   := ADR(newStack^[0]);
    s.xA   := s.x0 + s.xN * ADRSIZE(RefReferent);
    s.x1   := s.x0 + newStackN * ADRSIZE(RefReferent);
    s.data := newStack;
    s.xN   := newStackN;
  END ExpandStack;

(* PopStack pops an object off the stack. *)

PROCEDURE PopStack (s: Stacker): RefReferent =
  BEGIN
    DEC(s.xA, ADRSIZE(RefReferent));
    RETURN s.xA^;
  END PopStack;

(* StackEmpty tells if the stack is empty. *)

PROCEDURE StackEmpty (s: Stacker): BOOLEAN =
  BEGIN
    RETURN s.xA = s.x0;
  END StackEmpty;

(* Malloc returns the address of "size" bytes of untraced, zeroed storage *)

PROCEDURE Malloc (size: INTEGER): ADDRESS =
  VAR res: ADDRESS;
  BEGIN
    RTOS.LockHeap();
    BEGIN
      res := Cstdlib.malloc(size);
      IF res = NIL THEN
        RTMisc.FatalError(NIL, 0, "malloc failed, unable to get more memory");
      END;
    END;
    RTOS.UnlockHeap();
    RTMisc.Zero(res, size);
    RETURN res;
  END Malloc;

(* AllocForNew allocates space for a NEW. *)

PROCEDURE AllocForNew (dataSize, dataAlignment: CARDINAL): RefReferent =
  VAR
    alignment : INTEGER;
    nextPtr   : RefHeader;
  BEGIN
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       new.ptr? *)
    VAR referentTry := new.ptr + ADRSIZE(Header);
    BEGIN
      (* ---------------- see CheckTypes ---------------------------------
|      WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|        alignment := a - referentTry;
|        nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
         ------------------------------------------------------------------ *)
      alignment :=
        align[Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask),
              dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;
    (* If this is not ok, take the long route *)
    IF nextPtr > new.boundary THEN
      nextPtr := NIL;         (* clear in case of GC *)
      VAR
        res := LongAlloc(dataSize, dataAlignment, new,
                         allocMode := AllocMode.New, pure := FALSE);
      BEGIN
        RTOS.UnlockHeap();
        RETURN res;
      END;
    END;
    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(new.ptr, alignment);
      new.ptr := new.ptr + alignment;
    END;
    VAR res := LOOPHOLE(new.ptr + ADRSIZE(Header), RefReferent);
    BEGIN
      new.ptr := nextPtr;
      RTOS.UnlockHeap();
      RETURN res;
    END;
  END AllocForNew;

(* AllocForNewTransient allocates space for a NEW. *)

PROCEDURE AllocForNewTransient (dataSize, dataAlignment: CARDINAL)
  : RefReferent =
  VAR
    alignment : INTEGER;
    nextPtr   : RefHeader;
  BEGIN
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       newTransient.ptr? *)
    VAR referentTry := newTransient.ptr + ADRSIZE(Header);
    BEGIN
      (* ---------------- see CheckTypes ---------------------------------
|      WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|        alignment := a - referentTry;
|        nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
         ------------------------------------------------------------------ *)
      alignment :=
        align[Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask),
              dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;
    (* If this is not ok, take the long route *)
    IF nextPtr > newTransient.boundary THEN
      nextPtr := NIL;         (* clear in case of GC *)
      VAR
        res := LongAlloc(dataSize, dataAlignment, newTransient,
                         allocMode := AllocMode.New, pure := FALSE);
      BEGIN
        RTOS.UnlockHeap();
        RETURN res;
      END;
    END;
    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(newTransient.ptr, alignment);
      newTransient.ptr := newTransient.ptr + alignment;
    END;
    VAR res := LOOPHOLE(newTransient.ptr + ADRSIZE(Header), RefReferent);
    BEGIN
      newTransient.ptr := nextPtr;
      RTOS.UnlockHeap();
      RETURN res;
    END;
  END AllocForNewTransient;

(* AllocForCopy allocates space to copy an object from oldspace; it has the
   same logic as AllocForNew. *)

PROCEDURE AllocForCopy (dataSize, dataAlignment: CARDINAL;
                        VAR current: AllocRec;
                        pure: BOOLEAN) : RefReferent =
  VAR
    alignment       : INTEGER;
    nextPtr         : RefHeader;
    res, referentTry: RefReferent;
  BEGIN
    <* ASSERT collectorOn *>
    RTOS.LockHeap();
    (* Where would this heap object end if we were to allocate at
       current.ptr? *)
    referentTry := current.ptr + ADRSIZE(Header);

    (* ---------------- see CheckTypes ---------------------------------
|    WITH a = RTMisc.Align (referentTry, dataAlignment) DO
|      alignment := a - referentTry;
|      nextPtr := LOOPHOLE (a + dataSize, RefHeader); END;
       ------------------------------------------------------------------ *)
    VAR tmp := Word.And(LOOPHOLE(referentTry, INTEGER), MaxAlignMask);
    BEGIN
      alignment := align[tmp, dataAlignment];
      nextPtr := referentTry + (alignment + dataSize);
    END;

    (* If this is not ok, take the long route *)
    IF nextPtr > current.boundary THEN
      res := LongAlloc(dataSize, dataAlignment, current,
                       allocMode := AllocMode.Copy, pure := pure);
      RTOS.UnlockHeap();
      RETURN res;
    END;

    (* Align the referent *)
    IF alignment # 0 THEN
      InsertFiller(current.ptr, alignment);
      current.ptr := current.ptr + alignment;
    END;

    res := LOOPHOLE(current.ptr + ADRSIZE(Header), RefReferent);
    current.ptr := nextPtr;
    RTOS.UnlockHeap();
    RETURN res;
  END AllocForCopy;

TYPE AllocMode = {New, Copy};

PROCEDURE LongAlloc (              dataSize, dataAlignment : CARDINAL;
                     VAR (*INOUT*) current                 : AllocRec;
                                   allocMode               : AllocMode;
                                   pure                    : BOOLEAN)
  : RefReferent =
  VAR
    nbBytes := RTMisc.Upper(ADRSIZE(Header), dataAlignment) + dataSize;
    nbPages := (nbBytes + AdrPerPage - 1) DIV AdrPerPage;
    res     : RefReferent;
    notAfter: SET OF Note;
  BEGIN
    IF allocMode = AllocMode.New THEN
      CollectEnough();
      notAfter := SET OF Note{Note.Copied};
    ELSE
      notAfter := SET OF Note{Note.Allocated};
    END;

    VAR
      thisPage := FindFreePages(nbPages, notAfter := notAfter);
      (* thisPage points to a block of nbPages contiguous, free pages; just
         what we need! *)
      thisPtr                  := PageToHeader(thisPage);
      thisBoundary             := PageToHeader(thisPage + 1);
      persistent := current.db # NIL;
      gray := persistent OR allocMode = AllocMode.Copy AND NOT pure;
      generation  : Generation;
      note        : Note;
    BEGIN
      (* maybe we have to put a filler to align this thing *)
      res := RTMisc.Align(thisPtr + ADRSIZE(Header), dataAlignment);
      InsertFiller(thisPtr, res - ADRSIZE(Header) - thisPtr);

      (* allocate the object *)
      thisPtr := LOOPHOLE(res + dataSize, RefHeader);

      IF allocMode = AllocMode.New THEN
        generation := Generation.Younger;
        note := Note.Allocated;
      ELSE
        generation := copyGeneration;
        IF generation = Generation.Older THEN <* ASSERT gray OR pure *> END;
        note := Note.Copied;
      END;
      desc[thisPage - p0] :=
          Desc{space := Space.Current, generation := generation,
               pure := pure, note := note, gray := gray,
               access := Access.ReadWrite, continued := FALSE,
               resident := TRUE, dirty := NOT pure};
      IF persistent THEN
        VAR page := current.db.newPage();
        BEGIN
          map[thisPage - p0] := page;
          page.p := thisPage;
          page.lastReader := ThreadF.myTxn;
          page.writer := ThreadF.myTxn;
        END
      END;
      IF nbPages = 1 THEN
        CASE allocMode OF
        | AllocMode.New => INC(smallNewPages);
        | AllocMode.Copy => INC(smallCopyPages);
        END;
      ELSE
        CASE allocMode OF
        | AllocMode.New => INC(largeNewPages, nbPages);
        | AllocMode.Copy => INC(largeCopyPages, nbPages);
        END;
        FOR i := 1 TO nbPages - 1 DO
          desc[thisPage + i - p0] :=
              Desc{space := Space.Current, generation := generation,
                   pure := pure, note := note, gray := gray,
                   access := Access.ReadWrite, continued := TRUE,
                   resident := TRUE, dirty := NOT pure};
          IF persistent THEN
            VAR page := current.db.newPage();
            BEGIN
              map[thisPage + i - p0] := page;
              page.p := thisPage;
              page.lastReader := ThreadF.myTxn;
              page.writer := ThreadF.myTxn;
            END
          END
        END
      END;
      IF perfOn THEN PerfChange(thisPage, nbPages); END;

      IF nbPages = 1 THEN
        IF thisBoundary - thisPtr > current.boundary - current.ptr THEN
          (* more allocation space available on this page; fill and file
             the current page *)
          InsertFiller(current.ptr, current.boundary - current.ptr);
          IF current.page # Nil THEN
            <* ASSERT desc[current.page - p0].space = Space.Current *>
            desc[current.page - p0].link := current.stack;
            current.stack := current.page;
            IF allocMode = AllocMode.Copy THEN
              <* ASSERT desc[current.page - p0].gray
                   OR desc[current.page - p0].pure *>
            END;
          END;
          current.ptr := thisPtr;
          current.boundary := thisBoundary;
          current.page := thisPage;
        ELSE
          (* more allocation space available on current page; fill and file
             this page *)
          InsertFiller(thisPtr, thisBoundary - thisPtr);
          desc[thisPage - p0].link := current.stack;
          current.stack := thisPage;
        END;
      ELSE
        (* file this page *)
        desc[thisPage - p0].link := current.stack;
        current.stack := thisPage;
      END;
    END;
    RETURN res;
  END LongAlloc;

(*--------------------------------------------------*)

VAR
  backgroundWaiting   := FALSE;

(* The background thread may be present or not.  If it is present, it
   speeds collection asynchronously.  Because it makes progress slowly, it
   should impose only a small overhead when the mutator is running, but
   quickly complete a collection if the collector pauses. *)

PROCEDURE BackgroundThread (<* UNUSED *> closure: Thread.Closure): REFANY =
  BEGIN
    LOOP
      backgroundWaiting := TRUE; (* no locks, unfortunately *)
      WHILE collectorState = CollectorState.Zero DO Wait(); END;
      backgroundWaiting := FALSE;
      WHILE collectorState # CollectorState.Zero DO
        RTOS.LockHeap();
        BEGIN
          IF collectorState # CollectorState.Zero THEN
            CollectorOn();
            CollectSome();
            CollectorOff();
          END;
        END;
        RTOS.UnlockHeap();
        Thread.Pause(1.0d0);       (* one second *)
      END;
    END;
  END BackgroundThread;


(* --------------------------------------------------------- collector *)

PROCEDURE StartGC () =
  BEGIN
    StartCollection();
  END StartGC;

PROCEDURE FinishGC () =
  BEGIN
    FinishCollection();
  END FinishGC;

PROCEDURE Crash (): BOOLEAN =
  VAR result: BOOLEAN;
  BEGIN
    RTOS.LockHeap();        (* left incremented *)
    IF collectorState = CollectorState.Zero THEN
      (* no collection in progress *)
      collectorOn := TRUE;       (* left on *)
      result := TRUE;
    ELSIF NOT collectorOn THEN
      CollectorOn();             (* left on *)
      (* finish collection *)
      WHILE collectorState # CollectorState.Zero DO CollectSome(); END;
      result := TRUE;
    ELSE
      collectorOn := TRUE;       (* left on *)
      result := FALSE;
    END;
    (* unprotect all pages *)
    FOR p := p0 TO p1 - 1 DO
      IF desc[p - p0].access # Access.ReadWrite THEN Unprotect(p); END;
    END;
    RETURN result;
  END Crash;

(* --------------------------------------------------------- debugging *)

VAR
  protectedCheck, refCheck: RTHeapMap.Visitor;

PROCEDURE InstallSanityCheck () =
  BEGIN
    RegisterMonitor(
      NEW(MonitorClosure, before := SanityCheck, after := SanityCheck));
    IF (refCheck = NIL) THEN
      protectedCheck := NEW (RTHeapMap.Visitor,
                             apply := ProtectedOlderRefSanityCheck);
      refCheck := NEW (RTHeapMap.Visitor, apply := RefSanityCheck);
    END;
  END InstallSanityCheck;

(* SanityCheck checks the heap for correctness when no collection is in
   progress. *)

PROCEDURE SanityCheck (<*UNUSED*> self: MonitorClosure) =
  VAR p := p0;
  BEGIN
    WHILE p < p1 DO
      CASE desc[p - p0].space OF
      | Space.Unallocated => INC(p);
      | Space.Previous =>
        <* ASSERT FALSE *>
      | Space.Current =>
        <* ASSERT NOT desc[p - p0].gray *>
        <* ASSERT NOT desc[p - p0].continued *>
        IF desc[p - p0].resident THEN
          IF desc[p - p0].access # Access.None THEN
            IF map[p - p0] = NIL AND desc[p - p0].access = Access.ReadOnly THEN
              <* ASSERT NOT desc[p - p0].dirty *>
              <* ASSERT desc[p - p0].generation = Generation.Older *>
            END;
            (* visit the objects on the page *)
            VAR
              h  := PageToHeader(p);
              he := PageToHeader(p + 1);
            BEGIN
              IF    p = new.page          THEN he := new.ptr;
              ELSIF p = newTransient.page THEN he := newTransient.ptr; END;
              WHILE h < he DO
                (* check the references in the object *)
                IF map[p - p0] = NIL AND desc[p - p0].access = Access.ReadOnly
                 THEN
                  RTHeapMap.WalkRef (h, protectedCheck);
                ELSE
                  RTHeapMap.WalkRef (h, refCheck);
                END;
                INC(h, ADRSIZE(Header) + ReferentSize(h));
              END;
              IF h > he THEN
                <* ASSERT HeaderToPage(h - 1) = p + PageCount(p) - 1 *>
              ELSE
                <* ASSERT PageCount(p) = 1 *>
              END
            END
          ELSE
            (* don't visit objects since the page is protected *)
          END
        ELSE
          <* ASSERT desc[p - p0].access = Access.None *>
          <* ASSERT desc[p - p0].pure *>
          <* ASSERT map[p - p0].writer = NIL *>
        END;

        VAR
          n := PageCount(p);
          d := desc[p - p0];
        BEGIN
          <* ASSERT NOT d.continued *>
          LOOP
            INC(p);
            DEC(n);
            IF n = 0 THEN EXIT; END;
            VAR dd := desc[p - p0];
            BEGIN
              <* ASSERT dd.space      = d.space      *>
              <* ASSERT dd.generation = d.generation *>
              <* ASSERT dd.pure       = d.pure       *>
              <* ASSERT dd.note       = d.note       *>
              <* ASSERT dd.gray       = d.gray       *>
              <* ASSERT dd.access     = d.access     *>
              <* ASSERT dd.continued                 *>
              <* ASSERT dd.resident   = d.resident   *>
              (* ASSERT dd.dirty      = d.dirty      *)
              (* ASSERT dd.link       = d.link       *)
              <* ASSERT map[p - p0] = NIL OR map[p - p0].db # NIL *>
            END
          END
        END;
      | Space.Free =>
        <* ASSERT NOT desc[p - p0].continued *>
        INC(p);
      END;
    END;
    <* ASSERT p = p1 *>
  END SanityCheck;

PROCEDURE RefSanityCheck (<*UNUSED*>v: RTHeapMap.Visitor;  cp  : ADDRESS) =
  VAR ref := LOOPHOLE(cp, REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      VAR
        p := ReferentToPage(ref);
        h := HeaderOf(ref);
      BEGIN
        IF p0 <= p AND p < p1 THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          <* ASSERT NOT desc[p - p0].continued *>
          IF desc[p - p0].access # Access.None THEN
            VAR tc := h.typecode;
            BEGIN
              <* ASSERT (0 < tc AND tc < RT0u.nTypes)
                        OR tc = Fill_1_type
                        OR tc = Fill_N_type *>
            END
          END
        ELSE
          (* the compiler generates Text.T that are not in the traced heap *)
          <* ASSERT h.typecode = RT0.TextTypecode *>
        END;
      END;
    END;
  END RefSanityCheck;

PROCEDURE ProtectedOlderRefSanityCheck (<*UNUSED*> v  : RTHeapMap.Visitor;
                                                   cp : ADDRESS) =
  VAR ref := LOOPHOLE(cp, REF RefReferent)^;
  BEGIN
    IF ref # NIL THEN
      VAR
        p := ReferentToPage(ref);
        h := HeaderOf(ref);
      BEGIN
        IF p0 <= p AND p < p1 THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          <* ASSERT desc[p - p0].generation = Generation.Older *>
          <* ASSERT NOT desc[p - p0].continued *>
          IF desc[p - p0].access # Access.None THEN
            VAR tc := h.typecode;
            BEGIN
              <* ASSERT (0 < tc AND tc < RT0u.nTypes)
                        OR tc = Fill_1_type
                        OR tc = Fill_N_type *>
            END
          END;
        ELSE
          (* the compiler generates Text.T that are not in the traced heap *)
          <* ASSERT h.typecode = RT0.TextTypecode *>
        END;
      END;
    END;
  END ProtectedOlderRefSanityCheck;

(* ----------------------------------------------------------------------- *)

PROCEDURE VisitAllRefs (v: RefVisitor) =
  VAR tc: Typecode;
  BEGIN
    TRY
      Disable();
      FOR p := p0 TO p1 - 1 DO
        IF desc[p - p0].space = Space.Current
             AND NOT desc[p - p0].continued THEN
          VAR
            h             := PageToHeader(p);
            he            := PageToHeader(p + 1);
            size: INTEGER;
          BEGIN
            IF    p = new.page          THEN he := new.ptr;
            ELSIF p = newTransient.page THEN he := newTransient.ptr; END;
            WHILE h < he DO
              size := ReferentSize(h);
              tc := h.typecode;
              IF tc # Fill_1_type AND tc # Fill_N_type THEN
                IF NOT v.visit(
                         tc, LOOPHOLE(h + ADRSIZE(Header), REFANY), size) THEN
                  RETURN;
                END;
              END;
              INC(h, ADRSIZE(Header) + size);
            END;
          END;
        END;
      END;
    FINALLY
      Enable();
    END;
  END VisitAllRefs;

TYPE
  CountClosure = MonitorClosure OBJECT
                   tcs    : <*TRANSIENT*> REF ARRAY OF Typecode;
                   counts : <*TRANSIENT*> REF ARRAY OF CARDINAL;
                   visitor: RefVisitor;
                 OVERRIDES
                   after := CountRefsForTypecodes;
                 END;

TYPE
  CountAllClosure = MonitorClosure OBJECT
                      counts : <*TRANSIENT*> REF ARRAY OF CARDINAL;
                      visitor: RefVisitor;
                    OVERRIDES
                      after := CountRefsForAllTypecodes;
                    END;

TYPE
  CountVisitor =
    RefVisitor OBJECT cl: CountClosure OVERRIDES visit := One; END;

  CountAllVisitor =
    RefVisitor OBJECT cl: CountAllClosure OVERRIDES visit := All; END;

PROCEDURE One (           self: CountVisitor;
                          tc  : Typecode;
               <*UNUSED*> r   : REFANY;
               <*UNUSED*> size: CARDINAL      ): BOOLEAN =
  BEGIN
    FOR i := FIRST(self.cl.tcs^) TO LAST(self.cl.tcs^) DO
      IF self.cl.tcs[i] = tc THEN INC(self.cl.counts[i]); RETURN TRUE; END;
    END;
    RETURN TRUE;
  END One;

PROCEDURE All (           self: CountAllVisitor;
                          tc  : Typecode;
               <*UNUSED*> r   : REFANY;
               <*UNUSED*> size: CARDINAL         ): BOOLEAN =
  BEGIN
    INC(self.cl.counts[tc]);
    RETURN TRUE;
  END All;

PROCEDURE CountRefsForTypecodes (cl: CountClosure) =
  BEGIN
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      cl.counts[i] := 0;
    END;
    VisitAllRefs(cl.visitor);
    FOR i := FIRST(cl.tcs^) TO LAST(cl.tcs^) DO
      PutText("count[");
      PutInt(cl.tcs[i]);
      PutText("] = ");
      PutInt(cl.counts[i]);
      IF i # LAST(cl.tcs^) THEN PutText(",  "); END;
    END;
    PutText("\n");
  END CountRefsForTypecodes;

PROCEDURE CountRefsForAllTypecodes (cl: CountAllClosure) =
  BEGIN
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      cl.counts[i] := 0;
    END;
    VisitAllRefs(cl.visitor);
    FOR i := FIRST(cl.counts^) TO LAST(cl.counts^) DO
      IF cl.counts[i] > 1 THEN
        PutInt(i);
        PutText(": ");
        PutInt(cl.counts[i]);
        IF i # LAST(cl.counts^) THEN PutText(", "); END;
      END;
    END;
    PutText("\n");
  END CountRefsForAllTypecodes;

(* ---------------------------------------------------- showheap hooks *)

VAR
  perfW  : RTPerfTool.Handle;
  perfOn : BOOLEAN := FALSE;

CONST
  EventSize = (BITSIZE(RTHeapEvent.T) + BITSIZE(CHAR) - 1) DIV BITSIZE(CHAR);

PROCEDURE PerfStart () =
  VAR i, j: Page;
  BEGIN
    IF RTPerfTool.Start("showheap", perfW) THEN
      perfOn := TRUE;
      RTProcess.RegisterExitor(PerfStop);
      PerfGrow(p0, p1 - p0);

      i := p0;
      WHILE i # Nil AND i < p1 DO
        j := i + 1;
        WHILE j < p1 AND desc[j - p0].continued DO INC(j); END;
        IF desc[i - p0].space # Space.Free THEN PerfChange(i, j - i); END;
        i := j;
      END;
    END;
  END PerfStart;

PROCEDURE PerfFlip () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Flip};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfFlip;

PROCEDURE PerfPromotedRoots () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Roots};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfPromotedRoots;

PROCEDURE PerfStop () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Bye};
  BEGIN
    (* UNSAFE, but needed to prevent deadlock if we're crashing! *)
    EVAL RTPerfTool.Send (perfW, ADR(e), EventSize);
    RTPerfTool.Close (perfW);
  END PerfStop;

PROCEDURE PerfAllow (<*UNUSED*> n: INTEGER := 0) =
  VAR
    e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Off, nb :=
                       disableCount + disableMotionCount};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfAllow;

PROCEDURE PerfBegin () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Begin};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfBegin;

PROCEDURE PerfEnd () =
  VAR e := RTHeapEvent.T{kind := RTHeapEvent.Kind.End};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfEnd;

PROCEDURE PerfChange (first: Page; nb: CARDINAL) =
  VAR
    e := RTHeapEvent.T{kind := RTHeapEvent.Kind.Change, first := first,
                       nb := nb, desc := desc[first - p0]};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfChange;

PROCEDURE PerfGrow (firstNew: Page; nb: CARDINAL) =
  VAR
    e := RTHeapEvent.T{
           kind := RTHeapEvent.Kind.Grow, first := firstNew, nb := nb};
  BEGIN
    perfOn := RTPerfTool.Send (perfW, ADR(e), EventSize);
  END PerfGrow;

(* ----------------------------------------------------------------------- *)

(* RTWeakRef *)

(* weakTable contains four singly-linked lists: for entries in use (rooted
   at index weakLive0), entries with final cleanup (at weakFinal0), dead
   entries awaiting cleanup (at weakDead0), and free entries (at
   weakFree0).

   Entries in use contain the weak ref, the REF, and the procedure.  The
   "a" field of the weak ref is the index in the table; this speeds lookup.
   The "b" field is a unique value, taken from a 32-bit counter.

   Dead entries contain the same dields, but the "a" field of the weak ref
   is set to -1 to keep lookups from succeeding.  When the cleanup
   procedure is to be called, the original weak ref can still be
   reconstructed, since the "a" field was the index. *)

VAR
  weakTable: UNTRACED REF ARRAY OF WeakEntry; (* allocated in "Init" *)
             (* := NEW(UNTRACED REF ARRAY OF WeakEntry, 0); *)
  weakLive0  := -1;              (* the root of the in-use list *)
  weakFinal0 := -1;              (* the root of the thread-cleanup list *)
  weakDead0  := -1;              (* the root of the dead list *)
  weakFree0  := -1;              (* the root of the free list *)

TYPE
  Int32 = BITS 32 FOR [-16_7fffffff-1 .. 16_7fffffff];
  WeakRefAB = RECORD
                a: Int32;
                b: Int32;
              END;
  WeakEntry = RECORD
                t: WeakRefAB;    (* the weak ref, if well-formed *)
                r: RefReferent;  (* the traced reference *)
                p: ADDRESS;      (* a WeakRefCleanUpProc or a PROCEDURE(r:
                                    REFANY) *)
                next: INTEGER;   (* the next entry on the list *)
              END;

(* This is WeakRef.FromRef, which returns a new weak ref for an object. *)

VAR startedWeakCleaner := FALSE;

PROCEDURE WeakRefFromRef (r: REFANY; p: WeakRefCleanUpProc := NIL): WeakRef =
  VAR
    start           := FALSE;
    result: WeakRef;
  BEGIN
    <* ASSERT r # NIL *>
    RTOS.LockHeap();
    BEGIN
      (* create a WeakCleaner thread the first time through *)
      IF p # NIL AND NOT startedWeakCleaner THEN
        start := TRUE;
        startedWeakCleaner := TRUE;
      END;
      (* if necessary, expand weakTable *)
      IF weakFree0 = -1 THEN ExpandWeakTable(); END;
      IF p # NIL THEN
        (* mark the object as having a weak ref with non-nil cleanup *)
        VAR header := HeaderOf(LOOPHOLE(r, ADDRESS));
        BEGIN
          <* ASSERT NOT header^.weak *>
          header^.weak := TRUE;
        END;
      END;
      (* allocate a new entry *)
      VAR i := weakFree0;
      BEGIN
        weakFree0 := weakTable[i].next;
        (* generate a new weak ref *)
        VAR t := WeakRefAB{a := i, b := Word.Plus(weakTable[i].t.b, 1)};
        BEGIN
          <* ASSERT t.b # 0 *>
          (* set up the entry *)
          weakTable[i] :=
            WeakEntry{t := t, r := LOOPHOLE(r, RefReferent), p :=
                      LOOPHOLE(p, ADDRESS), next := weakLive0};
          weakLive0 := i;
          result := LOOPHOLE(t, WeakRef);
        END;
      END;
    END;
    RTOS.UnlockHeap();
    IF start THEN
      EVAL Thread.Fork(NEW(Thread.Closure, apply := WeakCleaner));
    END;
    RETURN result;
  END WeakRefFromRef;

PROCEDURE ExpandWeakTable () =
  VAR
    newTable := NEW(UNTRACED REF ARRAY OF WeakEntry,
                    2 * NUMBER(weakTable^) + 1);
  BEGIN
    SUBARRAY(newTable^, 0, NUMBER(weakTable^)) := weakTable^;
    FOR i := NUMBER(weakTable^) TO NUMBER(newTable^) - 1 DO
      WITH entry = newTable[i] DO
        entry.t.b := 0;
        entry.next := weakFree0;
        weakFree0 := i;
      END;
    END;
    weakTable := newTable;
  END ExpandWeakTable;

(* This is WeakRef.ToRef, which inverts FromRef *)

PROCEDURE WeakRefToRef (READONLY t: WeakRef): REFANY =
  VAR ab: WeakRefAB;  r: REFANY := NIL;
  BEGIN
    LOOPHOLE (ab, WeakRef) := t;
    RTOS.LockHeap();
    (* if the weak ref is not dead, we know the index *)
    IF ab.a < NUMBER(weakTable^) THEN
      WITH entry = weakTable[ab.a] DO
        (* check the weak ref there *)
        IF entry.t = ab THEN
          <* ASSERT entry.r # NIL *>
          IF collectorState # CollectorState.Zero THEN
            VAR p := ReferentToPage(entry.r);
            BEGIN
              <* ASSERT p # Nil *>
              IF desc[p - p0].space = Space.Previous THEN
                CollectorOn();
                Move(NIL, ADR(entry.r));
                CollectorOff();
              END;
            END;
          END;
          r := LOOPHOLE(entry.r, REFANY);
        END;
      END;
    END;
    RTOS.UnlockHeap();
    RETURN r;
  END WeakRefToRef;

(* This is RTHeapRef.RegisterFinalCleanup, which registers final cleanup
   for a heap object. *)

PROCEDURE RegisterFinalCleanup (r: REFANY; p: PROCEDURE (r: REFANY)) =
  BEGIN
    <* ASSERT r # NIL *>
    <* ASSERT p # NIL *>
    RTOS.LockHeap();
    BEGIN
      (* if necessary, expand weakTable *)
      IF weakFree0 = -1 THEN ExpandWeakTable(); END;
      (* allocate a new entry *)
      VAR i := weakFree0;
      BEGIN
        weakFree0 := weakTable[i].next;
        (* set up the entry, without a weak ref *)
        weakTable[i].r := LOOPHOLE(r, RefReferent);
        weakTable[i].p := LOOPHOLE(p, ADDRESS);
        weakTable[i].next := weakFinal0;
        weakFinal0 := i;
      END;
    END;
    RTOS.UnlockHeap();
  END RegisterFinalCleanup;

(* WeakCleaner waits for entries to be placed on the dead list, then cleans
   them up and puts them on the free list. *)

PROCEDURE WeakCleaner (<*UNUSED*> closure: Thread.Closure): REFANY =
  VAR
    i   : INTEGER;
    copy: WeakEntry;
  BEGIN
    LOOP
      (* get an entry to handle.  copy its contents, then put it on the
         free list. *)
      WHILE weakDead0 = -1 DO Wait(); END;
      RTOS.LockHeap();
      IF weakDead0 = -1 THEN
        RTOS.UnlockHeap();
      ELSE
        i := weakDead0;
        WITH entry = weakTable[i] DO
          <* ASSERT entry.t.a = -1 *>
          CollectorOn();
          Move(NIL, ADR(entry.r));
          CollectorOff();
          copy := entry;
          weakDead0 := entry.next;
          entry.next := weakFree0;
          weakFree0 := i;
        END;
        RTOS.UnlockHeap();
        (* call the registered procedure.  note that collections are
           allowed; the copy is kept on the stack so the object won't be
           freed during the call. *)
        IF copy.p # NIL THEN
          LOOPHOLE(copy.p, WeakRefCleanUpProc)(
            LOOPHOLE(WeakRefAB{a := i, b := copy.t.b}, WeakRef),
            LOOPHOLE(copy.r, REFANY));
        END;
        copy.r := NIL;           (* to help conservative collector *)
      END;
    END;
  END WeakCleaner;

(* ----------------------------------------------------------------------- *)

PROCEDURE FirstPage (p: Page): Page =
  BEGIN
    WHILE desc[p - p0].continued DO DEC(p); END;
    RETURN p;
  END FirstPage;

PROCEDURE PageCount (p: Page): CARDINAL =
  VAR n := 0;
  BEGIN
    <* ASSERT NOT desc[p - p0].continued *>
    REPEAT INC(p); INC(n); UNTIL p >= p1 OR NOT desc[p - p0].continued;
    RETURN n;
  END PageCount;

(* ----------------------------------------------------------------------- *)

PROCEDURE Protect (p: Page; access: Access) =
  VAR
    n := PageCount(p);
  BEGIN
    <* ASSERT collectorOn OR access = Access.ReadWrite OR map[p-p0] # NIL *>
    <* ASSERT RTHeapDep.VM *>
    <* NOWARN *><* ASSERT TRUE *>
    RTHeapDep.Protect(p, n,
                      access # Access.None,
                      access = Access.ReadWrite);
    FOR i := 0 TO n - 1 DO
      desc[p + i - p0].access := access;
    END;
    IF perfOn THEN PerfChange(p, n); END;
  END Protect;

PROCEDURE Unprotect (p: Page) =
  BEGIN
    Protect(p, Access.ReadWrite);
  END Unprotect;

TYPE Swizzler = RTTypeMap.Visitor OBJECT
  db: RTDB.T;
  page: RTDB.Page;
METHODS
  swizzleRef(db: RTDB.T; ref: RefReferent): RefReferent RAISES {Thread.Aborted}
  := SwizzleRef;
  map(page: RTDB.Page; nbPages: CARDINAL) RAISES {Thread.Aborted} := Map;
  swizzleType(db: RTDB.T; ref: RefReferent): TypeDefn RAISES {Thread.Aborted}
  := SwizzleType;
  swizzlePage(): BOOLEAN RAISES {Thread.Aborted} := SwizzlePage;
OVERRIDES
  apply := Swizzle
END;

PROCEDURE Swizzle (self: Swizzler; cp: ADDRESS; k: RTTypeMap.Kind)
  RAISES {Thread.Aborted} =
  VAR refref := LOOPHOLE(cp, UNTRACED REF RefReferent);
  BEGIN
    CASE k OF
    | RTTypeMap.Kind.UntracedRef, RTTypeMap.Kind.TransientRef =>
      refref^ := NIL;
    | RTTypeMap.Kind.Ref =>
      VAR ref := refref^;
      BEGIN
        IF ref # NIL THEN
          refref^ := self.swizzleRef(self.db, ref);
        END
      END
    ELSE
    END;
  END Swizzle;

PROCEDURE SwizzleRef (self: Swizzler; db: RTDB.T; ref: RefReferent)
  : RefReferent
  RAISES {Thread.Aborted} =
  VAR
    id: RTDB.Id := Word.RightShift(LOOPHOLE(ref, INTEGER), LogBytesPerPage);
    offset := Word.And(LOOPHOLE(ref, INTEGER), BytesPerPage - 1);
    nbPages := 1;
    p: Page;
  BEGIN
    (*
      Small object PID:
      +----------------+----------------+
      | page id        | byte offset  00|
      +----------------+----------------+

      Large object PID:
      +----------------+-----------+----+
      | page id        | # pages   |xxx1|
      +----------------+-----------+----+
      The offset is encoded by the value of xxx.
      If # pages = 0 then object size can only be inferred from its type.
    *)
    IF Word.And(offset, 1) # 0 THEN
      (* large object *)
      nbPages := Word.RightShift(offset, 4);
      CASE Word.And(offset, 2_1111) OF
      | 2_0001 => offset :=   4;
      | 2_0011 => offset :=   8;
      | 2_0101 => offset :=  16;
      | 2_0111 => offset :=  32;
      | 2_1001 => offset :=  64;
      | 2_1011 => offset := 128;
      | 2_1101 => offset := 256;
      | 2_1111 => offset := 512;
      ELSE
        <* ASSERT FALSE *>
      END;
    END;

    VAR page: RTDB.Page := db.mapPage(id);
    BEGIN
      IF page.p = Nil THEN
        self.map(page, nbPages);
      END;
      p := page.p;
    END;

    VAR pi := p - p0;
    BEGIN
      IF desc[pi].space = Space.Previous THEN
        IF desc[pi].pure THEN
          PromotePage(
              p, Desc{space := Space.Current, generation := copyGeneration,
                      pure := TRUE, note := Note.Persistent,
                      gray := FALSE, access := desc[pi].access,
                      continued := FALSE, resident := desc[pi].resident,
                      dirty := FALSE});
        ELSE
          IF collectorOn THEN
            IF desc[pi].access # Access.ReadWrite THEN Unprotect(p) END;
          ELSE
            Protect(p, Access.None);
          END;            
          PromotePage(
              p, Desc{space := Space.Current, generation := copyGeneration,
                      pure := FALSE, note := Note.Persistent,
                      gray := TRUE, access := desc[pi].access,
                      continued := FALSE, resident := TRUE,
                      dirty := TRUE});
          desc[pi].link := impureCopy.stack;
          impureCopy.stack := p;
        END
      END
    END;
    RETURN PageToAddress(p) + offset;
  END SwizzleRef;

PROCEDURE Map (self: Swizzler; page: RTDB.Page; nbPages: CARDINAL)
  RAISES {Thread.Aborted} =
  (* page has not yet been mapped *)
  VAR
    p: Page;
    notAfter: SET OF Note := SET OF Note{Note.Allocated};
    d := Desc{space := Space.Current, generation := copyGeneration,
              pure := TRUE, note := Note.Persistent, gray := FALSE,
              access := Access.ReadWrite, continued := FALSE,
              resident := FALSE, dirty := FALSE};
  PROCEDURE PeekPageCount(page: RTDB.Page;
                          READONLY data: RTHeapDep.PageData) =
    <*FATAL Thread.Aborted*>
    VAR
      h: RefHeader;
      type: ADDRESS;
      def: TypeDefn;
      dataSize, nbBytes: CARDINAL;
    BEGIN
      TRY
        ThreadF.SuspendOthers();
        h := LOOPHOLE(ADR(data[0]), RefHeader);
        type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
        IF type = LOOPHOLE(0, ADDRESS) THEN
          INC(h, ADRSIZE(Header));
          type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
        ELSIF type = LOOPHOLE(1, ADDRESS) THEN
          INC(h, LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^);
          type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
        END;
        def := self.swizzleType(page.db, type);
        IF def.nDimensions = 0 THEN
          dataSize := def.dataSize;
        ELSE
          dataSize := OpenArraySize(h, def);
        END;
        nbBytes := RTMisc.Upper(ADRSIZE(Header), def.dataAlignment) + dataSize;
        nbPages := (nbBytes + AdrPerPage - 1) DIV AdrPerPage;
      FINALLY
        ThreadF.ResumeOthers();
      END
    END PeekPageCount;
  BEGIN
    IF nbPages = 0 THEN 
      (* large object of unknown size: get size from first page *)
      TRY
        IF self.page # NIL THEN
          Protect(self.page.p, Access.None);
        END;
        ThreadF.ResumeOthers();
        Thread.Acquire(page);
        page.peek(PeekPageCount);
      FINALLY
        Thread.Release(page);
        ThreadF.SuspendOthers();
        IF self.page # NIL THEN
          Unprotect(self.page.p);
        END
      END;
    END;
    p := FindFreePages(nbPages, notAfter := notAfter);
    desc[p - p0] := d;
    map[p - p0] := page;
    page.p := p;
    IF nbPages > 1 THEN
      d.continued := TRUE;
      FOR i := 1 TO nbPages - 1 DO
        desc[p + i - p0] := d;
        page := page.db.mapPage(page.id + 1);
        map[p + i - p0] := page;
        page.p := p + i;
      END
    END;
    Protect(p, Access.None);
  END Map;

PROCEDURE SwizzleType (self: Swizzler; db: RTDB.T; ref: RefReferent): TypeDefn
  RAISES {Thread.Aborted} =
  VAR
    fpRef := LOOPHOLE(self.swizzleRef(db, ref), REF Fingerprint.T);
    fpAdr := ref;
    p: Page := Word.RightShift(LOOPHOLE(fpRef, INTEGER), LogBytesPerPage);
    fp: Fingerprint.T;
  PROCEDURE PeekType(page: RTDB.Page; READONLY data: RTHeapDep.PageData) =
    VAR p := page.p;
    BEGIN
      ThreadF.SuspendOthers();
      Unprotect(p);
      PageToData(p)^ := data;
      fp := fpRef^;
      Protect(p, Access.None);
      desc[p - p0].dirty := TRUE;
      ThreadF.ResumeOthers();
    END PeekType;
  BEGIN
    IF desc[p - p0].access # Access.ReadWrite THEN
      IF desc[p - p0].pure THEN
        (* don't fault page, just get fingerprint *)
        VAR page := map[p - p0];
        BEGIN
          <* ASSERT page # NIL *>
          <* ASSERT page # self.page *>
          TRY
            IF self.page # NIL THEN
              Protect(self.page.p, Access.None);
            END;
            ThreadF.ResumeOthers();
            Thread.Acquire(page);
            ThreadF.SuspendOthers();
            IF desc[p - p0].resident THEN
              fp := fpRef^;
              <* ASSERT desc[p - p0].access = Access.ReadOnly *>
            ELSIF desc[p - p0].dirty THEN
              Unprotect(p);
              fp := fpRef^;
              Protect(p, Access.None);
            ELSE
              TRY
                ThreadF.ResumeOthers();
                page.peek(PeekType);
              FINALLY
                ThreadF.SuspendOthers();
              END
            END
          FINALLY
            ThreadF.ResumeOthers();
            Thread.Release(page);
            ThreadF.SuspendOthers();
            IF self.page # NIL THEN
              Unprotect(self.page.p);
            END
          END
        END
      ELSIF desc[p - p0].gray THEN
        (* don't clean page, just get fingerprint *)
        Unprotect(p);
        fp := fpRef^;
        Protect(p, Access.None);
      ELSE
        fp := fpRef^;
        <* ASSERT desc[p - p0].access = Access.ReadOnly *>
      END
    ELSE
      fp := fpRef^;
    END;
    db.mapFP(fp, fpRef, fpAdr);
    RETURN RTType.Get(RTTypeFP.FromFingerprint(fp));
  END SwizzleType;

PROCEDURE SwizzlePage (self: Swizzler): BOOLEAN RAISES {Thread.Aborted} =
  VAR
    pure := TRUE;
    type: RefReferent;
    def: TypeDefn;
    referentSize: CARDINAL;
    p := self.page.p;
    h := PageToHeader(p);
    he := PageToHeader(p + 1);
  BEGIN
    WHILE h < he DO
      type := LOOPHOLE(h, UNTRACED REF ADDRESS)^;
      IF type = LOOPHOLE(0, ADDRESS) THEN
        h^ := Header{typecode := Fill_1_type};
        referentSize := 0;
      ELSIF type = LOOPHOLE(1, ADDRESS) THEN
        h^ := Header{typecode := Fill_N_type};
        referentSize := LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF INTEGER)^;
        DEC(referentSize, ADRSIZE(Header));
      ELSE
        def := self.swizzleType(self.db, type);
        <* ASSERT def.traced = 1 *>
        h^ := Header{typecode := def.typecode};
        IF def.nDimensions = 0 THEN
          referentSize := def.dataSize;
          IF def.defaultMethods # NIL THEN
            (* object: set up methods pointer *)
            LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF RT0.MethodSuite)^ :=
                def.defaultMethods;
          END;
        ELSE
          (* open array: set up the internal pointer *)
          LOOPHOLE(h + ADRSIZE(Header), UNTRACED REF ADDRESS)^ :=
              h + ADRSIZE(Header) + def.dataSize;
          referentSize := OpenArraySize(h, def);
        END;
        IF def.gc_map # NIL OR def.parent # NIL THEN
          pure := FALSE;
          RTTypeMap.DoWalkRef(def, <*NOWARN*> (* Thread.Aborted *)
                              h + ADRSIZE(Header),
                              RTTypeMap.Mask{RTTypeMap.Kind.Ref,
                                             RTTypeMap.Kind.UntracedRef,
                                             RTTypeMap.Kind.TransientRef},
                              self);
        END
      END;
      INC(h, ADRSIZE(Header) + referentSize);
    END;
    RETURN pure;
  END SwizzlePage;

PROCEDURE SwizzleRoot (db: RTDB.T): REFANY =
  (* root should be first object on page 1 *)  
  <*FATAL Thread.Aborted*>
  VAR
    rootId := LOOPHOLE(BytesPerPage, ADDRESS) + ADRSIZE(Header);
    swizzler := NEW(Swizzler, db := db, page := NIL);
  BEGIN
    TRY
      ThreadF.SuspendOthers();
      RETURN LOOPHOLE(swizzler.swizzleRef(db, rootId), REFANY);
    FINALLY
      ThreadF.ResumeOthers();
    END
  END SwizzleRoot;

PROCEDURE GrayFault (p: Page) =
  BEGIN
    <* ASSERT desc[p - p0].resident *>
    <* ASSERT desc[p - p0].access = Access.None *>
    <* ASSERT NOT desc[p - p0].pure *>
    CollectorOn();
    IF p = impureCopy.page THEN
      EVAL CopySome();
      IF desc[p - p0].gray THEN
        <* ASSERT p = impureCopy.page *>
        InsertFiller(impureCopy.ptr, impureCopy.boundary - impureCopy.ptr);
        impureCopy.page := Nil;
        impureCopy.ptr := NIL;
        impureCopy.boundary := NIL;
        CleanBetween(PageToHeader(p), PageToHeader(p + 1));
        desc[p - p0].gray := FALSE;
        desc[p - p0].dirty := FALSE;
        IF perfOn THEN PerfChange(p, 1) END;
        IF ThreadF.myTxn = NIL
          OR map[p - p0] = NIL
          OR map[p - p0].writer = ThreadF.myTxn
         THEN
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, Access.ReadOnly);
          END
        ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
          Protect(p, Access.ReadOnly);
        ELSE
          Protect(p, Access.None);
        END;
      END;
    ELSIF p = impureTransientCopy.page THEN
      EVAL CopySome();
      IF desc[p - p0].gray THEN
        <* ASSERT p = impureTransientCopy.page *>
        InsertFiller(
            impureTransientCopy.ptr,
            impureTransientCopy.boundary - impureTransientCopy.ptr);
        impureTransientCopy.page := Nil;
        impureTransientCopy.ptr := NIL;
        impureTransientCopy.boundary := NIL;
        CleanBetween(PageToHeader(p), PageToHeader(p + 1));
        desc[p - p0].gray := FALSE;
        desc[p - p0].dirty := FALSE;
        IF perfOn THEN PerfChange(p, 1); END;
        IF ThreadF.myTxn = NIL
          OR map[p - p0] = NIL
          OR map[p - p0].writer = ThreadF.myTxn
         THEN
          IF desc[p - p0].generation = Generation.Older THEN
            <* ASSERT desc[p - p0].space = Space.Current *>
            Protect(p, Access.ReadOnly);
          END
        ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
          Protect(p, Access.ReadOnly);
        ELSE
          Protect(p, Access.None);
        END
      END
    ELSE
      p := FirstPage(p);
      Unprotect(p);
      CleanBetween(PageToHeader(p), PageToHeader(p + 1));
      FOR i := 0 TO PageCount(p) - 1 DO
        desc[p + i - p0].gray := FALSE;
        desc[p + i - p0].dirty := FALSE;
      END;
      IF perfOn THEN PerfChange(p, PageCount(p)) END;
      IF ThreadF.myTxn = NIL
        OR map[p - p0] = NIL
        OR map[p - p0].writer = ThreadF.myTxn
       THEN
        IF desc[p - p0].generation = Generation.Older THEN
          <* ASSERT desc[p - p0].space = Space.Current *>
          Protect(p, Access.ReadOnly);
        END
      ELSIF map[p - p0].lastReader = ThreadF.myTxn THEN
        Protect(p, Access.ReadOnly);
      ELSE
        Protect(p, Access.None);
      END
    END;
    CollectorOff();
  END GrayFault;

PROCEDURE OlderFault (p: Page) =
  BEGIN
    <* ASSERT desc[p - p0].resident *>
    <* ASSERT NOT desc[p - p0].pure *>
    p := FirstPage(p);
    <* ASSERT NOT desc[p - p0].dirty *>
    <* ASSERT desc[p - p0].generation = Generation.Older *>
    desc[p - p0].dirty := TRUE;
    Unprotect(p);
  END OlderFault;

PROCEDURE CopyPage (page: RTDB.Page; READONLY data: RTHeapDep.PageData) =
  VAR
    p := page.p;
    q := FirstPage(p);
  BEGIN
    ThreadF.SuspendOthers();
    Unprotect(q);
    PageToData(p)^ := data;
    desc[p - p0].dirty := TRUE;
    Protect(q, Access.None);
    ThreadF.ResumeOthers();
  END CopyPage;

PROCEDURE PersistentFault (p: Page) RAISES { Thread.Aborted } =
  VAR page: RTDB.Page;
  BEGIN
    p := FirstPage(p);
    page := map[p - p0];
    <*ASSERT desc[p - p0].space = Space.Current*>
    IF desc[p - p0].resident THEN
      IF desc[p - p0].access = Access.None THEN
        IF page.lastReader # ThreadF.myTxn THEN
          TRY
            ThreadF.ResumeOthers();
            Thread.Acquire(page);
            page.readAccess();
          FINALLY
            Thread.Release(page);
            ThreadF.SuspendOthers();
          END;
          IF NOT desc[p - p0].resident THEN RETURN END;
          IF desc[p - p0].access # Access.None THEN RETURN END;
          page.lastReader := ThreadF.myTxn;
        END;
        Protect(p, Access.ReadOnly);
      ELSE
        <* ASSERT desc[p - p0].access = Access.ReadOnly *>
        IF page.writer # ThreadF.myTxn THEN
          TRY
            ThreadF.ResumeOthers();
            Thread.Acquire(page);
            page.writeAccess();
          FINALLY
            Thread.Release(page);
            ThreadF.SuspendOthers();
          END;
          IF NOT desc[p - p0].resident THEN RETURN END;
          IF desc[p - p0].access # Access.ReadOnly THEN RETURN END;
          page.writer := ThreadF.myTxn;
        END;
        desc[p - p0].dirty := NOT desc[p - p0].pure;
        Unprotect(p);
      END;
      RETURN;
    END;

    (* non-resident, so fault the page(s) *)
    <* ASSERT desc[p - p0].pure *>
    desc[p - p0].dirty := FALSE;
    FOR pp := p TO p + PageCount(p) - 1 DO
      IF desc[pp - p0].resident THEN RETURN END;
      IF desc[pp - p0].dirty THEN RETURN END;
      page := map[pp - p0];
      TRY
        ThreadF.ResumeOthers();
        Thread.Acquire(page);
        page.readAccess();
        page.read(CopyPage);
      FINALLY
        Thread.Release(page);
        ThreadF.SuspendOthers();
      END
    END;

    <* ASSERT NOT desc[p - p0].resident *>
    <* ASSERT desc[p - p0].dirty *>
    page := map[p - p0];
    <* ASSERT p = page.p *>
    VAR swizzler := NEW(Swizzler, db := page.db, page := page);
    BEGIN
      TRY
        Unprotect(p);
        desc[p - p0].pure := swizzler.swizzlePage();
      EXCEPT
      | Thread.Aborted =>
        Protect(p, Access.None);
        RAISE Thread.Aborted;
      END
    END;

    desc[p - p0].resident := TRUE;
    desc[p - p0].dirty := FALSE;
    VAR d := desc[p - p0];
    BEGIN
      d.continued := TRUE;
      FOR pp := p+1 TO p + PageCount(p) - 1 DO
        desc[pp - p0] := d;
      END
    END;
    map[p - p0].lastReader := ThreadF.myTxn;
    Protect(p, Access.ReadOnly);
  END PersistentFault;

PROCEDURE Fault (addr: ADDRESS): BOOLEAN =
  VAR p := LOOPHOLE(addr, INTEGER) DIV BytesPerPage;
  BEGIN
    TRY
      ThreadF.SuspendOthers();
      <* ASSERT RTHeapDep.VM *>
      <* NOWARN *><* ASSERT TRUE *>
      cycleCost :=
          cycleCost + RTHeapDep.VMFaultTime(); (* otherwise unaccounted for *)
      IF NOT (p0 <= p AND p < p1)
        OR desc[p - p0].space = Space.Unallocated THEN
        RETURN FALSE;			 (* not in heap *)
      END;
      IF desc[p - p0].access = Access.ReadWrite THEN
        RETURN TRUE;			 (* was protected, but not any more *)
      END;

      IF desc[p - p0].gray THEN
        GrayFault(p);
      ELSIF map[p - p0] # NIL THEN
        IF ThreadF.myTxn = NIL THEN
          IF desc[p - p0].resident THEN
            desc[p - p0].dirty := NOT desc[p - p0].pure;
            Unprotect(p);
          ELSE
            RETURN FALSE
          END
        ELSE
          PersistentFault(p); <*NOWARN*>   (* Thread.Aborted *)
        END
      ELSE
        OlderFault(p);
      END;
      RETURN TRUE;			 (* was protected, now cleared *)
    FINALLY
      ThreadF.ResumeOthers();
    END
  END Fault;

(* ----------------------------------------------------------------------- *)

(****** Page-level allocator ******)

(* The freelist is sorted by blocksize, linked through the first page in
   each block, using the "link" field in the "desc" array.  Page allocation
   is best-fit.  For elements of the same blocksize, they are sorted by
   page number, to make the showheap display more easily readable, and to
   slightly reduce fragmentation. *)

(* FindFreePages allocates a run of "n" free pages, which we would prefer
   not be near pages in the current space with notes in notAfter.  The
   allocator can thus be used to separate pages with different notes, since
   they will have different lifetimes.  This is a concern only when
   incremental and generational collection are combined. *)

PROCEDURE FindFreePages (n: INTEGER; notAfter: SET OF Note): Page =
  VAR p: Page;
  BEGIN
    IF collectorState = CollectorState.Zero THEN
      p := AllocateFreePagesFromBlock(n, SET OF Note{}, TRUE);
      IF p # Nil THEN RETURN p; END;
    ELSE
      p := AllocateFreePagesFromBlock(n, notAfter, TRUE);
      IF p # Nil THEN RETURN p; END;
      p := AllocateFreePagesFromBlock(n, SET OF Note{}, FALSE);
      IF p # Nil THEN RETURN p; END;
    END;
    GrowHeap(n);
    p := AllocateFreePagesFromBlock(n, SET OF Note{}, TRUE);
    <* ASSERT p # Nil *>
    RETURN p;
  END FindFreePages;

VAR free: Page;                  (* the head of the freelist *)

(* AllocateFreePagesFromBlock finds the first block large enough to satisfy
   the request.  "notAfter" is the set of page notes in the current space
   that the block allocated from must not immediately follow; this is used
   to separate Note.Allocated pages from Note.Copied pages.  If "front" is
   TRUE, the pages will be allocated from the beginning of the block, else
   from the end; this is also used to separate Note.Allocated Pages from
   Note.Copied pages.  If the block is bigger than the request, the
   remainder is left at the right point in the freelist.  If no block
   exists, Nil is returned. *)

PROCEDURE AllocateFreePagesFromBlock (n       : INTEGER;
                                      notAfter: SET OF Note;
                                      front   : BOOLEAN      ): Page =
  VAR
    p                   := free;
    prevP               := Nil;
    prevLength          := 0;
    length    : INTEGER;
  BEGIN
    LOOP
      IF p = Nil THEN RETURN Nil; END;
      length := FreeLength(p);
      IF length >= n
           AND NOT (p > p0 AND desc[(p - 1) - p0].space = Space.Current
                      AND desc[(p - 1) - p0].note IN notAfter) THEN
        EXIT;
      END;
      prevP := p;
      prevLength := length;
      p := desc[p - p0].link;
    END;
    IF length = n THEN
      IF prevP = Nil THEN
        free := desc[p - p0].link;
      ELSE
        desc[prevP - p0].link := desc[p - p0].link;
      END;
      RETURN p;
    ELSE
      VAR
        newP, fragP: Page;
        fragLength : CARDINAL;
      BEGIN
        IF front THEN
          newP := p;
          fragP := p + n;
        ELSE
          newP := p + length - n;
          fragP := p;
        END;
        fragLength := length - n;
        IF fragLength > prevLength THEN
          IF prevP = Nil THEN
            free := fragP;
          ELSE
            desc[prevP - p0].link := fragP;
          END;
          desc[fragP - p0].link := desc[p - p0].link;
        ELSE
          IF prevP = Nil THEN
            free := desc[p - p0].link;
          ELSE
            desc[prevP - p0].link := desc[p - p0].link;
          END;
          VAR
            pp     := free;
            prevPP := Nil;
          BEGIN
            LOOP
              IF pp = Nil THEN EXIT; END;
              VAR length := FreeLength(pp);
              BEGIN
                IF length > fragLength
                     OR (length = fragLength AND pp > fragP) THEN
                  EXIT;
                END;
              END;
              prevPP := pp;
              pp := desc[pp - p0].link;
            END;
            desc[fragP - p0].link := pp;
            IF prevPP = Nil THEN
              free := fragP;
            ELSE
              desc[prevPP - p0].link := fragP;
            END;
          END;
        END;
        RETURN newP;
      END;
    END;
  END AllocateFreePagesFromBlock;

(* RebuildFreelist rebuilds the free list, from the "desc" array.  It first
   links all free blocks into the free list, then it sorts the free list.
   The sort used is insertion sort, which is quadratic in the number of
   different block sizes, but only linear in the number of pages. *)

PROCEDURE RebuildFreelist () =
  BEGIN
    VAR
      prevP     := Nil;
      prevSpace := Space.Unallocated;
    BEGIN
      (* link together the first pages of all free blocks *)
      FOR p := p0 TO p1 - 1 DO
        VAR space := desc[p - p0].space;
        BEGIN
          IF space = Space.Free AND prevSpace # Space.Free THEN
            IF prevP = Nil THEN
              free := p;
            ELSE
              desc[prevP - p0].link := p;
            END;
            prevP := p;
          END;
          prevSpace := space;
        END;
      END;
      IF prevP = Nil THEN
        free := Nil;
      ELSE
        desc[prevP - p0].link := Nil;
      END;
    END;
    (* sort them, using insertion sort *)
    VAR
      n     := 1;                (* smallest block size *)
      p     := free;             (* start of sublist we're examining *)
      prevP := Nil;              (* element before sublist *)
    BEGIN
      LOOP
        VAR
          excess     := Nil;
          prevExcess := Nil;
        BEGIN
          (* separate off blocks over "n" long into excess list *)
          WHILE p # Nil DO
            VAR length := FreeLength(p);
            BEGIN
              <* ASSERT length >= n *>
              IF length > n THEN
                IF prevExcess = Nil THEN
                  excess := p;
                ELSE
                  desc[prevExcess - p0].link := p;
                END;
                IF prevP = Nil THEN
                  free := desc[p - p0].link;
                ELSE
                  desc[prevP - p0].link := desc[p - p0].link;
                END;
                prevExcess := p;
              ELSE
                prevP := p;
              END;
            END;
            p := desc[p - p0].link;
          END;
          (* maybe done *)
          IF excess = Nil THEN EXIT; END;
          <* ASSERT prevExcess # Nil *>
          (* link longer blocks onto end *)
          IF prevP = Nil THEN
            free := excess;
          ELSE
            desc[prevP - p0].link := excess;
          END;
          desc[prevExcess - p0].link := Nil;
          p := excess;
        END;
        (* find smallest element size of remaining bocks *)
        n := LAST(CARDINAL);
        VAR pp := p;
        BEGIN
          REPEAT
            VAR length := FreeLength(pp);
            BEGIN
              IF length < n THEN n := length; END;
            END;
            pp := desc[pp - p0].link;
          UNTIL pp = Nil;
        END;
      END;
    END;
  END RebuildFreelist;

(* FreeLength returns the number of free pages starting at page p. *)

PROCEDURE FreeLength (p: Page): CARDINAL =
  BEGIN
    <* ASSERT desc[p - p0].space = Space.Free *>
    VAR pp := p + 1;
    BEGIN
      LOOP
        IF pp >= p1 THEN EXIT; END;
        IF desc[pp - p0].space # Space.Free THEN EXIT; END;
        INC(pp);
      END;
      RETURN pp - p;
    END;
  END FreeLength;

(* GrowHeap adds a block of at least "minNewPages" free pages to the heap,
   and links it into the free list. *)

VAR fragment0, fragment1: ADDRESS := NIL;

CONST
  InitialBytes = 262144;         (* initial heap size is 256K *)
  MinNewBytes  = 262144;         (* grow the heap by at least 256K *)
  MinNewFactor = 0.2;            (* grow the heap by at least 20% *)

PROCEDURE GrowHeap (pp: INTEGER) =
  VAR
    newChunk    : ADDRESS;
    newSideSpan : INTEGER;
    firstNewPage: Page;
    lastNewPage : Page;
    newP0       : Page;
    newP1       : Page;
  BEGIN
    IF allocatedPages = 0 THEN
      pp := MAX(pp, (InitialBytes + BytesPerPage - 1) DIV BytesPerPage);
    ELSE
      pp := MAX(pp, (MinNewBytes + BytesPerPage - 1) DIV BytesPerPage);
      pp := MAX(pp, CEILING(FLOAT(allocatedPages) * MinNewFactor));
    END;
    VAR bytes := (pp + 1) * BytesPerPage;
    BEGIN
      newChunk := RTOS.GetMemory(bytes);
      IF newChunk = NIL OR newChunk = LOOPHOLE(-1, ADDRESS) THEN
        RTMisc.FatalError(NIL, 0, "gc: Could not extend the traced heap");
      END;
      IF fragment1 = newChunk THEN
        newChunk := fragment0;
        bytes := bytes + (fragment1 - fragment0);
      END;
      VAR excess := (-LOOPHOLE(newChunk, INTEGER)) MOD BytesPerPage;
      BEGIN
        INC(newChunk, excess);
        DEC(bytes, excess);
      END;
      VAR pages := bytes DIV BytesPerPage;
      BEGIN
        firstNewPage := LOOPHOLE(newChunk, INTEGER) DIV BytesPerPage;
        lastNewPage := firstNewPage + pages - 1;
        fragment0 :=
          LOOPHOLE((firstNewPage + pages) * BytesPerPage, ADDRESS);
        fragment1 := newChunk + bytes;
      END;
    END;
    (* determine the new boundaries of the allocated pages *)
    IF p0 = Nil THEN
      newP0 := firstNewPage;
      newP1 := lastNewPage + 1;
    ELSIF firstNewPage < p0 THEN
      newP0 := firstNewPage;
      newP1 := p1;
    ELSIF p1 <= lastNewPage THEN
      newP0 := p0;
      newP1 := lastNewPage + 1;
    ELSE
      newP0 := p0;
      newP1 := p1;
    END;
    (* extend the side arrays if necessary *)
    newSideSpan := newP1 - newP0;
    IF desc = NIL OR newSideSpan # NUMBER(desc^) THEN
      WITH newDesc  = NEW(UNTRACED REF ARRAY OF Desc,      newSideSpan) DO
      WITH newMap   = NEW(UNTRACED REF ARRAY OF RTDB.Page, newSideSpan) DO
        IF desc # NIL THEN
          <* ASSERT map # NIL *>
          FOR i := FIRST(desc^) TO LAST(desc^) DO
            newDesc [i + p0 - newP0] := desc[i];
            newMap  [i + p0 - newP0] := map[i];
          END;
          FOR i := p1 TO firstNewPage - 1 DO
            newDesc [i - newP0].space := Space.Unallocated;
            newMap  [i - newP0]       := NIL;
          END;
          FOR i := lastNewPage + 1 TO p0 - 1 DO
            newDesc [i - newP0].space := Space.Unallocated;
            newMap  [i - newP0]       := NIL;
          END;
          DISPOSE(desc);
          DISPOSE(map);
        END;
        desc := newDesc;
        map  := newMap;
      END
      END
    END;
    p0 := newP0;
    p1 := newP1;
    FOR i := firstNewPage TO lastNewPage DO
      desc[i - p0].space := Space.Free;
    END;
    IF perfOn THEN
      PerfGrow(firstNewPage, lastNewPage - firstNewPage + 1);
    END;
    INC(allocatedPages, lastNewPage - firstNewPage + 1);
    RebuildFreelist();
  END GrowHeap;

(*** SYNCHRONIZATION ***)

(* Wait() waits on a condition, implemented compatibly with RTOS.LockHeap().
   Wait() will be called only from background threads.  The caller
   will be in a critical section.  *)

VAR
  mutex     := NEW(MUTEX);
  condition := NEW(Thread.Condition);

PROCEDURE Wait () =
  BEGIN
    (* This procedure may not be called from inside the collector. *)
    LOCK mutex DO Thread.Wait(mutex, condition); END;
  END Wait;

(* Broadcast() broadcasts a condition, implemented compatibly with
   RTOS.LockHeap().  Broadcast will not be called from inside the collector, so
   that references to the heap will succeed. *)

PROCEDURE Broadcast () =
  BEGIN
    Thread.Broadcast(condition);
  END Broadcast;

(*** INITIALIZATION ***)

CONST MaxAlignment  = 8;
CONST MaxAlignMask  = 2_0111;     (* bit mask to capture MaxAlignment *)
TYPE  MaxAlignRange = [0 .. MaxAlignment - 1];

VAR align: ARRAY MaxAlignRange, [1 .. MaxAlignment] OF CARDINAL;
(* align[i,j] == RTMisc.Align (i, j) - i *)

PROCEDURE Init () =
  BEGIN
    weakTable := NEW(UNTRACED REF ARRAY OF WeakEntry, 0);

    (* initialize the alignment array *)
    FOR i := FIRST(align) TO LAST(align) DO
      FOR j := FIRST(align[0]) TO LAST(align[0]) DO
        align[i, j] := RTMisc.Upper(i, j) - i;
      END;
    END;
  END Init;

VAR
  stats: RECORD
    print := FALSE;
    accuratePages, ambiguousPages := 0;
    objects, bytes: UNTRACED REF ARRAY OF INTEGER;
  END;

BEGIN
  IF RTParams.IsPresent("nogc") THEN disableCount := 1; END;
  IF RTParams.IsPresent("novm") THEN disableVMCount := 1; END;
  IF RTParams.IsPresent("noincremental") THEN incremental := FALSE; END;
  IF RTParams.IsPresent("nogenerational") THEN generational := FALSE; END;
  IF RTParams.IsPresent("paranoidgc") THEN InstallSanityCheck(); END;
  IF RTParams.IsPresent("stats") THEN
    stats.print := TRUE;
    stats.objects := NEW(UNTRACED REF ARRAY OF INTEGER, RT0u.nTypes);
    stats.bytes   := NEW(UNTRACED REF ARRAY OF INTEGER, RT0u.nTypes);
  END;
  PerfStart();
END RTCollector.
