(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Last.m3                                               *)
(* Last Modified On Mon Sep 26 09:15:29 PDT 1994 By kalsow     *)
(*      Modified On Fri Dec 21 01:34:18 1990 By muller         *)

(*
 * HISTORY
 * 23-May-96  Wilson Hsieh (whsieh) at the University of Washington
 *	functional
 *
 * 07-Mar-96  Wilson Hsieh (whsieh) at the University of Washington
 *	wherever ArrayType.Split is called, call Type.Base so that
 *	 ALIGNED FOR works
 *
 * 25-Oct-95  Wilson Hsieh (whsieh) at the University of Washington
 *	make built-in operations non-bounded
 *
 *)

MODULE Last;

IMPORT CG, CallExpr, Expr, Type, Procedure, First, ArrayType;
IMPORT TypeExpr, IntegerExpr, EnumExpr, Int, ArrayExpr;
IMPORT Reel, LReel, EReel, ReelExpr, Target, TInt;

VAR Z: CallExpr.MethodList;

PROCEDURE Check (ce: CallExpr.T;  <*UNUSED*> VAR cs: Expr.CheckState) =
  BEGIN
    First.DoCheck ("LAST", ce);
  END Check;

PROCEDURE Prep (ce: CallExpr.T) =
  VAR
    e := ce.args[0];
    t, index, element: Type.T;
  BEGIN
    IF NOT TypeExpr.Split (e, t) THEN t := Expr.TypeOf (e) END;
    Type.Compile (t);
    IF ArrayType.Split (Type.Base(t), index, element) THEN t := index END;
 
    IF (t = NIL) THEN (* open array *)
      Expr.Prep (e);
    END;
  END Prep;

PROCEDURE Compile (ce: CallExpr.T) =
  VAR
    e := ce.args[0];
    t, index, element: Type.T;
    min, max: Target.Int;
  BEGIN
    IF NOT TypeExpr.Split (e, t) THEN t := Expr.TypeOf (e) END;
    IF ArrayType.Split (Type.Base(t), index, element) THEN t := index END;
 
    IF (t = NIL) THEN (* open array *)
      Expr.Compile (e);
      CG.Open_size (0);
      CG.Load_integer (TInt.One);
      CG.Subtract (CG.Type.Int);
    ELSIF Type.GetBounds (t, min, max) THEN (* ordinal type *)
      CG.Load_integer (max);
    ELSIF Type.IsEqual (t, Reel.T, NIL) THEN
      CG.Load_float (Target.Real.max);
    ELSIF Type.IsEqual (t, LReel.T, NIL) THEN
      CG.Load_float (Target.Longreal.max);
    ELSIF Type.IsEqual (t, EReel.T, NIL) THEN
      CG.Load_float (Target.Extended.max);
    ELSE
      <* ASSERT FALSE *>
    END;
  END Compile;

PROCEDURE Fold (ce: CallExpr.T): Expr.T =
  VAR t, index, elem: Type.T;  e: Expr.T;  min, max: Target.Int;
  BEGIN
    e := ce.args[0];
    IF TypeExpr.Split (e, t) THEN RETURN LastOfType (t) END;
    t := Expr.TypeOf (e);
    IF NOT ArrayType.Split (Type.Base(t), index, elem) THEN RETURN NIL END;
    IF (index = NIL) THEN
      (* LAST (open array value) => try for constant open array *)
      e := Expr.ConstValue (e);
      IF (e = NIL) THEN RETURN NIL END;
      IF ArrayExpr.GetBounds (e, min, max)
        THEN RETURN IntegerExpr.New (max);
        ELSE RETURN NIL;
      END;
    END;
    RETURN LastOfType (t);
  END Fold;

PROCEDURE LastOfType (t: Type.T): Expr.T =
  VAR min, max: Target.Int;   elem, t_base: Type.T;
  BEGIN
    IF ArrayType.Split (Type.Base(t), t, elem) AND (t = NIL) THEN
      RETURN NIL;
    END;
    t_base:= Type.Base (t);
    IF Type.GetBounds (t, min, max) THEN
      IF t_base = Int.T
        THEN RETURN IntegerExpr.New (max);
        ELSE RETURN EnumExpr.New (t, max);
      END;
    ELSIF t_base = Reel.T THEN
      RETURN ReelExpr.New (Target.Real.max, ReelExpr.Precision.Short);
    ELSIF t_base = LReel.T THEN
      RETURN ReelExpr.New (Target.Longreal.max, ReelExpr.Precision.Long);
    ELSIF t_base = EReel.T THEN
      RETURN ReelExpr.New (Target.Extended.max, ReelExpr.Precision.Extended);
    ELSE
      RETURN NIL;
    END;
  END LastOfType;

PROCEDURE Initialize () =
  BEGIN
    Z := CallExpr.NewMethodList (1, 1, TRUE, FALSE, FALSE, NIL,
                                 First.TypeOf,
                                 CallExpr.NotAddressable,
                                 Check,
                                 Prep,
                                 Compile,
                                 CallExpr.NoLValue,
                                 CallExpr.NoLValue,
                                 CallExpr.PrepNoBranch,
                                 CallExpr.NoBranch,
                                 Fold,
                                 CallExpr.IsNever, (* writable *)
                                 CallExpr.IsNever, (* designator *)
                                 CallExpr.NotWritable (* noteWriter *));
    Procedure.Define ("LAST", Z, TRUE,
                      isBounded := TRUE, isFunctional := TRUE);
  END Initialize;

BEGIN
END Last.
