/*
  READ.C:  Routines for tokenising the input stream.
  These routines have been taken from the Version 1 of read(X), and
  hence do not match the new tokeniser's scanning philosophy.  Rewriting
  to match would improve efficiency and remove peekch() calls.

  Different character sets have been catered for.  The Prolog internal
  character set is implicitly ASCII.  If the host machine wants EBCDIC,
  there is no problem:  all necessary conversions are macro-expanded.
  For example, character constants are expanded into the right
  character set.  And, the argument to tokenread() is always in ASCII
  since it was passed from Prolog.
*/


#include "zip.h"

#define peekchtype() (chtype[peekch()])


char chtype[] =
{
  BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK,
  BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK,
  BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK, BLANK,
  BLANK, BLANK, EOFCH, BLANK, BLANK, BLANK, BLANK, BLANK,
  BLANK, SOLCH, STRQT, SYMCH, SYMCH, COMCH, SYMCH, ATOQT,
  PUNCH, PUNCH, SYMCH, SYMCH, SOLCH, SYMCH, SYMCH, SYMCH,
  DIGIT, DIGIT, DIGIT, DIGIT, DIGIT, DIGIT, DIGIT, DIGIT,
  DIGIT, DIGIT, SYMCH, SOLCH, SYMCH, SYMCH, SYMCH, SYMCH,
  SYMCH, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE,
  UCASE, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE,
  UCASE, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE, UCASE,
  UCASE, UCASE, UCASE, PUNCH, SYMCH, PUNCH, SYMCH, ULINE,
  SYMCH, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE,
  LCASE, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE,
  LCASE, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE, LCASE,
  LCASE, LCASE, LCASE, PUNCH, PUNCH, PUNCH, SYMCH, BLANK
};


/*
   Return the next unsigned integer (base 1..10) or character code (0'c).
   Given first digit from tokeniser.  The caller will getch() for tokeniser.
*/

static int readinteger(b)
int b;
{
  int d,n;
  n = 0;
  b -= convch('0');
  if (peekch() == convch('\'')) skipch();
  else
  {
    n = b;
    b = 10;
  }
  if (b == 0) n = getch();
  else
  {
    while(peekchtype() == DIGIT)
    {
      d = getch() - convch('0');
      if (d < b) n = (b * n) + d; else break;
    }
  }
  return(makeword(INT, val(n)));
}


/* read all the characters of an atom or variable up to the buffer limit */

static int readatom(c)
int c;
{
  int ok;
  initbuff();
  ok = keepchar(c);
  while (ok && (peekchtype() <= ULINE)) ok = keepchar(getch());
  makeblock();
  return(findatom());
}


/*
  Read all of the characters of a quoted atom or string up to limit or eof.
  The quote argument is in the implicit character set.
*/

static readquoted(q)
int q;
{
  char scan;
  int c;
  initbuff();
  do
  {
    scan = (peekchtype() != EOFCH);
    if (scan)
    {
      c = getch();
      if (chtype[c] != q) scan = keepchar(c);
      else if (peekchtype() == q) scan = keepchar(getch());
      else scan = FALSE;
    }
  } while (scan);
  makeblock();
}


/* read all the characters of a comment */

static readcomment()
{
  char done;
  int c;
  done = FALSE;
  do
  {
    if (peekchtype() == EOFCH) done = TRUE;
    else
    { c = getch();
      if (c == convch('*') && (peekch() == convch('/')))
      {
        skipch();
        done = TRUE;
      }
    }
  } while (!done);
}


/*  Read and return a sign atom */

static int readsymbol(c)
int c;
{
  int ok;
  initbuff();
  ok = keepchar(c);
  while (ok && (peekchtype()== SYMCH)) ok = keepchar(getch());
  makeblock();
  return(findatom());
}


/*  Read something beginning with '/', returning an atom or a blank */

static int readsolidus(c)
int c;
{
  if (peekch() == convch('*'))
  {
    readcomment();
    return(makeword(INT,32));
  }
  else return(readsymbol(c));
}

/*  Read something beginning with '.', returning an atom or a blank */

static int readfullstop(c)
int c;
{
  if (peekchtype() == BLANK || peekchtype() == EOFCH) return(makeword(INT,32));
  else return(readsymbol(c));
}


/*
  Read and return a token.
  Given its first character in the implicit character set.
*/

int tokenread(c)
int c;
{
  switch(chtype[c])
  {
    case BLANK:
      while (peekchtype() == BLANK) skipch();
      return(makeword(INT,32));
    case COMCH:
      while (TRUE)
      {
        if (peekchtype() == EOFCH) return(makeword(INT,32));
        c = getch();
        if (c == CHPRNL) return(makeword(INT,32));
      }
    case DIGIT:
      return(readinteger(c));
    case LCASE:
    case UCASE:
    case ULINE:
      return(readatom(c));
    case SYMCH:
      if (c == convch('/')) return(readsolidus(c));
      else if (c == convch('.')) return(readfullstop(c));
      else return(readsymbol(c));
    case ATOQT:
      readquoted(convch(ATOQT));
      return(findatom());
    case STRQT:
      readquoted(convch(STRQT));
      return(makebox());
  }
}
