/*
    RESIDENT PROLOG COMPILER
    Grammar Rule preprocessor
    Arithmetic preprocessor
    Fast compiler for non-atomic unit clauses
    Normal compiler
*/


/* Grammar Rule preprocessing */

g_head((LP,List),S,SR,H) :- !, append(List,SR,List2), extend([S,List2],LP,H).
g_head(LP,S,SR,H) :- extend([S,SR],LP,H).

g_body(!,S,S,!) :- !.
g_body([],S,S1,S=S1) :- !.
g_body([X],S,SR,'C'(S,X,SR)) :- !.
g_body([X|R],S,SR,('C'(S,X,SR1),RB)) :- !, g_body(R,SR1,SR,RB).
g_body({T},S,S,T) :- !.
g_body((T,R),S,SR,(Tt,Rt)) :- !, g_body(T,S,SR1,Tt), g_body(R,SR1,SR,Rt).
g_body((T;R),S,SR,(Tt;Rt)) :- 
	!,
	g_body(T,S,S1,T1),
	g_fill(S,SR,S1,T1,Tt),
	g_body(R,S,S2,R1),
	g_fill(S,SR,S2,R1,Rt).
g_body(T,S,SR,Tt) :- extend([S,SR],T,Tt).

g_fill(S,SR,S1,T,(T,SR=S)) :- S1 == S, !.
g_fill(S,SR,SR,T,T).

extend(M,OT,NT) :- OT =.. OL, append(OL,M,NL), NT =.. NL.

append([],L,L) :- !.
append([H|T],L,[H|R]) :- append(T,L,R).

g_tidy((P1;P2),(Q1;Q2)) :- !, g_tidy(P1,Q1), g_tidy(P2,Q2).
g_tidy(((P1,P2),P3),Q) :- g_tidy((P1,(P2,P3)),Q).
g_tidy((P1,P2),(Q1,Q2)) :- !, g_tidy(P1,Q1), g_tidy(P2,Q2).
g_tidy(A,A).

'C'([X|S],X,S).

phrase(PD,WL) :- g_body(PD,WL,[],G), !, call(G).

/* ZIP Clause Preprocessor */

expand((H:-X),(H:-Z)) :- cb(X,[],Z,[],_), !.
expand(X,X).

cb(V,L,[V|C],C,L) :- var(V), !.
cb((A,B),L0,C0,C1,L1) :- !, cb(A,L0,C0,Cn,Ln), cb(B,Ln,Cn,C1,L1).
cb((A;B),L0,[(A1;B1)|Z],Z,L0) :- !, cb(A,[],A1,[],_), cb(B,[],B1,[],_).
cb((A->B),L0,[(A1->B1)|Z],Z,L0) :- !, cb(A,[],A1,[],_), cb(B,[],B1,[],_).
cb(A,L0,C0,C1,L1) :- cg(A,L0,C0,C1,L1).

cg(is(X,Y),L0,C0,C5,L) :- var(X), asschk(X,L0,V), !, c(Y,C0,C1,C1,C2,L0,L), C2 = ['%pushv%'(V), '%inteq%'|C5].
cg(is(X,Y),L0,C0,C5,[[X|X]|L]) :- var(X), !, c(Y,C0,C1,C1,C2,L0,L), C2 = ['%result%'(X)|C5].
cg(is(X,Y),L0,C0,C5,L) :- !, c(X,C0,C1,C2,C3,L0,L1), c(Y,C1,C2,C3,C4,L1,L), C4 = ['%inteq%'|C5].
cg(E,L0,C0,C5,L) :- compop(E,X,Y,Op,_), !, c(X,C0,C1,C2,C3,L0,L1), c(Y,C1,C2,C3,C4,L1,L), C4 = [Op|C5].
cg(G,L,[G|X],X,L).

c(I,E,E,[C|C1],C1,L,L) :- integer(I), !, pushint(I,C).
c(V,E0,E1,['%pushv%'(Z)|C1],C1,L0,L1) :- var(V), !, assoc(V,L0,E0,E1,Z,L1).
c(Exp,E0,E2,C0,C3,L0,L2) :- binop(Exp,A,B,Op,_), !, c(A,E0,E1,C0,C1,L0,L1), c(B,E1,E2,C1,C2,L1,L2), C2 = [Op|C3].
c(Exp,E0,E1,C0,C2,L0,L1) :- unop(Exp,A,Op,_), !, c(A,E0,E1,C0,C1,L0,L1), C1 = [Op|C2].
c(Exp,[is(X,Exp)|Z],Z,['%pushv%'(X)|C],C,L,L).

pushint(I,'%pushb%'(I)) :- I >= 0, I =< 255, !.
pushint(I,'%pushi%'(I)).

/* an association list holds eval'd var pairs */

asschk(V1,[[V|V2]|_],V2) :- V1 == V, !.
asschk(V1,[_|L],V2) :- asschk(V1,L,V2).

assoc(V1,[],['%eval%'(V1,V2)|C],C,V2,[[V1|V2]]) :- !.
assoc(V1,[[Va|Vb]|L],C,C,Vb,[[Va|Vb]|L]) :- V1 == Va, !.
assoc(V1,[X|L],C0,C1,V2,[X|L1]) :- assoc(V1,L,C0,C1,V2,L1).

binop(X+Y,X,Y,'%add%',add).
binop(X-Y,X,Y,'%sub%',sub).
binop(X/Y,X,Y,'%div%',div).
binop(X*Y,X,Y,'%mul%',mul).
binop(X mod Y,X,Y,'%mod%',mod).
binop(X /\ Y,X,Y,'%and%',and).
binop(X \/ Y,X,Y,'%or%',or).
binop(<<(X,Y),X,Y,'%shl%',shl).
binop(>>(X,Y),X,Y,'%shr%',shr).

unop(+X,X,true,_).
unop(-X,X,'%neg%',neg).
unop(\(X),X,'%not%',not).

compop(=:=(X,Y),X,Y,'%inteq%',eq).
compop(=\=(X,Y),X,Y,'%intne%',ne).
compop(X<Y,X,Y,'%intlt%',lt).
compop(X>Y,X,Y,'%intgt%',gt).
compop(X=<Y,X,Y,'%intle%',le).
compop(X>=Y,X,Y,'%intge%',ge).


/*
  ZIP compiler in Prolog
*/
/* Fast compilation and assertion of unit clauses */
/* Not quite speeded up yet. */

compassunit(C,N,D) :-
	primcom0,
	functor(C,F,Arity),
	do_var_each(0,Arity,head,outer,C,VL,VL1),
	assign_var_types(VL,0,NumLocals,0,NumTemps),
	actsize(A),		
	LocalOrigin is A+Arity,
	FS is LocalOrigin+NumLocals,
	MaxFS is FS+NumTemps,
	assign_offsets(VL,LocalOrigin,FS),
	c_term(0,Arity,0,C,head,outer,Code,[return(FS)]),
	primcom0,
	strad(Code,0,Code1),
	assemble(Code1,Bytes,[],0),
	arg(1,C,A1),
	findkey(A1,Key,Kind),
	primcom3(F,Arity,Key,Kind,MaxFS,Bytes,N,D),
	!.

/*  Now for the slow compiler */

legal_clause(C) :-  dest_clause(C,H,_), test_head(H), !.
legal_clause(_) :- primcom12.

test_head(H) :- atom(H), !, test_proc(H,0).
test_head(H) :- simpleterm(H), warning(48,H), !, fail.
test_head(H) :- functor(H,F,A), test_proc(F,A).

test_proc(F,A) :- primcom11(F,A).
test_proc(F,A) :- warning(45,F/A), fail.

dest_clause((H:-B),H,B) :- !.
dest_clause(H,H,true).

findkey(0,_,_,_) :- !.
findkey(_,H,K,L) :- arg(1,H,A), findkey(A,K,L).

findkey(V,_,_) :- functor(V,'$VAR',10), !.
findkey(A,A,_) :- atomic(A), !.
findkey(A,F,N) :- functor(A,F,N), !.
findkey(_,_,_).


doassert(Clause,N,D) :-
	dest_clause(Clause,Head,Body),
	functor(Head,F,Arity),
	do_var_each(0,Arity,head,outer,Head,VL,VL1),
	do_body_vars(Body,NewBody,last,VL1,[]),
	assign_var_types(VL,0,NumLocals,0,NumTemps),
	actsize(A),		
	LocalOrigin is A+Arity,
	FS is LocalOrigin+NumLocals,
	MaxFS is FS+NumTemps,
	assign_offsets(VL,LocalOrigin,FS),
	c_term(0,Arity,0,Head,head,outer,Code,BodyCode),
	gen_body_code(NewBody,FS,BodyCode),
	conseq(Code,OptCode),
	primcom0,
	strad(OptCode,0,C),
	labadd(C,0),
	assemble(C,Bytes,[],0),
	findkey(Arity,Head,Key,Kind),
	primcom3(F,Arity,Key,Kind,MaxFS,Bytes,N,D),
	!.

actsize(8).			  

/* Variable handling tables */

var_bits(head,outer,'$VAR'(1, _, _, _, _, _, _, _, O, _), I) :- !,
	(   nonvar(O), !	
	;   actsize(A),	
	    O is A+I-1 ).	
var_bits(head,inner,'$VAR'(_, 1, _, _, _, _, _, _, _, _), _) :- !.
var_bits(body,outer,'$VAR'(_, _, 1, _, _, _, _, _, _, _), _) :- !.
var_bits(body,inner,'$VAR'(_, _, _, 1, _, _, _, _, _, _), _) :- !.
var_bits(last,outer,'$VAR'(_, _, _, _, 1, _, _, _, _, _), _) :- !.
var_bits(last,inner,'$VAR'(_, _, _, _, _, 1, _, _, _, _), _).

var_index(multiple,V,Value) :- !, arg(7,V,Value).	
var_index(var_type,V,Value) :- !, arg(8,V,Value).
var_index(f_offset,V,Value) :- !, arg(9,V,Value).
var_index(first_fl,V,Value) :-    arg(10,V,Value).


assign_var_type('$VAR'(_, _, 0, _, 0, _, 0,  void, _, _), 0, 0) :- !.
assign_var_type('$VAR'(0, 1, 0, 0, 0, 0, 1,  temp, _, _), 0, 1) :- !.
assign_var_type('$VAR'(0, 0, _, _, _, _, _, b_loc, _, _), 1, 0) :- !.
assign_var_type('$VAR'(0, 1, _, _, _, _, 1, local, _, _), 1, 0) :- !.
assign_var_type('$VAR'(1, _, _, _, _, _, 1,   arg, _, _), 0, 0).



do_var_each(N,N,_,_,_,VL,VL) :- !.
do_var_each(I,N,Position,Depth,Term,VL0,VL2) :-
	succ(I,J),
	arg(J,Term,Arg),
	do_var(Position,Depth,Arg,J,VL0,VL1),
	do_var_each(J,N,Position,Depth,Term,VL1,VL2).

do_var(Posn,IorO,V,I,[V|VL],VL) :-  var(V), !, var_bits(Posn,IorO,V,I).
do_var(Posn,IorO,V,I,VL,VL) :- var_bits(Posn,IorO,V,I), !, var_index(multiple,V,1).
do_var(_,_,X,_,VL,VL) :- atomic(X), !.			
do_var(Posn,_,T,_,VL0,VL1) :- functor(T,_,A), do_var_each(0,A,Posn,inner,T,VL0,VL1).

do_body_vars([],[],_,V,V) :- !.
do_body_vars(Var,call(Var),Posn,VL0,VL1) :-
	var(Var), !,
	do_var_each(0,1,Posn,outer,call(Var),VL0,VL1).	
do_body_vars([P|Q],[NewP|NewQ],Posn,VL0,VL2) :- !,		
	do_body_vars(P,NewP,body,VL0,VL1),
	do_body_vars(Q,NewQ,Posn,VL1,VL2).
do_body_vars((P;Q),'%disjunct%'(NewP,NewQ,Vars),Posn,VL0,VL2) :- !,	
	do_body_vars(P,NewP,Posn,VL0,VL1),
	do_body_vars(Q,NewQ,Posn,VL1,VL2),
	real_list(VL0,VL2,Vars).
do_body_vars((P->Q),'%localcut%'(NewP,NewQ,V),Posn,[V|VL0],VL2) :- !,
	V = '$VAR'(0,1,1,0,0,0,1,local,_,_),	
	do_body_vars(P,NewP,body,VL0,VL1),	
	do_body_vars(Q,NewQ,Posn,VL1,VL2).
do_body_vars(V,call(V),P,VL0,VL1) :- functor(V,'$VAR',10), !, do_var_each(0,1,Posn,outer,call(V),VL0,Vl1).
do_body_vars(Goal,Goal,Posn,VL0,VL1) :- nonvar(Goal), !, functor(Goal,_,A), do_var_each(0,A,Posn,outer,Goal,VL0,VL1).

	
real_list(VL0,VLn,[]) :- VL0 == VLn, !. 
real_list([Var|VL0],VLn,[Var|List]) :- real_list(VL0,VLn,List).


/* This procedure works on the list of variables which is built by do_vars.
   (using an accumulator for each).  */

assign_var_types([],Locals,Locals,Temps,Temps) :- !.
assign_var_types([V|Z],L0,L2,T0,T2) :-
	assign_var_type(V,L_inc,T_inc),
	L1 is L0 + L_inc,
	T1 is T0 + T_inc,
	assign_var_types(Z,L1,L2,T1,T2).



assign_offsets([],_,_).
assign_offsets([V|Z],LocOff,TempOff) :-
	var_index(var_type,V,Type),			
	assign_offset(Type,V,LocOff,NewLoc,TempOff,NewTemp),
	assign_offsets(Z,NewLoc,NewTemp).
	
assign_offset(local,V,L0,L1,T,T) :- !, succ(L0,L1), var_index(f_offset,V,L0).
assign_offset(b_loc,V,L0,L1,T,T) :- !, succ(L0,L1), var_index(f_offset,V,L0).
assign_offset(temp,V,L,L,T0,T1) :- !, succ(T0,T1), var_index(f_offset,V,T0).
assign_offset(void,V,L,L,T,T) :- var_index(f_offset,V,0), !.
assign_offset(_,_,L,L,T,T).		


/* Body code generation.  FS is the size of frame excluding any
   temporary variables.  */

gen_body_code(true,FS,[return(FS)]) :- !.
gen_body_code(B,FS,[enter(FS)|BodyCode]) :-
	c_goals(B,FS,last,BodyCode,[exit]).

c_goals([Goal],FS,Posn,Code0,Code1) :- c_goal(Posn,Goal,FS,Code0,Code1).
c_goals([Goal|Z],FS,Posn,Code0,Code3) :- !,
	c_goal(body,Goal,FS,Code0,Code2),
	c_goals(Z,FS,Posn,Code2,Code3).

c_goal(body,Goal,FS,Code0,Code1) :- open_code(Goal,FS,body,Code0,Code1), !.
c_goal(body,Goal,_,Code0,Code1) :- !,
	functor(Goal,F,N),
	c_term(0,N,0,Goal,body,outer,Code0,[call(F,N)|Code1]).
c_goal(last,Goal,FS,Code0,Code1) :- open_code(Goal,FS,last,Code0,Code1), !.
c_goal(last,Goal,_,Code0,Code1) :-
	functor(Goal,F,N),
	c_term(0,N,0,Goal,last,outer,Code0,[depart(F,N)|Code1]).



c_term(N,N,_,_,_,_,L,L) :- !.
c_term(I,N,InPop,Term,Position,Depth,L0,L2) :-
	succ(I,J), arg(J,Term,Arg), L is J-InPop,
	c_arg(Arg,Position,Depth,L,N,L0,L1),
	c_term(J,N,InPop,Term,Position,Depth,L1,L2).

c_arg('$VAR'(_,_,_,_,_,_,_,Type,Offset,FirstFlag),Position,Depth,_,_,Instr,Z) :- !,
	c_arg_var(Type,Offset,FirstFlag,Position,Depth,Instr,Z).
c_arg([],_,_,_,_,[constnil|Z],Z) :- !.
c_arg(Int,_,_,_,_,[immed(Int)|Z],Z) :- integer(Int), 0 =< Int, Int =< 255, !.
c_arg(A,_,_,_,_,[constant(A)|Z],Z) :- atomic(A), !.
c_arg(Term,Position,_,N,N,[lastconslist|L0],L1) :-
	functor(Term,'.',2), !,
	c_term(0,2,0,Term,Position,inner,L0,L1).
c_arg(Term,Position,_,N,N,[lastfunctor(Name,Arity)|L0],L1) :- !,
	functor(Term,Name,Arity),
	c_term(0,Arity,0,Term,Position,inner,L0,L1).
c_arg(Term,Position,_,_,_,[conslist|L0],L1) :-
	functor(Term,'.',2), !,
	choose_pop(Position,Pop),
	c_term(0,2,1,Term,Position,inner,L0,[Pop|L1]).
c_arg(Term,Position,_,_,_,[functor(Name,Arity)|L0],L1) :-
	functor(Term,Name,Arity),
	choose_pop(Position,Pop),
	c_term(0,Arity,1,Term,Position,inner,L0,[Pop|L1]).

choose_pop(head,pop) :- !.
choose_pop(_,poparg).


/* Generate instruction for variable */


c_arg_var(void,_,_,_,_,[void|Z],Z) :- !.
c_arg_var(arg,Offset,FirstFlag,head,outer,[void|Z],Z) :- var(FirstFlag), !, FirstFlag=1.
c_arg_var(arg,Offset,_,body,outer,[vararg(Offset)|Z],Z) :- !.
c_arg_var(arg,Offset,_,last,outer,[vararg(Offset)|Z],Z) :- !.
c_arg_var(arg,Offset,_,_,_,[var(Offset)|Z],Z) :- !.
c_arg_var(b_loc,Offset,FirstFlag,body,outer,[firvararg(Offset)|Z],Z) :-
	var(FirstFlag), !,
	FirstFlag = ff(_).	
c_arg_var(b_loc,Offset,FirstFlag,body,inner,[firstvar(Offset)|Z],Z) :-
	var(FirstFlag), !,
	FirstFlag = ff(_).	
c_arg_var(b_loc,Offset,FirstFlag,last,outer,[glofirvar(Offset)|Z],Z) :-
	var(FirstFlag), !,	
	FirstFlag = ff(1).	
c_arg_var(b_loc,Offset,ff(FirstFlag),last,outer,[glovar(Offset)|Z],Z) :-
	var(FirstFlag), !,	
	FirstFlag = 1.
c_arg_var(_,Offset,FirstFlag,_,_,[firstvar(Offset)|Z],Z) :- var(FirstFlag), !, FirstFlag=1.
c_arg_var(_,Offset,_,body,outer,[vararg(Offset)|Z],Z) :- !.
c_arg_var(_,Offset,_,last,outer,[vararg(Offset)|Z],Z) :- !.
c_arg_var(_,Offset,_,_,_,[var(Offset)|Z],Z) :- !.



open_code(true,_,_,C,C) :- !.
open_code(!,FS,_,[cut(FS)|C],C) :- !.
open_code(fail,_,_,[fail|C],C) :- !.
open_code(succ(X,Y),_,BL,C0,C1) :- !, c_term(0,2,0,succ(X,Y),BL,outer,C0,[prosucc|C1]).
open_code(=(X,Y),FS,BL,C0,C1) :- !, c_term(0,2,0,=(X,Y),BL,outer,C0,[proequal|C1]).
open_code(arg(X,Y,Z),FS,BL,C0,C1) :- !, c_term(0,3,0,arg(X,Y,Z),BL,outer,C0,[proarg|C1]).
open_code(functor(X,Y,Z),FS,BL,C0,C1) :- !, c_term(0,3,0,functor(X,Y,Z),BL,outer,C0,[profunctor|C1]).
open_code('%pushb%'(I),_,_,[pushb(I)|Z],Z) :- !.
open_code('%pushi%'(I),_,_,[pushi(I)|Z],Z) :- !.
open_code('%result%'('$VAR'(_,_,_,_,_,_,_,T,O,F)),_,_,[Code|Z],Z) :- !,
	c_arg_var(T,O,F,body,outer,I,_),
	result_code(I,Code).
open_code('%eval%'('$VAR'(_,_,_,_,_,_,_,T1,O1,FF1),'$VAR'(_,_,_,_,_,_,_,T2,O2,FF2)),_,_,[eval(O1,O2)|Z],Z) :-
	c_arg_var(T1,O1,FF1,body,outer,_,_),
	c_arg_var(T2,O2,FF2,body,outer,_,_), !.
open_code('%disjunct%'('%localcut%'(P,Q,V),R,Vars),FS,BL,[savel(V_offset)|C0],C5) :- !,
	var_index(f_offset,V,V_offset),
	init_vars(Vars,C0,[disjunct(Offset1)|C1]),
	c_goals(P,FS,body,C1,[localcut(V_offset)|C2]),
	c_goals(Q,FS,BL,C2,[continue(Offset2),label(Offset1),endor|C4]),
	c_goals(R,FS,BL,C4,[label(Offset2)|C5]).
open_code('%disjunct%'(P,Q,Vars),FS,BL,C0,C4) :- !,		
	init_vars(Vars,C0,[disjunct(Offset1)|C1]),
	c_goals(P,FS,BL,C1,[continue(Offset2),label(Offset1),endor|C3]),
	c_goals(Q,FS,BL,C3,[label(Offset2)|C4]).
open_code('%localcut%'(P,Q,V),FS,BL,[savel(V_offset)|C0],C2) :- !,
	var_index(f_offset,V,V_offset),
	c_goals(P,FS,body,C0,[localcut(V_offset)|C1]),
	c_goals(Q,FS,BL,C1,C2).
open_code(put([X]),FS,BL,C0,C1) :- !, c_goal(BL,put(X),FS,C0,C1).
open_code(Goal,_,BL,[ProInst|C],C) :-
	var_instr(Goal,V,ProInst,Offset), !, functor(V,'$VAR',10),
	var_index(first_fl,V,Fflag), nonvar(Fflag),
	var_index(f_offset,V,Offset), Offset =\= 0.
open_code(F,_,_,[Op|Z],Z) :- binop(_,_,_,F,Op), !.
open_code(F,_,_,[Op|Z],Z) :- unop(_,_,F,Op), !.
open_code(F,_,_,[Op|Z],Z) :- compop(_,_,_,F,Op), !.

result_code([firvararg(A)|_],firstresult(A)) :- !.
result_code([vararg(A)|_],result(A)) :- !.


init_vars([],C,C).
init_vars(['$VAR'(_,_,_,_,_,_,_,b_loc,Offset,ff(_))|Z],[initvar(Offset)|C0],C1) :- !,
	init_vars(Z,C0,C1).
init_vars([_|Z],C0,C1) :- init_vars(Z,C0,C1).


var_instr(call(V),V,callx(N),N).
var_instr(var(V),V,provar(N),N).	
var_instr(atom(V),V,proatom(N),N).
var_instr(integer(V),V,proint(N),N).
var_instr(nonvar(V),V,prononvar(N),N).
var_instr(atomic(V),V,proatomic(N),N).
var_instr('%pushv%'(V),V,pushv(N),N).


/* Final processing:  peephole optimiser and assembler */

conseq([voidn(M),void|C0], C1):- !, N is M+1, conseq([voidn(N)|C0], C1).
conseq([void,void|C0],C1):- !, conseq([voidn(2)|C0],C1).
conseq([I|C0],[I|C1]) :- !, conseq(C0,C1).
conseq([], []).

instr(call(_,_),2) :- !.
instr(label(_),0) :- !.
instr(pad(X),X) :- !.
instr(X,N) :- functor(X,_,A), succ(A,N).

strad([],_,[]) :- !.
strad([H|T],P,[H|Z]) :- instr(H,I), P1 is P+I, stradx(P1,P2), !, strad(T,P2,Z).
strad([H|T],P,[pad(N),H|Z]) :- !, instr(H,I), N is 4-P, strad(T,I,Z).

stradx(0,0).
stradx(1,1).
stradx(2,2).
stradx(3,3).
stradx(4,0).
% fails for X > 4

labadd([],_) :- !.
labadd([label(X)|T],X) :- !, labadd(T,X).
labadd([H|T],N) :- instr(H,I), N1 is N + I, labadd(T,N1).



assemble([],L,L,_) :- !.
assemble([H|T],L1,L2,P1) :- ass(H,L1,L3,P1), instr(H,P2), P3 is P1+P2, assemble(T,L3,L2,P3).

ass(constant(A),[11,N|Z],Z,_) :- !, primcom1(0,A,_,N).
ass(functor(F,A),[9,N,A|Z],Z,_) :- !, primcom1(2,F,A,N).
ass(lastfunctor(F,A),[10,N,A|Z],Z,_) :- !, primcom1(2,F,A,N).
ass(depart(F,A),[16,N,A|Z],Z,_) :- !, primcom1(1,F,A,N).
ass(call(F,A),[17,N|Z],Z,_) :- !, primcom1(1,F,A,N).
ass(pushi(I),[43,N|Z],Z,_) :- !, primcom1(0,I,_,N).
ass(pad(1),[0|Z],Z,_).
ass(pad(2),[0,0|Z],Z,_).
ass(pad(3),[0,0,0|Z],Z,_).
ass(label(_),Z,Z,_).
ass(immed(A),[27,A|Z],Z,_).
ass(constnil,[35|Z],Z,_).
ass(conslist,[33|Z],Z,_).
ass(lastconslist,[34|Z],Z,_).
ass(void,[7|Z],Z,_).
ass(voidn(A),[32,A|Z],Z,_).
ass(firstvar(A),[5,A|Z],Z,_).
ass(glofirvar(A),[29,A|Z],Z,_).
ass(glovar(A),[30,A|Z],Z,_).
ass(var(A),[4,A|Z],Z,_).
ass(vararg(A),[2,A|Z],Z,_).
ass(firvararg(A),[40,A|Z],Z,_).
ass(pop,[1|Z],Z,_).
ass(poparg,[3|Z],Z,_).
ass(callx(A),[26,A|Z],Z,_).
ass(return(A),[13,A|Z],Z,_).
ass(enter(A),[12,A|Z],Z,_).
ass(exit,[25|Z],Z,_).
ass(cut(A),[15,A|Z],Z,_).
ass(fail,[19|Z],Z,_).
ass(savel(A),[14,A|Z],Z,_).
ass(disjunct(A),[18,N|Z],Z,P) :- N is A - (P + 2).
ass(continue(A),[8,N|Z],Z,P) :- N is A - (P + 2).
ass(endor,[28|Z],Z,_).
ass(localcut(A),[31,A|Z],Z,_).
ass(initvar(A),[63,A|Z],Z,_).
ass(provar(A),[20,A|Z],Z,_).
ass(prononvar(A),[21,A|Z],Z,_).
ass(proatom(A),[22,A|Z],Z,_).
ass(proint(A),[23,A|Z],Z,_).
ass(proatomic(A),[39,A|Z],Z,_).
ass(prosucc,[24|Z],Z,_).
ass(proarg,[36|Z],Z,_).
ass(profunctor,[37|Z],Z,_).
ass(proequal,[38|Z],Z,_).
ass(eval(A,B),[41,A,B|Z],Z,_).
ass(pushb(A),[42,A|Z],Z,_).
ass(pushv(A),[44,A|Z],Z,_).
ass(firstresult(A),[6,A|Z],Z,_).
ass(result(A),[45,A|Z],Z,_).
ass(add,[46|Z],Z,_).
ass(sub,[47|Z],Z,_).
ass(mul,[48|Z],Z,_).
ass(div,[49|Z],Z,_).
ass(mod,[50|Z],Z,_).
ass(shr,[51|Z],Z,_).
ass(shl,[52|Z],Z,_).
ass(and,[53|Z],Z,_).
ass(or,[54|Z],Z,_).
ass(not,[55|Z],Z,_).
ass(neg,[56|Z],Z,_).
ass(eq,[57|Z],Z,_).
ass(ne,[58|Z],Z,_).
ass(lt,[59|Z],Z,_).
ass(le,[60|Z],Z,_).
ass(gt,[61|Z],Z,_).
ass(ge,[62|Z],Z,_).
