/*  The ZAP Assembler */

#include "zap.h"

#ifdef ebcdic
extern char etoa[];
#endif

#define PSIKI		(-1)
#define PSIKF 	(-2)
#define PSIKA 	(-3)
#define PSIOP 	(-4)  /* NOT USED ANY MORE */
#define PSIEXR	(-5)
#define PSIG 		(-6)
#define PSII		(-7)
#define PSIA		(-8)
#define PSIP		(-9)
#define PSIF 		(-10)
#define PSIXRT	(-11)
#define PSIENDCL	(-12)
#define PSICL		(-13)
#define PSIPRIM	(-14)
#define PSIVIS 	(-15)
#define PSIUPD (-16)

char *op[] =
{
  "keyi", "keyf", "keya", "op",
  "endxrtable", "g", "i", "a", "p", "f", "xrtable", "endclause", "clause",
  "primitive", "visible",
  "updatable",
  "pop", "vararg", "poparg", "var", "firstvar", "firstresult", "void", "continue",
  "functor", "lastfunctor", "constant", "enter", "return", "savel", "cut",
  "glovar", "localcut", "voidn",
  "proint", "prosucc", "exit", "callx", "immed", "endor", "glofirvar",
  "depart", "call", "disjunct", "fail", "provar", "prononvar", "proatom",
  "conslist", "lastconslist", "constnil",
  "proarg", "profunctor", "proequal","proatomic","firvararg",
  "eval", "pushb", "pushi", "pushv", "result", "add", "sub", "mul", "div",
  "mod", "shr", "shl", "and", "or", "not", "neg", "eq", "ne", "lt", "le",
  "gt", "ge","initvar","pad",
  "Z"
};

int opnum[] =
{
  PSIKI, PSIKF, PSIKA, PSIOP,
  PSIEXR, PSIG, PSII, PSIA, PSIP, PSIF, PSIXRT, PSIENDCL, PSICL,
  PSIPRIM, PSIVIS,
  PSIUPD,
  IPOP, VARARG, POPARG, IVAR, FIRSTVAR, FIRRESULT, VOID, ICONTINUE,
  IFUNCTOR, LASTFUNCTOR, CONSTANT, ENTER, IRETURN, SAVEL, CUT,
  GLOVAR, LOCALCUT, VOIDN,
  PROINT, PROSUCC, IEXIT, CALLX, IMMED, ENDOR, GLOFIRVAR,
  DEPART, CALL, DISJUNCT, IFAIL, PROVAR, PRONONVAR, PROATOM,
  CONSLIST, LASTCONSLIST, CONSTNIL,
  PROARG, PROFUNCTOR, PROEQUAL,PROAIC,FIRVARARG,
  EVAL, PUSHB, PUSHI, PUSHV, RESULT, ISADD, ISSUB, ISMUL, ISDIV,
  ISMOD, ISSHR, ISSHL, ISAND, ISOR, ISNOT, ISNEG, ISEQ, ISNE,
  ISLT, ISLE, ISGT, ISGE,INITVAR,0,
  -999
};


#define LENNAMEBUF	65
#define MAXNAMEBUF 	&buffname[LENNAMEBUF-1]
#define LENHASH		199
#define MAXHEAP		50000

char l[256], *lp, *buffindx;
int code[256], xr[256], heap[MAXHEAP], maxfs, procword, currkey, xrp, cp, xrin;
int buffhash, currmod, buffname[LENNAMEBUF], hash[LENHASH], syswords[SYSWSIZE];
int topheap;
FILE *file;
int end_of_file;




main(argc,argv)
int argc;
char *argv[];
{
  int i;
  topheap = 0;
  for (i=0; i < LENHASH; i++) hash[i] = TERMIN;
  initstore();
  while (--argc > 0)
  {
    if ((file = fopen(*++argv,"r")) == NULL) zaperror("cannot open input file");
    xrin = FALSE;
    end_of_file = FALSE;
    assemble();
    fclose(file);
  }
#ifdef os370
  file = fopen("prologx.zapimage",
               "w,binary,recfm(u),blksize(6144),lrecl(0),space=(15,15,block)");
#else
  file = fopen("zap.out","w");
#endif
  dumpheap();
  fclose(file);
  exit(0);
}


/* allocate n words from the heap */

int alloc(n)
int n;
{
  int c;
  c = topheap;
  if (topheap + n > MAXHEAP) zaperror("heap too large");
  topheap += n;
  return(c);
}

/* error reporting */

zaperror(s)
char *s;
{
  printf("Error:  %s\n",s);
  exit(0);
}

/* initialise the name buffer */

initbuff()
{
  int i;
  buffhash = 0;
  buffindx = (char *) &buffname[1];
  for (i = 1; i < LENNAMEBUF; i++) buffname[i] = 0;
}


/* convert the name buffer into a block lookalike */

makeblock()
{
  int bc;
  bc = buffindx - (char *) &buffname[1];
  buffname[0] = makeword(BLOCK,bc);
  if (buffhash < 0) buffhash = -buffhash;
  buffhash %= LENHASH;
}


/* keep a character */

keepchar(c)
char c;
{
#ifdef ebcdic
  c = etoa[c];
#endif
  if (buffindx > (char *) MAXNAMEBUF) zaperror("name too long");
  *buffindx++ = c;
  buffhash = (buffhash * 5) ^ c;
}


/* return the atom having the name matching the buffname */

int findatom()
{
  int i,j,m,n,p,s;
  n = blocksize(val(buffname[0]));
  if (tag(hash[buffhash]) == TERMIN) return(hash[buffhash] = makeatom(n));
  else
  {
    p = val(hash[buffhash]);
    while (TRUE)
    {
      i = val(heap[p+ATOCHOFF]);
      j = 0;
      m = n;
      s = TRUE;
      while (s && m--) if (heap[i++] != buffname[j++]) s = FALSE;
      if (s) return(makeword(ATOM,p));
      else if (tag(heap[p+ATOHAOFF]) == TERMIN) return(heap[p+ATOHAOFF] = makeatom(n));
      else p = val(heap[p+ATOHAOFF]);
    }
  }
}


int makeatom(n)
int n;
{
  int c, t, b, a;
  c = alloc(n);
  t = c;
  b = 0;
  while (n--) heap[t++] = buffname[b++];
  a = alloc(ATOMLEN);
  heap[a+ATOHAOFF] = TERMIN;
  heap[a+ATOFUOFF] = TERMIN;
  heap[a+ATOCHOFF] = makeword(BOX,c);
  return(makeword(ATOM,a));
}


/* find a functor */

int findfunct(a,i,need)
int a,i,need;
{
  int p;
  p = heap[val(a)+ATOFUOFF];
  if (tag(p) == TERMIN)
    return((need) ? (heap[val(a)+ATOFUOFF] = makefunct(a,i)) : TERMIN);
  p = val(p);
  while (TRUE)
  {
    if (i == val(heap[p+FUNAROFF])) return(makeword(FUNCTOR,p));
    else if (tag(heap[p+FUNFUOFF]) == TERMIN)
      return((need) ? (heap[p+FUNFUOFF] = makefunct(a,i)) : TERMIN);
    p = val(heap[p+FUNFUOFF]);
  }
}


int makefunct(a,i)
int a,i;
{
  int h;
  h = alloc(FUNCLEN);
  heap[h+FUNATOFF] = a;
  heap[h+FUNAROFF] = makeword(INT,i);
  heap[h+FUNFUOFF] = TERMIN;
  heap[h+FUNPROFF] = TERMIN;
  return(makeword(FUNCTOR,h));
}


/* search for a procedure */
/* functor f, module m, check/construct need, need fixture fix */

int findproc(f,m,need,fix)
int f,m,need,fix;
{
  int p;
  p = heap[val(f)+FUNPROFF];
  if (tag(p) == TERMIN)
    return((need) ? (heap[val(f)+FUNPROFF] = makeproc(m,f,fix)) : TERMIN);
  p = val(p);
  while(TRUE)
  {
    if ((m == heap[p+PROVISOFF]) || (heap[p+PROFLAOFF] & PFOMNI))
      return(makeword(PROC,p));
    else if (tag(heap[p+PROPROOFF]) == TERMIN)
      return((need) ? (heap[p+PROPROOFF] = makeproc(m,f,fix)) : TERMIN);
    p = val(heap[p+PROPROOFF]);
  }
}


int makeproc(m,f,fix)
int m,f,fix;
{
  int a,b;
  a = alloc(PROCLEN);
  heap[a+PRODEFOFF] = heap[a+PROVISOFF] = m;
  heap[a+PROFUNOFF] = f;
  heap[a+PROFLAOFF] = makeword(INT,PFTRANS);
  heap[a+PROPROOFF] = heap[a+PROCLAOFF] = TERMIN;
  if (fix)
  {
    b = alloc(FIXTURELEN);
    heap[b+FIXTABOFF] = makeword(TABLE,FIXTURELEN);
    heap[b+FIXFIROFF] = heap[b+FIXLASOFF] = TERMIN;
    heap[a+PROCLAOFF] = makeword(TABREF,b);
  }
  return(makeword(PROC,a));
}


/*
  read the next line of characters into the line buffer, setting EOF
*/

readchars()
{
  int s, c;
  lp = l;
  s = TRUE;
  while (s)
  {
    if (lp > &l[255]) zaperror("input line exceeds 255");
    c = getc(file);
    if (c == EOF)
    {
      end_of_file = TRUE;
      s = FALSE;
    }
    else if (c == '\n')
    {
      *lp = '\0';
      s = FALSE;
    }
    else *lp++ = c;
  }
  lp = l;
}

readline()
{
  while (TRUE)
  {
    readchars();
    if (((*lp != '\0') && (*lp != ';')) || (end_of_file == TRUE)) return;
  }
}


/* skip spaces in the line buffer */

skipsp()
{
  while (*lp == ' ') lp++;
}


/* check for a delimiter */

inname(c)
char c;
{
  return((c != ' ') && (c != '\0'));
}


/*
  read the next name from the input line into the name buffer.
*/

readname()
{
  int c;
  initbuff();
  skipsp();
  if (*lp == '\'') readquoted();
  else while (inname(c = *lp++)) keepchar(c);
  makeblock();
}


readquoted()
{
  int scan,c;
  lp++;
  do
  {
    scan = (*lp != '\0');
    if (scan)
    {
      c = *lp++;
      if (c != '\'') keepchar(c);
      else if (*lp == '\'') keepchar(*lp++);
      else scan = FALSE;
    }
  } while (scan);
}


/* return the next number from the input line */

int readnum()
{
  int n;
  char c;
  n = 0;
  skipsp();
  if ((*lp < '0') || (*lp > '9')) zaperror("non-digit encountered");
  while (inname(c = *lp++)) n = 10 * n + (c - '0');
  return(n);
}


/* see if the next token is an opcode */

int getcode()
{
  char *p, *pt, t[255], c;
  int i;
  skipsp();
  pt = t;
  while (inname(c = *lp++)) *pt++ = c;
  *pt = '\0';
  i = 0;
  while (opnum[i] != -999)
  {
    p = op[i];
    pt = t;
    while ((c = *pt++) == *p++) if (c == '\0') return(i);
    i++;
  }
  printf("%s\n",t);
  zaperror("cannot identify as opcode");
}


/*  read and return a procedure from the input line */
/* 'fix' is whether the procedure should have a fixture (FALSE for primitives) */

int readproc(fix)
int fix;
{
  readname();
  return(findproc(findfunct(findatom(),readnum(),TRUE),currmod,TRUE,fix));
}

/* read a functor */

int readfunc()
{
  readname();
  return(findfunct(findatom(),readnum(),TRUE));
}


/* assemble the ZIPcodes from the input file */


assemble()
{
  int i, t, g, f, c;
  int pp;
  while (TRUE)
  {
    readline();
    if (end_of_file) return;
    i = getcode();
    printf("%s\n",op[i]);
    i = opnum[i];
    switch(i)
    {
      default:
        zaperror("unknown opcode while reading pseudo-op");
      case PSIPRIM:
        t = readproc(FALSE);
        heap[val(t)+PROCLAOFF] = makeword(INT,readnum());
        heap[val(t)+PROFLAOFF] |= makeword(INT,readnum());
        break;
      case PSIVIS:
        t = readproc(TRUE);
        heap[val(t)+PROFLAOFF] |= 7;
        heap[val(t)+PROFLAOFF] &= ~PFTRANS;
        break;
      case PSIUPD:
        t = readproc(TRUE);
        heap[val(t)+PROFLAOFF] |= 5;
        heap[val(t)+PROFLAOFF] &= ~PFTRANS;
        break;
      case PSICL:
        t = readproc(TRUE);
	if (tag(heap[val(t)+PROCLAOFF]) != TABREF) zaperror("attempt to update primitive");
        maxfs = readnum();
        procword = t;
        currkey = UNDEF;
        c = readclause();
        if ((currmod != heap[val(t)+PRODEFOFF])
          && (heap[val(t)+PROFLAOFF] & PFSACRED))
              zaperror("Attempt to update sacred procedure");
        f = heap[val(t)+PROCLAOFF];
        g = heap[val(f)+FIXLASOFF];
        if (tag(g) == TERMIN)
        {
          heap[val(f)+FIXFIROFF] = heap[val(f)+FIXLASOFF] = c;
          heap[val(c)+CLABACOFF] = f;
        }
        else
        {
          heap[val(c)+CLABACOFF] = g;
          heap[val(g)+CLAFOROFF] = heap[val(f)+FIXLASOFF] = c;
        }
        break;
    }
  }
}


/*  read and pass back a clause, constructing code and XR table */


int readclause()
{
  int i,cl;
  cp = 0;
  do
  {
    readline();
    i = getcode();
    printf("%s\t\t",op[i]);
    i = opnum[i];
    switch(i)
    {
      default:
        zaperror("unknown opcode while reading clause");
      case VOID:
      case IPOP:
      case ENDOR:
      case POPARG:
      case IEXIT:
      case IFAIL:
      case CONSLIST:
      case LASTCONSLIST:
      case CONSTNIL:
      case PROARG:
      case PROFUNCTOR:
      case PROSUCC:
      case PROEQUAL:
      case ISADD:  case ISSUB:  case ISMUL:  case ISDIV:  case ISMOD:
      case ISSHL:  case ISSHR:  case ISAND:  case ISOR:
      case ISNOT:
      case ISNEG:  case ISEQ:   case ISNE:   case ISLT:   case ISLE:
      case ISGT:   case ISGE:
        emit(i);
        break;
      case RESULT:  case FIRRESULT:
      case PUSHV:
      case PUSHB:
      case IVAR:
      case FIRSTVAR:
      case GLOFIRVAR:
      case GLOVAR:
      case ENTER:
      case IRETURN:
      case CALLX:
      case CUT:
      case VOIDN:
      case IMMED:
      case PROVAR:
      case PRONONVAR:
      case PROATOM:
      case PROINT:
      case PROAIC:
      case VARARG:
      case FIRVARARG:
      case SAVEL:
      case DISJUNCT:
      case ICONTINUE:
      case LOCALCUT:
      case INITVAR:
      case CALL:
      case CONSTANT:
      case PUSHI:
        emit(i);
        emit(readnum());
        break;
      case 0:   /* a pad(x) instruction */
        i = readnum();
        while (i--) emit(0);
        break;
      case DEPART:
      case IFUNCTOR:
      case LASTFUNCTOR:
      case EVAL:
        emit(i);
        emit(readnum());
        emit(readnum());
        break;
      case PSIXRT:
        xrp = 0;
        xrin = TRUE;
        break;
      case PSIKF:
        currkey = readfunc();
printf("func key %d\n", val(currkey));
        if (currkey == syswords[SWFUNDOT]) { currkey = makeword(CONS,0);
            printf("mod cons\n"); }
        break;
      case PSIKI:
        currkey = makeword(INT,readnum());
        break;
      case PSIKA:
        readname();
        currkey = findatom();
        break;
      case PSIF:
        exref(readfunc());
        break;
      case PSIP:
        exref(readproc(TRUE));
        break;
      case PSIA:
        readname();
        exref(findatom());
        break;
      case PSII:
        exref(makeword(INT,readnum()));
        break;
      case PSIG:
        exref(syswords[SWNIL]);
        break;
      case PSIEXR:
        xrin = FALSE;
        break;
      case PSIENDCL:
        cl = packup();
        break;
    }
    printf("\n");
  } while (i != PSIENDCL);
  return(cl);
}


/* emit a byte of code */

emit(b)
int b;
{
  if (cp > 255) zaperror("code block too large");
  printf("%d ",b);
  code[cp++] = b;
}


/* insert the next entry in the external references table */

exref(w)
int w;
{
  if (xrin == FALSE) zaperror("illegal insertion into XR table");
  if (xrp > 255) zaperror("XR table too large");
  xr[xrp++] = w;
}


/* construct the clause, XR table, code block, and return clause */

int packup()
{
  int c,b,x,i;
  char *bp;
  if (cp > 0)
  {
    c = alloc(CLAUSELEN + xrp);
    heap[c+CLAKEYOFF] = currkey;
printf("packing key %d\n",val(currkey));
    heap[c+CLAFLAOFF] = makeword(INT,maxfs);
    heap[c+CLAFOROFF] = TERMIN;
    heap[c+CLAPROOFF] = procword;
    heap[c+CLASIZOFF] = makeword(INT,CLAUSELEN+xrp);
    b = alloc(blocksize(cp));
    heap[b] = makeword(BLOCK,cp);
    bp = (char *) &heap[b+1];
    for (i = 0; i < cp; i++) *bp++ = code[i];
    while ((int) bp & 3) *bp++ = '\0';
    heap[c+CLATEXOFF] = makeword(BOX,b);
    if (xrp > 0) for (i = 0; i < xrp; i++) heap[c+CLAUSELEN+i] = xr[i];
    return(makeword(CLAUSE,c));
  }
  else zaperror("attempt to construct clause having empty code block");
}


/* initialising the system word table */

initstore()
{
  insys(SWROOTMOD,"root-module");
  currmod = syswords[SWROOTMOD];
  prsys(SWSTART,"start_handler",0);
  insys(SWUSER,"user");
  insys(SWDOT,".");
  fnsys(SWFUNDOT,".",2);
  fnsys(SWDATABASE,"%database%",3);
  fnsys(SWFUNCOMMA,",",2);
  fnsys(SWSOURCE,"%source%",6);
  insys(SWMINUS,"-");
  insys(SWNOCULP,"no-culprit");
  insys(SWRESLIM,"resource-limit");
  prsys(SWUNKNOWN,"unknown_handler",1);
  prsys(SWISPROC,"is",2);
  prsys(SWBREAK,"break_handler",0);
  insys(SWEOF,"end_of_file");
  insys(SWNIL,"[]");
  insys(SWCUT,"!");
  insys(SWSEMICOL,";");
  fnsys(SWFUNSEMICOLON,";",2);
  insys(SWBRACES,"{}");
  fnsys(SWFUNBRACES,"{}",1);
  insys(SWUNDER,"_");
  fnsys(SWFUNARROW,"->",2);
  fnsys(SWFUNCALL,"call",1);
}


insys(i,s)
int i;
char *s;
{
  makename(s);
  syswords[i] = findatom();
}


fnsys(i,s,a)
int a,i;
char *s;
{
  insys(i,s);
  syswords[i] = findfunct(syswords[i],a,TRUE);
}


prsys(i,s,a)
int a,i;
char *s;
{
  fnsys(i,s,a);
  syswords[i] = findproc(syswords[i],currmod,TRUE,TRUE);
}


/* construct a name from a zero-terminated string */

makename(s)
char *s;
{
  int c;
  initbuff();
  while ((c = *s++) != '\0') keepchar(c);
  makeblock();
}


/*  write out the heap in Prolog-X Heap Image Format */

dumpheap()
{
  int i,j;
  j = 0;
  for (i = 0; i < LENHASH; i++)
  {
    if (tag(hash[i]) != TERMIN) j++;
    putword(hash[i]);
  }
  printf("Hash table fill:  %d\n",j);
  for (i = 0; i < SYSWSIZE; i++) putword(syswords[i]);
  for (i = 0; i < topheap; i++) putword(heap[i]);
  printf("Heap size (words):  %d\n",topheap);
}

putword(i)
{
  putc((i >> 24) & 0377,file);
  putc((i >> 16) & 0377,file);
  putc((i >>  8) & 0377,file);
  putc(        i & 0377,file);
}
