/*
**	This file demonstrates the use of the connected client/server model.
**	In one window, start icp, consult this file and enter the command
**	    server.
**	You will get an output of the form
**	    | ?- server.
**	    Server started. You can start a client with the following command:
**		client(5569,2460554499) .
**	In one or more other windows, start icp, consult this file and enter
**	the command suggested by the server
**	An example session will be like this
**	    client(5569,2460554499) . 
**	    enter term, "stop" stops this channel>
**	Now in the server window you should see output of the form
**	    accepted From Port: 3688   Address: 2460554499 
**	    more accepts [y/n] >
**	In addition a window will be opened where data received from the
**	client will be displayed.
**	The server can continue accepting connections from other clients.
**	If this is required, enter
**	    y.
**	else, enter
**	    n.
**	to this prompt.
**	Prolog terms entered at the client prompt will be transmitted to the
**	server. To close the connection enter the atom 'stop'.
*/

/************************************************************************/
/**************************** connected server **************************/
/************************************************************************/

/************************************************************************/
/*	server								*/
/* This starts off the server, either using the icprolog port as	*/
/* defined in /etc/services, or any port selected by the system		*/
/************************************************************************/
server :-
	tcp_getport(icprolog, _, Port), !,
	server(Port).
server :- server(0).

/************************************************************************/
/*	server(Port)							*/
/* This starts off the server, using the specified port. The system	*/
/* then calls server_loop/2 which accepts connections from clients.	*/
/************************************************************************/
server(Port) :-
	tcp_server(Port, Socket),
	tcp_getsockaddr(Socket, RealPort, RealAddress),	% get our address
	writeseqnl(['Server started.',
		'You can start a client with the following command: ','\n\t',
		client(RealPort, RealAddress),'.']),
	server_loop(y, Socket).

/************************************************************************/
/*	server_loop(Flag, Socket)					*/
/* If Flag is 'y', accepts a connection on socket Socket and forks	*/
/* off the predicate process_request/1 which deals with communication	*/
/* with the client. The main server loop then prompts the user to	*/
/* determine whether to continue listenning for connections and loops.	*/
/* If the Flag is 'n', the server socket is closed and no new		*/
/* connections are accepted. Connected clients continue to be served	*/
/* by the forked processes.						*/
/************************************************************************/
server_loop(y, Socket) :- !,
	tcp_accept(Socket, NewSocket, Port, Address),
	writeseqnl(['accepted From Port:', Port, '  Address:', Address]),
	fork(process_request(NewSocket)),
	server_flag(Flag, [y,n]),	% prompt user for more accepts
	server_loop(Flag, Socket).
server_loop(_, Socket) :- !,
	writeseqnl(['closing server socket', Socket]),
	tcp_close(Socket).

/************************************************************************/
/*	process_request(Socket)						*/
/* A window is opened to display data received from the client and	*/
/* process_request/5 is called to communicate with the client.		*/ 
/************************************************************************/
process_request(Socket) :-
	tcp_getpeeraddr(Socket, Port, Address),		% get address of client
	pname(connection(Socket, Port, Address), Name),	% make title of xterm
	tty(Name, _, Stream),				% start xterm
	process_request(ok, Socket, Port, Address, Stream).

/************************************************************************/
/*	process_request(stop, Socket, Port, Address, Stream)		*/
/* This predicate receives data from the client and writes it out to	*/
/* the xterm for the connection.					*/
/* The connection is closed when the atom 'stop' is received or if an	*/
/* error occurs								*/
/************************************************************************/
process_request(stop, Socket, Port, Address, Stream) :- !,
	close(Stream),		% close xterm
	writeseqnl(['closing socket', Socket, 'Port:', Port, '  Address:', Address]),
	tcp_close(Socket).	% close connection
process_request(_, Socket, Port, Address, Stream) :-
	tcp_recv(Socket, Data),
	writeseqnl(Stream, ['got>', Data, '<', 'from Port:', Port, '  Address:', Address]),
	process_request(Data, Socket, Port, Address, Stream).
process_request(_, Socket, Port, Address, Stream) :-
	writeseqnl(['ERROR tcp_recv on  Port:', Port, '  Address:', Address]),
	process_request(stop, Socket, Port, Address, Stream).

/************************************************************************/
/*	server_flag(Flag, List)						*/
/* prompt user for whether to continue					*/
/************************************************************************/
server_flag(Flag, List) :-
	writeseq(['more accepts', List, '> ']), flush,
	read(Flag),
	on(Flag, List), !.
server_flag(Flag, List) :-
	server_flag(Flag, List).

/************************************************************************/
/************************** connected client ****************************/
/************************************************************************/
/************************************************************************/
/*	client(Port, Address)						*/
/* This starts off the client, connecting to the specified server port	*/
/* and address								*/
/************************************************************************/
client(Port, Address) :-
	tcp_client(Port, Address, Socket),
	client_loop(ok, Socket).

/************************************************************************/
/*	client_loop(Data, Socket)					*/
/* Prompts the user for data to be sent to the server and sends it. The	*/
/* connection is closed when the atom 'stop' is sent.			*/
/************************************************************************/
client_loop(stop, Socket) :- !, tcp_close(Socket).
client_loop(_, Socket) :-
	write('enter term, "stop" stops this channel> '), flush,
	read(Data),
	tcp_send(Socket, Data),
	client_loop(Data, Socket).

?- ensure_loaded(tcp).
?- writeseqnl(['use the command\n\t \033[1mserver.\033[0m\nto start the server']).
