/*****************************************************************************
LOGIC & OBJECTS ------------- IC PROLOG ][ version

written by F.G.McCabe (fgm@doc.ic.ac.uk)
modified for IC PROLOG ][ by Z.Bobolakis (zb@doc.ic.ac.uk)
last modified: 25 Jan 1993

Logic Programming Section
Dept. of Computing
Imperial College
London
******************************************************************************/

% L&O File

:- dynamic user_compile/3.
user_compile(lo, Window, Progs) :-
	loread(Window, Progs).

:- dynamic user_consult/3.
user_consult(lo, Window, Progs) :-
	loconsult_it(Window, Progs).


% loread/2 invokes the preprocessor and returns Prolog source code to the
% compiler
loread(File, Progs) :-
	loread(File, compile, Progs).

% consults an L&O source file and passes the code to the preprocessor
% the Prolog source code is treated as by the standard consult/1
loconsult_it(File, Progs) :-
	loread(File, consult, Progs).
	

% preprocessor is invoked
loread(File, Mode, All_Progs) :-
  read_all_formulae(LOFormulae, PrologForm), % read all the L&O formulae
  keysort(LOFormulae, Fs),	        % sort them
  collect_all(PrologForm, Fs, Classes, File, SuperObj), %bring class bodies&rules together
  tr_mode(Tr),                          % check if in trace mode
  tr_all_templates(Classes, Progs, Labels, Tr, SuperObj),  
  declarations(Progs, Labels, File, Commands), % produce 'housekeeping' code
  assemble_all_progs(Mode, Progs, Commands, All_Progs).



%
% sort formulae using  quicksort
keysort(F,Fs) :- qsort(F,Fs,@<).
%
%



%
% Input from file
%
read_all_formulae(F, PF) :-
  read_a_formula(F, PF, NF, NPF),
  !,
  read_all_formulae(NF, NPF).
read_all_formulae([], []).

read_a_formula(F, PF, NF, NPF) :-
  current_input(Str),
  get(_),
  cursor(Str, Ps),
  TempPos is Ps-1,
  cursor(Str, TempPos),
  read(Form),
  Form\=end_of_file,
  !,
  inspect_formula(Form, Ps, F, PF, NF, NPF, Str). 

inspect_formula((:-Comm), _, F, PF, NF, NPF, _) :-
	 Comm,!,        % there may be commands to exec
  read_a_formula(F, PF, NF, NPF).   
inspect_formula((L<=M:-B), Pos, [LbS/LbA-((L<=M:-B),Pos)|F], PF, F, PF, _) :-!,
  functor(L,LbS,LbA).
inspect_formula((L<<M:-B), Pos, [LbS/LbA-((L<<M:-B),Pos)|F], PF, F, PF, _) :-!,
  functor(L,LbS,LbA).
inspect_formula(L:{B}, Pos, [LbS/LbA-((L:{NB}),Pos)|F], PF, F, PF, Str):-!,
  functor(L,LbS,LbA),
  analyze_body_hook(B, Pos, NB, Str).
inspect_formula((L<=M), Pos, [LbS/LbA-((L<=M:-true),Pos)|F], PF, F, PF, _) :-!,
  functor(L,LbS,LbA).
inspect_formula((L<<M), Pos, [LbS/LbA-((L<<M:-true),Pos)|F], PF, F, PF, _) :-!,
  functor(L,LbS,LbA).
inspect_formula((H:-B), Pos, F, [(H:-B)-Pos|PF], F, PF, _) :- !.
inspect_formula(H, Pos, F, [(H:-true)-Pos|PF], F, PF, _).



analyze_body_hook(B, Pos, NB, Stream) :-
	defined(gnu_analyze_body_hook/4),
	!,
	gnu_analyze_body_hook(B, Pos, NB, Stream).
analyze_body_hook(B, _, NB, _) :-
	standard_analyze_body_hook(B, NB).


standard_analyze_body_hook('. '(Cl,Clauses), [Cl-_|NCls]) :-
	!,
	standard_analyze_body_hook(Clauses, NCls).
standard_analyze_body_hook((Cl..Clauses), [Cl-_|NCls]) :-
        !,
        standard_analyze_body_hook(Clauses, NCls).
standard_analyze_body_hook((Cl'. '), [Cl-_]) :- !.
standard_analyze_body_hook(Cl, [Cl-_]).




collect_all([], F, C, File, system) :-
	!,
	collect_classes(F, C, File).
collect_all(PFs, LOFs, C, File, File) :-
	C=[tpl(File/0,File,_,File:{PFs},[[true|(File<<system:-true)]])|Cls], 
	collect_classes(LOFs, Cls, File).

%
% classify code for each class
%
collect_classes([LbS-((L<=M),P)|Fs],[tpl(LbS,Id,Pb,Body,[['@w'(Id,P,1)|(L<=M:-true)]|Rules])|Progs],Id):-!,
	coll_template(Fs,RFs,LbS,Body,Pb,Rules,2,Id),
	collect_classes(RFs,Progs,Id).
collect_classes([LbS-((L<=M:-B),P)|Fs],[tpl(LbS,Id,Pb,Body,[['@w'(Id,P,1)|(L<=M:-B)]|Rules])|Progs],Id):-!,
	coll_template(Fs,RFs,LbS,Body,Pb,Rules,2,Id),
	collect_classes(RFs,Progs,Id).
collect_classes([LbS-((L<<M),P)|Fs],[tpl(LbS,Id,Pb,Body,[['@w'(Id,P,1)|(L<<M:-true)]|Rules])|Progs],Id):-!,
	coll_template(Fs,RFs,LbS,Body,Pb,Rules,2,Id),
	collect_classes(RFs,Progs,Id).
collect_classes([LbS-((L<<M:-B),P)|Fs],[tpl(LbS,Id,Pb,Body,[['@w'(Id,P,1)|(L<<M:-B)]|Rules])|Progs],Id):-!,
	coll_template(Fs,RFs,LbS,Body,Pb,Rules,2,Id),
	collect_classes(RFs,Progs,Id).
collect_classes([LbS-((L:{Body}),P)|Fs],[tpl(LbS,Id,P,(L:{Body}),Rules)|Progs],Id):-!,
	coll_template(Fs,RFs,LbS,_,_,Rules,1,Id),
	collect_classes(RFs,Progs,Id).
collect_classes([],[],_).


coll_template([LbS-((L<=M),P)|Fs],RFs, LbS, Body, Pb, [['@w'(Id,P,C)|(L<=M:-true)]|Rules],C,Id):-!,
	C1 is C+1,
 coll_template(Fs, RFs, LbS, Body, Pb, Rules,C1,Id).
coll_template([LbS-((L<<M),P)|Fs],RFs, LbS, Body, Pb, [['@w'(Id,P,C)|(L<<M:-true)]|Rules],C,Id):-!,
	C1 is C+1,
 coll_template(Fs, RFs, LbS, Body, Pb, Rules,C1,Id).
coll_template([LbS-((L<=M:-B),P)|Fs],RFs, LbS, Body, Pb, [['@w'(Id,P,C)|(L<=M:-B)]|Rules],C,Id):-!,
	C1 is C+1,
 coll_template(Fs, RFs, LbS, Body, Pb, Rules,C1,Id).
coll_template([LbS-((L<<M:-B),P)|Fs],RFs, LbS, Body, Pb, [['@w'(Id,P,C)|(L<<M:-B)]|Rules],C,Id):-!,
	C1 is C+1,
 coll_template(Fs, RFs, LbS, Body, Pb, Rules,C1,Id).
coll_template([LbS-((L:Body),P)|Fs], RFs, LbS, (L:Body),P, Rules,C,Id):-!,
	coll_template(Fs,RFs,LbS,_,_,Rules,C,Id).
coll_template(Fs, Fs, _, _, _, [], _, _).








%
% some housekeeping commands are produced here
%
declarations(Progs,Labels,File,Commands) :-
  declare_list(Progs,P), 
  Commands = ((get_prop(File,'#label_list#',OldL) ->
                   kill_lbls(OldL) ; true),
	      (get_prop(File,'#class_list#',OldP) ->
                   kill_all(OldP) ; true),
              set_prop(File,'#class_list#',P),
              set_prop(File,'#label_list#',Labels),
              MCommands),
  label_decls(Labels,MCommands).


declare_list([],[]).
declare_list([pr(Pred/_,_)|L],[Pred|L1]):-
  declare_list(L,L1).



label_decls([],true).
label_decls([LbL/LbA|L],Commands) :-
  label_symbol(LbL,LbA,Lb),
  access_rule(Lb,LbL,LbA,Rule),
  Commands = (assert(Rule),MCommands),
  label_decls(L,MCommands).





%
% produce final code: either to be passed to the compiler or interpreter
%
assemble_all_progs(consult, Progs, Commands, All_Progs) :-
  call(Commands),         % in consult mode, commands are executed
  solidify_all(Progs,All_Progs).

assemble_all_progs(compile, Progs, Commands, All_Progs) :-
  NProgs = [pr('<LOAD>'/0,[('<LOAD>' :- Commands)]) | Progs],
    % in compile mode, commands are passed to the compiler and are executed
    % when the program is loaded
  solidify_all(NProgs, All_Progs).




kill_all([]) :- !.
kill_all([P|Ps]) :- kill(P),
	kill_all(Ps).

kill_lbls([]).
kill_lbls([L/A|LbLs]):-
  functor(LbL,L,A),
  retract((?::?(LbL,_,_):-_)),!,
  kill_lbls(LbLs).
kill_lbls([_|LbLs]):-
  kill_lbls(LbLs).




% 
% solidify the code (compile mode)
%
solidify_all([],[]).
solidify_all([pr(Pred/A,Cl)|Progs],[pr(Pred/A,NCls)|NProgs]) :-
  solidify(Cl,NCls),
  solidify_all(Progs,NProgs).


solidify([],[]).
solidify([C|L],[V-Cl|Ls]):-
  toground(C, Cl, V),
  solidify(L,Ls).


  	

/*
evaluate_exp(G,E):-call(G),!.
evaluate_exp(G,'?no_value?').
*/


%
% the code listed below is used in the translation phase
%

/* occ_ident is identical list membership */
occ_ident(X,[Y|_]):- X==Y,!.
occ_ident(X,[_|List]):-
        occ_ident(X,List).


/* index_occ locate by identity, and return index */
index_occ(X, [Y|_], Ix, Ix):- X==Y,!.
index_occ(X, [_|L], Ix, Iy):-
  Ix1 is Ix+1,
  index_occ(X, L, Ix1, Iy).


/* add_occ add an element to the end of a list */
add_occ(X,Y,C,I,P,IX):-var(Y),!,Y=[['@x'(I,P,IX,C)|X]|_].
add_occ(X,[_|List],C,I,P,IX):-
 C1 is C+1,
	add_occ(X,List,C1,I,P,IX).

%
% termin is used to complete a list
%
termin([]):-!.
termin([_|L]):-termin(L).




%
% local_pred construct a local predicate from label symbol and 
% predicate symbol
%
local_pred(Prefix, Pred, Arity, Symbol):-
  conc([Prefix,Arity,Pred], Symbol).

%% local_pred_name searches the list of defined predicates for the correct name
%
local_pred_name(Defnd,Prog,Arity,Name):-
  arg(3,Defnd,Preds),
  occ(sym(Prog,Arity,Name),Preds).

%
% local_fun construct a local function from label symbol and 
% function symbol
%
local_fun(Prefix, Fun, Arity, Symbol):-
  conc([Prefix, Arity,Fun,'*'], Symbol).

%
% local_fun_name searches the list of defined functions for the correct name
%
local_fun_name(Defnd,Prog,Arity,Name):-
  arg(2,Defnd,Preds),
  occ(sym(Prog,Arity,Name),Preds).


%
% label_symbol constructs the label entry predicate symbol  
%
label_symbol(Label, Symbol):-!,
  functor(Label,Symb,Arity),
  conc([Arity,Symb,':'], Symbol).
label_symbol(Name,Arity,Symbol):-
  conc([Arity,Name,':'], Symbol).


%
% super_pred constructs a arity-less version
%
super_pred(Prefix, Symbol):-
  conc([Prefix, super], Symbol).


notnum(X):-number(X),!,fail.
notnum(_).




%
% triple construct a triple ...
%
triple(Pred,A1,A2,A3,Pred(A1,A2,A3)).
%
% Q-Prolog version:-
% triple(Pred,A1,A2,A3,Atom):-
% Atom=..[Pred,A1,A2,A3].
%

%
% unary construct a unary predicate ...
%
unary(Pred,A,Pred(A)).
%
% Q-Prolog version:-
% unary(Pred,A,Atom):-
% Atom=..[Pred,A].
%

%
% binary construct a binary predicate ...
%
binary(Pred,A,B,Pred(A,B)).
%
% Q-Prolog version:-
% binary(Pred,A,B,Atom):-
% Atom=..[Pred,A,B].
%




%
%
%
warning(X) :- nl, aux_warn(X), nl.
aux_warn([]).
aux_warn([X|R]) :- write(X), write(' '), aux_warn(R).
%
%
%
