/*****************************************************************************
LOGIC & OBJECTS ------------- IC PROLOG ][ version

written by F.G.McCabe (fgm@doc.ic.ac.uk)
modified for IC PROLOG ][ by Z.Bobolakis (zb@doc.ic.ac.uk)
last modified: 25 Jan 1993

Logic Programming Section
Dept. of Computing
Imperial College
London
******************************************************************************/

% L&O File

/*
   tr_term translates a term which may be a functional expression into
  	a regular term, together with a conjunction of goals.
*/

% already a variable?

% Is this a label argument?

tr_term(X, X, arg(Ix,LbArg,X), Defnd):-
  var(X),
  label_argument(X,Defnd,Ix,LbArg),!.

tr_term(X, X, true, _):-
  var(X),!.

tr_term(T,S,G,Defnd):-   % have to deal with the special case of a var functor
   functor(T,F,_),
   var(F),!,
	  T=..[F|A],
	  tr_list(A, B, [], G, Defnd),
   S=..[F|B].

% quoted expression - again no interpretation
tr_term(`X, Y, G, Defnd):-!,
        tr_quoted(X, Y, G, Defnd).            % have to look for unquotes though

tr_term(#F, T, G, Defnd):-
  functor(F,V,Ar),
  var(V),
  compound(F),!,
  F=..[_|A],
  tr_complex(V, Ar, A, T, G, Defnd).	
tr_term(#F, T, G, Defnd):-!,
   tr_term(F, T, G, Defnd).

%quote-hash quotes at the function level but unquotes the arguments
tr_term(`#X, Y, G, Defnd):-!,
  X=..[F|A],
  tr_list(A, AA, [], G, Defnd),
  Y=..[F|AA].

% external class template function call
tr_term(Class:Var, T, Goal, Defnd):-
  	var(Var),!,
  	tr_term(Class, Clb, SGL, Defnd),
   self_var(Defnd,Sf),
   simplify((SGL,
              Var=..[VF|VA], 
              append(VA, [T], VVA),
              '?apply?'(VF, VVA, Clb, Sf)), Goal).

tr_term(Class:Term, T, Goal, Defnd):-
  	Class\==super,!,
   Term=..[Fn|A],
   tr_fun_symbol(Fn,FS,FG),
   tr_term(Class, Lb, SGL, Defnd),
   tr_list(A,B,[T],SGC,Defnd),
   Cll=..[FS|B],
   (var(Lb)->(self_var(Defnd,Sf),
             simplify((SGL,SGC,FG,?:?(Cll,Lb,Sf)), Goal));
            label_symbol(Lb,LbSymb),
            triple(LbSymb,Cll,Lb,Lb,GL),
            simplify((SGL,SGC,FG,GL),Goal)).

tr_term(super:Term, T, Goal, Defnd):-!,
   tr_term(`#Term, Trm, SGC, Defnd),
  	Trm=..[Fn|A],
   tr_fun_symbol(Fn,FS,FG),
   append(A,[T],B),
   Cll=..[FS|B],
   label_n_self(Defnd, LbS, LbArg, SelfVar),
   super_pred(LbS, LocP),
   triple(LocP,Cll, LbArg, SelfVar,GL),
   simplify((SGC,FG,GL), Goal).


tr_term(:Term, T, G, Defnd):- !,
  	tr_term(super:Term, T, G, Defnd).

% self keyword reference
tr_term(self, SelfVar, true, Defnd):-!, self_var(Defnd,SelfVar).

% a list pair is examined internally
tr_term([H|T], [HS|HT], (GH,GT), Defnd):-!,
  tr_term(H, HS, GH, Defnd),
  tr_term(T, HT, GT, Defnd).

% a brace term
tr_term({L}, {LL}, G, Defnd):-
  tr_commas(L, LL, G, Defnd).

% number term - no interpretation
tr_term(X, X, true, _):-
	number(X),!.

% if a term is atomic - it may be a constant function
tr_term(T, S, G, Defnd):-
	atom(T),
 reducible(T,0,Defnd),!,
	form_fun_goal(T, 0, [S], G, Defnd).
tr_term(T, T, true, _):-
	atom(T),!.

% a complex term is ripped apart and rebuilt in a new guise
tr_term(T, S, (FG,G), Defnd):-
	compound(T),
 functor(T,F,Ar),
	T=..[F|A],
	tr_term(F, F0, FG, Defnd),
 tr_complex(F0, Ar, A, S, G, Defnd).	

% examine a complex term
% a reducible expression ...
tr_complex(F, _, A, S, (SG,'?apply?'(F, B, LbArg, SelfVar)), Defnd):-
  	var(F),!,
   tr_list(A, B, [S], SG, Defnd),
   label_n_self(Defnd,_,LbArg,SelfVar).
tr_complex(F, Ar, A, S, (SG,G), Defnd):-
   reducible(F,Ar,Defnd),!,
   tr_list(A, B, [S], SG, Defnd),
  	form_fun_goal(F, Ar, B, G, Defnd).
% a normal free function ...
tr_complex(F, _, A, S, GL, Defnd):-
  	tr_list(A, B, [], GL, Defnd),
   S=..[F|B].
  
% has the user has declared a symbol or variable to be the name of a function?
tr_fun_symbol(F, S, '?*?'(F,S)):-
  var(F),!.
tr_fun_symbol(F, Fn, true):-
  atom(F),!,
  conc([F,'*'],Fn).

% translate a list of terms...
tr_list([], L, L, true, _).
tr_list([T|L], [S|K], R, (G,GL), Defnd):-
  tr_term(T, S, G, Defnd),
  tr_list(L, K, R, GL, Defnd).

% translate a comma list of terms...  special case for dealing with {}'s
tr_commas(X, X, true, _):-
  var(X),!.
tr_commas((T,L), (S,K), (G,GL), Defnd):-
  notnum(L),notnum(T),!,
  tr_term(T, S, G, Defnd),
  tr_commas(L, K, GL, Defnd).
tr_commas(T, S, G, Defnd):-
  tr_term(T, S, G, Defnd).

% tr_quoted inspects a quoted term looking for the unquote symbol #
tr_quoted(X, X, true, _):-
  var(X),!.
tr_quoted(X(Y),X(Z),G,Defnd):-
  var(X),!,                % special case of variable functor
  tr_quoted(Y,Z,G,Defnd).
tr_quoted(X, X, true, _):-
  atomic(X),!.
tr_quoted(#X, Y, G, Defnd):-!,
  tr_term(X, Y, G, Defnd).  % unquote part of term
tr_quoted(X, Y, (FG,AG), Defnd):-
  X=..[F|A],
  tr_quoted(F, FY, FG, Defnd),
  tr_quoted_list(A, AY, AG, Defnd),
  Y=..[FY|AY].

tr_quoted_list([], [], true, _).
tr_quoted_list([T|L], [S|K], (G,GL), Defnd):-
  tr_quoted(T, S, G, Defnd),
  tr_quoted_list(L, K, GL, Defnd).

%
% form a sub-goal that will constrain the value of a variable
%
form_fun_goal(F, Ar, Args, Goal, defnd(_,Funs,_,_,SV,LbArg,_,_)):-
  occ(sym(F,Ar,LocP), Funs),!,   % locally defined function
  append(Args,[LbArg,SV],AArgs),
  Goal=..[LocP|AArgs].
form_fun_goal(F, 0, [Val], Vsymb(SV,Val), defnd(_,_,_,IVs,SV,_,_,_)):-
  occ(F,IVs),!,                          % instance variable?
  conc(['<',F,'>'],Vsymb).        % access the global var program
form_fun_goal(F, Ar, Args, Goal, _):-
  percentf(F,Ar,PerCent),!,                    % a percent f function?
  Goal=..[PerCent|Args].
form_fun_goal(F, Ar, Args, Goal, _):-
  primitive_fun(F,Ar),                         % a primitive function?
  Goal=..[F|Args].

%
% simplify the complex nesting of sub-goals (mostly trues of course)
%
simplify(G,S):-
	sim_left(G,L),
	sim_right(L,S).

sim_left((true,A), L):-
	sim_left(A, L).
sim_left(((A,B),C), L):-
	sim_left((A,B,C), L).
sim_left((A;B),(SA;SB)):-
	simplify(A,SA),
	simplify(B,SB).
sim_left((A->B),(SA->SB)):-
	simplify(A,SA),
	simplify(B,SB).
sim_left(\+A, \+SA):-
	simplify(A,SA).
sim_left(A,A).

sim_right((A,B), C):-
	simplify(B, SB),
	(SB=true->A=C;C= (A,SB)).
sim_right((true;A), B):-
 sim_right(A, B).
sim_right((A;true), B):-
  sim_right(A, B).
sim_right(A,A).

%
% test for reducibility
%
reducible(F,A,defnd(_,Funs,_,_,_,_,_,_)):-
  occ(sym(F,A,_), Funs),!.
reducible(F,0,defnd(_,_,_,IVs,_,_,_,_)):-
  occ(F, IVs),!.
reducible(F,A,_):-
  primitive_fun(F,A),!.
reducible(F,A,_):-
  percentf(F,A,_).

